# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
details file has the functions to operate on cluster overview and configuration page where
we can view application group, run backup, view jobs, run restore job ....

Overview:

    view_appgrp_jobs                   --      view application group jobs

    backup                             --      Initiates backup job for the selected application

    access_restore                     --      initiate restore job for the selected application
                                               group

    access_configuration               --      access configuration tab


Configuration:

    access_overview()                  --      access overview tab


Restore:

    select_volums_data                 --     open the volume and data restore page

    select_app_manifest()              --     open the manifest restore page

    select_full_application()          --     open the full application restore page

    restore_volume_data()              --     initiate volume and data resotre

    restore_application_manifest()     --     initiate application manifest restore

    restore_full_app()                 --     initiate full application restore

    select_access_node()               --     select access node on restore screen

    select_destinatino_cluster()       --     select restore destinatino cluster

    select_volume_path()               --     select destination volume path for volume data restore

    select_restore_path()              --     select restore destinatino path

    select_in_place_restore()          --     select in place restore checkbox

    select_out_of_place_restore()      --     select out of place restore checkbox

    enable_enter_namespace_toggle()    --     enable "enter namespace manually" toggle

    set_restore_namespace()            --     set restore destinatino namespace

    select_storage_class()             --     select storage class

    enable_unconditional_overwrite()   --     enable unconditional onverwrite toggle on full
                                              application restore

    confirm_overwrite()                --     confirm overwrite options

    submit_restore()                   --     submit restore job
"""

from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.browse import Browse
from Web.AdminConsole.Components.panel import Backup, ModalPanel, DropDown

from Web.Common.exceptions import CVWebAutomationException
from Web.Common.page_object import PageService, WebAction


class Overview:
    """
    Functions to operate on cluster overview page
    """

    def __init__(self, admin_console):
        self._admin_console = admin_console
        self.__table = Table(admin_console)
        self.__backup = Backup(admin_console)
        self.__panel = ModalPanel(admin_console)

    @PageService()
    def view_appgrp_jobs(self, app_grp_name):
        """
        view job history for the provided application group
        Args:
            app_grp_name                       (String)       --     application group name
        """
        self.__table.access_action_item(
            app_grp_name, self._admin_console.props['action.jobs'])

    @PageService()
    def backup(self, app_grp_name, backup_level=Backup.BackupType.FULL):
        """
        Initiate backup
        Args:
            app_grp_name                 (String)       --    Instance name
            backup_level                 (String)       --    Specify backup level from constant
                                                              present in OverView class
        """
        self.__table.access_action_item(
            app_grp_name, self._admin_console.props['label.globalActions.backup'])
        _job_id = self.__backup.submit_backup(backup_level)
        return _job_id

    @PageService()
    def access_restore(self, app_grp_name):
        """
        click application group restore action
        Args:
            app_grp_name                  (String)     --  application group name
        """
        self.__table.access_action_item(
            app_grp_name, self._admin_console.props['label.restore'])
        return Restore(self._admin_console)

    @PageService()
    def access_configuration(self):
        """Access configuration page"""
        self._admin_console.select_configuration_tab()
        return Configuration(self._admin_console)


class Configuration:
    """Operations on cluster configuration page"""

    def __init__(self, admin_console):
        self._admin_console = admin_console
        self.__table = Table(admin_console)

    @PageService()
    def access_overview(self):
        """Access overview page"""
        self._admin_console.select_overview_tab()
        return Overview(self._admin_console)


class Restore:
    """Restore operations"""

    def __init__(self, admin_console):
        self._admin_console = admin_console
        self.__panel = ModalPanel(admin_console)
        self.__browse = Browse(admin_console)
        self.__table = Table(admin_console)
        self.__dropdown = DropDown(admin_console)

    @WebAction()
    def __select_restore_type(self, res):
        """
        Selects the Restore type"""
        if self._admin_console.check_if_entity_exists(
                "xpath", f"//div[contains(text(),'{res}')]"):
            self._admin_console.driver.find_element_by_xpath(
                f"//div[contains(text(),'{res}')]").click()

    @WebAction()
    def __select_pvc(self, namespace, pvc):
        """
        Select pvc under provided namespace
        Args:
            namespace: the namespace where pvc will be selected
            pvc:  the pvc name that will be selected
        """
        self._admin_console.log.info(
            "click namespace %s on volume destination tree", namespace)
        xpath = f"//span[contains(text(),'{namespace}')]"
        if self._admin_console.check_if_entity_exists('xpath', xpath):
            self._admin_console.driver.find_element_by_xpath(xpath).click()
            self._admin_console.wait_for_completion()
        else:
            raise CVWebAutomationException(
                "There is no namespace named {0}.".format(namespace))

        self._admin_console.log.info(
            "select pvc %s on volume destination tree", pvc)
        xpath = f"//button[@class='leaf']/../span[contains(text(),'{pvc}')]"
        if self._admin_console.check_if_entity_exists('xpath', xpath):
            self._admin_console.driver.find_element_by_xpath(xpath).click()
            self._admin_console.wait_for_completion()
        else:
            raise CVWebAutomationException(
                "There is no pvc named {0}.".format(pvc))

    @WebAction()
    def _click_application(self, dbtype):
        """
        click the application on full app out of place restore options
        Args:
            dbtype: the application that need be clicked
                 (either cassandra or mongo)
        """
        xpath = "//ul[@class='vm-full-restore-list']/*/a[contains(text(),'" + \
            dbtype + "')]"
        self._admin_console.log.info(xpath)
        if self._admin_console.check_if_entity_exists('xpath', xpath):
            self._admin_console.driver.find_element_by_xpath(xpath).click()
            self._admin_console.wait_for_completion()
        else:
            raise CVWebAutomationException(
                "There is no application named {0}.".format(dbtype))

    @PageService()
    def select_volumes_data(self):
        """
        Opens the guest files restore page
        """
        #self.__select_restore_type('Volumes and data')
        self.__select_restore_type(
            self._admin_console.props['label.guestFileKUBERNETES'])
        self._admin_console.wait_for_completion()

    @PageService()
    def select_app_manifests(self):
        """
        Opens the virtual machine files restore page
        """
        #self.__select_restore_type('Application manifests')
        self.__select_restore_type(
            self._admin_console.props['label.fileRestoreKUBERNETES'])
        self._admin_console.wait_for_completion()

    @PageService()
    def select_full_application(self):
        """
        Opens the full VM restore page
        """
        #self.__select_restore_type('Full application')
        self.__select_restore_type(
            self._admin_console.props['label.fullVMKUBERNETES'])
        self._admin_console.wait_for_completion()

    @PageService()
    def restore_volume_data(self, podname):
        """
        initiate volume and data restore
        """
        self.select_volumes_data()
        self.__table.access_link_by_column('Name', podname)
        self.__browse.select_for_restore(all_files=True)
        self.__browse.submit_for_restore()

    @PageService()
    def restore_application_manifest(self, podname):
        """
        initiate application manifest restore
        """
        self.select_app_manifests()
        self.__table.access_link_by_column('Name', podname)
        self.__browse.select_for_restore(all_files=True)
        self.__browse.submit_for_restore()

    @PageService()
    def restore_full_app(self):
        """ Initiate full app restore """
        self.select_full_application()
        self.__browse.select_for_restore(all_files=True)
        self.__browse.submit_for_restore()

    @PageService()
    def select_access_node(self, dropdownid, accessnode):
        """select access node on restore screen
        Args:
            dropdownid      (String)       -- access node dropdown id
            accessnode      (String)       -- restore access node
        """
        self.__dropdown.select_drop_down_values(
            drop_down_id=dropdownid, values=[accessnode])

    @PageService()
    def single_select_accessnode(self, dropdownid, accessnode):
        """select access node on single select drop down
        Args:
            dropdownid      (String)       -- access node dropdown id
            accessnode      (String)       -- restore access node
        """
        self._admin_console.select_value_from_dropdown(dropdownid, accessnode)

    @PageService()
    def select_destination_cluster(self, dropdownid, servername):
        """
        select restore destination cluster
        Args:
            dropdownid      (String)       -- access node dropdown id
            accessnode      (String)       -- restore access node
        """
        self.__dropdown.select_drop_down_values(
            drop_down_id=dropdownid, values=[servername])

    @PageService()
    def single_select_cluster(self, dropdownid, servername):
        """
        select restore destination cluster
        Args:
            dropdownid      (String)       -- access node dropdown id
            accessnode      (String)       -- restore access node
        """
        self._admin_console.select_value_from_dropdown(dropdownid, servername)

    @PageService()
    def select_volume_path(self, namespace, pvc):
        """
        select destination volume path for volume data restore
        Args:
            namespace     (String)       -- namespace
            pvc          (String)        --  pvc name
        """
        # self._admin_console.click_button("Browse")
        self._admin_console.click_button(
            self._admin_console.props['action.browse'])
        self.__select_pvc(namespace, pvc)
        self._admin_console.click_button('OK')

    @PageService()
    def set_restore_path(self, restorepath):
        """
        setup restore destionpath
        Args:
            restorepath  (String)      -- restore destinatino path
        """
        self._admin_console.fill_form_by_id('restorePath', restorepath)

    @PageService()
    def set_path_on_restore_option(self, path):
        """
        set Path on volume data out of place restore screen
        Args:
             pth (String)  -- restore paht
        """
        self._admin_console.fill_form_by_id('path', path)

    @PageService()
    def select_in_place_restore(self):
        """Select in place restore checkbox"""
        self._admin_console.checkbox_select('inPlace')

    @PageService()
    def select_out_of_place_restore(self):
        """Select out of place restore checkbox"""
        self._admin_console.checkbox_select('outOfPlace')

    @PageService()
    def enable_enter_namespace_toggle(self):
        """Enable Enter namespace manually toggle"""
        self._admin_console.enable_toggle(
            toggle_id='vmFullRestoreKubernetes_toggle_manual_namespace')

    @PageService()
    def set_restore_namespace(self, namespace):
        """set restore destination namespoace """
        self._admin_console.fill_form_by_id('manualNamespaceInput', namespace)

    @PageService()
    def select_storage_class(self, storageclass):
        """ select storage class """
        self._admin_console.select_value_from_dropdown(
            "storageClass", storageclass)

    @PageService()
    def enable_unconditional_overwrite(self, toggle_index=0):
        """enable unconditional overwrite toggle on full application restore """
        # self._admin_console.enable_toggle(toggle_id='vmFullRestoreKubernetes_toggle_overwrite')
        self._admin_console.enable_toggle(index=toggle_index)

    @PageService()
    def confirm_overwrite(self):
        """ Confirm overwrite option """
        self._admin_console.click_button(
            self._admin_console.props['button.yes'])

    @PageService()
    def submit_restore(self):
        """submit restore job"""
        self._admin_console.click_button(
            self._admin_console.props['action.submit'])

    @PageService()
    def select_application(self, dbname):
        """click the applications on out of place restore options window"""
        self._admin_console.log.info(
            "inside restore click_app_on_restore_option method")
        self._click_application(dbname)
