# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the methods that can be done on the application groups page.

Classes:

    AppGroups --  This class contains all the methods for action in application groups page

Functions:

    select_app_group()			--	Opens the application group with the given name

    add_app_group()          	--  Creates a application group for the given cluster

    add_application_group_no_cluster_name()    --   from cluster details, add a new application
                                                    group

    add_application_group()      --   from application group tab, add new application group

    action_backup_app_groups()   --  Backs up the given application group

    action_delete_app_groups()   --  Deletes the given application groups

    action_restore_app_group()   --  Restores the given applicationgroup

    action_jobs_app_group()      --  Opens the jobs page of the application group

    has_app_group()              --  check if application group exists

    get_details_by_app_group()   -- get table content filtered by application group

"""
from Web.AdminConsole.Components.panel import Backup, DropDown
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.K8s.application_group_details import AppGroupDetails
from Web.Common.page_object import PageService
from Web.Common.exceptions import CVWebAutomationException


class AppGroups:
    """
    Class for the application groups page
    """

    def __init__(self, admin_console):
        self._admin_console = admin_console
        self.__table = Table(admin_console)
        self.__dropdown = DropDown(admin_console)
        self.__app_group_obj = AppGroupDetails(admin_console)

    @PageService()
    def select_app_group(self, app_group_name):
        """
        Opens the Application group with the given name

        Args:
            app_group_name (basestring) :  the name of the VM group to open

        """
        self.__table.access_link(app_group_name)
        # self.select_hyperlink(app_group_name)

    @PageService()
    def add_app_group(
            self,
            cluster_name,
            app_group_name,
            app_content,
            plan=None):
        """
        Adds a new Application group

        Args:
            cluster_name (basestring)     : the cluster

            app_group_name (basestring)       : the name of the application group to create


            app_content      (dict)           :   the content to be added to the application group


            plan            (basestring)    :   the plan to be associated with the subclient

        """
        self._admin_console.access_menu("Add Application group")
        self._admin_console.select_value_from_dropdown('client', cluster_name)
        self.__dropdown.select_drop_down_values(
            drop_down_id='client', values=[cluster_name])
        self._admin_console.fill_form_by_id("name", app_group_name)
        self.__dropdown.select_drop_down_values(
            drop_down_id="browseCollectionContent_isteven-multi-select_#0847",
            values=["Applications"])
        self.__app_group_obj.select_application_from_browse_tree(
            app_content)
        self._admin_console.cv_single_select("Plan", plan)
        self._admin_console.submit_form()
        self._admin_console.check_error_message()

    @PageService()
    def add_application_group_no_cluster_name(
            self, app_group_name, namespaces, plan=None):
        """
        from clusters, Adds a new Application group

        Args:
            app_group_name (basestring)       : the name of the application group to create
            namespace      (string)           :   the content to be added to the application group
            plan            (basestring)    :   the plan to be associated with the subclient

        """
        self._admin_console.fill_form_by_id("name", app_group_name)
        self.__dropdown.select_drop_down_values(
            drop_down_id="browseCollectionContent_isteven-multi-select_#0847",
            values=["Applications"])
        for namespace in namespaces:
            self.__app_group_obj.select_namespace_from_browse_tree(namespace)
        self._admin_console.cv_single_select("Plan", plan)
        self._admin_console.submit_form()
        self._admin_console.check_error_message()

    @PageService()
    def add_application_group(
            self,
            cluster_name,
            app_group_name,
            namespace,
            plan=None):
        """
        Adds a new Application group

        Args:
            cluster_name (basestring)     : the cluster

            app_group_name (basestring)   :   the name of the application group to create

            namespace      (arrays)       :   the namespaces to be added to the application
                                              group

            plan          (basestring)    :   the plan to be associated with the subclient
        """
        self._admin_console.access_menu("Add Application group")
        self.__dropdown.select_drop_down_values(
            drop_down_id='client', values=[cluster_name])
        self._admin_console.fill_form_by_id("name", app_group_name)
        self.__dropdown.select_drop_down_values(
            drop_down_id="browseCollectionContent_isteven-multi-select_#0847",
            values=["Applications"])
        self.__app_group_obj.select_namespace_from_browse_tree(namespace)
        self._admin_console.cv_single_select("Plan", plan)
        self._admin_console.submit_form()
        self._admin_console.check_error_message()

    @PageService()
    def action_backup_app_groups(self, app_group_name, backup_type):
        """
        Starts a backup of the given type for the specified collection

        Args:
           app_group_name (basestring)  : the name of the VM group to backup
           backup_type (BackupType)    : the type of backup to run, among the type in Backup.
                                        BackupType enum

        Returns:
            the backup job ID
        """
        self.__table.access_action_item(
            app_group_name, self._admin_console.props['action.backup'])
        backup = Backup(self)
        return backup.submit_backup(backup_type)

    @PageService()
    def action_delete_vm_groups(self, app_group_name):
        """
        Deletes a app group with the given name

        Args:
            app_group_name (basestring) : the application group to delete
        """

        self.__table.access_action_item(
            app_group_name, self._admin_console.props['label.delete'])
        self._admin_console.click_button(
            self._admin_console.props['label.yes'])

    @PageService()
    def action_restore_app_groups(self, app_group_name):
        """
        Opens the restore page of the vm group from the server details page

        Args:
            vm_group_name (basestring):  the VM group to restore

        """
        self.__table.access_action_item(
            app_group_name, self._admin_console.props['label.restore'])

    @PageService()
    def action_jobs_app_groups(self, app_group_name):
        """
        Lists all the jobs of the specific application group

        Args:
            app_group_name (basestring): the application group whose jobs should be opened
        """
        self.__table.access_action_item(
            app_group_name, self._admin_console.props['action.jobs'])

    @PageService()
    def has_app_group(self, app_group):
        """
        Check if app group exists
        Args:
            app_group               (str):   app group name

        Returns                    (bool): True if vm group exists or False otherwise
        """
        return self.__table.is_entity_present_in_column('Name', app_group)

    @PageService()
    def get_details_by_app_group(self, app_group):
        """
        Get table content filtered by vm group
        Args:
            app_group               (str):  app group name

        Returns:                   (Dict): table content of specified vm group

        """
        if self.has_app_group(app_group):
            return self.__table.get_table_data()
        raise CVWebAutomationException(
            "VM group [%s] not found in vm groups page" %
            app_group)
