# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides methods for all the actions that can be done of the application group
details page.


Classes:

   AppGroupDetails  --  This class contains all the methods for action in a particular application
   group details page

Functions:
   select_application_from_browse_tree()    --     select application from the browse tree

   select_namespace_from_browse_tree()      --     select namespace from browse tree

   backup()        --     run backup job

   job()          --     open the job page with all the running jobs for the application group

"""
from Web.AdminConsole.Components.panel import Backup, DropDown
from Web.AdminConsole.Components.table import Table
from Web.Common.exceptions import CVWebAutomationException
from Web.Common.page_object import PageService, WebAction

class AppGroupDetails:
    """
    This class contains all the methods for action in a particular server's page
    """

    def __init__(self, admin_console):
        """ """
        self._admin_console = admin_console
        self.__table = Table(admin_console)
        self.__dropdown = DropDown(admin_console)

    @WebAction()
    def __search_application(self, application_name):
        """
        Searches for an application in the cluster

        Args:
            application_name(string):   the name of the application to be searched

        Raises:
            Exception:
                if the given application is not found in the cluster

        """

        try:
            self._admin_console.log.info(
                "Searching for application %s",
                str(application_name))
            if self._admin_console.check_if_entity_exists(
                    'xpath', '//input[@placeholder="Search applications"]'):
                search = self._admin_console.driver.find_element_by_xpath(
                    '//input[@placeholder="Search applications"]')
            else:
                raise CVWebAutomationException(
                    "There is no option to search applications")
            search.clear()
            search.send_keys(application_name)
            search.send_keys(Keys.ENTER)
            self._admin_console.wait_for_completion()

            if self._admin_console.check_if_entity_exists(
                    "xpath", "//span[@title='{0}']".format(application_name)):
                self._admin_console.driver.find_element_by_xpath(
                    "//span[@title='{0}']".format(application_name)).click()
            else:
                raise CVWebAutomationException(
                    "There is no application named {0}.".format(application_name))
            self._admin_console.log.info(
                "Selected the application %s",
                str(application_name))
        except Exception as exp:
            raise CVWebAutomationException(
                "Exception occurred while searching and selecting VM. {0}".format(
                    str(exp)))

    @WebAction()
    def __select_namespace(self, namespace):
        """
        Select namespace

        Args:
            namespace: the namespace to be selected
        """
        try:
            self._admin_console.log.info("select namespace %s", namespace)
			xpath = "//span[@title='Namespace:" + namespace + "']"
            if self._admin_console.check_if_entity_exists('xpath', xpath):
                self._admin_console.driver.find_element_by_xpath(xpath).click()
            else:
                raise CVWebAutomationException(
                    "There is no namespace named {0}.".format(namespace))
        except Exception as exp:
            raise CVWebAutomationException(
                "Exception occurred while selecting namespace. {0}".format(
                    str(exp)))

    @PageService()
    def select_application_from_browse_tree(self, application_content):
        """
        Select content for subclient from the browse tree

        Args:
            application_content  (dict):     the content to be selected
                Sample value:   {'Applications':[app1,app2],
                                 'Labels':[label1,label2],
                                 'Volumes':[volume1,volume2]
                                }

        Raises:
            Exception:
                if the content could not be selected properly

        """
        selected = []
        all_content = []
        for key, value in application_content.items():
            all_content += value
            self.__dropdown.select_drop_down_values(1, [key])
            for _app in value:
                self.__search_application(_app)
                selected.append(_app)
            difference = list(set(all_content) - set(selected))
            if difference:
                raise CVWebAutomationException(
                    "Exception during selection of content. Some of the content "
                    "could not be selected")

    @PageService()
    def select_namespace_from_browse_tree(self, namespace):
        """
        Select content for subclient from the browse tree

        Args:
            namespace:  the namespace to be selected
        """
        self.__select_namespace(namespace)

    @PageService()
    def backup(self, bkp_type):
        """
        run backup job
        """
        self._admin_console.access_menu(
            self._admin_console.props['action.backup'])
        backup = Backup(self)
        return backup.submit_backup(bkp_type)

    @PageService()
    def jobs(self):
        """
        Opens the jobs page with all the running jobs for the server

        Raises:
            Exception:
                if the jobs page could not be opened

        """
        self._admin_console.access_menu(
            self._admin_console.props['label.nav.jobs'])
