# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""

This module provides the function or operations that can be used to run
basic operations on theme customization page.

CustomizeThemeMain : This class provides methods for theme customization related operations

 __init__()                                 --  Initialize object of TestCase class associated

 navigate_to_theme_customization_page()     --  To navigate to theme customization page of the admin console

 add_theme_customization()                  --  To set the theme customization on the commcell

 edit_theme_customization()                 --  To edit the commcell theme customization

 remove_theme_customization()               --  To remove the existing theme customization values

 validate_theme_customization()             --  To validate the theme customization settings

 validate_theme_logo()                      --  To validate commcell logo

"""
import os
from Web.AdminConsole.AdminConsolePages.theme import Theme


class CustomizeThemeMain:
    """Admin console helper for theme customization related operations"""

    def __init__(self, testcase):
        """
        Helper for schedule policy related files

        Args:
            testcase    (object)    -- object of TestCase class

        """
        self.driver = testcase.driver
        self.log = testcase.log
        self.commcell = testcase.commcell

        self.logo_file_path = None

        self.theme = None
        self.theme = Theme(self.driver)

    def navigate_to_theme_customization_page(self):
        """ To navigate to theme customization page of the admin console """
        self.theme.navigate_to_theme()
        self.log.info("successfully navigated to theme customization page")

    def add_theme_customization(self, primary_color=None, header_color=None, header_text_color=None,
                                navigation_color=None, link_color=None):
        """
        To set theme customization in the commcell

        Args:

            primary_color(str)          --  primary color of the commcell.      eg.#841a1a

            header_color(str)           --  header color of the commcell.       eg.#31b7a5

            header_text_color(str)      --  header text color of the commcell.  eg.#d6d8e9

            navigation_color(str)       --  navigation color of the commcell.   eg.#152a0f

            link_color(str)             --  logo color of the commcell.         eg.#d83ba7


        Returns:
                None
        Raises:
                Exception:
                    --- if failed to set the customization values

        """
        self.navigate_to_theme_customization_page()
        self.log.info("setting new logo")
        self.theme.set_logo(self.logo_file_path)
        self.log.info("adding new color settings")
        self.theme.set_color_settings(primary_color=primary_color,
                                      header_color=header_color,
                                      header_text_color=header_text_color,
                                      navigation_color=navigation_color,
                                      link_color=link_color)
        self.log.info("New theme settings were applied successfully")

    def edit_theme_customization(self, new_primary_color=None, new_header_color=None, new_header_text_color=None,
                                 new_navigation_color=None, new_link_color=None):
        """
        Method to edit the theme customization on the commcell

        Args:

            new_primary_color(str)          --  new primary color of the commcell.      eg.#1a8428

            new_header_color(str)           --  new header color of the commcell.       eg.#ac5ca6

            new_header_text_color(str)      --  new header text color of the commcell.  eg.#223c0b

            new_navigation_color(str)       --  new navigation color of the commcell.   eg.#cfc3d8

            new_link_color(str)             --  new logo color of the commcell.         eg.#d83ba7

        Returns:
                None
        Raises:
                Exception:
                    --- if failed to set the customization values

        """
        self.navigate_to_theme_customization_page()
        self.theme.set_color_settings(primary_color=new_primary_color,
                                      header_color=new_header_color,
                                      header_text_color=new_header_text_color,
                                      navigation_color=new_navigation_color,
                                      link_color=new_link_color)
        self.log.info("color settings were modified successfully ")

    def remove_theme_customization(self):
        """
        To reset the theme customization on the commcell

        Returns:
                None
        Raises:
                Exception:
                    --- if failed to reset the customization values

        """
        self.navigate_to_theme_customization_page()
        self.log.info("Resetting the theme customization")
        self.theme.reset_theme()
        self.log.info("Theme settings were reset successfully")

    def validate_theme_customization(self, primary_color='#2f4e66',
                                     header_color='#e4e7ea',
                                     header_text_color='#2f4e66',
                                     navigation_color='#eeeeee',
                                     link_color='#4B8DCC'):
        """
        method to validate the color settings

        Args:
            primary_color(str)          --  primary color of the commcell.      eg.#841a1a

            header_color(str)           --  header color of the commcell.       eg.#31b7a5

            header_text_color(str)      --  header text color of the commcell.  eg.#d6d8e9

            navigation_color(str)       --  navigation color of the commcell.   eg.#152a0f

            link_color(str)             --  logo color of the commcell.         eg.#d83ba7

        Returns:
                None
        Raises:
                Exception:
                    --- if failed to validate the theme customization values
        """
        self.navigate_to_theme_customization_page()
        theme_dictionary = self.theme.get_theme_values()
        if not theme_dictionary['primary_color_value'] == primary_color:
            exp = "primary color value was not set correctly"
            raise Exception(exp)

        if not theme_dictionary['header_color_value'] == header_color:
            exp = "header color value was not set correctly"
            raise Exception(exp)

        if not theme_dictionary['header_text_color_value'] == header_text_color:
            exp = "header text color value was not set correctly"
            raise Exception(exp)

        if not theme_dictionary['navigation_color_value'] == navigation_color:
            exp = "navigation color value was not set correctly"
            raise Exception(exp)

        if not theme_dictionary['link_color_value'] == link_color:
            exp = "link color value was not set correctly"
            raise Exception(exp)
        self.log.info("successfully validated the color settings")

    def validate_theme_logo(self, logo_file_path):
        """
        Method to validate the custom theme logo

        Args:
            logo_file_path(str) -- custom logo file path

        Returns:
            None
        Raise:
            Exception:
                    -- if failed to validate the custom logo

        """
        self.navigate_to_theme_customization_page()
        theme_dictionary = self.theme.get_theme_values()
        path, file_name = os.path.split(logo_file_path)
        if not theme_dictionary['logo_value'] == file_name:
            exp = "Logo was not set correctly"
            raise Exception(exp)
        self.log.info("successfully validated the commcell logo name")
