# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""This Module provides methods to run different template for Snap Operations

Class : SnapTemplate

Functions:
    cleanup()           : To cleanup entities created during execution
    create_entities()   : To create required entities like Plans, Subclient etc
    snaptemplate1()     : Template for Snap Test Cases

"""
import random
import time
import string
from AutomationUtils.machine import Machine
from AutomationUtils.database_helper import get_csdb
from cvpysdk.policies.storage_policies import StoragePolicies
from cvpysdk.policies.storage_policies import StoragePolicyCopy
from Web.AdminConsole.AdminConsolePages.Plans import Plans
from Web.AdminConsole.AdminConsolePages.PlanDetails import PlanDetails
from Web.AdminConsole.AdminConsolePages.Arrays import Arrays
from Web.AdminConsole.FileServerPages.file_servers import FileServers
from Web.AdminConsole.FileServerPages.fsagent import FsSubclient
from Web.AdminConsole.FileServerPages.fssubclientdetails import FsSubclientDetails
from Web.AdminConsole.Components.panel import Backup
from Web.AdminConsole.Components.table import Table
from Web.Common.exceptions import CVTestStepFailure
from Web.Common.page_object import TestStep


class SnapTemplate(object):
    """ SnapTemplate Class for Snap Cases"""
    test_step = TestStep()

    def __init__(self, testcase, admin_console):

        """Initializing the Test case file"""
        self.admin_console = admin_console
        self.tcinputs = testcase.tcinputs
        self.commcell = testcase.commcell
        self.testcase = testcase
        self.policies = StoragePolicies(self.commcell)
        self.plan_obj = Plans(self.admin_console)
        self.plan_details = PlanDetails(self.admin_console)
        self.fs_servers = FileServers(self.admin_console)
        self.fs_subclient = FsSubclient(self.admin_console)
        self.fs_scdetails = FsSubclientDetails(self.admin_console)
        self.arrays = Arrays(self.admin_console)
        self.table = Table(self.admin_console)
        self.log = testcase.log
        self.navigator = self.admin_console.navigator
        self._csdb = get_csdb()
        self._storagepool_name = {'pri_storage': self.tcinputs['StoragePoolName']}
        self.string = self.tcinputs['SnapEngine'].replace("/", "").replace(" ", "").replace("(", "").replace(")", "")
        self._plan_name = "CC_AutoPlan_{0}_{1}".format(self.string, self.testcase.id)
        self._subclient_name = "CC_AutoSC_{0}_{1}".format(self.string, self.testcase.id)
        self._backupset_name = 'defaultBackupSet'
        self.out_restore_path = "C:\\AutomationRestore{0}".format(self.string)
        self.restore_path = None
        self.storage_policy = None
        self.snap_primary = self._plan_name + " snap copy"
        self.first_node_copy = None
        self.second_node_copy = None
        self.type = self.tcinputs.get('ReplicationType', False)
        self.additional_storage = {
            'Add': True,
            'default_replica': True,
            'storage_name': self.first_node_copy,
            'source_copy': self.snap_primary,
            'storage_pool': self.tcinputs['StoragePoolName'],
            'copy_type': "vault",
            'snap_engine': self.tcinputs['SnapEngine'],
            'mappings': {'src_svm': self.tcinputs.get('PrimaryArray', None),
                         'dest_svm': self.tcinputs.get('SecondaryArray', None)}
        }
        self._snapshot_options = {'snap_recovery_points': '3',
                                  'sla_hours': '2',
                                  'sla_minutes': '30'}
        self.machine = Machine(self.tcinputs['ClientName'], self.commcell)
        self.testdata_dir = []
        self.get_sp = """SELECT DISTINCT AG.name FROM archGroup AG JOIN archGroupCopy AGC ON AG.id = AGC.archGroupId
                                                    WHERE AG.name LIKE '{a}'"""
        self.get_array_name = """SELECT SCH.SMArrayId FROM SMControlHost SCH
                                    WITH (NOLOCK) JOIN SMSnap SS ON SS.ControlHostId  = SCH.ControlHostId 
                                    JOIN SMVolSnapMap SVM ON SVM.SMSnapId  = SS.SMSnapId JOIN SMVolume SV
                                    ON SV.SMVolumeId = SVM.SMVolumeId WHERE SV.JobId = {a} AND SV.CopyId = {b}"""

    @property
    def storagepool_name(self):
        """Return Storage Pool Name"""
        return self._storagepool_name

    @storagepool_name.setter
    def storagepool_name(self, value):
        """Set Storage Pool name"""
        self._storagepool_name = value

    @property
    def plan_name(self):
        """Return Plan Name"""
        return self._plan_name

    @plan_name.setter
    def plan_name(self, value):
        """Set Plan name"""
        self._plan_name = value

    @property
    def subclient_name(self):
        """Return Subclient Name"""
        return self._subclient_name

    @subclient_name.setter
    def subclient_name(self, value):
        """Set Subclient Name"""
        self._subclient_name = value

    @property
    def backupset_name(self):
        """Return BackupSet Name"""
        return self._backupset_name

    @backupset_name.setter
    def backupset_name(self, value):
        """Set BackupSet Name"""
        self._backupset_name = value

    @property
    def snapshot_options(self):
        """Return Snap options"""
        return self._snapshot_options

    @snapshot_options.setter
    def snapshot_options(self, value):
        """Set Snap options"""
        self._snapshot_options = value

    def wait_for_job_completion(self, jobid):
        """Waits for Backup or Restore Job to complete"""
        job_obj = self.commcell.job_controller.get(jobid)
        return job_obj.wait_for_completion()

    @test_step
    def spcopy_obj(self, copy_name):
        """ Create storage Policy Copy object
        Arg:
            copy_name        (str)         -- Copy name
        """
        spcopy = StoragePolicyCopy(self.commcell, self.storage_policy, copy_name)
        return spcopy

    @test_step
    def cleanup(self):
        """To perform cleanup operation"""
        try:
            self.navigator.navigate_to_file_servers()
            self.fs_servers.access_server(self.tcinputs['ClientName'])
            self.fs_subclient.access_subclient_tab()
            if self.table.is_entity_present_in_column('Name', self.subclient_name):
                self.fs_subclient.delete_subclient(backupset_name=self.backupset_name,
                                                   subclient_name=self.subclient_name)
                self.admin_console.wait_for_completion()

            if self.type:
                self.log.info("Deleting: %s Configuration for secondary snap copies", self.type)
                self.navigator.navigate_to_plan()
                if self.table.is_entity_present_in_column('Plan name', self.plan_name):
                    self.plan_obj.select_plan(self.plan_name)
                    self.snap_copy_deletion(self.type)
                    self.log.info("Successfully Deleted: %s Configuration for secondary snap copies", self.type)
                    self.admin_console.wait_for_completion()
                else:
                    self.log.info("No such plan configuration exists")

            self.navigator.navigate_to_plan()
            if self.table.is_entity_present_in_column('Plan name', self.plan_name):
                self.plan_obj.delete_plan(self.plan_name)
                self.admin_console.wait_for_completion()
                self.log.info(f"Plan: {self.plan_name} deleted successfully.")

            self.policies.refresh()
            sp = self.execute_query(self.get_sp, {'a': self.plan_name})
            if not (sp in [[[]], [['']], ['']]):
                self.log.info(f"Deleting storage policy: {sp}")
                self.policies.delete(self.plan_name)

            if self.machine.check_directory_exists(self.out_restore_path):
                self.machine.remove_directory(self.out_restore_path)
                self.log.info(f"Cleaned up Outplace Restore Directory: {self.out_restore_path}")

            self.log.info("Cleanup has completed successfully.")

        except Exception as exp:
            raise CVTestStepFailure(f'Cleanup entities failed with error : {exp}')

    @test_step
    def snap_copy_creation(self, replica_type):
        """To create copies for the given replication types"""

        try:
            if replica_type == "pv_replica":
                self.first_node_copy = "Vault/Replica"
                self.additional_storage['storage_name'] = self.first_node_copy
                self.log.info("Creating Vault Copy: %s", self.first_node_copy)
                self.plan_details.edit_server_plan_storage_pool(self.additional_storage, edit_storage=False, snap=True)
                self.log.info("Successfully created Vault Copy: %s", self.first_node_copy)

            elif replica_type == "pm_replica":
                self.first_node_copy = "PM"
                self.additional_storage['storage_name'] = self.first_node_copy
                self.additional_storage['default_replica'] = False
                self.additional_storage['copy_type'] = "mirror"
                self.log.info("Creating Mirror Copy: %s", self.first_node_copy)
                self.plan_details.edit_server_plan_storage_pool(self.additional_storage, edit_storage=False, snap=True)
                self.log.info("Successfully created Mirror Copy: %s", self.first_node_copy)

            elif replica_type == "pvm_replica":
                self.first_node_copy = "PV"
                self.additional_storage['storage_name'] = self.first_node_copy
                self.log.info("Creating First Node Vault Copy: %s", self.first_node_copy)
                self.plan_details.edit_server_plan_storage_pool(self.additional_storage, edit_storage=False, snap=True)
                self.log.info("Successfully created First node Vault Copy: %s", self.first_node_copy)

                self.second_node_copy = "PVM"
                self.additional_storage['storage_name'] = self.second_node_copy
                self.additional_storage['source_copy'] = self.first_node_copy
                self.additional_storage['default_replica'] = False
                self.additional_storage['copy_type'] = "mirror"
                self.additional_storage['mappings']['src_svm'] = self.tcinputs.get('SecondaryArray', None)
                self.additional_storage['mappings']['dest_svm'] = self.tcinputs.get('TertiaryArray', None)
                self.log.info("Creating Second Node Mirror Copy: %s", self.second_node_copy)
                self.plan_details.edit_server_plan_storage_pool(self.additional_storage, edit_storage=False, snap=True)
                self.log.info("Successfully created Second Node Mirror Copy: %s", self.second_node_copy)

            elif replica_type == "pmv_replica":
                self.first_node_copy = "PM"
                self.additional_storage['storage_name'] = self.first_node_copy
                self.additional_storage['default_replica'] = False
                self.additional_storage['copy_type'] = "mirror"
                self.log.info("Creating First Node Mirror Copy: %s", self.first_node_copy)
                self.plan_details.edit_server_plan_storage_pool(self.additional_storage, edit_storage=False, snap=True)
                self.log.info("Successfully created First Node Mirror Copy: %s", self.first_node_copy)

                self.second_node_copy = "PMV"
                self.additional_storage['storage_name'] = self.second_node_copy
                self.additional_storage['source_copy'] = self.first_node_copy
                self.additional_storage['default_replica'] = True
                self.additional_storage['copy_type'] = "vault"
                self.additional_storage['mappings']['src_svm'] = self.tcinputs.get('SecondaryArray', None)
                self.additional_storage['mappings']['dest_svm'] = self.tcinputs.get('TertiaryArray', None)
                self.log.info("Creating Second Node Vault Copy: %s", self.second_node_copy)
                self.plan_details.edit_server_plan_storage_pool(self.additional_storage, edit_storage=False, snap=True)
                self.log.info("Successfully created second node Vault Copy: %s", self.second_node_copy)

            elif replica_type == "pmm_replica":
                self.first_node_copy = "PM"
                self.additional_storage['storage_name'] = self.first_node_copy
                self.additional_storage['default_replica'] = False
                self.additional_storage['copy_type'] = "mirror"
                self.log.info("Creating First Node Mirror Copy: %s", self.first_node_copy)
                self.plan_details.edit_server_plan_storage_pool(self.additional_storage, edit_storage=False, snap=True)
                self.log.info("Successfully created First Node Mirror Copy: %s", self.first_node_copy)

                self.second_node_copy = "PMM"
                self.additional_storage['storage_name'] = self.second_node_copy
                self.additional_storage['source_copy'] = self.first_node_copy
                self.additional_storage['default_replica'] = False
                self.additional_storage['copy_type'] = "mirror"
                self.additional_storage['mappings']['src_svm'] = self.tcinputs.get('SecondaryArray', None)
                self.additional_storage['mappings']['dest_svm'] = self.tcinputs.get('TertiaryArray', None)
                self.log.info("Creating Second Node Mirror Copy: %s", self.second_node_copy)
                self.plan_details.edit_server_plan_storage_pool(self.additional_storage, edit_storage=False, snap=True)
                self.log.info("Successfully created Second Node Mirror Copy: %s", self.second_node_copy)

            elif replica_type == "pvv_replica":
                self.first_node_copy = "PV"
                self.additional_storage['storage_name'] = self.first_node_copy
                self.log.info("Creating First Node Vault Copy: %s", self.first_node_copy)
                self.plan_details.edit_server_plan_storage_pool(self.additional_storage, edit_storage=False, snap=True)
                self.log.info("Successfully created First node Vault Copy: %s", self.first_node_copy)

                self.second_node_copy = "PVV"
                self.additional_storage['storage_name'] = self.second_node_copy
                self.additional_storage['source_copy'] = self.first_node_copy
                self.additional_storage['default_replica'] = False
                self.additional_storage['mappings']['src_svm'] = self.tcinputs.get('SecondaryArray', None)
                self.additional_storage['mappings']['dest_svm'] = self.tcinputs.get('TertiaryArray', None)
                self.log.info("Creating Second Node Vault Copy: %s", self.second_node_copy)
                self.plan_details.edit_server_plan_storage_pool(self.additional_storage, edit_storage=False, snap=True)
                self.log.info("Successfully created Second node Vault Copy: %s", self.second_node_copy)

        except Exception as exp:
            raise CVTestStepFailure(f'Create Snap Copies failed with error : {exp}')

    @test_step
    def snap_copy_deletion(self, replica_type):
        """To Delete copies for the given replication types"""

        try:
            if replica_type == "pv_replica":
                if self.plan_details.is_copy_present(self.first_node_copy):
                    self.log.info("Deleting First Node Vault Copy: %s", self.first_node_copy)
                    self.plan_details.delete_server_plan_storage_copy(self.first_node_copy)
                    self.log.info("Successfully deleted First node Vault Copy: %s", self.first_node_copy)
                    time.sleep(300)

            elif replica_type == "pm_replica":
                if self.plan_details.is_copy_present(self.first_node_copy):
                    self.log.info("Deleting First Node Mirror Copy: %s", self.first_node_copy)
                    self.plan_details.delete_server_plan_storage_copy(self.first_node_copy)
                    self.log.info("Successfully deleted First node Mirror Copy: %s", self.first_node_copy)
                    time.sleep(300)

            elif replica_type == "pvm_replica":
                if self.plan_details.is_copy_present(self.second_node_copy):
                    self.log.info("Deleting Second Node Mirror Copy: %s", self.second_node_copy)
                    self.plan_details.delete_server_plan_storage_copy(self.second_node_copy)
                    self.log.info("Successfully deleted Second node Mirror Copy: %s", self.second_node_copy)
                    time.sleep(300)
                if self.plan_details.is_copy_present(self.first_node_copy):
                    self.log.info("Deleting First Node Vault Copy: %s", self.first_node_copy)
                    self.plan_details.delete_server_plan_storage_copy(self.first_node_copy)
                    self.log.info("Successfully deleted First node Vault Copy: %s", self.first_node_copy)
                    time.sleep(300)

            elif replica_type == "pmv_replica":
                if self.plan_details.is_copy_present(self.second_node_copy):
                    self.log.info("Deleting Second Node Vault Copy: %s", self.second_node_copy)
                    self.plan_details.delete_server_plan_storage_copy(self.second_node_copy)
                    self.log.info("Successfully deleted Second node Vault Copy: %s", self.second_node_copy)
                    time.sleep(300)
                if self.plan_details.is_copy_present(self.first_node_copy):
                    self.log.info("Deleting First Node Mirror Copy: %s", self.first_node_copy)
                    self.plan_details.delete_server_plan_storage_copy(self.first_node_copy)
                    self.log.info("Successfully deleted First node Mirror Copy: %s", self.first_node_copy)
                    time.sleep(300)

            elif replica_type == "pmm_replica":
                if self.plan_details.is_copy_present(self.second_node_copy):
                    self.log.info("Deleting Second Node Mirror Copy: %s", self.second_node_copy)
                    self.plan_details.delete_server_plan_storage_copy(self.second_node_copy)
                    self.log.info("Successfully deleted Second node Mirror Copy: %s", self.second_node_copy)
                    time.sleep(300)
                if self.plan_details.is_copy_present(self.first_node_copy):
                    self.log.info("Deleting First Node Mirror Copy: %s", self.first_node_copy)
                    self.plan_details.delete_server_plan_storage_copy(self.first_node_copy)
                    self.log.info("Successfully deleted First node Mirror Copy: %s", self.first_node_copy)
                    time.sleep(300)

            elif replica_type == "pvv_replica":
                if self.plan_details.is_copy_present(self.second_node_copy):
                    self.log.info("Deleting Second Node Vault Copy: %s", self.second_node_copy)
                    self.plan_details.delete_server_plan_storage_copy(self.second_node_copy)
                    self.log.info("Successfully deleted Second node Vault Copy: %s", self.second_node_copy)
                    time.sleep(300)
                if self.plan_details.is_copy_present(self.first_node_copy):
                    self.log.info("Deleting First Node Vault Copy: %s", self.first_node_copy)
                    self.plan_details.delete_server_plan_storage_copy(self.first_node_copy)
                    self.log.info("Successfully deleted First node Vault Copy: %s", self.first_node_copy)
                    time.sleep(300)

        except Exception as exp:
            raise CVTestStepFailure(f'Delete Snap Copies failed with error : {exp}')

    @test_step
    def create_entities(self):
        """To create required entities for test case"""
        try:
            # To create a new plan
            self.log.info("Adding a new plan: %s", self.plan_name)
            self.navigator.navigate_to_plan()
            self.plan_obj.create_server_plan(plan_name=self.plan_name,
                                             storage=self.storagepool_name,
                                             snapshot_options=self.snapshot_options)
            self.log.info("successfully created plan: %s", self.plan_name)
            time.sleep(20)
            self.policies.refresh()
            self.storage_policy = self.policies.get(self.plan_name)
            # To add secondary snap copy configuration
            if self.type:
                self.log.info("Creating: %s Configuration for secondary snap copies", self.type)
                self.snap_copy_creation(self.type)
                self.log.info("Successfully created: %s Configuration for secondary snap copies", self.type)

            # To add a new Subclient
            self.log.info("Adding a new subclient %s", self.subclient_name)
            self.navigator.navigate_to_file_servers()
            self.fs_servers.access_server(self.tcinputs['ClientName'])
            self.fs_subclient.access_subclient_tab()
            self.fs_subclient.add_fs_subclient(backup_set=self.backupset_name,
                                               subclient_name=self.subclient_name,
                                               plan=self.plan_name,
                                               define_own_content=True,
                                               backup_data=self.tcinputs['SubclientContent'].split(','),
                                               remove_plan_content=True)
            self.log.info("Created a new subclient %s", self.subclient_name)
            self.fs_subclient.access_subclient(backupset_name=self.backupset_name,
                                               subclient_name=self.subclient_name)
            self.fs_scdetails.enable_snapshot_engine(enable_snapshot=True,
                                                     engine_name=self.tcinputs['SnapEngine'])

        except Exception as exp:
            raise CVTestStepFailure(f'Create entities failed with error : {exp}')

    @test_step
    def verify_backup(self, backup_type):
        """Verify Snapbackup"""
        try:
            self.navigator.navigate_to_file_servers()
            self.fs_servers.access_server(self.tcinputs['ClientName'])
            self.fs_subclient.access_subclient_tab()
            jobid = self.fs_subclient.backup_subclient(backupset_name=self.backupset_name,
                                                       subclient_name=self.subclient_name,
                                                       backup_type=backup_type)
            job_status = self.wait_for_job_completion(jobid)
            if not job_status:
                exp = "{0} Snap Job ID {1} didn't succeed".format(backup_type, jobid)
                raise Exception(exp)
            return jobid
        except Exception as exp:
            raise CVTestStepFailure(f'Snapbackup operation failed : {exp}')

    @test_step
    def verify_auxcopy(self):
        """Verify Aux Copy"""
        try:
            job = self.storage_policy.run_aux_copy()
            self.log.info("Successfully started aux copy job : {0}".format(job.job_id))
            job_status = self.wait_for_job_completion(job.job_id)
            if not job_status:
                exp = " Aux Copy Job ID {0} didn't succeed".format(job.job_id)
                raise Exception(exp)
            if job.status != 'Completed':
                raise Exception(
                    "job: {0} for snap operation is completed with errors, Reason: {1}".format(
                        job.job_id, job.delay_reason))
        except Exception as exp:
            raise CVTestStepFailure(f'Aux Copy operation failed : {exp}')

    @test_step
    def verify_restore(self, storage_copy_name=None, restore_aux_copy=False, inplace=False):
        """Verify Restore
        """
        try:
            if inplace:
                self.restore_path = None
            else:
                dir_name = "Res_" + ''.join([random.choice(string.ascii_letters) for _ in range(3)])
                self.restore_path = str(self.out_restore_path + '\\' + dir_name)
                self.machine.create_directory(self.restore_path)
                self.log.info("Successfully Created directory: {0}".format(self.restore_path))
            self.navigator.navigate_to_file_servers()
            self.fs_servers.access_server(self.tcinputs['ClientName'])
            self.fs_subclient.access_subclient_tab()
            rjobid = self.fs_subclient.restore_subclient(backupset_name=self.backupset_name,
                                                         subclient_name=self.subclient_name,
                                                         restore_aux_copy=restore_aux_copy,
                                                         storage_copy_name=storage_copy_name,
                                                         restore_path=self.restore_path)
            time.sleep(20)
            rjob_status = self.wait_for_job_completion(rjobid)
            if not rjob_status:
                exp = "Restore Job ID {0} didn't succeed".format(rjobid)
                raise Exception(exp)
        except Exception as exp:
            raise CVTestStepFailure(f'Restore operation failed : {exp}')

    @test_step
    def add_test_data(self):
        """add test data in subclient for verification"""
        try:
            hash = {}
            self.testdata_dir = []
            for drive in self.tcinputs['SubclientContent'].split(','):

                if self.machine.is_directory(drive) == 'True':
                    separator = "\\"
                    if self.machine.os_info == 'UNIX':
                        separator = "/"
                    if len(drive) < 4:
                        path = str(drive + "TestData")
                    else:
                        path = str(drive + separator + "TestData")

                    self.log.info("test data folder  is  {0}".format(path))
                    if self.machine.check_directory_exists(path):
                        self.log.info(
                            "TestData Folder already exists under {0}, deleting it and creating"
                            " new one!!".format(drive)
                        )
                        self.machine.remove_directory(path)
                    else:
                        self.log.info(
                            "TestData Folder does not exists under {0}, creating one!!".format(drive)
                        )
                    self.machine.create_directory(path)
                    self.machine.generate_test_data(path)
                    hash[str(drive)] = self.machine.get_folder_hash(path)
                    self.testdata_dir.append(drive.split(separator)[-1])
                    self.log.info("Created TestData Folder under {0}".format(drive))
            return hash
        except Exception as exp:
            raise CVTestStepFailure(f'Test Data generation failed : {exp}')

    @test_step
    def update_test_data(self, mode):
        """
        Update method to edit or add more test data
        Args:
            mode      --  mode values can be 'edit' or 'delete'
        """
        hash = {}
        if mode == 'edit':
            for drive in self.tcinputs['SubclientContent'].split(','):

                if self.machine.is_directory(drive) == 'True':
                    separator = "\\"
                    if self.machine.os_info == 'UNIX':
                        separator = "/"
                    if len(drive) < 4:
                        test_path = str(drive + "TestData")
                    else:
                        test_path = str(drive + separator + "TestData")
                    self.machine.modify_test_data(test_path, rename=True, modify=True)
                    hash[str(drive)] = self.machine.get_folder_hash(test_path)
                    self.log.info("Successfully Modified data at: {0}".format(test_path))
                else:
                    self.log.info("No data modified")
            return hash
        if mode == 'add':
            for drive in self.tcinputs['SubclientContent'].split(','):
                if self.machine.is_directory(drive) == 'True':
                    separator = "\\"
                    if self.machine.os_info == 'UNIX':
                        separator = "/"
                    if len(drive) < 4:
                        test_path = str(drive + "TestData")
                    else:
                        test_path = str(drive + separator + "TestData")
                    incr_path = test_path + separator + "TestDataIncr"

                    self.machine.create_directory(incr_path)
                    self.machine.generate_test_data(incr_path)
                    hash[str(drive)] = self.machine.get_folder_hash(test_path)
                    self.log.info(f"Successfully added more testdata at:{test_path}")

            return hash

    @test_step
    def inplace_validate(self, source_hash):
        """Compare two directories
            Args:
                source_hash: hash value of source data,
        """
        try:
            restore_location = self.tcinputs['SubclientContent'].split(',')
            for drive in restore_location:
                if self.machine.is_directory(drive) == 'True':
                    if len(drive) < 4:
                        path = str(drive + "TestData")
                    else:
                        path = str(drive + "\\TestData")
                        if self.machine.os_info == 'UNIX':
                            path = str(drive + "/TestData")

                    dest_hash = self.machine.get_folder_hash(path)
                    if dest_hash == source_hash[drive]:
                        self.log.info("Verified for %s", drive)
                        continue
                    raise CVTestStepFailure("Restore may have failed/skipped some files")
            self.log.info("Restore validation success")
        except Exception as exp:
            raise CVTestStepFailure(f'Validating Test Data failed : {exp}')

    @test_step
    def outplace_validate(self, source_hash, out_restore_location):
        """Compare two directories
            Args:
                source_hash: hash value returned from add_test_data(),
                out_restore_location: str: Data restore path"""
        try:
            source_files = self.tcinputs['SubclientContent'].split(',')
            separator = '\\'
            if self.machine.os_info == 'UNIX':
                separator = '/'
            if not out_restore_location:
                self.log.info("Please provide outplace restore path. \n Doing Inplace Validation"
                              " as no outplace path was provided.")
                self.inplace_validate(source_hash)
            else:
                out_restore_location = out_restore_location.replace('/', separator)
                for drive in source_files:
                    if self.machine.is_directory(drive) == 'True':
                        folder = drive.split(separator)[-1]
                        dest_drive = str(out_restore_location + separator + folder)
                        if len(out_restore_location) == 3:
                            dest_drive = str(out_restore_location + folder)
                        path = str(dest_drive + separator + "TestData")
                        if folder == "":
                            path = str(dest_drive + "TestData")
                        dest_hash = self.machine.get_folder_hash(path)
                        self.log.info(f"Source: {drive + separator + 'TestData'}, destination: {path}")
                        if dest_hash == source_hash[drive]:
                            self.log.info("Verified for %s", drive)
                            continue
                        raise CVTestStepFailure("Restore may have failed/skipped some Files")
                self.log.info("Restore validation success")
        except Exception as exp:
            raise CVTestStepFailure(f'Validating Test Data failed : {exp}')

    @test_step
    def delete_snap(self, job_id, copy_name):
        """ Deletes the snap, after restore is complete
            Args:
                job_id: string: job id
                copy_name: string: name of the copy"""
        try:
            spcopy = self.spcopy_obj(copy_name)
            self.navigator.navigate_to_arrays()
            array_name_list = self.execute_query(self.get_array_name, {'a': job_id, 'b': spcopy.copy_id})
            self.arrays.action_list_snaps(array_name_list[0][0])
            jobid = self.arrays.delete_snaps(job_id)
            job_status = self.wait_for_job_completion(jobid)
            if not job_status:
                exp = "Snap Job ID {0} didn't succeed".format(jobid)
                raise Exception(exp)
            self.log.info(f"Snap delete success for snap jobid {job_id}")
        except Exception as exp:
            raise CVTestStepFailure(f'deleting snap failed for jobid {job_id}: {exp}')

    def execute_query(self, query, my_options=None, fetch_rows='all'):
        """ Executes SQL Queries
            Args:
                query           (str)   -- sql query to execute

                my_options      (dict)  -- options in the query
                default: None

                fetch_rows      (str)   -- By default return all rows, if not return one row
            Return:
                    str : first column of the sql output

        """
        if my_options is None:
            self._csdb.execute(query)
        elif isinstance(my_options, dict):
            self._csdb.execute(query.format(**my_options))

        if fetch_rows != 'all':
            return self._csdb.fetch_one_row()[0]
        return self._csdb.fetch_all_rows()

    def snaptemplate1(self):
        """Main function for test case execution"""
        self.cleanup()
        self.create_entities()
        source_test_data = self.add_test_data()
        # Run Full Snapbackup and Restore
        full_job_id = self.verify_backup(Backup.BackupType.FULL)
        self.verify_restore(
            storage_copy_name=self.snap_primary, restore_aux_copy=False, inplace=True)
        self.inplace_validate(source_test_data)
        # Run INCR Snapbackup and Restore
        source_test_data2 = self.update_test_data(mode='add')
        inc_job_id = self.verify_backup(Backup.BackupType.INCR)
        self.verify_restore(storage_copy_name=self.snap_primary, restore_aux_copy=False)
        self.outplace_validate(source_test_data2, self.restore_path)
        self.delete_snap(inc_job_id, self.snap_primary)
        self.delete_snap(full_job_id, self.snap_primary)

    def snaptemplate2(self):
        """Main function for Replication test case execution"""
        self.cleanup()
        self.create_entities()
        source_test_data = self.add_test_data()
        # Run Full Snapbackup and Restore
        full_job_id = self.verify_backup(Backup.BackupType.FULL)
        # Run aux copy
        self.verify_auxcopy()
        # Inplace Restore from first node copy
        self.verify_restore(
            storage_copy_name=self.first_node_copy, restore_aux_copy=True, inplace=True)
        self.inplace_validate(source_test_data)
        # Inplace Restore from Second node copy
        if self.type not in ["pv_replica", "pm_replica"]:
            self.verify_restore(
                storage_copy_name=self.second_node_copy, restore_aux_copy=True, inplace=True)
            self.inplace_validate(source_test_data)
        # Run INCR Snapbackup and Restore
        source_test_data2 = self.update_test_data(mode='add')
        inc_job_id = self.verify_backup(Backup.BackupType.INCR)
        # Run aux copy
        self.verify_auxcopy()
        # Outplace Restore from first node copy
        self.verify_restore(
            storage_copy_name=self.first_node_copy, restore_aux_copy=True)
        self.outplace_validate(source_test_data2, self.restore_path)
        # Outplace Restore from Second node copy
        if self.type not in ["pv_replica", "pm_replica"]:
            self.verify_restore(
                storage_copy_name=self.second_node_copy, restore_aux_copy=True)
            self.outplace_validate(source_test_data2, self.restore_path)
        # deleting snaps from second node copy
        if self.type in ["pvv_replica", "pmv_replica"]:
            self.delete_snap(inc_job_id, self.second_node_copy)
            self.delete_snap(full_job_id, self.second_node_copy)
        # deleting snaps from first node copy
        if self.type in ["pv_replica", "pvm_replica", "pvv_replica"]:
            self.delete_snap(inc_job_id, self.first_node_copy)
            self.delete_snap(full_job_id, self.first_node_copy)
        # deleting snaps from primary
        self.delete_snap(inc_job_id, self.snap_primary)
        self.delete_snap(full_job_id, self.snap_primary)
