# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be used to run
basic operations on Service commcell page.

Class:

    ServiceCommcellMain()

Functions:

    __init__()                      --      function to initialize the class object

    create_service_commcell()       --      function to register a new service commcell

    validate_service_commcell()     --      function to validate if the service commcell is created successfully

    delete_service_commcell()       --      function to unregister a service commcell

"""

from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.AdminConsolePages.service_commcell import ServiceCommcell


class ServiceCommcellMain(object):
    """
        Helper for service commcell page
    """

    def __init__(self, admin_console, csdb=None, commcell=None):
        """
            Initializes the company helper module

            Args:
                admin_console   (object)        --  AdminConsole class object

                csdb            (Object)        --  CommServe database object

                commcell        (Object)        --  CommServe object
        """

        self.__csdb = csdb
        self.__admin_console = admin_console
        self.__commcell = commcell
        self.__table = Table(self.__admin_console)
        self.__log = admin_console.log
        self.__service_commcell = ServiceCommcell(self.__admin_console)

        self.__host_name = None
        self.__user_name = None
        self.__password = None
        self.__register_as_IdP = False

    @property
    def host_name(self):
        """ Get Host name"""
        return self.__host_name

    @host_name.setter
    def host_name(self, value):
        """ Set Host name"""
        self.__host_name = value

    @property
    def user_name(self):
        """ Get User name"""
        return self.__user_name

    @user_name.setter
    def user_name(self, value):
        """ Set User name"""
        self.__user_name = value

    @property
    def password(self):
        """ Get Password"""
        return self.__password

    @password.setter
    def password(self, value):
        """ Set Password"""
        self.__password = value

    @property
    def configure_as_IdP(self):
        """ Get flag for configure as IdP"""
        return self.__register_as_IdP

    @configure_as_IdP.setter
    def configure_as_IdP(self, value):
        """ Set flag for configure as IdP"""
        self.__register_as_IdP = value

    def create_service_commcell(self):
        """Method to register a new service"""
        self.__admin_console.navigator.navigate_to_service_commcell()
        self.__service_commcell.register_commcell(self.host_name, self.user_name, self.password, self.configure_as_IdP)

    def validate_service_commcell(self):
        """Method to validate the service commcell"""
        self.__admin_console.navigator.navigate_to_service_commcell()
        self.__table.apply_filter_over_column('Service CommCell host name', self.host_name)
        commcell_name = self.__table.get_column_data('CommCell name')[0]
        commcell_host_name = self.__table.get_column_data('Service CommCell host name')[0]
        if commcell_host_name != self.host_name:
            exp = 'Service commcell hostname({}) does not match with the provided hostname({})'.format(
                commcell_host_name, self.host_name)
            raise Exception(exp)
        _query1 = "select id from app_client where net_hostname='{}'".format(commcell_host_name)
        _query2 = "select id from app_commcell where aliasName='{}'".format(commcell_name)
        self.__csdb.execute(_query1)
        commcell_clientId = self.__csdb.fetch_one_row()[0]
        self.__csdb.execute(_query2)
        commcellId = self.__csdb.fetch_one_row()[0]
        xml_execute_command = """
                                <App_CheckPodCellConnectivityReq commClientID="{}" commcellID="{}">

                                </App_CheckPodCellConnectivityReq>

                                """.format(commcell_clientId, commcellId)
        response = self.__commcell.qoperation_execute(xml_execute_command)
        if bool(response) and response['isAccessible'] == 1:
            self.__log.info('The service commcell is online')
        else:
            exp = 'Error checking service commcell status or the service commcell is offline'
            raise Exception(exp)

    def delete_service_commcell(self):
        """Method to delete the service commcell"""
        self.__admin_console.navigator.navigate_to_service_commcell()
        self.__service_commcell.delete_registered_commcell(self.host_name)
