"""
This module provides the function or operations that can be used to run
basic operations on security page.

To initialize the class variables, pass the instance object to the appropriate
definition of AdminConsoleInfo

Call the required definition using the instance object, with no arguments to
be passed, only the flags needs to be passed into the method call while calling
the member function.

Class:

   SecurityMain() -> Roles() -> object()

        __init__()

Functions:

add_security_role()                         -- Adds a security role
edit_security_role()                        -- Edits a security role
delete_security_role()                      -- Delete a security role
verify_permissions()                        -- Verifies permissions are selected/deselected
verify_enable_role_and_visible_to_all()     -- Verifies state of enable role
                                               and visible to all checkboxes
get_available_permissions()                 -- Gets all available permissions
_generate_sample_list()                     -- Gets a random selection of elements from
                                               the given list
_generate_random_categories()               -- Generates a random list of categories
generate_random_permissions()               -- Generates a random list of permissions

"""
from AutomationUtils import logger
from Web.AdminConsole.AdminConsolePages.Roles import Roles
from random import choice, sample

from Web.AdminConsole.Components.table import Table


class RolesMain:
    """
        Helper for securities page
    """

    def __init__(self, adminpage=None, commcell=None, csdb=None):
        """
            Initializes the security helper module
            :param
                driver -- the web driver object
                commcell -- comcell object
                csdb -- csdb object
        """
        self.driver = adminpage.driver
        self.csdb = csdb
        self.commcell = commcell
        self.log = logger.get_log()
        if not self.driver:
            raise Exception('Driver is not provided')
        self._role_name = None
        self._new_role_name = None
        self._list_of_permissions = None
        self._list_of_permissions_to_edit = None
        self.roles_obj = None

        self._enable_role = True
        self._visible_to_all = False
        self._all_permissions = None
        self.__table = Table(adminpage)

    @property
    def role_name(self):
        """ Gets the role name"""
        return self._role_name

    @role_name.setter
    def role_name(self, value):
        """ Sets the role_name"""
        self._role_name = value

    @property
    def new_role_name(self):
        """ Gets the role name"""
        return self._new_role_name

    @new_role_name.setter
    def new_role_name(self, value):
        """ Sets the role_name"""
        self._new_role_name = value

    @property
    def enable_role(self):
        """ Gets the list of permissions used while editing the role"""
        return self._enable_role

    @enable_role.setter
    def enable_role(self, value):
        """ Sets the list of permissions used while editing the role"""
        self._enable_role = value

    @property
    def visible_to_all(self):
        """ Gets the list of permissions used while editing the role"""
        return self._visible_to_all

    @visible_to_all.setter
    def visible_to_all(self, value):
        """ Sets the list of permissions used while editing the role"""
        self._visible_to_all = value

    def add_security_roles(self):
        """ Adds the security roles """
        self.roles_obj = Roles(self.driver)
        self.roles_obj.navigate_to_roles()
        self.roles_obj.wait_for_completion()

        self.get_available_permissions()
        self._generate_random_categories()
        self._list_of_permissions = self.generate_random_permissions()

        if self.roles_obj.check_if_entity_exists('id', 'searchInput'):
            self.__table.search_for(self.role_name)
        if self.roles_obj.check_if_entity_exists('link', self.role_name):
            self.log.info("Role already exists")
            return
        self.roles_obj.add_role(
            self.role_name,
            self._list_of_permissions,
            self.enable_role,
            self.visible_to_all
        )

        # Verify that permissions are associated with the role properly
        self.__table.access_link(self.role_name)
        self.roles_obj.checkbox_select("showSelected")
        self.verify_permissions(self._list_of_permissions, status="added")
        self.verify_enable_role_and_visible_to_all()
        self.roles_obj.cancel_form()

    def edit_security_role(self):
        """ Edits the security role """

        edit_add_roles = dict()
        edit_add_roles["Add"] = self.generate_random_permissions()

        edit_remove_roles = dict()
        edit_remove_roles["Remove"] = self.generate_random_permissions(self._list_of_permissions)

        self.__table.search_for(self.role_name)
        if self.roles_obj.check_if_entity_exists("link", self.role_name):
            self.roles_obj.select_hyperlink(self.role_name)

            self.roles_obj.edit_role(self.new_role_name, edit_add_roles)
            self.__table.access_link(self.new_role_name)
            self.verify_permissions(edit_add_roles['Add'], status="added")
            self.role_name = self.new_role_name

        self.verify_enable_role_and_visible_to_all()
        self.roles_obj.cancel_form()

        if self.roles_obj.check_if_entity_exists("link", self.role_name):
            self.roles_obj.select_hyperlink(self.role_name)
            self.roles_obj.edit_role(self.new_role_name, edit_remove_roles)
            self.__table.access_link(self.new_role_name)
            self.verify_permissions(edit_remove_roles['Remove'], status="removed")

        self.roles_obj.cancel_form()

        self.roles_obj.wait_for_completion()
        self.roles_obj.check_error_message()

    def delete_security_role(self):
        """ Deletes the security role """

        self.roles_obj.action_delete_role(self.new_role_name)

        self.__table.search_for(self.new_role_name)
        if self.roles_obj.check_if_entity_exists("link", self.new_role_name):
            raise Exception(f"Role {self.new_role_name} was not deleted")
        else:
            self.log.info(f"Role {self.new_role_name} is successfully deleted")

    def verify_permissions(self, permissions, status="added"):
        """
        Method to validate permissions added/removed to the role.

        permissions (list): List of permissions to verify
        status       (str): Pass as "added" or "removed" to verify
                            if permissions are added or removed
        """

        self.roles_obj.checkbox_select("showSelected")

        for permission in permissions:

            if isinstance(permission, str):

                if 'Operations on Storage Policy' in permission:
                    permission = "Operations on Storage Policy \\  Copy"

                if status.lower() == "added":
                    self.log.info(f"Checking if permission {permission} is {status}")
                    if not self.roles_obj.check_if_entity_exists("xpath",
                                                                 f"//div[@title='{permission}']"):
                        raise Exception(f"The chosen permission {permission} was not associated with the role."
                                        " Please check the logs.")

                elif status.lower() == "removed":
                    self.log.info(f"Checking if permission {permission} is {status}")
                    if self.roles_obj.check_if_entity_exists("xpath",
                                                             f"//div[@title='{permission}']"):
                        raise Exception(f"The permission {permission} to be removed is still associated "
                                        "with the role. Please check the logs.")

            elif isinstance(permission, dict):

                for key, value in permission.items():

                    self.verify_permissions(value, status)

    def verify_enable_role_and_visible_to_all(self):
        """
        Method to validate state of "enable role" and "visible to all"
            checkboxes against setter values
        """
        enable_role_status = self.driver.find_element_by_xpath(
            "//input[@id='rolesEnabled']").is_selected()
        visible_to_all_status = self.driver.find_element_by_xpath(
            "//input[@id='visibleToAll']").is_selected()

        if not(self.enable_role == enable_role_status):
            raise Exception("Enable role does not match input.")

        if not (self.visible_to_all == visible_to_all_status):
                raise Exception("Visible to all does not match input.")

    def get_available_permissions(self):
        """ Get all categories and permissions listed on the add role pane """
        self.roles_obj = Roles(self.driver)
        self.roles_obj.select_hyperlink("Add role")
        self.roles_obj.wait_for_completion()
        self._all_permissions = self.roles_obj.scrape_available_permissions()
        self.roles_obj.cancel_form()

    def _generate_sample_list(self, source_list):
        """Generates a sample with random elements from the given list

        Args:
            source_list (list): List of any items from which you need a random sample

        Returns:
            sample_list(list) : List of random items
        """
        list_length = len(source_list)
        no_of_items = choice(range(0, list_length))
        sample_list = sample(source_list, no_of_items)
        return sample_list

    def _generate_random_categories(self):
        """Generates a list of random categories to be selected"""
        all_categories = []
        for item in self._all_permissions:
            for key, value in item.items():
                all_categories.append(key)

        self._all_categories = self._generate_sample_list(all_categories)

    def generate_random_permissions(self, all_permissions=None):
        """Generates a list of random permissions to be selected

        Args:
            all_permissions (list): List of permissions from which random permissions are
                                    to be selected
                             E.g.[
                                    "Access Policies",
                                    {"Developer Tools":["Workflow",
                                                        {"Datasource":["Add Datasource",
                                                                        "Edit Datasource"]
                                                        }]
                                    },
                                    {"Monitoring Policy":["Edit Monitoring Policy",
                                                          "Search Share"]
                                    }
                                ]
        Returns:
                random_list (list) : List of random permissions
        """
        random_list = []
        leaf_permissions = []
        if not all_permissions:
            all_permissions = self._all_permissions

        for permission in all_permissions:
            if isinstance(permission, dict):
                perm_dict = dict()
                for key, value in permission.items():
                    if key in self._all_categories:
                        random_list.append(key)
                    else:
                        sub_list = self.generate_random_permissions(value)
                        if sub_list:
                            perm_dict[key] = sub_list
                            random_list.append(perm_dict)

            elif isinstance(permission, str):
                leaf_permissions.append(permission)

        if leaf_permissions:
            random_list = self._generate_sample_list(leaf_permissions)

        return random_list