# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""

Helper file for kubernetes cluster related operations on command center.

Classes defined in this file:

    K8sHelper:         class for Kubenertes cluster command center related operations.

        __init__()                 --  Constructor for creating a k8s cluster on command center

        create_k8s_cluster()       --  create kubernetes clusters from command center

        add_application_group()    --  add application group

        delete_k8s_cluster()       --  delete kubernetes cluster from command center

        populate_test_data()       --  populatest test data

        populate_cassandra_mongo_db()      --  populate test data for cassandra and mongo pod

        verify_backup()            --  initiate backup job and verify job completed

        verify_volumedata_restore_job()                --  drop table

        verify_manifest_restore_job()      --  initiate manifest restore and verify job complete

        verify_restored_manifest()         --  verify restored manifest files

        verify_fullapp_restore_job()       --  initiate full app restore and verify job complete

        verify_restore_data()              --  verify data are restored correctly

        delete_and_recreate_namespace()    --  delete and recreate namespace

        download_src_pod_data()            --  download source pod data to master node

        download_rst_pod_data()            --  download restore pod data to master node

        drop_db_data()                     --  cleanup cassandra db data

        verify_db_restore_data()           --  verify cassandra/mongo db data are restored correctly

        cleanup_testdata()                 --  clean up test data

        cleanup_db_testdata()              --  clean up cassandra/mongo test data


"""

import os
import time
from VirtualServer.VSAUtils import VirtualServerUtils
from VirtualServer.VSAUtils.KubernetesHelper import KubernetesHelper
from Web.Common.page_object import TestStep
from Web.AdminConsole.Components.panel import Backup
from Web.AdminConsole.K8sPages.clusters import K8SClusters, K8sServer
from Web.AdminConsole.K8sPages.cluster_details import Overview
from Web.AdminConsole.K8sPages.application_groups import AppGroups
from Web.Common.exceptions import CVTestStepFailure
from AutomationUtils import machine


class K8sHelper:
    """
    Helper class for Kubenertes cluster related operations
    """
    test_step = TestStep()

    def __init__(self, admin_console, testcase, db=False):
        """Constructor for creating the cassandra object"""
        self._admin_console = admin_console
        self.__clusters = K8SClusters(admin_console)
        self.__k8s_server = K8sServer(self._admin_console)
        self.__app_group = AppGroups(self._admin_console)
        self.__overview = Overview(self._admin_console)
        self.__commcell = testcase.commcell
        self.__k8s = None
        self.server_name = testcase.k8s_server_name
        self.app_grp_name = testcase.k8s_application_group_name
        self.namespace = testcase.namespace
        self.restore_namespace = testcase.restore_namespace
        self.api_server_endpoint = testcase.tcinputs.get("api_server_endpoint")
        self.authentication = testcase.tcinputs.get("authentication")
        self.serviceaccount = testcase.tcinputs.get("secretName")
        self.servicetoken = testcase.tcinputs.get("secretKey")
        self.accessnode = testcase.tcinputs.get("access_node")
        self.plan = testcase.tcinputs.get("plan")
        self.storageclass = testcase.tcinputs.get("StorageType")
        self.testcaseid = testcase.id
        self.utils_path = VirtualServerUtils.UTILS_PATH
        self.dockerregkeyserver = testcase.tcinputs.get("regkeyserver")
        self.dockerusername = testcase.tcinputs.get("dockerusername")
        self.dockerpassword = testcase.tcinputs.get("dockerpassword")
        self.dockeremail = testcase.tcinputs.get("dockeremail")
        if db:
            self.cassandra_namespace = testcase.tcinputs.get(
                "cassandra_namespace")
            self.cassandra_restore_namespace = testcase.tcinputs.get(
                "cassandra_restore_namespace")
            self.cassandra_keyspace = testcase.tcinputs.get(
                "cassandra_keyspace")
            self.cassandra_tablename = testcase.tcinputs.get(
                "cassandra_tablename")
            self.mongo_namespace = testcase.tcinputs.get("mongo_namespace")
            self.mongo_restore_namespace = testcase.tcinputs.get(
                "mongo_restore_namespace")
            self.cassandra_pod_create = "cassandrapoddeployment.yml"
            self.mongo_pod_create = "mongopoddeployment.yml"
            self.cassandra_pod_deployment_path = os.path.join(
                self.utils_path, self.cassandra_pod_create)
            self.mongo_pod_deployment_path = os.path.join(
                self.utils_path, self.mongo_pod_create)
        else:
            self.pod_create = "poddeployment.yml"
            self.pod_deployment_path = os.path.join(
                self.utils_path, self.pod_create)
        self.kubehelper = KubernetesHelper(testcase)
        self.testcase = testcase
        self.src_pod_path = None
        self.rst_pod_path = None
        self.master_machine = machine.Machine(
            testcase.tcinputs.get('MasterNode'),
            username=testcase.tcinputs.get('Username'),
            password=testcase.tcinputs.get('Password'))

    @test_step
    def create_k8s_cluster(self):
        """create kubernetes cluster """
        self._admin_console.navigator.navigate_to_k8s_clusters()
        self._admin_console.refresh_page()
        if self.__clusters.is_cluster_exists(self.server_name):
            self.delete_k8s_cluster()
        _k8s_server = self.__clusters.add_k8s_cluster()
        _k8s_server.set_hostname(self.api_server_endpoint)
        _k8s_server.set_servername(self.server_name)
        _k8s_server.select_authentication(self.authentication)
        _k8s_server.set_service_account(self.serviceaccount)
        _k8s_server.set_service_token(self.servicetoken)
        _k8s_server.select_access_nodes(self.accessnode)
        _k8s_server.save()
        time.sleep(60)
        if db:
            self.__app_group.add_application_group_no_cluster_name(
                self.app_grp_name, [self.cassandra_namespace, self.mongo_namespace], self.plan)
        else:
            self.__app_group.add_application_group_no_cluster_name(
                self.app_grp_name, [self.namespace], self.plan)
        self._admin_console.access_tab('Clusters')
        self._admin_console.refresh_page()

        if not self.__clusters.is_cluster_exists(self.server_name):
            raise CVTestStepFailure(
                "[%s] kubernetes cluster is not getting created " %
                self.server_name)
        self._admin_console.log.info(
            "Successfully Created [%s] kubernetes cluster",
            self.server_name)

    @test_step
    def add_application_group(self):
        """add application group """
        self._admin_console.navigator.navigate_to_k8s_appgroup()
        self.__app_group.add_application_group(
            self.server_name,
            self.app_grp_name,
            self.namespace,
            plan=self.plan)

        self._admin_console.navigator.navigate_to_k8s_clusters()
        self._admin_console.refresh_page()
        if not self.__clusters.is_cluster_exists(self.server_name):
            raise CVTestStepFailure(
                "[%s] kubernetes cluster is not getting created " %
                self.server_name)
        self._admin_console.log.info(
            "Successfully Created [%s] kubernetes cluster",
            self.server_name)

    @test_step
    def delete_k8s_cluster(self):
        """Delete kubernetes cluster and verify cluster is deleted"""
        self._admin_console.navigator.navigate_to_k8s_clusters()
        self.__clusters.delete_cluster_name(self.server_name)
        self._admin_console.refresh_page()
        if self.__clusters.is_cluster_exists(self.server_name):
            raise CVTestStepFailure(
                "[%s] kubernetes cluster is not getting deleted" %
                self.server_name)
        self._admin_console.log.info(
            "Deleted [%s] cluster from command center successfully",
            self.server_name)
        self.__commcell.clients.refresh()
        try:
            self.__commcell.clients.delete(self.server_name)
        except Exception as exp:
            self._admin_console.log.info(
                "cluster {0} is successfully deleted from commcell console".format(exp))

    @test_step
    def cleanup_testdata(self):
        """clean up test bed """
        self.kubehelper.create_name_space(delete=True)
        if self.master_machine.check_directory_exists(self.src_pod_path):
            self._admin_console.log.info(
                "Directory Exists {0} cleaning it ".format(str(self.src_pod_path)))
            self.master_machine.remove_directory(self.src_pod_path)
        if self.master_machine.check_directory_exists(self.rst_pod_path):
            self._admin_console.log.info(
                "Directory Exists {0} cleaning it ".format(str(self.rst_pod_path)))
            self.master_machine.remove_directory(self.rst_pod_path)

    @test_step
    def populate_test_data(self, jobtype='FULL'):
        """populate test data """

        path = "/tmp/automation_{0}".format(self.testcaseid)
        if jobtype == 'FULL':
            self.kubehelper.populate_tc_inputs(self.testcase)
            self.kubehelper.create_name_space()
            self.kubehelper.create_pvc_yaml(self.pod_deployment_path, self.storageclass)
            self.kubehelper.populate_data(self.pod_deployment_path)
            self.kubehelper.create_pod_yaml(self.pod_deployment_path)
            self.kubehelper.populate_data(self.pod_deployment_path)

        pods = self.kubehelper.get_pods(self.namespace)
        test_pod = pods[0]
        time.sleep(60)
        self.kubehelper.upload_pod_data(
            test_pod, path, self.testcaseid, job_type=jobtype)

        list_before_backup = self.kubehelper.get_pods_services(self.namespace)
        self._admin_console.log.info(
            "Listing pods services and replication controllers before backup "
            "in namespace %s", str(list_before_backup))
        if len(list_before_backup) == 0:
            Exception("Failed to get namespace entities or failed to create")

    @test_step
    def verify_backup(self, backup_level=Backup.BackupType.FULL):
        """Initiate the backup and verify backup job is completed"""
        # Initiate backup job
        self._admin_console.log.info("run [%s] backup job", backup_level)
        self._admin_console.navigator.navigate_to_k8s_clusters()
        self.__clusters.open_cluster(self.server_name)
        self._admin_console.refresh_page()
        time.sleep(60)
        _job_id = self.__overview.backup(self.app_grp_name, backup_level)

        # Wait till backup job to complete
        job = self.__commcell.job_controller.get(_job_id)

        self._admin_console.log.info("Wait for [%s] job to complete", str(job))
        if job.wait_for_completion(300):  # wait for max 5 minutes
            self._admin_console.log.info(
                "Backup job completed with job id:[%s]", _job_id)
        else:
            err_str = "[%s] job id for backup job failed" % _job_id
            raise CVTestStepFailure(err_str)

    @test_step
    def verify_volumedata_restore_job(self, inplace=True):
        """Initiate restore and verify restore job is completed"""
        # Initiate restore job
        self._admin_console.navigator.navigate_to_k8s_clusters()
        self.__clusters.open_cluster(self.server_name)
        self._admin_console.refresh_page()
        _restore = self.__overview.access_restore(self.app_grp_name)
        test_pod = (self.kubehelper.get_pods(self.namespace))[0]
        _restore.restore_volume_data(test_pod)
        if inplace:
            self.restore_namespace = self.namespace
            _restore.select_in_place_restore()
        else:
            _restore.select_out_of_place_restore()
            _restore.select_destination_cluster(
                'vmDiskRestore_isteven-multi-select_#5406', self.server_name)
            _restore.select_access_node(
                'vmDiskRestore_isteven-multi-select_#4918',
                self.accessnode)
            testpvc = (self.kubehelper.get_pvcs(self.restore_namespace))[0]
            _restore.select_volume_path(self.restore_namespace, testpvc)
            #path = "/tmp/automation_{0}".format(self.testcaseid)
            # _restore.set_path_on_restore_option(path)
        _restore.enable_unconditional_overwrite()
        _restore.submit_restore()

        # Wait for restore job to complete
        _job_id = self._admin_console.get_jobid_from_popup()
        job = self.__commcell.job_controller.get(_job_id)
        self._admin_console.log.info(
            "Wait for [%s] restore job to complete", str(job))
        if job.wait_for_completion(300):  # wait for max 5 minutes
            self._admin_console.log.info(
                "Restore job completed with job id:[%s]", _job_id)
        else:
            err_str = "[%s] job id for restore job failed" % _job_id
            raise CVTestStepFailure(err_str)
        self._admin_console.log.info("Restore completed successfully")

     @test_step
    def verify_manifest_restore_job(self):
        """Initiate manifest restore and verify restore job is completed"""
        self._create_rst_path()
        self._admin_console.navigator.navigate_to_k8s_clusters()
        self.__clusters.open_cluster(self.server_name)
        self._admin_console.refresh_page()
        time.sleep(60)
        _restore = self.__overview.access_restore(self.app_grp_name)
        test_pod_name = (self.kubehelper.get_pods(self.namespace))[0]
        test_pod = (test_pod_name.split("-"))[0]
        _restore.restore_application_manifest(test_pod)
        _restore.select_access_node(
            'vmFileRestore_isteven-multi-select_#2697',
            self.accessnode)
        _restore.set_restore_path(self.rst_pod_path)
        _restore.submit_restore()
        # Wait for restore job to complete
        _job_id = self._admin_console.get_jobid_from_popup()
        job = self.__commcell.job_controller.get(_job_id)
        self._admin_console.log.info(
            "Wait for [%s] restore job to complete", str(job))
        if job.wait_for_completion(300):  # wait for max 5 minutes
            self._admin_console.log.info(
                "Restore job completed with job id:[%s]", _job_id)
        else:
            err_str = "[%s] job id for restore job failed" % _job_id
            raise CVTestStepFailure(err_str)
        self._admin_console.log.info("Restore completed successfully")

    @test_step
    def verify_restored_manifest(self):
        """ Validate restore manifest files """
        self.kubehelper.validate_restored_manifest(self.rst_pod_path)

    @test_step
    def verify_fullapp_restore_job(self, inplace=False, app=False, dbs=None):
       """
        Function to initiate full app restore and verify restore job is completed
        Args:
            inplace    (str):    true means in place restore, false means
                                 out of place restore

            app    (str):     true if it's cassandra and mongo db restore

            dbs    (list):    list of applications (cassandra, mongo)
        """
        # Initiate restore job
        self._admin_console.navigator.navigate_to_k8s_clusters()
        self.__clusters.open_cluster(self.server_name)
        self._admin_console.refresh_page()
        time.sleep(60)
        _restore = self.__overview.access_restore(self.app_grp_name)
        _restore.restore_full_app()
        if inplace:
            _restore.select_in_place_restore()
            _restore.enable_unconditional_overwrite(toggle_index=0)
        else:
            _restore.select_out_of_place_restore()
            _restore.single_select_cluster('serverId', self.server_name)
            _restore.single_select_accessnode(
                'destinationProxy', self.accessnode)
            if app:
                for db in dbs:
                    self._admin_console.log.info(db)
                    _restore.select_application(db)
                    _restore.enable_enter_namespace_toggle()
                    if db == 'cassandra':
                        _restore.set_restore_namespace(
                            self.cassandra_restore_namespace)
                        _restore.select_storage_class(self.storageclass)
                    else:
                        _restore.set_restore_namespace(
                            self.mongo_restore_namespace)
                        _restore.select_storage_class(self.storageclass)
            else:
                _restore.enable_enter_namespace_toggle()
                _restore.set_restore_namespace(self.restore_namespace)
                _restore.select_storage_class(self.storageclass)
            _restore.enable_unconditional_overwrite(toggle_index=1)
        _restore.confirm_overwrite()
        _restore.submit_restore()


        # Wait for restore job to complete
        _job_id = self._admin_console.get_jobid_from_popup()
        time.sleep(30)
        job = self.__commcell.job_controller.get(_job_id)
        self._admin_console.log.info(
            "Wait for [%s] restore job to complete", str(job))
        if job.wait_for_completion(300):  # wait for max 5 minutes
            self._admin_console.log.info(
                "Restore job completed with job id:[%s]", _job_id)
        else:
            err_str = "[%s] job id for restore job failed" % _job_id
            raise CVTestStepFailure(err_str)
        self._admin_console.log.info("Restore completed successfully")

    @test_step
    def delete_and_recreate_namespace(self, namespace):
        """cleanup source data before in place restore"""
        self.kubehelper.delete_and_recreate_namespace(namespace)

    def _create_rst_path(self):
        """create restore destination path """
        self.rst_pod_path = "/tmp/automation_restore_{0}_{1}".format(
            self.testcaseid, self.namespace)

        if self.kubehelper.master_machine.check_directory_exists(
                self.rst_pod_path):
            self._admin_console.log.info(
                "Directory Exists %s cleaning it ", str(
                    self.rst_pod_path))
            self.kubehelper.master_machine.remove_directory(self.rst_pod_path)
        output = self.kubehelper.master_machine.execute_command(
            "mkdir -p {0}".format(str(self.rst_pod_path)))
        if output.exit_code == 0:
            self._admin_console.log.info(
                "Directory created %s", str(
                    self.rst_pod_path))
        else:
            self._admin_console.log.info(
                "create directory %s failed to create restore destination path")
            raise Exception('Create restore directory failed')

    @test_step
    def download_src_pod_data(self):
        """download source pod data to master node"""
        test_pod = (self.kubehelper.get_pods(self.namespace))[0]
        self.src_pod_path = self.kubehelper.download_pod_data(
            test_pod, self.namespace, self.testcaseid)

    @test_step
    def download_rst_pod_data(self):
        """download restore pod data path to master node"""
        test_pod = (self.kubehelper.get_pods(self.namespace))[0]
        list_pods_services_rst = self.kubehelper.get_pods_services(
            self.restore_namespace)
        self._admin_console.log.info(
            "list of pods post restore {0} on namespace {1} " .format(
                list_pods_services_rst, self.restore_namespace))
        self.rst_pod_path = self.kubehelper.download_pod_data(
            test_pod, self.restore_namespace, self.testcaseid)

    @test_step
    def verify_restore_data(self):
        """verified the data are restored correctly"""

        result, diff_output = self.master_machine.compare_meta_data(
            self.src_pod_path,
            self.rst_pod_path
        )
        if result:
            self._admin_console.log.info("Meta data comparison successful")
        else:
            self._admin_console.log.error("Meta data comparison failed")
            self._admin_console.log.info("Diff output: \n%s", diff_output)
            raise Exception("Meta data comparison failed")

        result, diff_output = self.master_machine.compare_checksum(
            self.src_pod_path, self.rst_pod_path
        )
        if result:
            self._admin_console.log.info("Checksum comparison successful")
        else:
            self._admin_console.log.error("Checksum comparison failed")
            self._admin_console.log.info("Diff output: \n%s", diff_output)
            raise Exception("Checksum comparison failed")

    @test_step
    def cleanup_db_testdata(self):
        """clean up test bed """
        self.namespace = self.cassandra_namespace
        self.restore_namespace = self.cassandra_restore_namespace
        self.kubehelper.create_name_space(delete=True)
        self.namespace = self.mongo_namespace
        self.restore_namespace = self.mongo_restore_namespace
        self.kubehelper.create_name_space(delete=True)

    @test_step
    def populate_cassandra_mongo_db(self, jobtype='FULL'):
        """populate cassandra and mongo db test data """

        self._admin_console.log.info("create cassandra db pod")
        if jobtype == 'FULL':
            self.kubehelper.populate_db_tc_inputs(self.testcase)
            self.kubehelper.generate_imagepull_secret(
                self.dockerregkeyserver,
                self.dockerusername,
                self.dockerpassword,
                self.dockeremail)
            self.kubehelper.namespace = self.cassandra_namespace
            self.kubehelper.restore_namespace = self.cassandra_restore_namespace
            self.kubehelper.create_name_space()
            self.kubehelper.create_db_pvc_yaml(
                self.cassandra_pod_deployment_path, self.storageclass)
            self.kubehelper.populate_data(self.cassandra_pod_deployment_path)
            self.kubehelper.create_cassandra_statefulset(
                self.cassandra_pod_deployment_path)
            self.kubehelper.populate_data(self.cassandra_pod_deployment_path)
            self.kubehelper.create_cassandra_svc_yaml(
                self.cassandra_pod_deployment_path)
            self.kubehelper.populate_data(self.cassandra_pod_deployment_path)

            self._admin_console.log.info("create mongo db pod")
            self.kubehelper.namespace = self.mongo_namespace
            self.kubehelper.restore_namespace = self.mongo_restore_namespace
            self.kubehelper.create_name_space()
            self.kubehelper.create_db_pvc_yaml(
                self.mongo_pod_deployment_path, self.storageclass, type="Mongo")
            self.kubehelper.populate_data(self.mongo_pod_deployment_path)
            self.kubehelper.create_mongo_statefulset(
                self.mongo_pod_deployment_path)
            self.kubehelper.populate_data(self.mongo_pod_deployment_path)
            self.kubehelper.create_mongo_svc_yaml(
                self.mongo_pod_deployment_path)
            self.kubehelper.populate_data(self.mongo_pod_deployment_path)

        self._admin_console.log.info("populate cassandra mongo db data")
        self.kubehelper.populate_cassandra_mongo_db(self.__commcell, jobtype)

    @test_step
    def drop_db_data(self, cassandra_namespace):
        """
        Function to clean up the casandra db data before in place restore
        Args:
            cassandra_namesapce    (str):     name of cassandra namespace
            mongo_namespace    (str):     name of mongo namespace

        """
        self.kubehelper.drop_cassandra_data(cassandra_namespace)

    @test_step
    def verify_db_restore_data(self, cassandra_namespace, mongo_namespace):
        """
        Function to verify cassanrda and mongo db data are restored correctly
        Args:
            cassandra_namespace   (string): the name of cassandra namespace
            mongo_namespace    (string): the name of mongo namespace
        """
        self.kubehelper.validate_db_restore_data(
            self.__commcell, cassandra_namespace, mongo_namespace)

