# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be used to run
basic operations on identity servers page.

Class:

    IdentityServersMain:

        __init__()                  --Initializes the Identity servers helper module

IdentityServersMain():

    check_if_saml_app_exists()	    --searches for the given app with search box

    create_saml_app()				--Helper method for creating SAML app in admin console

    open_saml_app()				    --Searches for a given SAML app and opens it

    edit_saml_rule_or_mappings()	--Edit the redirect rule or mappings in SAML app

    modify_saml_general_settings()--Edit the redirect rule or mappings in SAML app

    delete_app()					--Deletes a given SAML app

    validate_saml_app()			    --Checks for the displayed values with the given values

    download_spmetadata()			--Downloads metadata of app

    edit_saml_idp()				    --Edits a given SAML app

    edit_trust_party_adfs()		--Create or delete relying trust party created at the AD machine

    initiate_saml_login()				    --Initiates SAML login

    initiate_saml_logout()				--Performs a SAML logout

    add_associations_on_saml_app()  -- Adds associations for a user on a SAML app

"""
import base64

from AutomationUtils import logger
from AutomationUtils.machine import Machine
from Web.AdminConsole.Components.table import Table
from Web.IdentityProvider.identity_provider_support import IDPSupport
from Web.AdminConsole.AdminConsolePages.identity_servers import IdentityServers
from Web.AdminConsole.AdminConsolePages.identity_server_details import IdentityServerDetails
from Web.Common.exceptions import CVWebAutomationException


class IdentityServersMain:
    """
        Helper for IdentityServers page
    """

    def __init__(self, admin_console, commcell=None, csdb=None):
        """
            Initializes the Identity servers helper module

             Args:

                admin_console   (object) -- AdminPage class object

                commcell -- comcell object

                csdb -- csdb object

            Returns : None
        """
        self.__admin_console = admin_console
        self.csdb = csdb
        self.commcell = commcell
        self.__navigator = admin_console.navigator
        self.saml_details = IdentityServerDetails(self.__admin_console)
        self.saml = IdentityServers(self.__admin_console)
        self.identity_provider_obj = IDPSupport(admin_console=self.__admin_console, options_type='ADFS')
        self.okta_obj = IDPSupport(self.__admin_console, 'OKTA')
        self.log = logger.get_log()
        self._app_name = None
        self._enable_company = False
        self._redirect_rule = None
        self._associations = None
        self._attribute_mapping = None
        self.__table = Table(self.__admin_console)

    @property
    def app_name(self):
        """ Get SAML app name"""
        return self._app_name

    @app_name.setter
    def app_name(self, value):
        """
        Get SAML app name

        Args:
            value   (str)   -- name of the app
        """
        self._app_name = value

    @property
    def redirect_rule(self):
        """ Get redirect rule"""
        return self._redirect_rule

    @redirect_rule.setter
    def redirect_rule(self, value):
        """ Set redirect rule

        Args:
            value   (dict)   -- Redirect rule to be set

                    Sample:

                        {
                            "domain":"redirect.loc"
                        }
        """
        self._redirect_rule = value

    @property
    def associations(self):
        """ Get SAML app associations"""
        return self._associations

    @associations.setter
    def associations(self, value):
        """ Get SAML app associations

         Args:
            value   (List)   -- List of associations to be set

                    Sample: ['user1,user2']
        """
        self._associations = value

    @property
    def attribute_mapping(self):
        """ Get attribute mapping"""
        return self._attribute_mapping

    @attribute_mapping.setter
    def attribute_mapping(self, value):
        """ Set attribute mapping

        Args:
            value   (dict)   -- Attribute mappings to be set

                    Sample:

                        {
                            "user attribute":"SAML attribute"
                        }
        """
        self._attribute_mapping = value

    def check_if_saml_app_exists(self, app_name):
        """
        searches for the given app with search box

        Args:

        app_name    (str)   --   SAML app name

        Returns:     None
        """
        self.__table.search_for(app_name)
        return self.__admin_console.check_if_entity_exists("link", app_name)

    def create_saml_app(self, idp_metadata_path, smtp_address,
                        webconsole_url, enable_company,
                        company_name, jks_file_path=None, alias_name=None,
                        keystore_password=None, key_password=None, auto_generate_key=True):
        """
        creating SAML app in admin console

        Args:

        idp_metadata_path    (str)    :IDP metadata file path

        smtp_address        (str)      :SMTP address of the identity server

        webconsole_url        (str)   :webconsole url to edit

        enable_company        (bool)    :True if created for company

        company_name            (str)    :name of company to be added

        jks_file_path        (str)    :keystore file path

        alias_name            (str)    :alias name for
                                        the .jks file

        keystore_password    (str)    :keystore password for
                                        the .jks file

        key_password        (str)     : key password for the
                                         .jks file

        auto_generate_key    (bool)    :True if key is auto generated

        Returns: None
        """
        is_app_created = self.saml.add_saml_app(self.app_name, idp_metadata_path,
                                                smtp_address, webconsole_url,
                                                enable_company, company_name,
                                                jks_file_path, alias_name,
                                                keystore_password, key_password,
                                                auto_generate_key)
        if is_app_created:
            if self.redirect_rule:
                self.saml_details.edit_redirect_rule(self.redirect_rule)
                self.__admin_console.wait_for_completion()
            if self.attribute_mapping:
                self.saml_details.edit_attribute_mappings(self.attribute_mapping)
                self.__admin_console.wait_for_completion()

    def open_saml_app(self, app_name):
        """
            Searches for a given SAML app and opens it

            Args:

                app_name    (str)   --   SAML app name

            Returns:     None
        """
        self.saml.select_identity_server(app_name)

    def edit_saml_rule_or_mappings(self, app_name, redirect_rule=None, mappings=None):
        """
            Edit the redirect rule or mappings in SAML app

            Args:

                app_name    (str)   --   SAML app name

                redirect_rule(str)   -- the redirect rule that has to be set

                mappings (str)   --   the attribute mappings that have to be set

            Returns:     None

        """
        if redirect_rule:
            self.saml_details.edit_redirect_rule(redirect_rule)
        if mappings:
            self.saml_details.edit_attribute_mappings(mappings)

    def modify_saml_general_settings(self, app_name, modify_app_state=False, enable_app=True,
                                     modify_auto_create=False, auto_create_user=True,
                                     add_default_usergroup=False,
                                     modify_user_group=False, user_group=None):
        """
            Edit the general settings in SAML app

            Args:

                app_name            (str)   --   SAML app name

                modify_app_state    (bool)   -- Changes the app state to disabled/enabled

                enable_app          (bool)   --  Changes the state of the app

                modify_auto_create  (bool)   --  Changes teh auto user create option

                auto_create_user    (bool)   --  Changes the auto create user to enabled /disabled

                add_default_usergroup(bool)  --  Add's Default User group

                modify_user_group   (bool)   --  Modifies the user group

                user_group          (str)   -- Sets the given user group

            Returns:     None

        """
        self.log.info(" Editing saml app general settings ")
        if modify_app_state:
            self.saml_details.modify_enable_app(enable_app)
        if modify_auto_create:
            self.saml_details.modify_auto_create_user(auto_create_user)
        if add_default_usergroup:
            self.saml_details.add_default_usergroup(user_group)
        if modify_user_group:
            if user_group:
                self.saml_details.modify_user_group(user_group)
            else:
                self.log.info("please give a valid user group")

    def delete_app(self):
        """
            Deletes a given SAML app
        """

        self.log.info("Deleting the app ")
        self.__navigator.navigate_to_identity_servers()
        self.open_saml_app(self.app_name)
        self.saml_details.delete_saml_app()

    def validate_saml_app(self, validate_key_dict):
        """
            Checks for the displayed values with the given values

            Args:
                validate_key_dict(dict) --Dictionary of values to validate with displayed values
                                          Sample:{'AppName': 'app name',
                            'Auto user create flag': True,
                            'Default user group': 'company name\\Tenant Users',
                            'Company name': 'company name',
                            'SP entity ID': 'https://hostname:443/webconsole',
                            'SSO URL':
                            'https://hostname:443/webconsole/samlAcsIdpInitCallback.do?samlAppKey
                            =NDVDQ0E2NUMwMkEzNDkz',
                            'Associations': None,
                            'Redirect rule': {'domain name', 'smtp address'}
                            'Attribute mapping': None}

            Returns:     None

        """
        displayed_val = self.saml_details.saml_app_info()

        self.log.info("Validating the SAML app created")
        for key, value in validate_key_dict.items():
            if displayed_val[key]:
                if isinstance(value, (str, bool)):
                    if displayed_val[key] == validate_key_dict[key]:
                        self.log.info("%s displayed for %s matches with %s given",
                                      displayed_val[key], key, validate_key_dict[key])
                    else:
                        raise CVWebAutomationException("%s displayed for %s does not match with %s given",
                                                       displayed_val[key], key, validate_key_dict[key])
                elif isinstance(value, list):
                    if set(displayed_val[key]) == set(validate_key_dict[key]):
                        self.log.info("%s displayed for %s matches with %s given",
                                      displayed_val[key], key, validate_key_dict[key])
                    else:
                        raise CVWebAutomationException("%s displayed for %s does not match with %s given",
                                                       displayed_val[key], key, validate_key_dict[key])
                else:
                    self.log.info('Entity displayed_val :%s', value)
                    for item, value_dict in value.items():
                        d_val = displayed_val[key][item].replace(" ", "")
                        key_val = validate_key_dict[key][item].replace(" ", "")
                        if d_val == key_val:
                            self.log.info("%s values match", item)
                        else:
                            raise CVWebAutomationException("%s displayed for %s does not match with %s given", d_val,
                                                           item, key_val)
            else:
                self.log.info("Displayed value for %s has None", validate_key_dict[key])
                if validate_key_dict[key] is None:
                    self.log.info("%s displayed for %s matches with %s given",
                                  displayed_val[key], key, validate_key_dict[key])

    def download_spmetadata(self, download_dir):
        """
        Downloads metadata of app

        Args:

            download_dir    (str)   --  Download location of the  SP metadata

        Returns:     None
        """
        downloaded_path = self.saml_details.download_sp_metadata(self.app_name, download_dir)
        return downloaded_path

    def edit_saml_idp(self, app_name, idp_meta_path=None, web_console_url=None,
                      jks_file_path=None, key_password=None, keystore_password=None,
                      alias_name=None):
        """
            Edits a given SAML app

        Args:
            app_name           (str)  :name of the identity server app

            idp_meta_path   (str)   :Location of IDP metadata path

            web_console_url        (str)   :webconsole url to edit

            jks_file_path        (str)    :keystore file path

            key_password        (str)     :key password for the .jks file

            keystore_password    (str)    :keystore password for the .jks file

            alias_name            (str)    :alias name for the .jks file

        Returns:     None
        """
        self.saml_details.edit_idp_details(idp_meta_path, web_console_url, jks_file_path,
                                           key_password, keystore_password, alias_name)

    def edit_trust_party_adfs(self, app_name, ad_host_ip, ad_machine_user, ad_machine_password,
                              sp_metadata_location=None, operation="Create"):
        """
            Create or delete the relying trust party created at the AD machine

            Args:

                app_name            (str)    : name of ADFS app

                ad_host_ip        (str)    :IP/name of AD machine

                ad_machine_user        (str)   :AD machine username

                ad_machine_password        (str)    :AD machine password

                sp_metadata_location    (str)    :path of SP metdata file

                operation                (str)    :Create-if app is created
                                                    Delete-if app is deleted
                                                    Default value is create

            Returns:     None
        """
        machine = Machine(machine_name=ad_host_ip, username=ad_machine_user,
                          password=ad_machine_password)
        if 'Create' in operation:
            output = machine.get_registry_value(
                value="ProgramFilesDir",
                win_key=r"HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion")
            if output:
                destination_path = output.replace("\r\n", "")
                destination_file_path = machine.join_path(
                    destination_path, f"SPMetadata_{app_name}.xml")
                if machine.check_file_exists(destination_file_path):
                    machine.delete_file(destination_file_path)
                machine.copy_from_local(sp_metadata_location, destination_path)
                output = machine.execute_command(
                    'Add-AdfsRelyingPartyTrust -Name "ADFS_check" '
                    '-MetadataFile "%s" -IssuanceAuthorizationRules \'@RuleTemplate = '
                    '"AllowAllAuthzRule"=> issue(Type = '
                    '"http://schemas.microsoft.com/authorization/claims/permit", Value '
                    '= "true");\' -IssuanceTransformRules \'@RuleTemplate = '
                    '"LdapClaims"@RuleName = "r1"c:[Type =="http://schemas.microsoft.com/ws/'
                    '2008/06/identity/claims/windowsaccountname",Issuer== "AD AUTHORITY"]=> '
                    'issue(store = "Active Directory", '
                    'types =("http://schemas.xmlsoap.org/ws/2005/05/identity/claims/'
                    'nameidentifier"), query=";mail;{0}", '
                    'param = c.Value);\'' % destination_file_path)
                if output.exception:
                    raise CVWebAutomationException("Create operation Failed at ADFS")
                if machine.check_file_exists(destination_file_path):
                    machine.delete_file(destination_file_path)
            else:
                raise CVWebAutomationException("Destination path not found")
        elif 'Delete' in operation:
            output = machine.execute_command('Remove-adfsRelyingPArtyTrust -TargetName ADFS_check')
            if output.exception:
                raise CVWebAutomationException("Delete operation Failed at ADFS")
        else:
            raise CVWebAutomationException("Invalid operation given")

    def initiate_saml_login(self, is_idp_initiated, ad_name, web_console_url, adfs_app_name, user,
                            password, tab_off_approach=True, verify_sso=True):
        """
        Initiates SAML login

        Args:
            is_idp_initiated: True if the login has to be initiated form the AD site

            ad_name         (str)   --    AD machien name /IP

            web_console_url(str)   -- web console URL of the client

            adfs_app_name  (str)   --  relying party trust app name

            user           (str)   --   user to login

            password        (str)   --password of the user

            tab_off_approach    (bool)  --True if SP init login is with tab approach

            verify_sso          (bool)  --verifies if SSO is successful

        Returns:True if the SAML login succeeds and false otherwise

        """
        if is_idp_initiated:
            status = self.identity_provider_obj.identity_provider_initiated_login(ad_name,
                                                                                  web_console_url,
                                                                                  adfs_app_name,
                                                                                  user, password,
                                                                                  verify_sso)
        else:
            if tab_off_approach:
                status = self.identity_provider_obj.service_provider_initiated_login(
                    ad_name,
                    web_console_url,
                    user,
                    password,
                    tab_off_approach,
                    verify_sso)
            else:
                _query = "select appKey from App_ThirdPartyApp where appname = '{0}' and " \
                         "appType = 2".format(self.app_name)
                self.csdb.execute(_query)
                app_key = self.csdb.fetch_one_row()
                if app_key:
                    saml_app_key = base64.b64encode(bytes(app_key[0], 'utf-'))
                else:
                    raise CVWebAutomationException("No such app exists")
                sp_initiated_link = web_console_url + "/initiateSaml.do?samlAppKey=" \
                    + str(saml_app_key.decode("utf-8"))

                status = self.identity_provider_obj.service_provider_initiated_login(
                    ad_name,
                    web_console_url,
                    user,
                    password,
                    verify_sso,
                    tab_off_approach,
                    sp_initiated_link
                )
        return status

    def initiate_saml_logout(self, is_idp_initiated, ad_name, web_console_url,
                             verify_single_logout=True):
        """
        Performs a SAML logout

        Args:
            is_idp_initiated    (bool)   -- True if it is a IDP initiated logout

            ad_name             (str)   --Name/IP of the AD machine

            web_console_url     (str)   --web console URL of the client

            verify_single_logout(bool)   --True ,if single logout is to be verified

        Returns:True if logout succeeds or not otherwise False

        """
        if is_idp_initiated:
            status = self.identity_provider_obj.identity_provider_initiated_logout(
                ad_name,
                web_console_url,
                verify_single_logout)
        else:
            status = self.identity_provider_obj.service_provider_initiated_logout(
                ad_name,
                web_console_url,
                verify_single_logout)
        return status

    def add_associations_on_saml_app(self, user):
        """
        Adds associations for a user on a SAML app

        Args:

            user    (str)   -- user to be added under associations

        Returns:None

        """
        _query = "select appKey from App_ThirdPartyApp where appname = '{0}' and " \
                 "appType = 2".format(self.app_name)
        self.csdb.execute(_query)
        app_key = self.csdb.fetch_one_row()[0]
        if not app_key:
            raise CVWebAutomationException("No such app exists")

        _query = "select props from App_ThirdPartyApp where appname = '{0}' and " \
                 "appType = 2".format(self.app_name)
        self.csdb.execute(_query)
        props = self.csdb.fetch_one_row()[0]

        self.commcell.add_associations_to_saml_app(self.app_name, app_key, props, user)

    def get_sso_url(self):
        """
        returns the SSO url of SAML app
        """
        return self.saml_details.copy_sso_url()

    def get_sp_entity_id(self):
        """
        returns the SP entity_id of SAML app
        """
        return self.saml_details.copy_sp_entity_id()

    def login_to_okta_and_edit_general_settings(self, okta_url, username, pwd, app_name,
                                                sso_url, sp_entity_id, name_id_format=None,
                                                attributes=None, group_attribute=None,
                                                slo=False, single_logout_url=None,
                                                sp_issuer=None, certificate=None):
        """
        Login to OKTA and edit the general settings

        okta_url (str)          : OKTA web url
        username (str)          : username to login
        pwd (str)               : password
        app_name (str)          : app name whose details are to be edited
        sso_url (str)           : Single sign on URL
        sp_entity_id (str)      : SP entity ID
        name_id_format (str)    : Type of value to be received from SAML response
        attributes (dict)       : Attribute Mappings
        group_attribute (dict)  : Group Attribute Mappings
        """
        self.__admin_console.browser.open_url_in_new_tab(okta_url)
        self.__admin_console.wait_for_completion()
        self.okta_obj.login(username, pwd)
        is_logged_in = self.okta_obj.check_if_login_successful()

        if is_logged_in:
            self.okta_obj.edit_general_settings(app_name,
                                                sso_url,
                                                sp_entity_id,
                                                name_id_format,
                                                attributes,
                                                group_attribute,
                                                slo, single_logout_url,
                                                sp_issuer, certificate)

    def logout_from_okta(self):
        """
        Logout from OKTA
        """
        self.okta_obj.logout_from_okta()

    def single_logout(self, okta_url):
        """
        Performs single logout

        okta_url (str)   : OKTA web url
        """
        self.__admin_console.browser.open_url_in_new_tab(okta_url)
        self.__admin_console.wait_for_completion()
        self.okta_obj.single_logout()

    def initiate_saml_login_with_okta(
            self,
            webconsole_url,
            hostname,
            okta_url,
            username,
            pwd,
            app_name,
            is_idp_initiated,
            tab_off_approach=True,
            sp_initiated_link=None):
        """
        Initiates SAML login with OKTA as Identity provider

        webconsole_url (str)    :   webconsole url
        hostname (str)          :   commcell hostname
        okta_url (str)          :   OKTA web url
        username (str)          :   username to login
        pwd (str)               :   password
        app_name (str)          :   SAML app name in OKTA
        is_idp_initiated (bool) :   True/False
        tab_off_approach (bool) :   redirects to other url on tab
        sp_initiated_link (str) :   link to initiate SAML login

        return  :   current url after SP/IDP initiated SAML login
        """
        if is_idp_initiated:
            status = self.okta_obj.idp_initiated_login(hostname, okta_url, app_name,
                                                       username, pwd, verify_sso=False)

        else:
            status = self.okta_obj.sp_initiated_login(webconsole_url,
                                                      hostname,
                                                      okta_url,
                                                      username,
                                                      pwd,
                                                      tab_off_approach=tab_off_approach,
                                                      verify_sso=False,
                                                      sp_initiated_link=sp_initiated_link)

        return status

    def initiate_saml_logout_with_okta(self, hostname):
        """
        initiates SAML logoutloaded_url,

        loaded_url (str) :   current url after SAML login
        hostname (str)   :   webconsole hostname
        """
        self.okta_obj.saml_logout(hostname)

    def delete_mapping(self, mapping_dict):
        """
        Deletes Attribute Mappings

        mapping_dict (dict)  :   Mapping dictionary
        """
        self.saml_details.delete_attribute_mapping(mapping_dict, from_edit=False)

    def check_if_login_is_successful(self):
        """
        Checks if Login is successful or not
        """
        self.okta_obj.check_if_login_successful()

    def check_slo(self, okta_url):
        """
        Check single logout status
        """
        return self.okta_obj.check_slo_status(okta_url)
