from Web.AdminConsole.Components.panel import ModalPanel
from Web.AdminConsole.Components.table import Table


class GlobalSearchHelper:
    """Admin console helper for global search operations"""

    def __init__(self, admin_console):
        """
                Helper for global search related actions

                Args:

                    admin_console   (object)    -- admin_console object

        """

        self.__admin_console = admin_console
        self.__navigator = self.__admin_console.navigator
        self.__driver = self.__admin_console.driver
        self.log = self.__admin_console.log
        self.__panel = ModalPanel(self.__admin_console)
        self.__table = Table(self.__admin_console)

    def verify_action_launches_panel(self, action_dict, entity, display_name):
        """Method to verify if action launches correct panel"""
        for action, header in action_dict.items():

            self.__navigator.manage_entity_from_search_bar(entity, display_name, action)
            self.__admin_console.wait_for_completion()
            if header == self.__panel.title():
                self.log.info(f"{entity} action {action} launches correct panel")
                self.__admin_console.cancel_form()
            else:
                raise Exception(f"{entity} action {action} does not launch correct panel")

    def verify_action_redirects(self, action_dict, entity, display_name):
        """Method to verify action redirects to correct page"""
        for action, header in action_dict.items():

            self.__navigator.manage_entity_from_search_bar(entity, display_name, action)
            self.__admin_console.wait_for_completion()
            if self.verify_page_header(header):
                self.log.info(f"{entity} action {action} launches correct panel")

                if self.__admin_console.check_if_entity_exists("xpath",
                                                               "//button[contains(.,'Cancel')]"):
                    self.__admin_console.click_button_using_text("Cancel")
                elif self.__admin_console.check_if_entity_exists("xpath",
                                                                 "//button[contains(.,'No')]"):
                    self.__admin_console.click_button_using_text("No")
                else:
                    pass

            else:
                raise Exception(f"{entity} action {action} does not launch correct panel")

    def verify_page_header(self, header_text):
        """Returns header text on the page"""

        if self.__admin_console.check_if_entity_exists("xpath",
                                                       f"//h1[contains(text(),'{header_text}')]"):
            return True
        elif self.__admin_console.check_if_entity_exists("xpath",
                                                        f"//h2[contains(text(),'{header_text}')]"):
            return True
        elif self.__admin_console.check_if_entity_exists("xpath",
                                                        f"//h4[contains(text(),'{header_text}')]"):
            return True
        elif self.__admin_console.check_if_entity_exists("xpath",
                                                        f"//h5[contains(text(),'{header_text}')]"):
            return True
        else:
            return False

    def verify_fs_hypervisor_actions(self, display_name, entity_flag=0):
        """
        Method to verify global search launches correct panel for actions of file servers

        Args:

            display_name   (str):  Display name of file server

            entity_flag    (int): Pass 0 for file server, 1 for hypervisor

        """
        # action_panels_dict contains action and panel header value pairs for actions that launch a panel
        actions_panels_dict = {"Add software": "Add software",
                               "Edit plan association": "Edit plan",
                               "Restore": "Restore",
                               "Back up": "Backup",
                               "Uninstall software": "Uninstall software",
                               "Release license": "Release license"
                               }
        # actions_redirect_dict contains action and page header value pairs for actions that redirect to a page
        actions_redirect_dict = {"View jobs": "Job history",
                                 "Update software": "Confirm software update",
                                 "Check readiness": "Check readiness",
                                 "Send logs": "Send log files of",
                                 "Retire": "Confirm retire client"}

        if entity_flag == 0:
            entity = "File servers"
        elif entity_flag == 1:
            entity = "Hypervisors"
        else:
            raise Exception("Invalid entity flag")

        self.verify_action_launches_panel(actions_panels_dict, entity, display_name)
        self.verify_action_redirects(actions_redirect_dict, entity, display_name)

    def verify_server_group_actions(self, server_group):
        """
        Method to verify server group actions

        Args:

            server_group        (str):  Name of server group

        """

        # actions_dict contains action and panel header value pairs for actions that launch a panel
        actions_dict = {"Change company": "Change company",
                        "Edit associations": "Edit associations",
                        "Add software": "Add software",
                        "Clone": "Clone server group"
                        }

        # actions_redirect_dict contains action and page header value pairs for actions that redirect to a page
        actions_redirect_dict = {
            "Update software": "Confirm software update",
            "Delete": "Delete server group",
            "View jobs": "Job history",
            "Push network configuration": " Confirm push network configuration"
        }

        self.verify_action_launches_panel(actions_dict, "Server groups", server_group)
        self.verify_action_redirects(actions_redirect_dict, "Server groups", server_group)

    def verify_user_actions(self, user):
        """
        Method to verify user actions

        Args:

            user        (str):  Name of user

        """

        # actions_dict contains action and panel header value pairs for actions that launch a panel
        actions_dict = {"Delete": "Confirm delete"}
        self.verify_action_redirects(actions_dict, "Users", user)

    def verify_user_group_actions(self, user_group):
        """
        Method to verify user group actions

        Args:

            user_group      (str):  Name of user group

        """

        # actions_dict contains action and panel header value pairs for actions that launch a panel
        actions_dict = {"Add users": "Add users"}
        # actions_redirect_dict contains action and page header value pairs for actions that redirect to a page
        actions_redirect_dict = {"Delete": "Confirm delete"}
        self.verify_action_launches_panel(actions_dict, "User groups", user_group)
        self.verify_action_redirects(actions_redirect_dict, "User groups", user_group)

    def verify_vm_actions(self, vm_name):
        """
        Method to verify virtual machine actions

        Args:

            vm_name     (str):  Name of the virtual machine

        """

        # actions_dict contains action and panel header value pairs for actions that launch a panel
        actions_dict = {"Configure replication": "Select replication group"}

        # actions_redirect_dict contains action and page header value pairs for actions that redirect to a page
        actions_redirect_dict = {"Restore": "Select restore type",
                                 "View jobs": "Job history",
                                 "View active mounts": "Active mounts"}

        self.verify_action_launches_panel(actions_dict, "Virtual machines", vm_name)
        self.verify_action_redirects(actions_redirect_dict, "Virtual machines", vm_name)

    def verify_vm_group_actions(self, vm_group_name):
        """
        Method to verify virtual machine group actions

        Args:

            vm_group_name (str)     : VM group name

        """

        # actions_dict contains action and panel header value pairs for actions that launch a panel
        actions_dict = {"Back up": "Select backup level"}

        # actions_redirect_dict contains action and page header value pairs for actions that redirect to a page
        actions_redirect_dict = {"View jobs": "Job history",
                                 "Delete": "Confirm delete"}

        self.verify_action_launches_panel(actions_dict, "VM groups", vm_group_name)
        self.verify_action_redirects(actions_redirect_dict, "VM groups", vm_group_name)
