# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be used to run
basic operations on Manage - > Commcell page.

Class:

 commcellHelper()

Functions:

 Details from API calls:

    get_all_details_from_api()             -- Fetch data from different APIs and updates resultant dictionary
    __get_about_details()                  -- Gets the about details from API
    __get_security_details()               -- Gets the security details from API
    __get_passkey_details                  -- Gets the passkey details from API
    __get_activity_control_details()       -- Gets the activity control details from API
    __get_email_settings_details()         -- Gets the email settings details from API
    __get_password_encryption_details()    -- Gets the password encryption details from API
    __get_default_plans_details()          -- Gets the default plans details from API
    __get_general_privacy_details()        -- Gets the general and privacy properties of commcell from API

 Edit details:

    edit_details()                         -- Edits details by toggling, adding/removing for the tiles.
    edit_specific_setting()                -- Allow editing of specific settings
    toggle_button()                        -- This static method helps to enable disable toggle buttons

 Extracting and validating details:

    extract_and_validate()                 -- Extract information and validate it even if it is edited
    extract_all_details_on_page()          -- Retrieves all plan information displayed in Commcell page
    validate_all_details()                 -- Validates if the details are displayed correctly
    generate_updated_details()             -- Update the changed settings into dictionary
"""

from Web.AdminConsole.AdminConsolePages.Commcell import Commcell
from Web.AdminConsole.Components.panel import ModalPanel, Security


class CommcellHelper:
    """ Helper for Commcell page """

    def __init__(self, admin_console, commcell):
        """ Initializes the commcell helper module """
        self.admin_console = admin_console
        self.commcell = commcell
        self.services = self.commcell._services
        self.log = admin_console.log
        self._initial_details = None
        self._all_displayed_details = None
        self._all_api_details = None
        self._edit_for_settings = None
        self.commcell_obj = Commcell(self.admin_console)
        self.__panel = ModalPanel(admin_console)

    def extract_all_details_on_page(self):
        """
            Retrieves all plan information displayed in Commcell page
            Returns:
                All displayed details
        """
        self._all_displayed_details = self.commcell_obj.extract_all_displayed_details()
        self.log.info(self._all_displayed_details)
        return self._all_displayed_details

    def get_all_details_from_api(self):
        """
            Calls various functions to get details from API and updates resultant dictionary
        """
        self._all_api_details = {}

        self.__get_about_details()
        self.__get_activity_control_details()
        self.__get_email_settings_details()
        self.__get_general_privacy_details()
        self.__get_security_details()
        self.__get_passkey_details()
        self.__get_password_encryption_details()
        self.__get_default_plans_details()

        self.log.info(self._all_api_details)

    def validate_all_details(self, validation_dict):
        """
            Validates if the details are displayed correctly
        """
        displayed_val = self._all_displayed_details
        for key_dict, val_dict in validation_dict.items():
            if isinstance(val_dict, list):
                self.log.info('Entity given_val "%s"', tuple(val_dict))
                if set(displayed_val[key_dict]) == set(validation_dict[key_dict]):
                    self.log.info(
                        "%s displayed for %s matches with %s given",
                        displayed_val[key_dict], key_dict, validation_dict[key_dict])
                else:
                    raise Exception("{0} displayed for {1} does not match with {2} given ".format(
                        displayed_val[key_dict], key_dict, validation_dict[key_dict]))
            elif isinstance(val_dict, str):
                self.log.info('Entity given_val "%s"', val_dict)
                if displayed_val[key_dict] == validation_dict[key_dict]:
                    self.log.info(
                        "%s displayed for %s matches with %s given",
                        displayed_val[key_dict], key_dict, validation_dict[key_dict])
                else:
                    raise Exception("{0} displayed for {1} does not match with {2} given ".format(
                        displayed_val[key_dict], key_dict, validation_dict[key_dict]))
            else:
                # dict
                self.log.info('Entity given_val :%s', val_dict)
                for item, value_dict in val_dict.items():
                    d_val = displayed_val[key_dict][item]
                    key_val = validation_dict[key_dict][item]
                    if d_val == key_val:
                        self.log.info("%s values match", item)
                    else:
                        raise Exception("{0} displayed for {1} does not match with {2} given ".format(
                            d_val, item, key_val))

    def __get_about_details(self):
        """
            Gets the about details from API
        """
        version = self.commcell.version.replace(' ', '').replace('(BUILD80)', '').replace('R2', '')\
            .replace('.0SP', '.').replace('SP', '.').replace('HPK', '.').replace('+', '').replace('-', '')\
            .replace('a', '.1').replace('b', '.2').replace('c', '.3')
        version = version + '.0'*(3 - len(version.split('.')))

        temp_details = {
            "CommServe": self.commcell.commserv_hostname,
            "Version": version,
        }
        self._all_api_details.update({"About": temp_details})

    def __get_security_details(self):
        """
            Gets the security details from API
        """
        security_associations = self.commcell.get_security_associations()
        self._all_api_details.update({'Security': security_associations})

    def __get_activity_control_details(self):
        """
            Gets the activity control details from API
        """
        temp_details = {}
        template_dict = {
            "DATA MANAGEMENT": "Data backup",
            "DATA RECOVERY": "Data restore",
            "ALL ACTIVITY": "All job activity",
            "SCHEDULER": "Scheduler"
        }
        activity_control = self.commcell.activity_control

        for key, name in template_dict.items():
            if activity_control.is_enabled(key):
                # API response of 0 is true and 1 is false
                value = "OFF"
            else:
                value = "ON"
            temp_details.update({name: value})

        self._all_api_details.update({"Activity control": temp_details})

    def __get_email_settings_details(self):
        """
            Gets the email setting details from API
        """
        temp_details = {}
        template_dict = {
            "smtpServer": "SMTP server",
            "smtpPort": "SMTP port",
            "senderInfo":
                {
                    "senderAddress": "Sender email",
                    "senderName": "Sender name"
                }
        }

        email_settings = self.commcell.get_email_settings()

        for key in template_dict:
            value = email_settings.get(key)
            if isinstance(value, dict):
                for sub_key, sub_value in value.items():
                    temp_details.update({template_dict[key][sub_key]: sub_value})
            else:
                temp_details.update({template_dict[key]: str(value)})

        self._all_api_details.update({"Email settings": temp_details})

    def __get_password_encryption_details(self):
        """
            Gets the password encryption details from API
        """
        temp_details = {}
        template_dict = {
            "keyProviderName": "Key Management Server"
        }
        pass_config = self.commcell.get_password_encryption_config()
        for key, value in pass_config.items():
            if key in template_dict.keys():
                temp_details.update({template_dict[key]: value})
        self._all_api_details.update({"Password encryption": temp_details})

    def __get_passkey_details(self):
        """
            Gets the passkey details from API
        """
        temp_details = {}
        template_dict = {
            "authType": "Require passkey for restores",
            "authorizeForRestore": "Authorize for restore",
            "allowUsersToEnablePasskey": "Allow users to enable passkey"
        }

        organization_properties = self.commcell.get_commcell_organization_properties()

        if organization_properties.get('advancedPrivacySettings').get('authType') == 2:
            temp_details.update({template_dict.get('authType'): "ON"})
            if organization_properties.get('advancedPrivacySettings').get('passkeySettings') \
                    .get('enableAuthorizeForRestore') == "true":
                temp_details.update({template_dict.get('enableAuthorizeForRestore'): "ON"})
            else:
                temp_details.update({template_dict.get('enableAuthorizeForRestore'): "OFF"})
        else:
            temp_details.update({template_dict.get('authType'): "OFF"})
        if organization_properties.get('allowUsersToEnablePasskey') == "true":
            temp_details.update({template_dict.get('allowUsersToEnablePasskey'): "ON"})
        else:
            temp_details.update({template_dict.get('allowUsersToEnablePasskey'): "OFF"})

        self._all_api_details.update({"Passkey": temp_details})

    def __get_default_plans_details(self):
        """
            Gets the default plans details from API
        """
        default_plans = self.commcell.get_default_plan()
        default_plans_details = []
        for list_item in default_plans:
            default_plans_details.append({list_item.get('subtype'): list_item.get('plan').get('planName')})

        self._all_api_details.update({"Default plans": default_plans_details})

    def __get_general_privacy_details(self):
        """
            Gets the general and privacy properties of commcell from API
        """
        temp_details1 = {}
        temp_details2 = {}
        template_dict1 = {
            "enableTwoFactorAuthentication": "Enable two factor authentication",
            "useUPNForEmail": "Use UPN instead of e-mail",
            "enableSharedLaptopUsage": "Requires authcode for installation"
        }
        # "enableSharedLaptopUsage" is used for "Requires authcode for installation"
        template_dict2 = {
            "enablePrivacy": "Allow users to enable data privacy"
        }
        commcell_properties = self.commcell.get_commcell_properties()
        for key, value in commcell_properties.items():
            if key in template_dict1.keys():
                if value:
                    temp_details1.update({template_dict1[key]: "ON"})
                else:
                    temp_details1.update({template_dict1[key]: "OFF"})
            elif key in template_dict2.keys():
                if value:
                    temp_details2.update({template_dict2[key]: "ON"})
                else:
                    temp_details2.update({template_dict2[key]: "OFF"})
        self._all_api_details.update({"General": temp_details1})
        self._all_api_details.update({"Privacy": temp_details2})

    def edit_details(self, to_edit, revert=False):
        """
            Edits details by toggling, adding/removing for the tiles.
        Args:
            to_edit (list): Name of tiles and the information in it to edit
                ['Activity control', {'Security': {'master': [['View']]}}, 'General', 'Privacy', 'Default plans']
            revert (boolean): True for reverting changes
        """
        disable_list = {
            'All job activity': self.commcell_obj.disable_all_job_activity,
            'Data backup': self.commcell_obj.disable_backup,
            'Data restore': self.commcell_obj.disable_restore,
            'Scheduler': self.commcell_obj.disable_schedule,
            'Requires authcode for installation': self.commcell_obj.disable_authcode_for_installation,
            'Use UPN instead of e-mail': self.commcell_obj.disable_upn,
            'Enable two factor authentication': self.commcell_obj.disable_tfa,
            'Allow users to enable data privacy': self.commcell_obj.disable_data_privacy
        }
        enable_list = {
            'All job activity': self.commcell_obj.enable_all_job_activity,
            'Data backup': self.commcell_obj.enable_backup,
            'Data restore': self.commcell_obj.enable_restore,
            'Scheduler': self.commcell_obj.enable_schedule,
            'Requires authcode for installation': self.commcell_obj.enable_authcode_for_installation,
            'Use UPN instead of e-mail': self.commcell_obj.enable_upn,
            'Enable two factor authentication': self.commcell_obj.enable_tfa,
            'Allow users to enable data privacy': self.commcell_obj.enable_data_privacy
        }
        enable_disable_list = [enable_list, disable_list]
        self._edit_for_settings = to_edit
        for edit_item in to_edit:
            if isinstance(edit_item, str):
                for key, value in self._all_displayed_details.get(edit_item).items():
                    self.toggle_button(enable_disable_list, key, revert, value)
            elif isinstance(edit_item, dict):
                for key, value in edit_item.items():
                    self.edit_specific_setting(enable_disable_list, key, revert, value)

    def edit_specific_setting(self, enable_disable_list, key, revert, value):
        """
            Allow editing of specific settings
        Args:
            enable_disable_list: Super List of all function to be called if toggling off/on a button
            key: Name of Tile
            revert: (Boolean) Tells if we are doing revert of all changes
            value: Content of Tile
        """
        if key == 'Email settings':
            if revert:
                self.commcell_obj.edit_email_settings(self._initial_details.get('Email settings'))
            else:
                self.commcell_obj.edit_email_settings(value)
        elif key == 'Security':
            if revert:
                Security(self.admin_console).edit_security_association(value, False)
            else:
                Security(self.admin_console).edit_security_association(value)
        elif key == 'Default plans':
            if revert:
                self.commcell_obj.edit_default_plans(self._initial_details.get('Default plans'))
            else:
                self.commcell_obj.edit_default_plans(value)
        elif key == 'Password encryption':
            if revert:
                self.commcell_obj.edit_password_encryption(self._initial_details.get('Password encryption'))
            else:
                self.commcell_obj.edit_password_encryption(value)
        else:
            for sub_key, sub_value in value.items():
                self.toggle_button(enable_disable_list, sub_key, not revert, sub_value)

    @staticmethod
    def toggle_button(enable_disable_list, key, revert, value):
        """
            This static method helps o enable disable toggle buttons
        Args:
            enable_disable_list: Super List of all function to be called if toggling off/on a button
            key: Tile heading
            revert: (Boolean) Tells if we are doing revert of all changes
            value: Tile content

        Returns:
            The corresponding function for toggling off/on button
        """
        if revert:
            if value == 'ON':
                enable_disable_list[0].get(key)()
            else:
                enable_disable_list[1].get(key)()
        else:
            if value == 'ON':
                enable_disable_list[1].get(key)()
            else:
                enable_disable_list[0].get(key)()

    def extract_and_validate(self, save_initial=False, edited=False):
        """
            Combined function to extract information and validate it even if it is edited
        Args:
            save_initial: (Boolean) Save the initial settings
            edited: (Boolean) True if any settings has been changed
        """
        self.get_all_details_from_api()
        details = self.extract_all_details_on_page()
        if save_initial:
            self._initial_details = details
        if edited:
            validation_details = self.generate_updated_details(details)
            self.log.info(validation_details)
        else:
            self.get_all_details_from_api()
            validation_details = self._all_api_details
        self.validate_all_details(validation_details)

    def generate_updated_details(self, old_details):
        """
            Update the changed settings into dictionary
        Args:
            old_details {dict}: Old dictionary of settings
        Returns:
            Updated dictionary of settings
        """
        for item in self._edit_for_settings:
            if isinstance(item, str):
                for key, value in old_details.get(item).items():
                    if value == 'ON':
                        old_details[item][key] = 'OFF'
                    elif value == 'OFF':
                        old_details[item][key] = 'ON'
            elif isinstance(item, dict):
                for key, value in item.items():
                    for sub_key, sub_value in value.items():
                        if isinstance(sub_value, list):
                            old_details[key][sub_key].append(sub_value)
                        else:
                            old_details[key][sub_key] = sub_value
        return old_details
