# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""

Helper file for cassandra cluster related operations.

Classes defined in this file:

    CassandraHelper:         Class for connecting to cql host and performing cqlsh queries.

        __init__()                 --  Constructor for creating a connection to Cassandra cluster

        _getnodeip()               --  return list of cql host ips/names

        _connect()                 --  create cluster object and initiate the connection

        connection()               --  return the cassandra cluster connection object

        createkeyspace()           --  execute cql query to create a test keyspace

        dropkeyspace()             --  execute cql query to drop the test keyspace

        createtable()              --  create table

        droptable()                --  drop table

        truncatetable()            --  truncate table

        check_if_keyspace_exists() --  check if the specified keyspace exist

        check_if_table_exists()    --  check if the specified table exist

        get_rows()                 --  get the rows in the table

        populate_test_data         --  insert test data into the specified table

        close_connection()         --  shutdown the db connection


    Cassandra:               Class for creating/deleting instances and running backup/restore
                                 for cassandra instance Under Big data, It also has the methods to
                                 connect/disconnect to cql host, generate/modify/drop test data
                                 and validate restored data

        __init__()                  --  constructor for creating cassandra object

        create_cassandra_instance() --  create cassandra pesudo client

        verify_backup()             --  run backup job and verify backup job complete

        verify_restore()            --  run restore job and verify restore job complete

        delete_cassandra_instance() --  delete cassandra instance and pseudo client

        generate-test_data()        --  generate test data

        drop_keyspace()             --  drop keysapce

        drop_table()                --  drop table

        truncate_table()            --  truncate table

        validate_restore()          --  verified expected data are restored to cassandra cluster db

        close_dbconnection()        --  shutdown cassandra db connection
"""

from cassandra.auth import PlainTextAuthProvider
from cassandra.cluster import Cluster
from cassandra.policies import DCAwareRoundRobinPolicy

from Web.Common.page_object import TestStep

from Web.AdminConsole.Bigdata.instances import Instances, CassandraServer
from Web.AdminConsole.Bigdata.details import Overview
from Web.Common.exceptions import CVTestStepFailure
from AutomationUtils import logger


class CassandraHelper():
    """
        Helper class for Cassandra cluster related operations
    """

    def __init__(self, cql_host, cql_username, cql_password, cql_port):
        """
        Constructor for creating the cassandra cluster connection
        Args:
                cql_host (str)        -- cql host name or ip

                cql_username (str)    -- cql username

                cql_password (str)    -- cql user password

            Returns:
                object  -   connection object to the cassandra cluster

            Raises:
                Exception:
                    if failed to connect to the database

        """
        self.cql_host = cql_host
        self.cql_username = cql_username
        self.cql_password = cql_password
        self.cql_port = int(cql_port)
        self.log = logger.get_log()
        self._cluster = None
        self._connection = None
        self._connect()

    def _getnodeip(self):
        """return list of cql host ips"""
        node_ips = [self.cql_host]
        return node_ips

    @property
    def connection(self):
        """return cassandra cluster connection object"""
        return self._connection

    def _connect(self):
        """initiation Cluster object and connect to cassandra cql host """
        try:
            auth_provider = PlainTextAuthProvider(
                username=self.cql_username,
                password=self.cql_password)
            self._cluster = Cluster(
                self._getnodeip(),
                load_balancing_policy=DCAwareRoundRobinPolicy(),
                auth_provider=auth_provider,
                port=self.cql_port)
            self._connection = self._cluster.connect()
            self.log.info("cassandra cluster connection is create")
        except Exception as excp:
            raise Exception(
                'Failed to connect to cassandra server\nERror: "{0}"'.format(excp))

    def createkeyspace(self, keyspace):
        """
        create keyspace
        Args:
                keyspace (str)        -- new keyspace name

            Raises:
                Exception:
                    if failed to create keyspace
        """
        try:
            cmd = "CREATE KEYSPACE IF NOT EXISTS " + keyspace + \
                " WITH replication = {'class': 'SimpleStrategy', \
                'replication_factor': '1'}  AND durable_writes = true;"
            self._connection.execute(cmd)
            self.log.info("keyspce created")
        except Exception:
            raise Exception("unable to create keyspace")

    def dropkeyspace(self, keyspace):
        """
        drop keyspace
        Args:
                keyspace (str)        -- new keyspace name

            Raises:
                Exception:
                    if failed to drop keyspace
        """
        try:
            cmd = "drop keyspace " + keyspace
            self._connection.execute(cmd)
            self.log.info("keyspace dropped")
        except Exception:
            raise Exception("Unable to drop keyspace")

    def createtable(self, keyspace, tablename):
        """
        create table in keyspace
        Args:
                keyspace (str)        -- keyspace name
                tablename (str)       -- new table name

            Raises:
                Exception:
                    if failed to create table
        """
        try:
            cmd = "CREATE TABLE IF NOT EXISTS " + keyspace + "." + \
                tablename + " ( id int PRIMARY KEY, fname text, lname text)"
            self._connection.execute(cmd)
            self.log.info("table created")
        except Exception:
            raise Exception("unable to create table")

    def droptable(self, keyspace, table):
        """
        drop table from keyspace
        Args:
                keyspace (str)        -- keyspace name
                table (str)       -- table name

            Raises:
                Exception:
                    if failed to drop table
        """
        try:
            cmd = "drop table " + keyspace + "." + table
            self._connection.execute(cmd)
            self.log.info("table dropped")
        except Exception:
            raise Exception("unable to drop table")

    def truncatetable(self, keyspace, table):
        """
        truncate table in keyspace
        Args:
                keyspace (str)        -- keyspace name
                tablename (str)       -- new table name

            Raises:
                Exception:
                    if failed to truncate table
        """
        try:
            cmd = "truncate table " + keyspace + "." + table
            self._connection.execute(cmd)
            self.log.info("table truncated")
        except Exception:
            raise Exception("failed to truncate table")

    def check_if_keyspace_exists(self, keyspace):
        """
        check if keyspace exist
        Args:
                keyspace (str)        -- keyspace name
            Return:
                True if keyspace exist
        """
        keyspaces = self._connection.execute(
            'select keyspace_name from system_schema.keyspaces')
        if keyspace not in keyspaces:
            return False
        return True

    def check_if_table_exists(self, keyspace, table):
        """
        check if table exist
        Args:
                keyspace (str)        -- keyspace name
                table (str)         -- table name
            Return:
                True if table exist
        """
        cmd = "select table_name from system_schema.tables where keyspace_name=" + keyspace
        tables = self._connection.execute(cmd)
        if table not in tables:
            return False
        return True

    def get_rows(self, keyspace, table):
        """
        get rows ResultSet from table under keyspace
        Args:
                keyspace (str)        -- keyspace name
                table (str)         -- table name
            Return:
                ResultSets for rows in table
        """
        try:
            cmd = "select * from " + keyspace + "." + table
            rows = self._connection.execute(cmd)
            return rows
        except Exception:
            raise Exception("failed to get table rows")

    def populate_test_data(self, keyspace, table, rows):
        """
        populate test data
        Args:
                keyspace (str)        -- keyspace name
                table (str)         -- table name
            Raises:
                Exceptions if populating test data failed
        """
        try:
            self.truncatetable(keyspace, table)
            cmd1 = "insert into " + keyspace + "." + \
                table + "(id, fname, lname) values("
            cmd2 = ", 'test', 'test')"
            i = 1
            while i <= rows:
                cmd = cmd1 + str(i) + cmd2
                self._connection.execute(cmd)
                i += 1
            self.log.info("test data populated")
        except Exception:
            raise Exception("failed to populate test data")

    def close_connection(self):
        """close the connection to cassandra cluster"""
        if self.connection is not None:
            self._connection.shutdown()
            self._connection = None


class Cassandra:
    """
    class has the functions to operate Cassandra instances Under Big data Apps
    """
    test_step = TestStep()

    def __init__(self, admin_console, testcase):
        """Constructor for creating the cassandra object"""
        self._admin_console = admin_console
        self.__instances = Instances(admin_console)
        self.__cassandra_server = CassandraServer(self._admin_console)
        self.__overview = Overview(self._admin_console)
        self.__commcell = testcase.commcell
        self.__cassandra = None
        self.cassandra_server_name = testcase.cassandra_server_name
        self.gatewaynode = testcase.tcinputs.get("gateway_node")
        self.config_file_path = testcase.tcinputs.get("config_file_path")
        self.cql_host = testcase.tcinputs.get("cql_host")
        self.plan_name = testcase.tcinputs.get("plan")
        self.cql_username = testcase.cql_username
        self.cql_password = testcase.cql_password
        self.cql_port = testcase.tcinputs.get("cql_port")
        self.jmx_port = testcase.tcinputs.get("jmx_port")
        self.jmx_username = testcase.jmx_username
        self.jmx_password = testcase.jmx_password
        self.staging_path = testcase.tcinputs.get("staging_path")

    @test_step
    def create_cassandra_instance(self):
        """create cassandra instance """
        self._admin_console.navigator.navigate_to_big_data()
        if self.__instances.is_instance_exists(self.cassandra_server_name):
            self.delete_cassandra_instance()
        _cassandra_server = self.__instances.add_cassandra_server()
        _cassandra_server.set_name(self.cassandra_server_name)

        _cassandra_server.set_config_file_path(
            self.config_file_path)
        _cassandra_server.set_cql_host(self.cql_host)
        _cassandra_server.select_plan(self.plan_name)
        _cassandra_server.select_gateway_node(self.gatewaynode)
        _cassandra_server.expand_authentication_settings()
        if self.cql_port != '9042':
            _cassandra_server.set_cql_port(self.cql_port)
        if self.jmx_port != '7199':
            _cassandra_server.set_jmx_port(self.jmx_port)
        _cassandra_server.enable_cql(self.cql_username, self.cql_password)
        _cassandra_server.enable_jmx(self.jmx_username, self.jmx_password)
        _cassandra_server.save()
        if not self.__instances.is_instance_exists(self.cassandra_server_name):
            raise CVTestStepFailure(
                "[%s] Cassandra server is not getting created " %
                self.cassandra_server_name)
        self._admin_console.log.info("Successfully Created [%s] cassandra server instance",
                                     self.cassandra_server_name)

    @test_step
    def verify_backup(self):
        """Initiate the backup and verify backup job is completed"""
        # Initiate backup job
        self._admin_console.navigator.navigate_to_big_data()
        self.__instances.access_instance(self.cassandra_server_name)
        _job_id = self.__overview.backup_now()

        # Wait till backup job to complete
        job = self.__commcell.job_controller.get(_job_id)

        self._admin_console.log.info("Wait for [%s] job to complete", str(job))
        if job.wait_for_completion(300):  # wait for max 5 minutes
            self._admin_console.log.info(
                "Backup job completed with job id:[%s]", _job_id)
        else:
            err_str = "[%s] job id for backup job failed" % _job_id
            raise CVTestStepFailure(err_str)

        # Verify backup is completed by checking the node in configuration tab
        _configuration = self.__overview.access_configuration()
        if _configuration.is_nodes_exists():
            self._admin_console.log.info("Data center backed up successfully")
            return
        raise CVTestStepFailure(
            "Nodes are not present in configuration tab for [%s] "
            "instance" %
            self.cassandra_server_name)

    @test_step
    def discover_node(self):
        """discover cassandra node"""
        self._admin_console.navigator.navigate_to_big_data()
        self.__instances.access_instance(self.cassandra_server_name)
        config = self.__overview.access_configuration()
        config.edit_cassandra_node()
        config.discover_cassandra_node()

    @test_step
    def verify_restore(self, stagefree=True, sstableloader=True, clusterview=False, inplace=True):
        """Initiate restore and verify restore job is completed"""
        # Initiate restore job
        self._admin_console.navigator.navigate_to_big_data()
        self.__instances.access_instance(self.cassandra_server_name)
        self.__instances.access_restore(instance='default')
        if clusterview:
            self._admin_console.enable_toggle(index=0, cv_toggle=True)
        _restore = self.__overview.restore_all()
        if inplace:
            _restore.select_in_place_restore()
        else:
            _restore.select_out_of_place_restore()
        if stagefree:
            _restore.select_stage_free_restore()
        if sstableloader:
            _restore.use_sstableloader_tool()
            _restore.set_staging_location(self.staging_path)
        _restore.submit_restore()
        # Wait for restore job to complete
        _job_id = self._admin_console.get_jobid_from_popup()
        job = self.__commcell.job_controller.get(_job_id)
        self._admin_console.log.info(
            "Wait for [%s] restore job to complete", str(job))
        if job.wait_for_completion(300):  # wait for max 5 minutes
            self._admin_console.log.info(
                "Restore job completed with job id:[%s]", _job_id)
        else:
            err_str = "[%s] job id for restore job failed" % _job_id
            raise CVTestStepFailure(err_str)
        self._admin_console.log.info("Restore completed successfully")

    @test_step
    def drop_keyspace_inplacerestore(self, keyspace, tablename, rows):
        """drop keyspace, DB veiw in place restore w/o sstableloader, validate restored data"""
        self.drop_keyspace(keyspace)
        self.verify_restore(stagefree=False, sstableloader=False)
        self.validate_restoredata(keyspace, tablename, rows)

    @test_step
    def drop_keyspace_inplacerestore_sstableloader(
            self, keyspace, tablename, rows):
        """drop keyspace, DB view in place restore w/ sstableloaer, validate restored data"""
        self.drop_keyspace(keyspace)
        self.verify_restore(stagefree=False)
        self.validate_restoredata(keyspace, tablename, rows)

    @test_step
    def drop_table_inplacerestore(self, keyspace, tablename, rows):
        """drop table, DB view in place restore w/o sstableloader, validate restored data"""
        self.drop_table(keyspace, tablename)
        self.verify_restore(stagefree=False, sstableloader=False)
        self.validate_restoredata(keyspace, tablename, rows)

    @test_step
    def drop_table_inplacerestore_sstableloader(
            self, keyspace, tablename, rows):
        """drop table, DB view in place restore w/ sstableloader, validate restored data"""
        self.drop_table(keyspace, tablename)
        self.verify_restore(stagefree=False)
        self.validate_restoredata(keyspace, tablename, rows)

    @test_step
    def drop_keyspace_clusterview_inplacerestore(
            self, keyspace, tablename, rows):
        """drop keyspace, Cluster veiw in place restore w/o sstableloader, validate restored data"""
        self.drop_keyspace(keyspace)
        self.verify_restore(
            stagefree=False,
            sstableloader=False,
            clusterview=True)
        self.validate_restoredata(keyspace, tablename, rows)

    @test_step
    def drop_keyspace_clusterview_inplacerestore_sstableloader(
            self, keyspace, tablename, rows):
        """drop keyspace, cluster view in place restore w/ sstableloaer, validate restored data"""
        self.drop_keyspace(keyspace)
        self.verify_restore(stagefree=False, clusterview=True)
        self.validate_restoredata(keyspace, tablename, rows)

    @test_step
    def drop_table_clusterview_inplacerestore(self, keyspace, tablename, rows):
        """drop table, cluster view in place restore w/o sstableloader,validate restored data"""
        self.drop_table(keyspace, tablename)
        self.verify_restore(
            stagefree=False,
            sstableloader=False,
            clusterview=True)
        self.validate_restoredata(keyspace, tablename, rows)

    @test_step
    def drop_table_clusterview_inplacerestore_sstableloader(
            self, keyspace, tablename, rows):
        """drop table, cluster view, in place restore with sstableloader, validate restored data """
        self.drop_table(keyspace, tablename)
        self.verify_restore(stagefree=False, clusterview=True)
        self.validate_restoredata(keyspace, tablename, rows)

    @test_step
    def delete_cassandra_instance(self):
        """Delete cassandra instance and verify instance is deleted"""
        self._admin_console.navigator.navigate_to_big_data()
        self.__instances.delete_instance_name(self.cassandra_server_name)
        if self.__instances.is_instance_exists(self.cassandra_server_name):
            raise CVTestStepFailure(
                "[%s] Cassandra server is not getting deleted" %
                self.cassandra_server_name)
        self._admin_console.log.info(
            "Deleted [%s] instance successfully",
            self.cassandra_server_name)

    @test_step
    def connect_to_db(self):
        """initiate connection to cassandra cql host"""
        self.__cassandra = CassandraHelper(self.cql_host,
                                           self.cql_username,
                                           self.cql_password,
                                           self.cql_port)

    @test_step
    def generate_test_data(self, keyspace, tablename, rows):
        """generate test data"""
        self.__cassandra.createkeyspace(keyspace)
        self.__cassandra.createtable(keyspace, tablename)
        self.__cassandra.populate_test_data(keyspace, tablename, rows)

    @test_step
    def drop_keyspace(self, keyspace):
        """drop keyspace"""
        self.__cassandra.dropkeyspace(keyspace)

    @test_step
    def drop_table(self, keyspace, table):
        """drop table"""
        self.__cassandra.droptable(keyspace, table)

    @test_step
    def truncate_table(self, keyspace, table):
        """truncate table"""
        self.__cassandra.truncatetable(keyspace, table)

    @test_step
    def validate_restoredata(self, keyspace, table, rows):
        """"validate restore data in DB"""
        origids = list(range(1, rows + 1))
        results = self.__cassandra.get_rows(keyspace, table)
        count = 0
        for result in results:
            count += 1
            if (result.id not in origids) or (
                    result.fname != 'test') or (result.lname != 'test'):
                raise CVTestStepFailure(
                    "restored data does not match with original data")
        if count == rows:
            self._admin_console.log.info("restored data match original data")
        else:
            raise CVTestStepFailure("number of rows in restored data does not match original data")

    @test_step
    def close_dbconnection(self):
        """close cassandra cluster connection"""
        self.__cassandra.close_connection()
