# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be used to run VSA
AdminConsole Automation test cases.

Class:

    AdminConsoleVirtualServer()

    VMWareAdminConsole() -> AdminConsoleVirtualServer()

    HyperVAdminConsole() -> AdminConsoleVirtualServer()

    OracleCloudAdminConsole() -> AdminConsoleVirtualServer()

    AliCloudAdminConsole() -> AdminConsoleVirtualServer()

    OracleCloudInfrastructureAdminConsole() -> AdminConsoleVirtualServer()

    GoogleCloudAdminConsole() -> AdminConsoleVirtualServer()

"""

import os
import time
import re
import socket
import zipfile

from abc import ABCMeta, abstractmethod
import ast
from past.builtins import basestring
from cvpysdk.client import Client
from cvpysdk.job import JobController
from cvpysdk.policies.storage_policies import StoragePolicies
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.webdriver.common.by import By
from Server.JobManager.jobmanager_helper import JobManager

from Web.AdminConsole.Components.panel import Backup
from Web.Common.exceptions import CVWebAutomationException
from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.Custom import viewer
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.DR.replication import ReplicationGroup
from Web.AdminConsole.DR.recovery_targets import RecoveryTargets, TargetDetails
from Web.AdminConsole.AdminConsolePages.dashboard import Dashboard
from Web.AdminConsole.VSAPages.enduser_fullvmrestore import EndUserFullVMRestore
from Web.AdminConsole.VSAPages.enduser_guestfiles_restore import EndUserGuestFilesRestoreSelectFolder
from Web.AdminConsole.VSAPages.guest_files_restore_select_folder import GuestFilesRestoreSelectFolder
from Web.AdminConsole.VSAPages.guest_files_restore_select_volume import GuestFilesRestoreSelectVolume
from Web.AdminConsole.VSAPages.manage_content import ManageContent
from Web.AdminConsole.VSAPages.virtual_machine_files_restore import VirtualMachineFilesRestore
from Web.AdminConsole.VSAPages.configure_vsa_replication_group import ConfigureVSAReplicationGroup
from Web.AdminConsole.VSAPages.disk_level_restore import DiskLevelRestore
from Web.AdminConsole.VSAPages.full_vm_restore import FullVMRestore
from Web.AdminConsole.VSAPages.hypervisor_details import HypervisorDetails
from Web.AdminConsole.VSAPages.hypervisors import Hypervisors
from Web.AdminConsole.VSAPages.live_mount import LiveMount
from Web.AdminConsole.VSAPages.replication_target_details import ReplicationTargetDetails
from Web.AdminConsole.VSAPages.replication_targets import ReplicationTargets
from Web.AdminConsole.VSAPages.select_restore import SelectRestore
from Web.AdminConsole.VSAPages.virtual_machines import VirtualMachines
from Web.AdminConsole.VSAPages.vm_details import VMDetails
from Web.AdminConsole.VSAPages.vsa_search_restore import VsaSearchRestore
from Web.AdminConsole.VSAPages.vsa_subclient_details import VsaSubclientDetails
from AutomationUtils import logger, cvhelper, constants, machine
from AutomationUtils import config
from VirtualServer.VSAUtils import VirtualServerUtils, VirtualServerConstants
from VirtualServer.VSAUtils.HypervisorHelper import Hypervisor
from VirtualServer.VSAUtils.VirtualServerHelper import AutoVSACommcell
from .LoadModule import load_module


class AdminConsoleVirtualServer:
    """ Adminconsole helper for VSA agent """

    __metaclass__ = ABCMeta

    def __new__(cls, instance, driver=None, commcell=None, csdb=None):
        """
        Initialize the object based in instance_type

        Args:
            instance    (object)   --  object of the instance class

            driver      (object)   --  the browser object

            commcell    (object)   --  an instance of the commcell class

            csdb        (object)   --  the cs DB object

        """

        hv_type = VirtualServerConstants.hypervisor_type
        if instance.instance_name.lower() == hv_type.MS_VIRTUAL_SERVER.value.lower():
            return object.__new__(HyperVAdminConsole)
        elif instance.instance_name.lower() == hv_type.VIRTUAL_CENTER.value.lower():
            return object.__new__(VMwareAdminConsole)
        elif instance.instance_name.lower() == hv_type.Oracle_Cloud_Classic.value.lower():
            return object.__new__(OracleCloudAdminConsole)
        elif instance.instance_name.lower() == hv_type.Alibaba_Cloud.value.lower():
            return object.__new__(AlibabaCloudAdminConsole)
        elif instance.instance_name.lower() == hv_type.AZURE_V2.value.lower():
            return object.__new__(AzureRMAdminConsole)
        elif instance.instance_name.lower() == hv_type.AMAZON_AWS.value.lower():
            return object.__new__(AmazonAdminConsole)
        elif instance.instance_name.lower() == hv_type.Vcloud.value.lower():
            return object.__new__(VcloudAdminConsole)
        elif instance.instance_name.lower() == hv_type.ORACLE_CLOUD_INFRASTRUCTURE.value.lower():
            return object.__new__(OracleCloudInfrastructureAdminConsole)
        elif instance.instance_name.lower() == hv_type.Google_Cloud.value.lower():
            return object.__new__(GoogleCloudAdminConsole)
        else:
            raise Exception("The given hypervisor has not been automated yet")

    def __init__(self, instance, browser=None, commcell=None, csdb=None):
        """
        A class that represents the VSA functions that can be performed for AdminConsole
            Automation

        Args:
            instance    (object)   --  the instance object

            browser      (object)   --  the browser object

            commcell    (object)   --  the commcell object

            csdb        (object)   --  the commcell database object

        """

        if not browser:
            raise Exception('Driver', -1)
        self.driver = browser.driver

        self.log = logger.get_log()
        self.csdb = csdb
        self._login_obj = None
        self.hvobj = None
        self.instance_obj = instance
        self.subclient_obj = None
        self.commcell = commcell
        self._subclient = None
        self._hypervisor = None
        self._destination_datastore = None
        self._destination_host = None
        self._instance = instance.instance_name
        self._co_ordinator = None
        self._server = None
        self._vms = None
        self._content = None
        self._restore_proxy = None
        self._restore_path = None
        self._restore_client = None
        self._agentless_vm = None
        self._destination_vm_object = None
        self.restore_destination_client = None
        self._restore_instance_type = instance.instance_name
        self._inplace_overwrite = False
        self._power_on_vm_after_restore = True
        self._overwrite_vm = False
        self._vm_restore_prefix = "DeleteMe_AC_"
        self.user_name = None
        self.password = None
        self._storage_policy = None
        self.run_aux = False
        self.aux_copy = "aux"
        self._ci_enabled = False
        self.backup_method = "Regular"
        self.snap_restore = False
        self.download_directory = None
        self.indexing_v2 = False
        self.oci_finger_print = None
        self.oci_private_key_password = None
        self.oci_region_name = None
        self.oci_tenancy_id = None
        self.oci_user_id = None
        self.recovery_target = None
        self.backup_job = None
        self.timestamp = None
        self._azure_cross_region_restore = False
        self.config = config.get_config()
        self.admin_console = AdminConsole(browser, self.commcell.webconsole_hostname)
        self.navigator = self.admin_console.navigator
        self.recovery_targets_obj = RecoveryTargets(self.admin_console)
        self.target_details_obj = TargetDetails(self.admin_console)
        self.replication_group_obj = ReplicationGroup(self.admin_console)
        self.content_obj = ManageContent(self.admin_console)
        self.restore_vol_obj = GuestFilesRestoreSelectVolume(self.admin_console)
        self.restore_files_obj = GuestFilesRestoreSelectFolder(self.admin_console)
        self.enduser_restore_files_obj = EndUserGuestFilesRestoreSelectFolder(self.admin_console)
        self.enduser_fullvm_obj = EndUserFullVMRestore(self.admin_console)
        self.vm_files_restore_obj = VirtualMachineFilesRestore(self.admin_console)
        self.full_vm_restore_obj = FullVMRestore(self.admin_console)
        self.disk_level_restore_obj = DiskLevelRestore(self.admin_console)
        self.hypervisor_ac_obj = Hypervisors(self.admin_console)
        self.vsa_sc_obj = VsaSubclientDetails(self.admin_console)
        self.hypervisor_details_obj = HypervisorDetails(self.admin_console)
        self.select_restore_obj = SelectRestore(self.admin_console)
        self.virtual_machines_obj = VirtualMachines(self.admin_console)
        self.vm_details_obj = VMDetails(self.admin_console)
        self.vsa_search_obj = VsaSearchRestore(self.admin_console)
        self.configure_vsa_replication_group_obj = ConfigureVSAReplicationGroup(self.admin_console)
        self.recovery_target_obj = ReplicationTargets(self.admin_console)
        self.recovery_target_details_obj = ReplicationTargetDetails(self.admin_console)
        self.live_mount_obj = LiveMount(self.admin_console)

        self.jobs = load_module(
            'Jobs',
            constants.AUTOMATION_DIRECTORY +
            '\\Web\\AdminConsole\\AdminConsolePages'
        )
        self.job_obj = self.jobs.Jobs(self.admin_console)

        self.replication_groups = load_module(
            'replication_groups',
            constants.AUTOMATION_DIRECTORY +
            '\\Web\\AdminConsole\\AdminConsolePages'
        )
        self.replication_group_obj = self.replication_groups.ReplicationGroup(self.driver)

        self.replication_group_details = load_module(
            'replication_group_details',
            constants.AUTOMATION_DIRECTORY +
            '\\Web\\AdminConsole\\AdminConsolePages'
        )
        self.replication_group_details_obj = self.replication_group_details.ReplicationGroupDetails(self.driver)

        self.configure_replication_group = load_module(
            'configure_replication_group',
            constants.AUTOMATION_DIRECTORY +
            '\\Web\\AdminConsole\\AdminConsolePages'
        )
        self.configure_replication_group_obj = self.configure_replication_group. \
            ConfigureReplicationGroup(self.driver)

        # backup options
        self._backup_type = Backup.BackupType.FULL
        self.granular_recovery = True
        self._testdata_path = None
        self.backup_folder_name = None
        self.cleanup_testdata_before_backup = True
        self._agentless_dict = None
        self.generate_testdata = True
        self.skip_testdata = False
        self._copy_precedence = "default"
        self.controller_machine = machine.Machine(socket.gethostname(), commcell)

    class VmValidation:
        """
        Validates the source and restore VMs
        """

        def __init__(self, vmobj):
            self.vm = vmobj

        def __eq__(self, other):
            """compares the source vm and restored vm"""
            self.log = logger.get_log()
            self.log.info("Source VM:{0}::\nCPU:{1}\nDisk count:{2}\nMemory:{3}".format(
                self.vm.vm_name, self.vm.no_of_cpu, self.vm.disk_count, self.vm.memory
            ))
            self.log.info("Destination VM:{0}::\nCPU:{1}\nDisk count:{2}\nMemory:{3}".format(
                other.vm.vm_name, other.vm.no_of_cpu, other.vm.disk_count, other.vm.memory
            ))
            return (self.vm.no_of_cpu == other.vm.no_of_cpu and
                    self.vm.disk_count == other.vm.disk_count and
                    self.vm.memory == other.vm.memory)

    @property
    def azure_cross_region_restore(self):
        """
        Returns if cross region restore or not
        Returns:
            true/false     (boolenan):  value of _cross_region_restore

        """
        return self._azure_cross_region_restore

    @azure_cross_region_restore.setter
    def azure_cross_region_restore(self, value):
        """
        Sets the value of  cross region restore fro Azure

        Args:
            true/false   (boolean): value for cross region restore
        """
        self._azure_cross_region_restore = value
        self.skip_testdata = value
        if value:
            self.generate_testdata = False
        else:
            self.generate_testdata = True

    @property
    def copy_precedence(self):
        """
        Returns the copy precedence while restore

        Returns:
            _copy_precedence      (basestring)  --  'primary' for backup copy

        """
        return self._copy_precedence

    @copy_precedence.setter
    def copy_precedence(self, value):
        """
        Sets the copy precedence during restore

        Args:
            value   (basestring)  --  'primary' for backup copy

        """
        self._copy_precedence = value

    @property
    def backup_type(self):
        """
        Returns the backup type

        Returns:
            _backup_type    (backupType enum)   --  the backup level in Backup.BackupType

        """
        return self._backup_type

    @backup_type.setter
    def backup_type(self, value):
        """
        Sets the backup level of the subclient

        Args:
            value    (basestring)   --  the level of backup to be run on the subclient

        """
        try:
            for data in Backup.BackupType:
                if data.name == value.upper():
                    self._backup_type = data
                    break
        except Exception:
            type_list = [data.name for data in Backup.BackupType]
            raise CVWebAutomationException(f"backup type : {value}, isn't "
                                           f"among the types in [{type_list}]")
        self.backup_folder_name = value

    @property
    def collect_metadata(self):
        """
        Returns the metadata collection property

        Returns:
            granular_recovery   (bool)   --   true / false for collecting metadata

        """
        return self.granular_recovery

    @collect_metadata.setter
    def collect_metadata(self, value):
        """
        Sets the metadata collection property

        Args:
            value    (bool)  --  True / False for collecting metadata during backup

        """
        self.granular_recovery = value

    @property
    def testdata_path(self):
        """
        Returns the testdata path

        Returns:
            testdata_path  (basestring)   --  the location where testdata needs to be copied

        """
        return self._testdata_path

    @testdata_path.setter
    def testdata_path(self, value):
        """
        Sets the testdata path where files needs to be copied before backup

        Args:
            value    (basestring)   --  the location for copying test data

        """
        self._testdata_path = value

    @property
    def subclient(self):
        """
        Returns the subclient name

        Returns:
            subclient  (basestring)   --  the name of the subclient to be backed up or restored

        """
        return self._subclient

    @subclient.setter
    def subclient(self, value):
        """
        Sets the subclient name

        Args:
            value    (basestring)   --  the name of the subclient to be backed up or restored

        """
        self._subclient = value

    @property
    def hypervisor(self):
        """
        Returns the hypervisor name

        Returns:
            hypervisor  (basestring)   --  the name of the hypervisor

        """
        return self._hypervisor

    @hypervisor.setter
    def hypervisor(self, value):
        """
        Sets the hypervisor name

        Args:
            value   (basestring)   --  the name of the hypervisor

        """
        self._hypervisor = value

    @property
    def destination_datastore(self):
        """
        Returns the datastore name

        Returns:
             datastore (basestring) --  the name of the datastore
        """
        return self._destination_datastore

    @destination_datastore.setter
    def destination_datastore(self, value):
        """
        Sets the datastore name

        Args:
             value (basestring) --  the name of the hypervisor
        """
        self._destination_datastore = value

    @property
    def destination_host(self):
        """
        Returns the destination host ip

        Returns:
             destination_host destination_host (basestring) --  the ip of the destination host
        """
        return self._destination_host

    @destination_host.setter
    def destination_host(self, value):
        """
        Sets the destinnation host ip

        Args:
             value (basestring) --  the ip of the destination host
        """
        self._destination_host = value

    @property
    def restore_proxy(self):
        """
        Returns the restore proxy name

        Returns:
            restore_proxy  (basestring)   --  the name of the restore proxy

        """
        return self._restore_proxy.machine_name

    @restore_proxy.setter
    def restore_proxy(self, value):
        """
        Sets the restore proxy value

        Args:
            value   (basestring / obj / tuple)  -- the restore proxy value to be set

        """
        if isinstance(value, basestring):
            self._restore_proxy = machine.Machine(value)
        elif isinstance(value, machine.Machine):
            self._restore_proxy = value
        elif isinstance(value, tuple):
            self.restore_destination_client = value[0]
            self._restore_proxy = machine.Machine(value[1])
        else:
            raise Exception("Please pass the correct type for the restore proxy value")
        if self._restore_proxy.os_info.lower() == "windows":
            _storage = self._restore_proxy.get_storage_details()
            for drive in _storage:
                if isinstance(_storage[drive], dict):
                    self.restore_path = drive + ":\\"
                    break
        else:
            self.restore_path = '/tmp'

    @property
    def restore_client(self):
        """
        Returns the restore client name

        Returns:
            restore_client  (basestring)   --  the name of the restore client

        """
        return self._restore_client

    @restore_client.setter
    def restore_client(self, value):
        """
        Sets the restore client value

        Args:
            value   (basestring)   --   the name of the restore client

        """
        self._restore_client = value

    @property
    def restore_path(self):
        """
        Returns the restore path location

        Returns:
            restore_path  (basestring)   --  the location to restore to

        """
        return self._restore_path

    @restore_path.setter
    def restore_path(self, value):
        """
        Sets the restore path

        Args:
            value   (basestring)   --   the path where files are to be restored

        """
        self._restore_path = value

    @property
    def agentless_vm(self):
        """
        Returns the agentless name

        Returns:
            agentless_vm  (basestring)   --  the name of the agentless VM where files needs to be
                                           restored to

        """
        return self._agentless_vm

    @agentless_vm.setter
    def agentless_vm(self, value):
        """
        Sets the agentless VM where the backed up files are to be restored during agentless restore

        Args:
            value   (basestring)   --  the name of the agentless VM

        Raises:
            Exception:
                if the type of the input variable is not a basestring

        """
        if isinstance(value, str):
            self._agentless_vm = value
            if self.restore_client is None:
                self.restore_client = self.hypervisor

            if self.restore_client == self.hypervisor:
                self.restore_destination_client = self.hvobj
            else:
                self.restore_destination_client, \
                self._restore_instance_type = self._create_hypervisor_object(
                    self.restore_client
                )

            if value not in self._vms:
                self.restore_destination_client.VMs = self._agentless_vm
                self.restore_destination_client.VMs[value].update_vm_info('All', True, True)

        else:
            raise Exception("Please pass the correct type for the agentless vm variable")

    @property
    def instance(self):
        """
        Returns the instance name

        Returns:
            instance  (basestring)   --  the name of the instance

        """
        return self._instance

    @instance.setter
    def instance(self, value):
        """
        Sets the instance name

        Args:
            value   (basestring)   -- the name of the instance

        """
        self._instance = value

    @property
    def co_ordinator(self):
        """
        Returns the coordinator name

        Returns:
            co_ordinator  (basestring)   --  the name of the first proxy in the instance

        """
        return self._co_ordinator.machine_name

    @co_ordinator.setter
    def co_ordinator(self, value):
        """
        Sets the co-ordinator of the client

        Args:
            value   (basestring)   --   the name of the co-ordinator proxy client

        """
        self._co_ordinator = machine.Machine(value, self.commcell)

    @property
    def server(self):
        """
        Returns the server name

        Returns:
            server  (basestring)   --  the name of the server where commands will be executed

        """
        return self._server

    @server.setter
    def server(self, value):
        """
        Sets the server name

        Args:
            value   (basestring)    --  sets the name of the server

        """
        self._server = value

    @property
    def content(self):
        """
        Returns the content

        Returns:
            VMs  (basestring)   --  the list of VMs to be backed up and restored

        """
        return self._vms

    @content.setter
    def content(self, value):
        """
        Sets the content of the subclient

        Args:
            value   (basestring)   --  like    [VM]=startswith=test*,test1
                                        [VM] - represent type
                                                        like [DNS],[HN]
                                        startswith  represent equality in admin console
                                                like endswith,equals
                                        test* - include all VM starts with test
                                               adding dynamic content in admin console
                                        , - to include multiple content
                                        test1 -  non-dynamic content

        """
        self._content = value
        self.set_subclient_content()

    @property
    def power_on_after_restore(self):
        """
        Returns the power on after restore variable

        Returns:
            power_on_vm_after_restore  (bool)   --  True / False to power on the VM
                                                          after restore

        """
        return self._power_on_vm_after_restore

    @power_on_after_restore.setter
    def power_on_after_restore(self, value):
        """
        Sets the power on after restore variable

        Args:
            value   (bool)  -- True / False to power on the VM after restore

        """
        self._power_on_vm_after_restore = value

    @property
    def unconditional_overwrite(self):
        """
        Returns the unconditional overwrite variable

        Returns:
            overwrite_vm  (bool)   --  True / False to overwrite the VM during restore

        """
        return self._overwrite_vm

    @unconditional_overwrite.setter
    def unconditional_overwrite(self, value):
        """
        Sets the unconditional overwrite variable

        Args:
            value   (bool)  --  True / False to overwrite the VM during restore

        """
        self._overwrite_vm = value

    @property
    def vm_restore_prefix(self):
        """
        Returns the prefix to be attached to the restore VM name

        Returns:
            vm_restore_prefix  (basestring)   --  the prefix to tbe attached to the restore VM name

        """
        return self._vm_restore_prefix

    @vm_restore_prefix.setter
    def vm_restore_prefix(self, value):
        """
        Sets the prefix to be attached to the restore VM name

        Args:
            value   (basestring)    --  the prefix to be attached to the restore VM name

        """
        self._vm_restore_prefix = value

    @property
    def full_vm_in_place(self):
        """
        Returns the full VM in place variable

        Returns:
            inplace_overwrite  (bool)   --  True / False to do an inplace restore

        """
        return self._inplace_overwrite

    @full_vm_in_place.setter
    def full_vm_in_place(self, value):
        """
        Sets the full VM inplace variable

        Args:
            value   (bool)  --  True / False to do an inplace restore

        """
        self._inplace_overwrite = value

    @property
    def storage_policy(self):
        """
        Returns the storage policy sdk object

        Returns:
            storage_policy  (object):   the object of the storage policy class

        """
        return self._storage_policy

    @storage_policy.setter
    def storage_policy(self, value):
        """
        Sets the storage policy value

        Args:
            value   (basestring):   name of the storage policy

        Returns:
            None

        """
        if not isinstance(value, basestring):
            raise Exception("The storage policy name should be a string")
        policies = StoragePolicies(self.commcell)
        self._storage_policy = policies.get(value)

    @property
    def ci_enabled(self):
        """
        Returns the ci flag whether it is enabled or disabled

        Returns:
            ci_enabled  (bool): whether content indexing is enabled or not

        """
        return self._ci_enabled

    @ci_enabled.setter
    def ci_enabled(self, value):
        """
        enables / disables content indexing

        Args:
            value   (bool):     enable / disable content indexing

        """
        if not isinstance(value, bool):
            raise Exception("The content indexing value should be True / False")
        self._ci_enabled = value

    @property
    def vms(self):
        """
        Returns the list of all VMs in the subclient content

        Returns:
            vms (list): list of all VMs in subclient content

        """
        return self._vms

    def _set_agentless_dict(self, agentless_dict):
        """
        Sets the agentless VM where the backed up files are to be restored during agentless restore for individual VMs

        Args:
            agentless_dict   (dict)   -- VMname :  the name of the agentless VM

        Raises:
            Exception:
                iffails to set agentless VM

        """
        self._agentless_dict = agentless_dict
        self.agentless_vm = self._agentless_dict[next(iter(self._agentless_dict))]

    def __deepcopy__(self, temp_obj):
        """
        over ride deepcopy method to copy every attribute of an object to other

        Args:
            temp_obj    (object)   --  the object to be copied

        Raises:
            Exception:
                if deep copy fails to set the object

        """
        try:
            cls = temp_obj.__class__
            result = cls.__new__(cls, temp_obj, temp_obj.vm_name)
            for key, value in temp_obj.__dict__.items():
                setattr(result, key, value)
            return result

        except Exception as exp:
            self.log.exception("Failed to deepcopy Exception: %s", str(exp))
            raise exp

    def _create_hypervisor_object(self, client_name=None):
        """
        Create Hypervisor Object

        Args:
            client_name  (basestring)   --  the name of the client to create the hypervisor
                                                object for
                    default :   None

        Raises:
            Exception:
                if initialization fails in creating object

        Returns:
            hvobj   (object)   --  the object of the hypervisor class of the corresponding instance

            instance_name   (basestring)    --  the instance name

        """
        try:
            if client_name is None:
                client_name = self.hypervisor
                instance = self.instance_obj
            else:
                client = self.commcell.clients.get(client_name)
                agent = client.agents.get('Virtual Server')
                instance_keys = next(iter(agent.instances._instances))
                instance = agent.instances.get(instance_keys)

            host_machine = instance.co_ordinator
            host_machine1 = socket.gethostbyname_ex(socket.gethostname())[2][0]
            server_host_name = instance.server_host_name

            self._compute_credentials(client_name)
            if self._instance == 'azure resource manager':
                hvobj = Hypervisor(server_host_name, self.user_name,
                                   self.password1, instance.instance_name, self.commcell, host_machine1)
            elif self._instance == 'Amazon':
                hvobj = Hypervisor(server_host_name, self.user_name,
                                   self.password2, instance.instance_name,
                                   self.commcell, host_machine1)
            elif self._instance == "Oracle Cloud Infrastructure":
                self.server_host_name = server_host_name
                oci_dict = {'oci_tenancy_id': self.oci_tenancy_id,
                            'oci_user_id': self.oci_user_id,
                            'oci_finger_print': self.oci_finger_print,
                            'oci_private_file_name': eval('self.config.Virtualization.oci.key_file_path'),
                            'oci_region_name': self.oci_region_name,
                            'oci_private_key_password': self.oci_private_key_password
                            }
                server_host_name = ["None"]
                hvobj = Hypervisor(server_host_name, oci_dict, '',
                                   self._instance.lower(), self.commcell, host_machine1)

            else:
                hvobj = Hypervisor(server_host_name, self.user_name,
                                   self.password, instance.instance_name, self.commcell, host_machine)

            return hvobj, instance.instance_name

        except Exception as exp:
            self.log.exception(
                "An Exception occurred in creating the Hypervisor object  %s", str(exp))
            raise exp

    def _compute_credentials(self, client_name):
        """Compute the credentials required to call the server

        Args:
            client_name (basestring)   --   the name of the client whose credentials are
                                            to be obtained

        Raises:
            Exception:
                if compute credentials fails to get the credentials from the database


        """
        try:
            _query = "select ac.*from (select Row_Number() Over ( Order By id ) as num, \
                     * from app_Instanceprop where componentNameid =( \
                     select TOP 1 instance  from APP_Application where clientId= ( \
                     Select TOP 1 id from App_Client where name = '%s') and appTypeId = '106'))ac \
                     where ac.num between 6 and 9" % client_name

            # """
            self.csdb.execute(_query)
            _results = self.csdb.fetch_all_rows()
            if not _results:
                raise Exception("An exception occurred getting server details")

            self.user_name = _results[0][5].strip()
            _password = _results[1][5].strip()
            id1 = _results[2][5].strip()
            id2 = _results[3][5].strip()

            _aws_access_key = cvhelper.format_string(self.commcell,
                                                     _results[0][5].strip())
            _aws_secret_key = cvhelper.format_string(self.commcell,
                                                     _results[2][5].strip())

            self.password = cvhelper.format_string(self.commcell, _password)
            self.password1 = (self.password, id1, id2)
            self.password2 = (_aws_access_key, _aws_secret_key)

        except Exception as err:
            self.log.exception(
                "An Exception occurred in getting credentials for Compute Credentials  %s", err)
            raise err

    @staticmethod
    def _get_disk_list(disk_list=None):
        """
        Gets the names of the disks to be restored

        Args:
            disk_list    (list)  --  the list containing all the disks to be restored

        Returns:
            final_disk_list     (list)  --  the list of disk names to be restored

        """
        try:
            final_disk_list = []
            if not disk_list:
                raise Exception("The disk list is empty. Please check the logs.")
            for disk in disk_list:
                final_disk_list.append(disk.split("\\")[-1])
            return final_disk_list
        except Exception as exp:
            raise exp

    def _navigate_to_vmgroup(self):
        """
        Navigates to VM group details page to submit backup
        """
        self.navigator.navigate_to_hypervisors()
        self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
        self.hypervisor_details_obj.open_subclient(self.subclient)

    def _navigate_to_vm_restore(self, vm_name):
        """
        Navigates to the individual VM page to trigger restore from VM level

        Args:
            vm_name     (basestring):   the name of the VM to restore

        Raises:
            Exception:
                if the restore page from VM level could not be opened

        """
        self.navigator.navigate_to_virtual_machines()
        self.virtual_machines_obj.open_vm(vm_name)
        self.vm_details_obj.vm_restore()

    def _navigate_to_active_mounts(self, vm_name):
        """
        Navigates to the individual VM to access active mounts from VM level

        Args:
            vm_name     (basestring):   the name of the VM

        Raises:
            Exception:
                if the Active Mounts page from VM level could not be opened

        """
        self.navigator.navigate_to_virtual_machines()
        self.virtual_machines_obj.action_list_mounts(vm_name)

    def delete_live_mount(self, vm_name):
        """
            Navigates to the individual VM to delete active mounts from VM level

            Args:
                vm_name     (basestring):   the name of the VM

            Raises:
                Exception:
                    if the Active Mounts page from VM level could not be opened

        """
        self._navigate_to_active_mounts(vm_name)
        return self.virtual_machines_obj.delete_active_live_mount(vm_name)

    def logout_command_center(self):
        """
        Logs out of the command center session
        """
        try:
            elements = self.driver.find_elements_by_xpath("//a[@class='modal__close-btn']")
            for element in elements:
                element.click()
                self.admin_console.wait_for_completion()
            self.navigator.logout()
        except Exception as exp:
            self.log.error("Could not logout of command center")
            raise exp

    def get_all_vms(self):
        """
        Get the list of all VMs from the subclient content

        Raises:
            Exception:
                if it fails to get the list of all VMs in the subclient from adminconsole

        """
        try:
            self.navigator.navigate_to_hypervisors()

            self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
            if self.subclient_obj:
                proxies = self.subclient_obj._get_subclient_proxies()
                if proxies:
                    self.co_ordinator = proxies[0]

            if not self._co_ordinator:
                proxy_list = self.hypervisor_details_obj.proxy_info()
                if proxy_list != self.instance_obj.associated_clients:
                    raise Exception("The proxy list does not match.")
                self.co_ordinator = proxy_list[0]

            self.log.info("Coordinator is %s", self._co_ordinator)
            if self.subclient_obj:
                proxy_list = self.subclient_obj._get_subclient_proxies()
                if proxy_list:
                    self.co_ordinator = proxy_list[0]

            if not self._co_ordinator:
                proxy_list = self.hypervisor_details_obj.proxy_info()
                if proxy_list != self.instance_obj.associated_clients:
                    raise Exception("The proxy list does not match.")

                self.co_ordinator = proxy_list[0]

            self.log.info("Coordinator is %s", self.co_ordinator)

            self.hypervisor_details_obj.open_subclient(self.subclient)

            self.vsa_sc_obj.manage_content()

            self._vms = self.content_obj.preview().keys()

            self.storage_policy = self.vsa_sc_obj.get_storage_policy()
        except Exception as exp:
            self.log.exception("Exception while getting all the VMs "
                               "in the subclient content: %s", str(exp))
            raise exp

    def set_subclient_content(self):
        """
        Set the content for the given subclient

        Raises:
            Exception:
                if it fails to set the subclient content via admin console

        """
        try:
            self.log.info("Setting the subclient content")

            self.navigator.navigate_to_hypervisors()
            self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
            self.hypervisor_details_obj.open_subclient(self.subclient)
            self.vsa_sc_obj.manage_content()

            content = ast.literal_eval(self._content)
            if "vm" in content.keys():
                self.content_obj.add_vm(content["vm"])
            if "rule" in content.keys():
                rules_list = content["rule"]
                for rule in rules_list:
                    rule_type, expression, name = rule.split(":")
                    self.content_obj.add_rule(rule_type, expression, name)

        except Exception as exp:
            self.log.exception("Exception while setting subclient content: %s", str(exp))
            raise exp

    def vsa_discovery(self):
        """
        VSA discovery to copy all test data

        Raises:
            Exception:
                if the discovery fails

        """
        try:
            VirtualServerUtils.decorative_log("Creating Hypervisor and VM objects")

            self.get_all_vms()

            self.hvobj = self._create_hypervisor_object()[0]
            for each_vm in self._vms:
                if not hasattr(self.hvobj.VMs, each_vm):
                    self.hvobj.VMs = each_vm

            if not self.azure_cross_region_restore:
                if not self.testdata_path:
                    self.testdata_path = VirtualServerUtils.get_testdata_path(
                        self.hvobj.machine)
                    self.timestamp = self.testdata_path.rpartition('\\')[-1]

                if self.cleanup_testdata_before_backup:
                    self.cleanup_testdata()

                if self.generate_testdata:
                    generate = self.controller_machine.generate_test_data(self.testdata_path)
                    self.log.info("Generating TestData at path".format(self.testdata_path))
                    if not generate:
                        raise Exception(generate)

            for _vm in self._vms:

                self.log.info("Creating object for VM %s", _vm)
                self.hvobj.VMs[_vm].update_vm_info('All', True, True)

                if not self.azure_cross_region_restore and not self.skip_testdata:
                    for _drive in self.hvobj.VMs[_vm].drive_list.values():
                        self.log.info("Copying Testdata to Drive %s", _drive)
                        self.hvobj.copy_test_data_to_each_volume(
                            _vm, _drive, self.backup_folder_name, self.testdata_path)
                    if self.ci_enabled:
                        _drive = next(iter(self.hvobj.VMs[_vm].drive_list.values()))
                        self.hvobj.copy_content_indexing_data(_vm, _drive, self.backup_folder_name)

        except Exception as exp:
            self.log.exception("Exception while copying test data: %s", str(exp))
            raise exp

    def backup(self):
        """
        Run backup for the given subclient from admin console

        Raises:
            Exception:
               If backup job does not complete successfully

        """
        try:
            self.vsa_discovery()

            VirtualServerUtils.decorative_log("Running backup for the subclient")

            if self.backup_type == Backup.BackupType.SYNTH:
                _bc_types = [Backup.BackupType.INCR, Backup.BackupType.SYNTH]
            else:
                _bc_types = [self.backup_type]

            for _bc_type in _bc_types:
                if _bc_type == Backup.BackupType.SYNTH:
                    self._navigate_to_vmgroup()
                _backup_jobs = self.vsa_sc_obj.backup_now(_bc_type)
                if not isinstance(_backup_jobs, list):
                    _backup_jobs = [_backup_jobs]
                for _backup_job in _backup_jobs:
                    self.backup_job = _backup_job
                    job_details = self.get_job_status(self.backup_job)

            if self.backup_type.value == Backup.BackupType.SYNTH and self.backup_method == "Regular":
                if job_details['Backup type'].lower() == "synthetic full":
                    self.indexing_v2 = True
                else:
                    self.verify_synth_full_backup()

            if self.run_aux and len(self.storage_policy.copies) > 1:
                if self.storage_policy.aux_copies:
                    self.aux_copy = self.storage_policy.aux_copies[0]
                    aux_job = self.storage_policy.run_aux_copy(self.aux_copy)
                    aux_job = aux_job.job_id
                    job_details = self.get_job_status(aux_job)

            if self.backup_method.lower() == "snap":
                bkpcopy_job = self.storage_policy.run_backup_copy()
                bkpcopy_job = bkpcopy_job.job_id
                job_details = self.get_job_status(bkpcopy_job)

            if self.ci_enabled:
                self.navigator.navigate_to_hypervisors()
                self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
                self.hypervisor_details_obj.open_subclient(self.subclient)
                ci_job = self.vsa_sc_obj.run_file_indexing()
                job_details = self.get_job_status(ci_job)

        except Exception as exp:
            self.log.exception("Exception while submitting backup: %s", str(exp))
            raise exp

    def get_job_status_from_jobspage(self, job_id):
        """
        Get the status of the given job ID

        Args:
            job_id   (basestring)   --  the job id whose status should be collected

        """
        try:
            self.log.info("Getting the status for the job %s", job_id)
            return self.job_obj.job_completion(job_id)
        except Exception as exp:
            self.log.exception("Exception occurred in getting the job status: %s", str(exp))
            raise exp

    def verify_synth_full_backup(self):
        """
        Verify whether synth full job is triggered after incremental backup

        Raises:
            Exception:
                if it fails to obtain the synth full job ID

        """
        try:
            self.log.info("Getting the synth full job ID")
            _query = "Select jobId from JMBkpJobInfo where bkpLevel=64 and applicationId=(" \
                     "select id from App_Application where subclientName='{0}' and clientId=(" \
                     "select id from App_Client where displayName='{1}'))".format(self.subclient,
                                                                                  self.hypervisor)

            self.csdb.execute(_query)
            job_id = str(self.csdb.fetch_one_row()[0])

            job_details = self.get_job_status(job_id)
            if job_details['Status'] not in ['Completed',
                                             'Completed w/ one or more errors']:
                raise Exception("Exception occurred during synth full job."
                                "Please check the logs for more details.")
        except Exception as exp:
            self.log.exception("Exception occurred while getting the synth full job ID:"
                               " %s", str(exp))
            raise exp

    def fs_testdata_validation(self, dest_client, dest_path):
        """
        Validates the testdata restored

        Args:
            dest_client      (object)   --  the machine class object of the client where
                                            the testdata was restored to

            dest_path        (basestring)   --  the path in the destination client where the
                                                testdata was restored to

        Raises:
            Exception:
                if test data validation fails

        """
        try:
            self.log.info("Validating the testdata")
            if self.indexing_v2 and self.backup_type.value == Backup.BackupType.SYNTH:
                self.log.info("Skipping test data validation since there is no backup")
                return

            difference = self.controller_machine.compare_folders(dest_client,
                                                                 self.testdata_path,
                                                                 dest_path)
            if difference:
                self.log.exception("Checksum mismatch failed for source and destination. %s",
                                   difference)
                raise Exception("Folder comparison failed for source {0} and destination "
                                "{1}".format(self.testdata_path, dest_path))

            self.log.info("Validation completed successfully")
        except Exception as exp:
            self.log.exception("Exception occurred during testdata validation. %s", str(exp))
            raise exp

    def file_level_restore(self, vm_level=False):
        """
        File level restore to guest agent in the commcell

        Args:
            vm_level    (bool):     Performs file level restore to guest agent from VM level

        Raises:
            Exception:
                if the file level restore fails

        """
        try:
            self.log.info("*" * 10 + "Performing file level restore to guest agent" + "*" * 10)

            if self._restore_proxy is None:
                self.restore_proxy = self._co_ordinator

            for _vm in self._vms:
                for _drive, _folder in self.hvobj.VMs[_vm].drive_list.items():
                    restore_path = self._restore_proxy.join_path(
                        self.restore_path, "AdminConsole", _vm, _drive)
                    validate_path = self._restore_proxy.join_path(
                        restore_path, self.backup_type.name, "TestData", self.timestamp)

                    if self._restore_proxy.check_directory_exists(restore_path):
                        self._restore_proxy.remove_directory(restore_path)

                    if self.ci_enabled:
                        self.navigator.navigate_to_virtual_machines()
                        self.virtual_machines_obj.open_vm(_vm)
                        self.vm_details_obj.vm_search_content(self.backup_folder_name)
                        self.vsa_search_obj.search_for_content(
                            file_name=self.backup_folder_name, contains=_drive,
                            include_folders=True)
                        self.vsa_search_obj.select_data_type("Folder")
                        self.vsa_search_obj.validate_contains(self.backup_folder_name,
                                                              name=True)
                        self.vsa_search_obj.validate_contains(_drive, name=False,
                                                              folder=True)
                    else:
                        if not vm_level:
                            self.navigator.navigate_to_hypervisors()
                            self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
                            self.hypervisor_details_obj.open_subclient(self.subclient)
                            self.vsa_sc_obj.restore()
                        else:
                            self._navigate_to_vm_restore(_vm)

                        if self.backup_method.lower() == "snap" and self.snap_restore:
                            self.select_restore_obj.select_source_copy(
                                self.storage_policy.snap_copy
                            )
                        elif self.run_aux:
                            self.select_restore_obj.select_source_copy(self.aux_copy)
                        elif self.backup_method.lower() == "snap" and not self.snap_restore:
                            self.select_restore_obj.select_source_copy("primary")
                        self.select_restore_obj.select_guest_files()
                        self.select_restore_obj.latest_backups()
                        if _folder == "/":
                            self.restore_vol_obj.select_volume(_vm, _folder)
                        else:
                            self.restore_vol_obj.select_volume(_vm, _drive)
                    restore_job = self.restore_files_obj.submit_guest_agent_restore(
                        [self.backup_type.name], restore_path, self.restore_proxy
                    )
                    job_details = self.get_job_status(restore_job)
                    if not job_details:
                        raise Exception("Restore job failed. Please check the logs")

                    self.fs_testdata_validation(self._restore_proxy, validate_path)

        except Exception as exp:
            self.log.exception("Exception occurred during file level restore. %s", str(exp))
            raise exp

    def guest_files_restore(self, in_place=False, vm_level=False, end_user=False):
        """
        File level restore to source VM

        Args:
            in_place    (bool):     if the files should be restored to the source path
            vm_level    (bool):     if the restore should be initiated from VM level
            end_user    (bool):     if the restore is initiated by end user

        Raises:
            Exception:
                if the file level restore fails

        """
        try:
            if self._restore_proxy is None:
                self.restore_proxy = self._co_ordinator

            for _vm in self._vms:
                self.agentless_vm = _vm

                for _drive, _folder in self.hvobj.VMs[_vm].drive_list.items():
                    if in_place:
                        restore_path = _folder
                        validate_path = self.hvobj.VMs[self.agentless_vm].machine.join_path(
                            restore_path, self.backup_type.name, "TestData", self.timestamp)
                    else:
                        restore_path = self.hvobj.VMs[self.agentless_vm].machine.join_path(
                            _folder, "AdminConsole", self.agentless_vm, _drive)
                        validate_path = self.hvobj.VMs[self.agentless_vm].machine.join_path(
                            restore_path, self.backup_type.name, "TestData", self.timestamp)

                    if self.hvobj.VMs[_vm].machine.check_directory_exists(restore_path):
                        if restore_path != _folder:
                            self.hvobj.VMs[_vm].machine.remove_directory(restore_path)

                    if self.ci_enabled:
                        self.navigator.navigate_to_virtual_machines()
                        self.virtual_machines_obj.open_vm(_vm)
                        self.vm_details_obj.vm_search_content(self.backup_folder_name)
                        self.vsa_search_obj.search_for_content(
                            file_name=self.backup_folder_name, contains=_drive,
                            include_folders=True)
                        self.vsa_search_obj.select_data_type("Folder")
                        self.vsa_search_obj.validate_contains(self.backup_folder_name,
                                                              name=True)
                        self.vsa_search_obj.validate_contains(_drive, name=False,
                                                              folder=True)
                    else:
                        if not vm_level:
                            self.navigator.navigate_to_hypervisors()
                            self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
                            self.hypervisor_details_obj.open_subclient(self.subclient)
                            self.vsa_sc_obj.restore()
                        else:
                            self._navigate_to_vm_restore(_vm)
                        if self.run_aux:
                            self.select_restore_obj.select_source_copy(self.aux_copy)
                        elif self.backup_method.lower() == "snap" and self.snap_restore:
                            self.select_restore_obj.select_source_copy(
                                str(self.storage_policy.snap_copy).title()
                            )
                        elif self.backup_method.lower() == "snap" and not self.snap_restore:
                            self.select_restore_obj.select_source_copy("primary")
                        self.select_restore_obj.select_guest_files()
                        self.select_restore_obj.latest_backups()
                        if _folder == "/":
                            self.restore_vol_obj.select_volume(_vm, _folder)
                        else:
                            self.restore_vol_obj.select_volume(_vm, _drive)

                    if not end_user:
                        restore_job = self.restore_files_obj.submit_this_vm_restore(
                            [self.backup_type.name], self.agentless_vm,
                            self.restore_proxy,
                            self.hvobj.VMs[_vm].machine.username,
                            self.hvobj.VMs[_vm].machine.password,
                            restore_path
                        )
                    else:
                        restore_job = self.enduser_restore_files_obj.enduser_files_restore(
                            [self.backup_type.name], self.agentless_vm,
                            self.hvobj.VMs[_vm].machine.username,
                            self.hvobj.VMs[_vm].machine.password,
                            restore_path
                        )

                    job_details = self.get_job_status(restore_job)
                    if not job_details:
                        raise Exception("Agentless restore job failed. Please check the logs")

                    self.fs_testdata_validation(self.hvobj.VMs[_vm].machine, validate_path)
        except Exception as exp:
            self.log.exception("Exception occurred during agentless restore. %s", str(exp))
            raise exp

    def agentless_restore(self, vm_level=False):
        """
        File level restore to VM in server which does not have Content Store installed

        Args:
            vm_level    (bool):     if the restore should be initiated from VM level

        Raises:
            Exception:
                if the restore to a VM without content store fails

        """
        try:
            if self._restore_proxy is None:
                self.restore_proxy = self._co_ordinator

            for _vm in self._vms:
                if not self._agentless_dict:
                    self._agentless_dict = {_vm: self._agentless_vm}
                elif not self._agentless_dict.get(_vm, None):
                    self._agentless_dict[_vm] = self._agentless_vm

                self._destination_vm_object = self.restore_destination_client.VMs[
                    self._agentless_dict[_vm]]
                destination_drive = list(self._destination_vm_object.drive_list.keys())[0]
                destination_folder = self._destination_vm_object.drive_list[destination_drive]

                for _drive, _folder in self.hvobj.VMs[_vm].drive_list.items():
                    if self.hvobj.VMs[_vm].guest_os.lower() != 'windows':
                        _drive_for_validation = _drive.strip().split("/")
                        _drive_for_validation = (self._destination_vm_object.machine.join_path
                                                 (*_drive_for_validation)).strip("\\")
                    else:
                        _drive_for_validation = _drive
                    restore_path = self._destination_vm_object.machine.join_path(
                        destination_folder, "AdminConsole", _vm, _drive_for_validation)

                    validate_path = self._destination_vm_object.machine.join_path(
                        restore_path, self.backup_type.name, "TestData", self.timestamp)

                    if self._destination_vm_object.machine.check_directory_exists(
                            restore_path):
                        self._destination_vm_object.machine.remove_directory(restore_path)

                    if self.ci_enabled:
                        self.navigator.navigate_to_virtual_machines()
                        self.virtual_machines_obj.open_vm(_vm)
                        self.vm_details_obj.vm_search_content(self.backup_folder_name)
                        self.vsa_search_obj.search_for_content(
                            file_name=self.backup_folder_name, contains=_drive,
                            include_folders=True)
                        self.vsa_search_obj.select_data_type("Folder")
                        self.vsa_search_obj.validate_contains(self.backup_folder_name,
                                                              name=True)
                        self.vsa_search_obj.validate_contains(_drive, name=False,
                                                              folder=True)
                    else:
                        if not vm_level:
                            self.navigator.navigate_to_hypervisors()
                            self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
                            self.hypervisor_details_obj.open_subclient(self.subclient)
                            self.vsa_sc_obj.restore()
                        else:
                            self._navigate_to_vm_restore(_vm)
                        if self.run_aux:
                            self.select_restore_obj.select_source_copy(self.aux_copy)
                        elif self.backup_method.lower() == "snap" and self.snap_restore:
                            self.select_restore_obj.select_source_copy(
                                self.storage_policy.snap_copy
                            )
                        elif self.backup_method.lower() == "snap" and not self.snap_restore:
                            self.select_restore_obj.select_source_copy("primary")
                        self.select_restore_obj.select_guest_files()
                        self.select_restore_obj.latest_backups()
                        if _folder == "/":
                            self.restore_vol_obj.select_volume(_vm, _folder)
                        else:
                            self.restore_vol_obj.select_volume(_vm, _drive)

                    restore_job = self.restore_files_obj.submit_other_vm_restore(
                        [self.backup_type.name], self.restore_proxy,
                        self.restore_client, self._agentless_dict[_vm],
                        self._destination_vm_object.machine.username,
                        self._destination_vm_object.machine.password,
                        restore_path
                    )

                    job_details = self.get_job_status(restore_job)

                    self.fs_testdata_validation(self._destination_vm_object.machine, validate_path)
        except Exception as exp:
            self.log.exception("Exception occurred during agentless restore. %s", str(exp))
            raise exp

    def disk_validation(self, vm_obj, disk_restore_destination):
        """
        Performs Disk Validation by mounting the restored disk on the Host

        Args:

            vm_obj                      (object)       --  instance of the VM class

            disk_restore_destination    (basestring)   --   restore path of all the disk

        Raises:
            Exception:
                if validation fails

        """
        try:
            self.log.info("Performing disk validation")
            _drive_letter = None

            if self.restore_client != self.hypervisor:
                self.restore_destination_client = self._create_hypervisor_object(
                    self.restore_client
                )
            else:
                self.restore_destination_client = self.hvobj

            _list_of_disks = self.restore_destination_client.get_disk_in_the_path(
                disk_restore_destination)

            _vm_disk_list = vm_obj.disk_list
            if not _vm_disk_list:
                raise Exception(
                    "Cannot validate the Disk as we cannot find the disk attached to the VM")

            if not ((_list_of_disks is None) or (_list_of_disks == [])):
                _final_mount_disk_list = []
                for each_disk in _vm_disk_list:
                    each_disk_name = os.path.basename(each_disk).split(".")[0]
                    for disk_path in _list_of_disks:
                        if each_disk_name in disk_path:
                            _final_mount_disk_list.append(disk_path)
            else:
                raise Exception(
                    "The Disk cannot be validated as we cannot find disk with Hypervisor"
                    " extension, could be converted disk")

            if not _final_mount_disk_list:
                _final_mount_disk_list = _list_of_disks

            for _file in _final_mount_disk_list:
                self.log.info("Validation Started For Disk :[%s]", _file)
                _file = disk_restore_destination + "\\" + _file
                _drive_letter = self.restore_destination_client.mount_disk(vm_obj, _file)
                if _drive_letter != -1:
                    for each_drive in _drive_letter:
                        dest_folder_path = VirtualServerConstants.get_folder_to_be_compared(
                            self.backup_type.name, each_drive)
                        self.log.info("Folder comparison started...")
                        time.sleep(5)
                        self.fs_testdata_validation(
                            self.restore_destination_client.machine, dest_folder_path)
                else:
                    self.log.error("ERROR - Error mounting VMDK %s", _file)
                    raise Exception("Exception at Mounting Disk ")

                self.restore_destination_client.un_mount_disk(vm_obj, _file)

        except Exception as exp:
            if _drive_letter and _drive_letter != -1:
                self.restore_destination_client.un_mount_disk(vm_obj, _file)
            self.log.exception("Exception occurred during disk validation. Please check the logs.")
            raise exp

    def disk_level_restore(self, vm_level=False):
        """
        Disk level restore to another VM in the server

        Args:
            vm_level    (bool):     if the restore should be initiated from VM level

        Raises:
            Exception:
                if the disk level restore or validation fails

        """
        try:
            self.log.info("*" * 10 + "Performing disk level restore" + "*" * 10)
            for _vm in self._vms:
                if not vm_level:
                    self.navigator.navigate_to_hypervisors()
                    self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
                    self.hypervisor_details_obj.open_subclient(self.subclient)
                    self.vsa_sc_obj.restore()
                else:
                    self._navigate_to_vm_restore(_vm)
                self.select_restore_obj.select_disk_restore()
        except Exception as exp:
            self.log.exception("Exception occurred during Attach disk restore. %s", str(exp))
            raise exp

    def virtual_machine_files_restore(self, vm_level=False):
        """
        Disks restores to a client in the commcell

        Args:
            vm_level    (bool):     if the restore should be initiated from VM level

        Raises:
            Exception:
                if the virtual machine files restore or validation fails

        """
        try:
            self.log.info("*" * 10 + "Performing virtual machine files restore" + "*" * 10)

            if self._restore_proxy is None:
                self.restore_proxy = self._co_ordinator

            if self.restore_client is None:
                self.restore_client = self.hypervisor

            for _vm in self._vms:
                restore_path = self._restore_proxy.join_path(
                    self.restore_path, "AdminConsole", _vm)

                # Clearing the restore path
                if self._restore_proxy.check_directory_exists(restore_path):
                    self._restore_proxy.remove_directory(restore_path)

                disks = self._get_disk_list(self.hvobj.VMs[_vm].disk_list)
                if not vm_level:
                    self.navigator.navigate_to_hypervisors()
                    self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
                    self.hypervisor_details_obj.open_subclient(self.subclient)
                    self.vsa_sc_obj.restore()
                else:
                    self._navigate_to_vm_restore(_vm)
                self.select_restore_obj.select_vm_files()

                restore_job = self.vm_files_restore_obj.vm_files_restore(
                    _vm,
                    disks,
                    self.restore_proxy,
                    restore_path
                )

                job_details = self.get_job_status(restore_job)

                self.disk_validation(self.hvobj.VMs[_vm], restore_path)

        except Exception as exp:
            self.log.exception("Exception occurred during Virtual Machine Files restore."
                               " %s", str(exp))
            raise exp

    @abstractmethod
    def full_vm_restore(self):
        """
        Performs Full VM restore
        """
        self.log.info("Performing full VM restore")

    def attach_disk_restore(self):
        """
        Performs attach_disk_restore
        """
        self.log.info("Performing attach_disk_restore")

    def vm_restore_validation(self, vm_name, restore_vm):
        """
        Validates the full VM restore job

        Args:
            vm_name          (basestring)   --  the name of the source VM that was backed up

            restore_vm       (basestring)   --  the name of the restored VM

        Raises:
            Exception:
                if full VM / instance restore validation fails

        """
        try:
            VirtualServerUtils.decorative_log("Validating full VM restore")
            time.sleep(120)
            self.restore_destination_client.update_hosts()
            on_premise = VirtualServerConstants.on_premise_hypervisor(
                self._restore_instance_type.lower())

            if self.full_vm_in_place:
                self.log.info("It is Inplace restore")
                source_obj = self.__deepcopy__((self.hvobj.VMs[vm_name]))
            else:
                src_attempt = 1
                while src_attempt < 5:
                    self.hvobj.VMs[vm_name].update_vm_info('All', True, True)
                    source_obj = self.hvobj.VMs[vm_name]
                    if source_obj.ip is not None or source_obj.ip != "":
                        src_attempt = 5
                    else:
                        self.log.info("Attempt number %s failed. Waiting for 2 mins for VM to"
                                      "come up", str(src_attempt))
                        time.sleep(120)
                        src_attempt += 1

            if self.power_on_after_restore:
                self.restore_destination_client.VMs = restore_vm
                attempt = 1
                while attempt < 5:
                    if self.instance == 'azure resource manager':
                        self.restore_destination_client.VMs[restore_vm].resource_group_name = self.resource_group
                    self.restore_destination_client.VMs[restore_vm].update_vm_info('All',
                                                                                   True, False)
                    restore_obj = self.restore_destination_client.VMs[restore_vm]

                    if restore_obj.ip is not None:
                        attempt = 5
                    else:
                        self.log.info("Attempt number %s failed. Waiting for 2 mins for VM to"
                                      "come up", str(attempt))
                        time.sleep(120)
                        attempt += 1

                _source = self.VmValidation(source_obj)
                _dest = self.VmValidation(restore_obj)

                if _source == _dest:
                    self.log.info("Config validation is successful")
                else:
                    self.log.error("Error while validating configuration")
                    raise Exception

                if self.full_vm_in_place and on_premise:
                    if not restore_obj.guid == source_obj.guid:
                        raise Exception(
                            "The GUID id of the in place restored VM does not match the source VM")

                if not self.azure_cross_region_restore:
                    if (not (re.match(VirtualServerConstants.Ip_regex, "%s" % restore_obj.ip)) and
                            restore_obj.ip is not None):
                        for each_drive in source_obj.drive_list:
                            dest_location = source_obj.machine.join_path(
                                source_obj.drive_list[each_drive],
                                self.backup_type.name, "TestData",
                                self.timestamp)
                            self.fs_testdata_validation(
                                restore_obj.machine, dest_location)

                    else:
                        self.log.info(
                            "This is Linux VM and the destination Ip is not proper ,"
                            "so no Data Validation cannot be performed")

                    if self.hvobj.instance_type.lower() == "amazon" and self.instance.lower() == "amazon":
                        if not self._overwrite_vm:
                            source_obj.validate_name_tag = False
                            restore_obj.validate_name_tag = False
                        else:
                            source_obj.validate_name_tag = True
                            restore_obj.validate_name_tag = True
                        if source_obj == restore_obj:
                            self.log.info("AWS EC2 instance specific validation successful")

            if not self.power_on_after_restore and (not self.azure_cross_region_restore):
                if ((self.hvobj.VMs[vm_name].guest_os.lower() == "windows") and (
                        not self.full_vm_in_place) and on_premise):

                    if self.hvobj.instance_type == "hyper-v":
                        dest_location = os.path.join(self.restore_path, restore_vm,
                                                     "Virtual Hard Disks")
                        self.disk_validation(restore_obj, dest_location)
                    else:
                        self.log.info("Skipping disk validation for powered off restore VM.")

        except Exception as exp:
            self.log.exception("Exception occurred during full VM restore validation. %s",
                               str(exp))
            raise exp

    def cleanup_testdata(self):
        """
        Cleans up testdata that is copied from each vm in the subclient

        Raises:
            Exception:
                if clean up of testdata from the backup VMs fails

        """
        try:
            self.log.info("Testdata cleanup from subclient started")
            if not self.backup_folder_name:
                self.backup_folder_name = self.backup_type.name
            if self.azure_cross_region_restore:
                self.log.info("No need for cleanup testdata in Azure cross region restore")
            else:

                if self._vms:
                    for _vm in self._vms:
                        self.log.info("VM selected is %s", _vm)
                        if not self.hvobj.VMs.get(_vm):
                            continue
                        self.hvobj.VMs[_vm].update_vm_info('All', True, True)
                        if self.cleanup_testdata_before_backup:
                            for name, _drive in self.hvobj.VMs[_vm].drive_list.items():
                                for folder_name in ["FULL", "INCR", "SYNTH"]:
                                    _testdata_path = self.hvobj.VMs[_vm].machine.join_path(_drive,
                                                                                           folder_name)
                                    self.log.info("Cleaning up %s", _testdata_path)
                                    if self.hvobj.VMs[_vm].machine.check_directory_exists(
                                            _testdata_path):
                                        self.hvobj.VMs[_vm].machine.remove_directory(_testdata_path)
                        else:
                            for _drive in self.hvobj.VMs[_vm].drive_list.values():
                                _testdata_path = VirtualServerConstants.get_folder_to_be_compared(
                                    folder_name=self.backup_folder_name, _driveletter=_drive)
                                self.log.info("Cleaning up %s", _testdata_path)
                                if self.hvobj.VMs[_vm].machine.check_directory_exists(_testdata_path):
                                    self.hvobj.VMs[_vm].machine.remove_directory(_testdata_path)

            self.log.info("Cleaning testdata in the controller")
            self.controller_machine.remove_directory(self.testdata_path)
        except Exception as exp:
            self.log.exception(
                "Exception while doing cleaning up test data directory: %s", str(exp))
            raise exp

    def search_and_download(self):
        """
        Searches for and downloads content

        Returns:

        """
        if not self.ci_enabled:
            raise Exception("Content Indexing is not enabled. Search will be live browse")

        for _vm in self._vms:
            for _drive, _folder in self.hvobj.VMs[_vm].drive_list.items():
                self.navigator.navigate_to_virtual_machines()
                self.virtual_machines_obj.open_vm(_vm)
                self.vm_details_obj.vm_search_content(self.backup_folder_name)
                self.vsa_search_obj.search_for_content(
                    file_name=self.backup_folder_name, contains=_drive,
                    include_folders=True)
                self.vsa_search_obj.select_data_type("Folder")
                self.vsa_search_obj.validate_contains(self.backup_folder_name,
                                                      name=True)
                self.vsa_search_obj.validate_contains(_drive, name=False,
                                                      folder=True)
                self.restore_files_obj.download_content([self.backup_type.name])

                WebDriverWait(self.driver, 600).until(EC.invisibility_of_element_located(
                    (By.ID, "download-tracker")
                ))

                if not self.controller_machine.check_file_exists(
                        self.download_directory) and not os.listdir(self.download_directory):
                    raise Exception("Download could not be completed.")

                files = self.controller_machine.get_files_in_path(self.download_directory)
                for file in files:
                    with zipfile.ZipFile(file, 'r') as zip_file:
                        zip_file.extractall(self.download_directory)

                    validate_path = self.controller_machine.join_path(self.download_directory,
                                                                      self.backup_type.name,
                                                                      "TestData",
                                                                      self.timestamp)
                    self.fs_testdata_validation(self.controller_machine, validate_path)
                break

    def ci_download_file(self):
        """
        Searches for and downloads a file

        Returns:

        """

        files_path = VirtualServerUtils.get_content_indexing_path(self.controller_machine)
        file_path_beforebackup = ''.join(self.controller_machine.get_files_in_path(files_path)[0])
        file = os.path.split(file_path_beforebackup)[1]
        download_folder = os.path.expanduser("~" + "/Downloads/")
        file_path_afterdownload = self.controller_machine.join_path(download_folder, file)

        if not self.ci_enabled:
            raise Exception("Content Indexing is not enabled. Search will be live browse")

        for _vm in self._vms:
            self.navigator.navigate_to_virtual_machines()
            self.virtual_machines_obj.open_vm(_vm)
            self.vm_details_obj.vm_search_content(file)
            self.vsa_search_obj.select_data_type(file)
            self.restore_files_obj.download_content([file], select_one=True)

            WebDriverWait(self.driver, 600).until(EC.invisibility_of_element_located(
                (By.ID, "download-tracker")))

            if not self.controller_machine.check_file_exists(file_path_afterdownload):
                raise Exception("Download could not be completed.")

            self.log.info("Validating file checksum")
            try:
                if file in os.listdir(download_folder):
                    file_path_afterdownload = self.controller_machine.join_path(download_folder, file)

                self.controller_machine.compare_files(self.controller_machine, file_path_beforebackup,
                                                      file_path_afterdownload)
                self.log.info("File checksum validation completed successfully")
                self.controller_machine.delete_file(file_path_afterdownload)
            except Exception as exp:
                self.log.error("File checksum failed %s", str(exp))

    def get_job_status(self, job_id):
        """
        Returns the job status via SDK. To be used as a workaround when facing issues with
        job progress in Command Center.

        Args:
            job_id             (str) : Job ID whose status has to be checked.

        Returns:
            Job Status
        """
        job_obj = JobManager(job_id, self.commcell)
        return job_obj.wait_for_state(expected_state=["completed", "completed w/ one or more errors"], time_limit=20000)

    def post_restore_clean_up(self, source_vm=False, status=False):

        """
            Cleans up VM and its resources after out of place restore

            Args:
                    source_vm                       (bool): whether  source vm has to be powered
                                                            off or not
                    status                          (bool) : whether the tc has passed ot failed
            Raises:
             Exception:
                If unable to clean up VM and its resources
        """
        try:
            for each_vm in self._vms:
                if self.hvobj.VMs.get(each_vm, None):
                    if source_vm:
                        self.log.info("Powering off VM {0}".format(each_vm))
                        self.hvobj.VMs[each_vm].power_off()
                    restore_vm_name = self.vm_restore_prefix + each_vm
                    if status:
                        self.log.info("Cleaning up VM {0}".format(restore_vm_name))
                        self.restore_destination_client.VMs[restore_vm_name].clean_up()
                    else:
                        self.log.info("Powering off VM {0}".format(restore_vm_name))
                        self.restore_destination_client.VMs[restore_vm_name].power_off()

        except Exception as err:
            self.log.exception(
                "Exception while doing cleaning up VM resources: " + str(err))
            raise err


class VMwareAdminConsole(AdminConsoleVirtualServer):
    """
    VMware class to perform VMware related adminconsole tasks
    """

    def __init__(self, instance, driver=None, commcell=None, csdb=None):
        """
        Initialize any VMware related variables

        Args:
            instance        (object)    --  the instance class object

            driver          (object)    --  the browser driver object

            commcell        (object)    --  the commcell object

            csdb            (object)    --  the commserve DB object

        """

        super(VMwareAdminConsole, self).__init__(instance, driver, commcell, csdb)
        self.dest_client_vm = None
        self._disk_provisioning = "Thin"
        self._transport_mode = "Auto"
        self._live_recovery = False
        self._live_recovery_options = {}
        self.live_recovery_datastore = []
        self._different_vcenter = False
        self._different_vcenter_info = {}
        self._full_vm_restore_options = {}
        self.datastore = None

    @property
    def live_recovery(self):
        """
        Returns the live recovery enable/disable option

        Returns:
            _live_recovery     (bool)  --  True / False to enable or disable live recovery

        """
        return self._live_recovery

    @live_recovery.setter
    def live_recovery(self, value):
        """
        Sets the live recovery option during restore

        Args:
            value   (bool)  --  True / False to enable or disable live recovery during restore

        """
        self._live_recovery = value

    @property
    def live_recovery_options(self):
        """
        Returns the dict containing the live recovery options

        Returns:
            _live_recovery_options     (dict)  --  dict containing the live recovery options

        """
        return self._live_recovery_options

    @property
    def redirect_datastore(self):
        """
        Returns the redirect datatore option

        Returns:
            _live_recovery_options['redirect_datastore']   (str)   --  the name of the
                                                                        datastore where writes
                                                                        will be redirected

        """
        return self._live_recovery_options['redirect_datastore']

    @redirect_datastore.setter
    def redirect_datastore(self, value):
        """
        Sets the redirect datastore option for live recovery

        Args:
            value   (basestring)    --  the name of the datastore to redirect writes during live
                                            recovery restore

        """
        self._live_recovery_options['redirect_datastore'] = value

    @property
    def delay_migration(self):
        """
        Returns the delay migration value

        Returns:
            _live_recovery_options['delay_migration']  (int)   --  the amount of time in hrs
                                                                    the migration has to be
                                                                    delayed

        """
        return self._live_recovery_options['delay_migration']

    @delay_migration.setter
    def delay_migration(self, value):
        """
        Sets the delay migration hours

        Args:
            value   (basestring / int)    --    the no of hours the migration has to be delayed
                                                during live recovery

        """
        self._live_recovery_options['delay_migration'] = str(value)

    @property
    def transport_mode(self):
        """
        Returns the transport mode to be used for restore

        Returns:
            _transport_mode    (basestring)   --  the transport mode to be used for restore
                    default:    Auto

        """
        return self._transport_mode

    @transport_mode.setter
    def transport_mode(self, value):
        """
        Sets the transport mode to be used for restore

        Args:
            value   (basestring)    --  the transport mode to be used

        """
        self._transport_mode = value

    @property
    def disk_provisioning(self):
        """
        Returns the disk provisioning option to be used during restore

        Returns:
            _disk_provisioning     (basestring)   --  the type of disk provisioning to be used
                                                      during restore
                        default :   Thin

        """
        return self._disk_provisioning

    @disk_provisioning.setter
    def disk_provisioning(self, value):
        """
        Sets the disk provisioning option

        Args:
            value   (basestring)    --  the disk provisioning to be used for the disks in the
                                        restore VM

        """
        self._disk_provisioning = value

    @property
    def different_vcenter(self):
        """
        Returns the option to restore to a different vcenter or not

        Returns:
            full_vm_restore_options['different_vCenter']   (bool)  --  restore to a different
                                                                        new vcenter

        """
        return self._different_vcenter

    @different_vcenter.setter
    def different_vcenter(self, value):
        """
        Sets the different vcenter flag

        Args:
            value   (basestring)    --  if a different vcenter has to be used during restore

        """
        self._different_vcenter = value

    @property
    def different_vcenter_info(self):
        """
        Returns the information about the new vcenter client to create during restore

        Returns:
            _different_vcenter_info (dict)   --  different vcenter username and password

        """
        return self._different_vcenter_info

    @property
    def different_vcenter_name(self):
        """
        Returns the name of the new vcenter to restore to

        Returns:
            _different_vcenter_info['vcenter_hostname']   (basestring)  --  name of the new vcenter

        """
        return self._different_vcenter_info['vcenter_hostname']

    @different_vcenter_name.setter
    def different_vcenter_name(self, value):
        """
        Sets the name of the different vcenter

        Args:
            value   (basestring)    --  the name of the different vcenter to be used during restore

        """
        self._different_vcenter_info['vcenter_hostname'] = value

    @property
    def different_vcenter_username(self):
        """
        Returns the username of the new vcenter to restore to

        Returns:
            _different_vcenter_info['vcenter_username'] (basestring)  --  username of the
                                                                            new vcenter

        """
        return self._different_vcenter_info['vcenter_username']

    @different_vcenter_username.setter
    def different_vcenter_username(self, value):
        """
        Sets the different vcenter user name

        Args:
            value   (basestring)    --  the username for the different vcenter

        """
        self._different_vcenter_info['vcenter_username'] = value

    @property
    def different_vcenter_password(self):
        """
        Returns the password of the new vcenter to restore to

        Returns:
            _different_vcenter_info['vcenter_password']   (basestring)  --  password of the
                                                                            new vcenter

        """
        return self._different_vcenter_info['vcenter_password']

    @different_vcenter_password.setter
    def different_vcenter_password(self, value):
        """
        Sets the password for the different vcenter

        Args:
            value   (basestring)    --  the password of the different vcenter

        """
        self._different_vcenter_info['vcenter_password'] = value

    def _get_nfs_datastore_name(self, restore_vm):
        """
        Gets the NFS datastore name in the given ESX

        Args:
            restore_vm   (basestring)   --  the name of the restored VM

        Raises:
            Exception:
                if it fails to get the name of the NFS datastore that are mounted

        """
        try:
            self.log.info("Getting the name of the 3DFS datastore where VM is mounted")
            datastore_list = self.restore_destination_client.VMs[restore_vm].datastore.split(',')
            for datastore in datastore_list:
                datastore = datastore.strip()
                if '3DFS_GX_BACKUP_0' in datastore:
                    self.live_recovery_datastore.append(datastore)
            self.log.info("The 3DFS datastore is %s", str(self.live_recovery_datastore))
        except Exception as exp:
            raise exp

    def live_recovery_validation(self, restore_vm, vm_name):
        """
        Validates the live VM recovery job

        Args:
            restore_vm  (str)   :   the name of the restored VM

            vm_name     (str)   :   the name of the backed up VM

        Raises:
             Exception:
                if validation failed

        """
        try:
            self.log.info("Starting live VM recovery validation")
            ds_exists = self.restore_destination_client.VMs[restore_vm].live_browse_ds_exists(
                self.live_recovery_datastore
            )
            if ds_exists:
                raise Exception("The datastore is not unmounted yet. Cleanup failed")

            if self.datastore and self.restore_destination_client.VMs[restore_vm].datastore != \
                    self.datastore:
                raise Exception("Something went wrong during migration. The destination"
                                " datastore is wrong.")
            self.log.info("Live VM recovery validation completed successfully")
        except Exception as exp:
            raise Exception("Live VM recovery validation failed. {0}".format(str(exp)))

    def _set_vm_restore_options(self):
        """
        Sets the VM restore options for each VM to be restored
        Raises:
            Exception:
                if there is any exception in setting the restore options

        """
        for each_vm in self._vms:
            self.restore_destination_client.VMs[each_vm] = self.__deepcopy__(
                self.hvobj.VMs[each_vm])
        if not self.recovery_target:
            resources = self.restore_destination_client.compute_free_resources(self._vms)
            for _vm in self._vms:
                self._full_vm_restore_options[_vm]['network'] = {}
                self._full_vm_restore_options[_vm]['network']['destination'] = resources[4]
                if self._destination_host:
                    self._full_vm_restore_options[_vm]['host'] = self._destination_host
                else:
                    self._full_vm_restore_options[_vm]['host'] = resources[1][0]
                if self._destination_datastore:
                    self._full_vm_restore_options[_vm]['datastore'] = self._destination_datastore
                else:
                    self._full_vm_restore_options[_vm]['datastore'] = resources[0]
                self._full_vm_restore_options[_vm]['respool'] = "/"
                self._full_vm_restore_options[_vm]['prefix'] = self.vm_restore_prefix
        else:
            for _vm in self._vms:
                self._full_vm_restore_options[_vm]['network'] = {}
                self._full_vm_restore_options[_vm]['network']['destination'] = \
                    self.recovery_target._destination_network
                if self._destination_host:
                    self._full_vm_restore_options[_vm]['host'] = self._destination_host
                else:
                    self._full_vm_restore_options[_vm]['host'] = self.recovery_target._destination_host
                if self._destination_datastore:
                    self._full_vm_restore_options[_vm]['datastore'] = self._destination_datastore
                else:
                    self._full_vm_restore_options[_vm]['datastore'] = self.recovery_target._datastore
                self._full_vm_restore_options[_vm]['respool'] = self.recovery_target._resource_pool
                self._full_vm_restore_options[_vm]['prefix'] = self.recovery_target._vm_prefix

    def _restore_job(self, vm_list):
        """
        Submits a restore job with the given inputs
        Args:
            vm_list     (list):    VMs to restore

        """
        dest_target = False
        if self.recovery_target:
            dest_target = True
        restore_job = self.full_vm_restore_obj.full_vm_restore(
            vm_list,
            self.full_vm_in_place,
            proxy=self.restore_proxy,
            destination_server=self.restore_client,
            vm_info=self._full_vm_restore_options,
            different_vcenter=self.different_vcenter,
            different_vcenter_info=self.different_vcenter_info,
            disk_prov=self.disk_provisioning,
            transport_mode=self.transport_mode,
            power_on=self.power_on_after_restore,
            over_write=self.unconditional_overwrite,
            live_recovery=self.live_recovery,
            live_recovery_options=self.live_recovery_options,
            dest_target=dest_target,
            copy_precedence=self.copy_precedence
        )

        if self.live_recovery:
            time.sleep(180)
            # Copy testdata when the VM is migrating
            for _vm in vm_list:
                if self.full_vm_in_place:
                    restore_vm = _vm
                else:
                    restore_vm = self.vm_restore_prefix + _vm
                self.log.info("Setting Restore Destination Client")

                self.restore_destination_client.VMs = restore_vm
                retry = 0
                while retry < 5:
                    try:

                        self.log.info("Collecting properties of destination client")
                        self.restore_destination_client.VMs[restore_vm].update_vm_info('All',
                                                                                       True, True)
                        break
                    except Exception as err:
                        self.log.info("VM Info couldn't be updated. Trying again. Try No {0}".format(retry))
                        time.sleep(240)
                        retry = retry + 1

                for _drive in self.hvobj.VMs[_vm].drive_list.values():
                    self.log.info("Copying TDFS data to Drive %s", _drive)
                    self.restore_destination_client.copy_test_data_to_each_volume(
                        restore_vm, _drive, "TDFSDataToTestWrite", self.testdata_path)
                    self.log.info("Done copying TDFS data")

                self._get_nfs_datastore_name(restore_vm)

        job_details = self.get_job_status(restore_job)

    def _end_user_restore_job(self, vm_list):
        """
        Submits a restore job with the given inputs for end user
        Args:
        vm_list     (list):    VMs to restore
        """
        restore_job = self.enduser_fullvm_obj.enduser_full_vm_restore(
            vm_list[0],
            restore_prefix=self.vm_restore_prefix
        )

        job_details = self.get_job_status(restore_job)

    def live_mount(self,
                   replication_target,
                   copy_precedence="Default copy"):
        """
        performs Live Mount for each VM in subclient

        Args:
            replication_target (string):    name of the replication target

            copy_precedence    (string):    supports "snap","backupcopy" as of now

        Raises:
            Exception:
                if Live Mount or Validation Fails

        """
        try:
            self.navigator.navigate_to_replication_targets()
            try:
                self.recovery_targets_obj.access_target(replication_target)
                rep_target_dict = self.target_details_obj.get_target_summary()
            except Exception as exp:
                self.log.exception("Exception occurred during getting summary/accessing recovery target. %s", str(exp))
                raise exp
            if copy_precedence.lower() == "snap":
                copy_precedence = self.storage_policy.snap_copy
            elif copy_precedence.lower() == "backupcopy":
                copy_precedence = "Primary"

            vm_names = []
            live_mount_jobs = []
            for _vm in self._vms:
                self._navigate_to_vm_restore(_vm)
                self.select_restore_obj.select_live_mount()
                if rep_target_dict['Destination network'] == "Not Connected":
                    rep_target_dict['Destination network'] = "Original network"

                live_mount_job_id = self.live_mount_obj.submit_live_mount(replication_target,
                                                                          rep_target_dict['Destination network'],
                                                                          copy_precedence)
                if not self.get_job_status(live_mount_job_id):
                    self.log.exception("Exception occurred during Live Mount Job")
                    raise Exception

                vm_names.append(_vm)
                live_mount_jobs.append(live_mount_job_id)
            return vm_names, live_mount_jobs

        except Exception as exp:
            self.log.exception("Exception occurred during Live Mount. %s", str(exp))
            raise exp

    def full_vm_restore(self, vm_level=False):
        """
        Performs full VM restore for a VMware subclient

        Args:
            vm_level    (bool):     if the restore should be initiated from VM level

        Raises:
            Exception:
                if full VM restore or validation fails

        """
        try:
            VirtualServerUtils.decorative_log("Performing VMware Full VM restore")

            if self._restore_proxy is None:
                self.restore_proxy = self._co_ordinator

            if self.restore_client is None:
                self.restore_client = self.hypervisor
                self.restore_destination_client = self.hvobj
            elif self.commcell.clients.has_client(self.restore_client):
                if not self.restore_destination_client:
                    self.restore_destination_client, \
                    self._restore_instance_type = self._create_hypervisor_object(
                        self.restore_client)
            else:
                if self.commcell.recovery_targets.has_recovery_target(self.restore_client):
                    self.recovery_target = self.commcell.recovery_targets.get(self.restore_client)
                    self.restore_destination_client, \
                    self._restore_instance_type = self._create_hypervisor_object(
                        self.recovery_target.destination_hypervisor)
                    self.vm_restore_prefix = self.recovery_target._vm_prefix
            self._full_vm_restore_options = dict.fromkeys(self._vms, {})
            self._set_vm_restore_options()

            if vm_level:
                for _vm in self._vms:
                    self._navigate_to_vm_restore(_vm)
                    self.select_restore_obj.select_full_vm_restore()
                    self._restore_job([_vm])
            else:
                self.navigator.navigate_to_hypervisors()
                self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
                self.hypervisor_details_obj.open_subclient(self.subclient)
                self.vsa_sc_obj.restore()
                self.select_restore_obj.select_full_vm_restore()

                self._restore_job(self._vms)

            for _vm in self._vms:
                if self.full_vm_in_place:
                    restore_vm = _vm
                else:
                    restore_vm = self.vm_restore_prefix + _vm
                self.vm_restore_validation(_vm, restore_vm)

                if self.live_recovery:
                    source_obj = self.hvobj.VMs[_vm]
                    for each_drive in source_obj.drive_list:
                        dest_location = self.controller_machine.join_path(
                            source_obj.drive_list[each_drive],
                            "TDFSDataToTestWrite",
                            "TestData",
                            self.timestamp)
                        self.fs_testdata_validation(
                            self.restore_destination_client.VMs[restore_vm].machine, dest_location)
                    self.live_recovery_validation(restore_vm, _vm)

        except Exception as exp:
            self.log.exception("Exception occurred during full VM restore. %s", str(exp))
            raise exp

    def conversion_validation(self, restore_obj, source_vm, restore_vm, restore_option):
        """
        Performs validation for VM restore to OCIinstance fromVMWare
        Args:
            restore_obj    (vsa object):     for VMware to Oci conversion ,pass OCi object
            source_vm     : name of source vm
            restore_vm  : name of restored vm
            restore_option : restore hypervisor properties
        Raises:
            Exception:
                if validation fails
        """
        VirtualServerUtils.decorative_log("Validating full VM restore")
        time.sleep(120)
        restore_obj.hvobj.update_hosts()
        self.hvobj.VMs[source_vm].update_vm_info('All', True, True)
        source_obj = self.hvobj.VMs[source_vm]
        if self.power_on_after_restore:
            restore_obj.hvobj.VMs = restore_vm
            attempt = 1
            while attempt < 5:
                restore_obj.hvobj.VMs[restore_vm].update_vm_info('All', True, True)
                restore_vm_obj = restore_obj.hvobj.VMs[restore_vm]
                if restore_vm_obj.ip is not None:
                    attempt = 5
                else:
                    self.log.info("Attempt number %s failed. Waiting for 2 mins for VM to"
                                  "come up", str(attempt))
                    time.sleep(120)
                    attempt += 1
        dest = restore_vm_obj.VmConversionValidation(restore_vm_obj)
        if dest.conversion_validation(restore_option):
            self.log.info("config validation is successful")
        else:
            self.log.error("error while configuration validation")
            raise Exception("Error while configuration validation")
        if (not (re.match(VirtualServerConstants.Ip_regex, "%s" % restore_vm_obj.ip)) and
                restore_vm_obj.ip is not None):
            for each_drive in source_obj.drive_list:
                dest_location = source_obj.machine.join_path(
                    source_obj.drive_list[each_drive],
                    self.backup_type.name, "TestData",
                    self.timestamp)
                self.fs_testdata_validation(
                    restore_vm_obj.machine, dest_location)
        else:
            self.log.info(
                "This is Linux VM and the destination Ip is not proper ,"
                "so no Data Validation cannot be performed")

    def full_vm_oci_restore(self, restore_obj=None):
        """
        Performs full VM restore for a VMware subclient

        Args:
            restore_obj    (vsa object):     for VMware to Oci conversion ,pass OCi object

        Raises:
            Exception:
                if full VM restore or validation fails

        """
        try:
            VirtualServerUtils.decorative_log("Performing VMware Full VM restore")

            restore_obj.get_all_vms()
            restore_obj._instance = "Oracle Cloud Infrastructure"
            restore_obj.hvobj, \
            self._restore_instance_type = restore_obj._create_hypervisor_object(
                self.restore_client)
            self.restore_destination_client = restore_obj.hvobj
            if self._restore_proxy is None:
                self.restore_proxy = restore_obj._co_ordinator
            for each_vm in restore_obj._vms:
                if not hasattr(restore_obj.hvobj.VMs, each_vm):
                    restore_obj.hvobj.VMs = each_vm
            vm_obj = restore_obj.hvobj.VMs[list(restore_obj._vms)[0]]
            self.navigator.navigate_to_hypervisors()
            self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
            self.hypervisor_details_obj.open_subclient(self.subclient)
            self.vsa_sc_obj.restore()
            self.select_restore_obj.select_full_vm_restore()
            restore_job = self.full_vm_restore_obj.full_vm_restore(
                self._vms,
                self.full_vm_in_place,
                restore_as="Oracle Cloud Infrastructure",
                proxy=self.restore_proxy,
                destination_server=self.restore_client,
                vm_info=vm_obj,
                power_on=self.power_on_after_restore,
                over_write=self.unconditional_overwrite,
            )
            try:
                time.sleep(360)
                job_details = self.get_job_status(restore_job)
            except Exception as exp:
                time.sleep(720)
                job_details = self.get_job_status(restore_job)

            if not job_details:
                raise Exception("Restore job failed. Please check the logs")

            for vm_name in self._vms:
                restore_vm = vm_name
                self.conversion_validation(restore_obj, vm_name, restore_vm, vm_obj)

        except Exception as exp:
            self.log.exception("Exception occurred during full VM restore. %s", str(exp))
            raise exp

    def end_user_full_vm_restore(self):
        """
            Performs full VM restore at VM level as end user
                Raises:
                    Exception:
                        if full VM restore or validation fails

        """
        for _vm in self._vms:
            self._navigate_to_vm_restore(_vm)
            self.select_restore_obj.select_full_vm_restore()
            self._end_user_restore_job([_vm])

        for _vm in self._vms:
            if self.full_vm_in_place:
                restore_vm = _vm
            else:
                restore_vm = self.vm_restore_prefix + _vm
            self.vm_restore_validation(_vm, restore_vm)

    def setup_backup_validation(self, backup_validation_options):
        """
        Setting up details for backup validation

        Args:
            backup_validation_options   (dict):     Username, password etc options needed
                                                    for backup validation

        Raises:
            Exception:
                If setting backup validation fails
        """
        try:
            VirtualServerUtils.decorative_log("Setting up Backup validation")
            self.navigator.navigate_to_hypervisors()
            self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
            self.hypervisor_details_obj.open_subclient(self.subclient)
            self.vsa_sc_obj.set_backup_validation(backup_validation_options)
        except Exception as exp:
            self.log.exception("Exception occurred during setting validation backup. %s", str(exp))
            raise exp

    def run_validate_backup_vmgroup(self, backup_validation_options):
        """
        Trigger backup validation from vm group level

        Args:
            backup_validation_options   (dict):     Recovery target etc needed for validation backup validation

        """
        VirtualServerUtils.decorative_log("Running Validate Backup from vm group")
        self.navigator.navigate_to_hypervisors()
        self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
        self.hypervisor_details_obj.open_subclient(self.subclient)
        backup_validation_job = self.vsa_sc_obj.run_validate_backup()
        if backup_validation_options.get('report_validation'):
            self.fail_backup_validation_job(backup_validation_job)
        else:
            self.validate_backup(backup_validation_options, backup_validation_job)
        # Add the key to have minimum of 15 mins of wait time before the job gets over

    def run_validate_backup_vm(self, backup_validation_options):
        """
        Trigger backup validation from vm level

        Args:
            backup_validation_options   (dict):     Recovery target etc needed for validation backup validation

        """
        VirtualServerUtils.decorative_log("Running Validate Backup from vm level")
        for each_vm in self._vms:
            self.navigator.navigate_to_virtual_machines()
            self.virtual_machines_obj.open_vm(each_vm)
            backup_validation_job = self.virtual_machines_obj.run_validate_backup()
            self.validate_backup(backup_validation_options, backup_validation_job, each_vm)
            # Add the key to have minimum of 15 mins of wait time before the job gets over

    def run_validate_backup_schedule(self, backup_validation_options):
        """
        Validating and triggering backup validation from backup schedule

        Args:
            backup_validation_options   (dict):     Recovery target etc needed for validation backup validation

        Raises:
            Exception:
                Exception in running job from schedule
        """
        self.navigator.navigate_to_hypervisors()
        self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
        self.hypervisor_details_obj.open_subclient(self.subclient)
        self.vsa_sc_obj.schedule_backup_validation(only_schedule=True, after_time=300)
        self.log.info("Sleeping for 6 mins for schedule to trigger")
        time.sleep(360)
        job_controller = self.commcell.job_controller
        vm_jobs = job_controller.all_jobs(client_name=None, job_filter='VM_Management')
        for _job in vm_jobs:
            _job_detail = job_controller.get(_job)
            if _job_detail.details['jobDetail']['generalInfo']['jobStartedFrom'] == 'Scheduled':
                if _job_detail.details['jobDetail']['generalInfo']['scheduleName'] == self.driver.session_id:
                    self.log.info("{} was started by schedule".format(_job))
                    self.validate_backup(backup_validation_options, str(_job))
                    return
        raise Exception("Backup validation job not triggered")

    def fail_backup_validation_job(self, backup_validation_job):
        """
        Fails Backup validation job

        Args:
            backup_validation_job      (string):     backup validation job

        Raises:
            Exception:
                If deleting the vms failed
        """
        try:
            time.sleep(300)
            vm_list = [*self._vms]
            for each_vm in vm_list:
                dest_vm = each_vm + '_LM_' + backup_validation_job
                self.hvobj.VMs = dest_vm
                if self.hvobj.check_vms_exist([dest_vm]):
                    self.log.info("--- Deleting the VM {} ---".format(dest_vm))
                    restore_obj = self.hvobj.VMs[dest_vm]
                    restore_obj.backup_validation()
                    restore_obj.delete_vm()
                    time.sleep(30)
                    self.log.info("--- Deleted VM {} Successfully ---".format(dest_vm))
        except Exception as exp:
            self.log.exception("Exception occurred deletion of LM vms %s", str(exp))
            raise exp

    def validate_backup(self, backup_validation_options, backup_validation_job, each_vm=None):
        """
        Runs and validates Backup validation job and status

        Args:
            backup_validation_options   (dict):     Recovery target etc needed for validation backup validation

            backup_validation_job      (string):     backup validation job

            each_vm                    (string):     Vm for which backup validation has been run

        Raises:
            Exception:
                If running and validating backup validation fails
        """
        try:
            if each_vm:
                vm_list = [each_vm]
            else:
                vm_list = [*self._vms]
            if backup_validation_options.get('recovery_target'):
                self.navigator.navigate_to_life_cycle_policies()
                self.recovery_target_obj.select_replication_target(backup_validation_options['recovery_target'])
                target_details = self.recovery_target_details_obj.replication_target_summary()
                target_client = target_details['Destination hypervisor']
                dest_client = self._create_hypervisor_object(target_client)[0]
                target_esx = target_details['Destination host']
            else:
                dest_client = self.hvobj

            VirtualServerUtils.decorative_log("sleeping for 5 mins so that all the vms are mounted and booted")
            time.sleep(300)
            # Commenting out validation of output of script for now
            # if backup_validation_options.get('unc_path'):
            #     output_file = os.path.split(backup_validation_options['unc_path'])[0] + '\\automation_out.txt'
            #     line = open(output_file).readline()
            #     # Will correct this log better
            #     if 'This file is created via automation' in line:
            #         self.log.info("Custom script executed properly")
            #     else:
            #         raise Exception("Custom script didn't ran")
            _job = JobManager(backup_validation_job, self.commcell)
            if _job.job.status.lower() in (
                    'killed', 'completed', 'completed w/ one or more errors', 'failed'):
                raise Exception(
                    "The Backup validation job has either failed/killed or completed too early."
                    "Please check the vm management log")
            no_of_vms_mounted = 0
            mounted_data_stores = []
            for each_vm in vm_list:
                dest_vm = each_vm + '_LM_' + backup_validation_job
                dest_client.VMs = dest_vm
                if dest_client.check_vms_exist([dest_vm]):
                    no_of_vms_mounted = no_of_vms_mounted + 1
                    if no_of_vms_mounted > 5:
                        raise Exception("Maximum number of mounted vms is 5")
                    self.log.info("--- Validation for vm {} ---".format(dest_vm))
                    restore_obj = dest_client.VMs[dest_vm]
                    restore_obj.backup_validation()
                    mounted_data_stores.append(restore_obj.datastore)
                    if restore_obj.power_state == 'poweredOn':
                        self.log.info("VM: {} is powered on".format(dest_vm))
                        if restore_obj.tools != 'running':
                            raise Exception("Boot up properly failed")
                        self.log.info("VM: {} booted properly".format(dest_vm))
                    else:
                        raise Exception("Power on failed")
                    if backup_validation_options.get('recovery_target'):
                        if restore_obj.esx_host == target_esx:
                            self.log.info("VM recovery target validated: {}".format(target_esx))
                    else:
                        if restore_obj.esx_host == self.hvobj.VMs[each_vm].esx_host:
                            self.log.info("VM recovery target validated")
            if len(vm_list) < 5:
                if not no_of_vms_mounted == len(vm_list):
                    raise Exception("All vms are not mounted")
            else:
                if no_of_vms_mounted < 5:
                    raise Exception("Total number of vms mounted at a time should be 5.")
            unique_datastores = (list(set(mounted_data_stores)))
            self.log.info('backup_validation_job: {0} backup_job: {1}'.
                          format(backup_validation_job, self.backup_job))
            if self.get_job_status(backup_validation_job):
                job_controller = self.commcell.job_controller
                _job_detail = job_controller.get(backup_validation_job)
                if _job_detail.job_type != 'Virtual Machine Management':
                    raise Exception("Job type should be Virtual Machine Management")
                self.log.info("{} job is of Virtual Machine Management".format(backup_validation_job))

            VirtualServerUtils.decorative_log("Checking if the Vms are deleted")
            for each_vm in vm_list:
                dest_vm = each_vm + '_LM_' + backup_validation_job
                self.log.info("--- Verifying Cleanup of the vm: {} ---".format(dest_vm))
                if dest_client.check_vms_exist([dest_vm]):
                    raise Exception('VMs is not cleaned up')
                self.log.info("VM:{} is cleaned".format(dest_vm))

            VirtualServerUtils.decorative_log("Checking if the Datastores are deleted")
            if dest_client.check_ds_exist(unique_datastores):
                raise Exception('Some Mounted Datastores are not cleaned up')
            self.log.info("Datastores are all cleaned")

            # validating for status for each vm
            VirtualServerUtils.decorative_log("Validating the backup validation results")
            for each_vm in vm_list:
                self.navigator.navigate_to_virtualization()
                self.virtual_machines_obj.open_vm(each_vm)
                details = self.vm_details_obj.backup_validation_results()
                if not (details['Boot status'] == 'Success'
                        and details['Last validation job ID'] == backup_validation_job
                        and details['Backup validated'] == self.backup_job):
                    self.log.error("failure in backup validation of vm{} : {}".format(each_vm, details))
                    raise Exception("failure in backup validations")
                else:
                    self.log.info("Backup validation verified for vm {}".format(each_vm))
                script_result = self.vm_details_obj.backup_validation_scripts()
                if not script_result:
                    if backup_validation_options.get('agent') or backup_validation_options.get(
                            'unc_path_win') or backup_validation_options.get('unc_path_unix'):
                        self.log.error("VM {} has agent installed or custom script added. "
                                       "Script result is empty".format(each_vm))
                        raise Exception("failure in backup validations")
                else:
                    for _script_status in script_result:
                        if _script_status[1] == 'Success':
                            self.log.info("Name: {}  Script status: {}".format(_script_status[0],
                                                                               _script_status[1]))
                        else:
                            raise Exception("Name: {}  Failure reason: {}".format(_script_status[0],
                                                                                  _script_status[2]))
        except Exception as exp:
            self.log.exception("Exception occurred during validation backup %s", str(exp))
            raise exp

    def backup_validation_report_validation(self, testcase_obj, test_status):
        """
        Validates Report and the actual data is correct

        Args:
            testcase_obj   (object):     Testcase object

            test_status     (string):       Status of the validation job

        Raises:
            Exception:
                If Report data mismatches

        """
        try:
            board = Dashboard(testcase_obj.admin_console)
            board.navigate_to_given_dashboard('Virtualization Dashboard')
            if board.driver.find_elements_by_xpath("//span[contains(text(),'VALIDATION FAILED')]"):
                if test_status == 'success':
                    self.log.error("VALIDATION FAILED should not be visible "
                                   "when there is no failed backup validation job")
                    raise Exception("Report validation failed")
            board.access_details_page('VMs', 'VALIDATION FAILED')
            web_c = WebConsole(testcase_obj.browser, testcase_obj.commcell.webconsole_hostname)
            viewer_obj = viewer.CustomReportViewer(web_c)
            table_obj = viewer.DataTable("Actively protected Virtual machines")
            viewer_obj.associate_component(table_obj)
            _data = table_obj.get_table_data()
            report = [dict(zip(_data, t)) for t in zip(*_data.values())]
            vms_in_vm_group = [*self._vms]
            for _vm in report:
                if _vm in vms_in_vm_group:
                    self.navigator.navigate_to_virtual_machines()
                    _query = "select name from APP_Client where displayName=\'%s\'" % (_vm['VM Name'])
                    self.csdb.execute(_query)
                    _results = self.csdb.fetch_all_rows()
                    if not _results or len(_results) > 1:
                        raise Exception("An exception occurred getting server details")
                    _vm_name = _results[0]
                    self.virtual_machines_obj.open_vm(_vm_name)
                    details = self.vm_details_obj.backup_validation_results()
                    if not (details['Boot status'] == _vm['Status']
                            and details['Last validation job ID'] == _vm['Last validation job ID']
                            and details['Backup validated'] == _vm['Validated Backup Job ID']):
                        self.log.error("failure in backup validation of vm{} : {}".format(_vm, details))
                        raise Exception("failure in backup validations")
            self.log.info("Report matches for all the VMs")
        except Exception as exp:
            self.log.exception("Exception occurred during report validation %s", str(exp))
            raise exp

    def get_replication_job(self, replication_group_name=None):
        """
        Return Live Sync job in progress for the replication group
        Args:
            replication_group_name: Name of replication Group

        Returns: JOB ID

        """
        self.navigator.navigate_to_replication_groups()
        self.replication_group_obj.access_group(replication_group_name)
        replication_job_id = self.replication_group_details_obj.get_replication_job()
        return replication_job_id

    def validate_snapshot_pruning(self, replication_group_name=None):
        """
            replication_group_name (string) : name of the replication group
            Raises:
             Exception:
                If validation fails
        """
        try:
            self.navigator.navigate_to_replication_groups()
            self.replication_group_obj.access_group(replication_group_name)
            auto_commcell_obj = AutoVSACommcell(self.commcell, self.csdb)
            # get control host id of the configured storage array
            control_host_id = auto_commcell_obj.get_control_host_id(replication_group_name)

            # xml to get to get the snapshot list on the storage array
            deploy_xml = """<?xml version='1.0' encoding='UTF-8'?><EVGui_GetVolumesOfSnapBackupReq jobId="0" copyId="0" clientId="0" controlHostId="{0}"/>""". \
                format(control_host_id)
            # executing qoperation by passing above xml
            snaphots_response = self.commcell.qoperation_execute(deploy_xml)
            snapshot_jobids = set()
            for response_index in range(len(snaphots_response['listOfVolumes'])):
                snapshot_jobids.add(snaphots_response['listOfVolumes'][response_index]['jobId'])

            snapshot_retention = 3  # default value
            for _vm in self._vms:
                snap_backup_jobids = []

                job_controller_obj = JobController(self.commcell)
                client_obj = Client(self.commcell, _vm)
                client_id = int(client_obj.client_id)

                # getting finished job id's
                jobs_details = job_controller_obj.finished_jobs(client_name=self.destination_client,
                                                                lookup_time=48,
                                                                job_filter='snapbackup',
                                                                job_summary='full',
                                                                limit=100)
                # filtering jobs with client id
                for job_id in jobs_details.keys():
                    if jobs_details[job_id]['subclient']['clientId'] == client_id:
                        snap_backup_jobids.append(int(job_id))

                # getting backup job id's yet to replicate
                pending_backup_jobids = auto_commcell_obj.get_backup_pending_jobs_to_replicate(_vm)

                jobids_exist = []
                jobids_notexist = []

                # logic to get the list of job id's to be there and list of job id's not to be there
                snap_backup_jobids.sort(reverse=True)
                if pending_backup_jobids[0] != '':  # if we are having backup jobs yet to replicate
                    # converting job id's string to int
                    for jobid_index in range(len(pending_backup_jobids)):
                        pending_backup_jobids[jobid_index] = int(pending_backup_jobids[jobid_index])

                    # adding all backup job id's yet to replicate to
                    # the list of snapshot job id's to be present on the storage array
                    for jobid in pending_backup_jobids:
                        jobids_exist.append(jobid)
                        snapshot_retention = snapshot_retention - 1

                for jobid_index in range(len(snap_backup_jobids)):
                    if jobid_index < snapshot_retention:
                        jobids_exist.append(snap_backup_jobids[jobid_index])
                    else:
                        jobids_notexist.append(snap_backup_jobids[jobid_index])

                # validating snapshot job id's to be present on the storage array
                for jobid in jobids_exist:
                    if jobid not in snapshot_jobids:
                        raise Exception(f'snapshot corresponding to jobid:{jobid} is not found on array')
                # validating snapshot job id's not to be present on the storage array
                for jobid in jobids_notexist:
                    if jobid in snapshot_jobids:
                        raise Exception(
                            f'snapshot corresponding to jobid:{jobid} is found on array and should be pruned')
            return True
        except Exception as exp:
            self.log.error(exp)
            raise Exception("Failed to complete snapshot pruning validation.")

    def validate_status(self, status_list, value):
        """
        Validate values in status list
        Raises:
             Exception:
                If validation fails
        """
        for status in status_list:
            if status.lower() != value:
                return False
        return True

    def validate_live_sync(self, replication_group_name=None, live_sync_direct=False):
        """
                Validates Live Sync functionality for VMW.

                Args:
                    replication_group_name (basestring):     Name of the replication group, if not set,
                    a replication group is automatically set
                    live_sync_direct (bool) :   true if it is a live sync direct validation

                Raises:
                    Exception:
                        If validation fails.

        """
        self.navigator.navigate_to_replication_groups()
        self.replication_group_obj.access_group(replication_group_name)
        recovery_target_name = self.replication_group_details_obj.get_recovery_target()
        self.navigator.navigate_to_replication_targets()
        self.recovery_target_obj.select_replication_target(recovery_target_name)

        destination_client_name = self.recovery_target_obj.get_destination_hypervisor()

        self.navigator.navigate_to_replication_groups()
        self.replication_group_obj.access_group(replication_group_name)

        sync_status = self.replication_group_details_obj.get_sync_status()
        assert self.validate_status(sync_status, "in sync"), "Sync status is not 'In Sync'"
        try:
            vm_pairs = self.replication_group_details_obj.get_source_and_destination()

            source_vms = vm_pairs[0]
            destination_vms = vm_pairs[1]

            if destination_client_name[-3:] == "_DN":
                destination_client_name = destination_client_name[:-3]

            self.destination_client = self._create_hypervisor_object(destination_client_name)[0]

            for source_vm_name, destination_vm_name in zip(source_vms, destination_vms):
                self.destination_client.VMs = destination_vm_name
                dest_vm = self.destination_client.VMs[destination_vm_name]
                source_vm = self.hvobj.VMs[source_vm_name]

                dest_vm.power_on()
                self.log.info('Successfully powered on VM: "%s"', destination_vm_name)

                # Wait for IP to be generated
                wait = 10

                while wait:
                    self.log.info('Waiting for 60 seconds for the IP to be generated')
                    time.sleep(60)
                    try:
                        dest_vm.update_vm_info('All', os_info=True, force_update=True)
                    except Exception:
                        pass

                    if dest_vm.ip and VirtualServerUtils.validate_ip(dest_vm.ip):
                        break
                    wait -= 1
                else:
                    self.log.error('Valid IP not generated within 10 minutes')
                    raise Exception(f'Valid IP for VM: {destination_vm_name} not generated within 5 minutes')
                self.log.info('IP is generated')

                _source = self.VmValidation(source_vm)
                _dest = self.VmValidation(dest_vm)

                # Validate Configuration
                assert _source == _dest and dest_vm.NicName, "error while configuration validation"
                self.log.info("config validation is successful")
                if not live_sync_direct:
                    assert '__GX_BACKUP__' in dest_vm.VMSnapshot, 'Snapshot validation failed'
                    self.log.info('Snapshot validation successful')

                # To validate test data between source and destination
                for label, drive in dest_vm.drive_list.items():
                    dest_path = self.controller_machine.join_path(drive, self.backup_folder_name, "TestData",
                                                                  self.timestamp)
                    self.fs_testdata_validation(dest_vm.machine, dest_path)
                self.log.info('Testdata validation successful')

                dest_vm.power_off()
                self.log.info('Successfully powered off VM: "%s"', destination_vm_name)

            if live_sync_direct:
                assert self.validate_snapshot_pruning(replication_group_name), "snapshot pruning validation failed"
                self.log.info("snapshot pruning validation successful")

            self.log.info('Validation completed successfully')

        except Exception as exp:
            self.log.error(exp)
            raise Exception("Failed to complete live sync validation.")


class HyperVAdminConsole(AdminConsoleVirtualServer):
    """
    Hyper-V class to perform Hyper-V related adminconsole tasks
    """

    def __init__(self, instance, driver=None, commcell=None, csdb=None):
        """
        Initialize any Hyper-V related variables

        Args:
            instance    (object)   --  object of the instance class

            driver      (object)   --  the browser object

            commcell    (object)   --  an instance of the commcell class

            csdb        (object)   --  the cs DB object

        """
        super(HyperVAdminConsole, self).__init__(instance, driver, commcell, csdb)
        self._register_to_failover = False
        self._restore_location = None

    @property
    def register_to_failover(self):
        """
        Returns the register to failover option

        Returns:
            register_to_failover   (bool)  --  True / False to register to failover cluster

        """
        return self._register_to_failover

    @register_to_failover.setter
    def register_to_failover(self, value):
        """
        Sets the register to failover option

        Args:
            value   (bool)  --  True / False to register the VM to failover after restore

        """
        self._register_to_failover = value

    @property
    def restore_location(self):
        """
        Returns the restore location, hyper-v default folder or new folder or original location

        Returns:
            restore_location   (basestring)   --  the location to restore the VM to

        """
        return self._restore_location

    @restore_location.setter
    def restore_location(self, value):
        """
        Sets the location to restore the VM to

        Args:
            value   (basestring)    --  the location where the VM should be restored

        """
        self._restore_location = value

    def full_vm_restore(self):
        """
        Perform full VM restore for hyper-V

        Raises:
            Exception:
                if full VM restore or validation fails

        """
        try:
            if self._restore_proxy is None:
                self.restore_proxy = self._co_ordinator
            if self.restore_client is None:
                self.restore_client = self.hypervisor

            self.navigator.navigate_to_hypervisors()
            self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
            self.hypervisor_details_obj.open_subclient(self.subclient)
            self.vsa_sc_obj.restore()
            self.select_restore_obj.select_full_vm_restore()

            if self.full_vm_in_place:
                self.restore_location = "Original folder"
                restore_path = None
                self.vm_restore_prefix = None

            if self.restore_location is None:
                self.restore_location = "Hyper-V default folder"
                restore_path = None

            if self.restore_location == "Select a folder":
                if not self.restore_path:
                    raise Exception("The restore path was not provided")
                restore_path = self.restore_path

            restore_job = self.full_vm_restore_obj.hv_full_vm_restore(
                self._vms,
                self.restore_proxy,
                self.restore_client,
                self.restore_location,
                self.vm_restore_prefix,
                restore_path,
                self.power_on_after_restore,
                self.unconditional_overwrite,
                self.register_to_failover
            )

            job_details = self.get_job_status(restore_job)
            if job_details['Status'] not in ['Completed',
                                             "Completed w/ one or more errors"]:
                raise Exception("Restore job failed. Please check the logs")

            for _vm in self._vms:
                if self.full_vm_in_place:
                    restore_vm = _vm
                else:
                    restore_vm = str(self.vm_restore_prefix) + _vm
                self.vm_restore_validation(_vm, restore_vm)
            self.vm_restore_prefix = self.vm_restore_prefix + 1

        except Exception as exp:
            self.log.exception("Exception occurred during Hyper-v Full VM restore. %s", str(exp))
            raise exp


class OracleCloudAdminConsole(AdminConsoleVirtualServer):
    """
    This module is for performing Oracle Cloud related actions from AdminConsole
    """

    def __init__(self, instance, driver=None, commcell=None, csdb=None):
        """
        Initializes the Oracle Cloud related inputs

        Args:
            instance    (object)   --  object of the instance class

            driver      (object)   --  the browser object

            commcell    (object)   --  an instance of the commcell class

            csdb        (object)   --  the cs DB object

        """

        super(OracleCloudAdminConsole, self).__init__(instance, driver, commcell, csdb)
        self._user_account = None
        self._security_groups = None
        self._ssh_keys = None
        self._networks = None
        self._instance_shape = "Auto"

    @property
    def instance_shape(self):
        """
        Returns the instance shape option

        Returns:
            _instance_shape     (basestring)    --  the shape to be used for the restore instance

        """
        return self._instance_shape

    @instance_shape.setter
    def instance_shape(self, value):
        """
        Sets the instance shape option

        Args:
            value   (basestring)    --  the instance shape to be used for the restore VMs

        """
        if not isinstance(value, basestring):
            raise Exception("The instance shape should be a string")
        self._instance_shape = value

    @property
    def user_account(self):
        """
        Returns the user account name

        Returns:
            _user_account   (basestring)    --  the name of the user account to restore to

        """
        return self._user_account

    @user_account.setter
    def user_account(self, value):
        """
        Sets the user account

        Args:
            value   (basestring)    --  the user account where the instances are to be restored

        """
        if not isinstance(value, basestring):
            raise Exception("The user account name should be a string")
        self._user_account = value

    @property
    def security_groups(self):
        """
        Returns the security groups

        Returns:
            _security_groups    (list)  --  list of all security groups to be associated with
                                            the restored instances

        """
        return self._security_groups

    @security_groups.setter
    def security_groups(self, value):
        """
        Sets the security groups

        Args:
            value   (list)  --  list of all security groups to be associated with the
                                restored instances

        """
        if isinstance(value, list):
            self._security_groups = value
        elif isinstance(value, basestring):
            self._security_groups = [value]
        else:
            raise Exception("The security group should either be a list or string.")

    @property
    def ssh_keys(self):
        """
        Returns the ssh keys

        Returns:
            _ssh_keys   (list)  --  list of all ssh keys to be associated with instances

        """
        return self._ssh_keys

    @ssh_keys.setter
    def ssh_keys(self, value):
        """
        Sets the ssh keys

        Args:
            value   (list)  --  list of all ssh keys to be associated with the restored instances

        """
        if isinstance(value, list):
            self._ssh_keys = value
        elif isinstance(value, basestring):
            self._ssh_keys = [value]
        else:
            raise Exception("The ssh keys should either be a list or string.")

    @property
    def networks(self):
        """
        Returns the networks list

        Returns:
            _networks   (list)  --  list of all networks to be attached to instances

        """
        return self._networks

    @networks.setter
    def networks(self, value):
        """
        Sets the networks list

        Args:
            value   (list)  --  list of all networks to be attached to the restored instances

        """
        if isinstance(value, list):
            self._networks = value
        elif isinstance(value, basestring):
            self._networks = [value]
        else:
            raise Exception("The networks should either be a list or string.")

    def _set_restore_options(self):
        """
        Sets the restore options for full VM restore

        Raises:
            Exception:
                if there is any error in setting the restore options

        """
        try:
            self.log.info("Setting the restore options")
            dest_vm = list(self._vms)[0]

            if not self.user_account:
                self.user_account = self.hvobj._get_instance_user_name(dest_vm)

            if not self.security_groups:
                self.security_groups = self.hvobj._get_security_list_of_instance(dest_vm)

            if not self.ssh_keys:
                self.ssh_keys = self.hvobj._get_ssh_keys_of_instance(dest_vm)

        except Exception as exp:
            raise Exception("An error occurred while setting the restore options. "
                            "{0}".format(str(exp)))

    def full_vm_restore(self):
        """
        Full instance restore of Oracle Cloud instances

        Raises:
            Exception:
                if full instance restore or validation fails

        """
        try:
            self.log.info("Restoring the Oracle Cloud Instance")
            if self._restore_proxy is None:
                self.restore_proxy = self._co_ordinator

            if self.restore_client is None:
                self.restore_client = self.hypervisor
                self.restore_destination_client = self.hvobj
            else:
                if not self.restore_destination_client:
                    self.restore_destination_client, \
                    self._restore_instance_type = self._create_hypervisor_object(
                        self.restore_client)

            self._set_restore_options()

            self.navigator.navigate_to_hypervisors()
            self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
            self.hypervisor_details_obj.open_subclient(self.subclient)
            self.vsa_sc_obj.restore()
            self.select_restore_obj.select_full_vm_restore()

            restore_job = self.full_vm_restore_obj.opc_full_vm_restore(
                self._vms,
                self.restore_proxy,
                self.restore_client,
                self.power_on_after_restore,
                self.vm_restore_prefix,
                self.user_account,
                self.instance_shape,
                self.networks,
                self.security_groups,
                self.ssh_keys
            )

            job_details = self.get_job_status(restore_job)
            if job_details['Status'] not in ['Completed',
                                             "Completed w/ one or more errors"]:
                raise Exception("Restore job failed. Please check the logs")

            for _vm in self._vms:
                restore_vm = self.vm_restore_prefix + _vm
                self.vm_restore_validation(_vm, restore_vm)

        except Exception as exp:
            self.log.exception("Exception occurred during Oracle Cloud Full VM restore."
                               " %s", str(exp))
            raise exp


class AlibabaCloudAdminConsole(AdminConsoleVirtualServer):
    """
    Class for Ali Cloud related operations to be done in admin console
    """

    def __init__(self, instance, driver=None, commcell=None, csdb=None):
        """
        Initializes the Oracle Cloud related inputs

        Args:
            instance    (object)   --  object of the instance class

            driver      (object)   --  the browser object

            commcell    (object)   --  an instance of the commcell class

            csdb        (object)   --  the cs DB object

        """

        super(AlibabaCloudAdminConsole, self).__init__(instance, driver, commcell, csdb)
        self._full_vm_restore_options = None
        self.restore_proxy_input = None

    def _set_instance_restore_options(self):
        """
        Sets the restore options for each instance to be restored

        Raises:
            Exception:
                if there is any exception in setting the restore options

        """
        availability_zone, network, security_groups = self.hvobj.compute_free_resources(
            self.restore_proxy)
        for vm in self._vms:
            # instance_type will always come from the backup VM
            # availability_zone, network and security_groups will be obtained from proxy

            self._full_vm_restore_options[vm]['availability_zone'] = availability_zone
            self._full_vm_restore_options[vm]['instance_type'] = self.hvobj.VMs[vm].instance_type
            self._full_vm_restore_options[vm]['network'] = network
            self._full_vm_restore_options[vm]['security_groups'] = security_groups

    def _restore_job(self, vm_list):
        """
        Submits a restore job with the given inputs

        Args:
            vm_list     (list):    VMs to restore

        """
        restore_job = self.full_vm_restore_obj.ali_cloud_full_vm_restore(
            vm_list,
            destination_client=self.restore_client,
            destination_proxy=self.restore_proxy,
            power_on=self.power_on_after_restore,
            overwrite_instance=self.unconditional_overwrite,
            in_place=self.full_vm_in_place,
            restore_prefix=self.vm_restore_prefix,
            instance_options=self._full_vm_restore_options
        )

        job_details = self.get_job_status(restore_job)
        if not job_details:
            raise Exception("Restore job failed. Please check the logs")

    def full_vm_restore(self, vm_level=False):
        """
        Runs a full instance restore of Alibaba cloud instances

        Args:
            vm_level    (bool):     if the VM should be restored from the VM level

        """
        try:
            if self.restore_proxy_input is not None:
                self.restore_proxy = machine.Machine(self.restore_proxy_input, self.commcell)
            if self._restore_proxy is None:
                self.restore_proxy = self._co_ordinator

            if self.restore_client is None:
                self.restore_client = self.hypervisor
                self.restore_destination_client = self.hvobj
            else:
                if not self.restore_destination_client:
                    self.restore_destination_client, \
                    self._restore_instance_type = self._create_hypervisor_object(
                        self.restore_client)

            proxy_info = self.hvobj.to_vm_object(self.restore_proxy)
            self.log.info("Collecting proxy disk count before restore for HotAdd validation")
            proxy_before_restore_disks_count = proxy_info.storage_disks()['TotalCount']
            self.log.info('disk count : %s ', str(proxy_before_restore_disks_count))
            self._full_vm_restore_options = {vm: dict() for vm in self._vms}
            self._set_instance_restore_options()

            if vm_level:
                for _vm in self._vms:
                    self._navigate_to_vm_restore(_vm)
                    self.select_restore_obj.select_full_vm_restore()
                    self._restore_job([_vm])
            else:
                self.navigator.navigate_to_hypervisors()
                self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
                self.hypervisor_details_obj.open_subclient(self.subclient)
                self.vsa_sc_obj.restore()
                self.select_restore_obj.select_full_vm_restore()
                self._restore_job(self.vms)

            for _vm in self._vms:
                if self.full_vm_in_place:
                    restore_vm = _vm
                else:
                    restore_vm = self.vm_restore_prefix + _vm
                self.vm_restore_validation(_vm, restore_vm)

            self.log.info("Collecting proxy disk count after restore for HotAdd validation")
            proxy_after_restore_disks_count = proxy_info.storage_disks()['TotalCount']
            self.log.info('disk count : %s ', str(proxy_after_restore_disks_count))
            try:
                assert proxy_before_restore_disks_count == proxy_after_restore_disks_count
                self.log.info('Hot Add validation successful')
            except:
                self.log.exception("HotAdd Validation failed")

        except Exception as exp:
            self.log.exception("Exception occurred during full VM restore. %s", str(exp))
            raise exp


class AzureRMAdminConsole(AdminConsoleVirtualServer):
    """
    Azure RM class to perform AzureRM related adminconsole tasks
    """

    def __init__(self, instance, driver=None, commcell=None, csdb=None):
        """
        Initialize any Azure RM related variables

        Args:
            instance    (object):  object of the instance class

            driver      (object):  the browser object

            commcell    (object):  an instance of the commcell class

            csdb        (object): the cs DB object

        """
        super(AzureRMAdminConsole, self).__init__(instance, driver, commcell, csdb)

        self._full_vm_restore_options = {}
        self._resource_group = None
        self._storage_account = None
        self._region = None
        self._vm_size = None
        self._network_interface = None
        self._security_group = None
        self._create_public_ip = False
        self._managed_vm = None

    @property
    def resource_group(self):
        """
        Returns the live resource_group

        Returns:
            resource_group     (str):    value of resource group

        """
        return self._resource_group

    @resource_group.setter
    def resource_group(self, value):
        """
        Sets the value of  resource_group

        Args:
            value   (str):  value for resource_group
        """
        self._resource_group = value

    @property
    def managed_vm(self):
        """
        Returns the live Managed VM value

        Returns:
            managed_     (bool): True / False to enable or disable managed_vm
        """
        return self._managed_vm

    @managed_vm.setter
    def managed_vm(self, value):
        """
        Sets the  Managed VM value

        Args:
            value   (bool)  :  value for managed_v
        """
        self._managed_vm = value

    @property
    def create_public_ip(self):
        """
        Returns the live create public IP value

        Returns:
            create_public_ip     (bool): True / False to enable or disable Create Public IP
        """
        return self._create_public_ip

    @create_public_ip.setter
    def create_public_ip(self, value):
        """
        Sets the  create public IP value

        Args:
            value   (bool)  :  value for _create_public_ip
        """
        self._create_public_ip = value

    @property
    def storage_account(self):
        """
        Returns the live storage_account
        Returns:
            storage_account     (st):  value of storage_account

        """
        return self._storage_account

    @property
    def region(self):
        """
        Returns the live region
        Returns:
            region     (st):  value of region

        """
        return self._region

    @storage_account.setter
    def storage_account(self, value):
        """
        Sets the value of  storage_account

        Args:
            value   (str): value for storage_account
        """
        self._storage_account = value

    @region.setter
    def region(self, value):
        """
        Sets the value of  region

        Args:
            value   (str): value for region
        """
        self._region = value

    @property
    def vm_size(self):
        """
        Returns the live vmsize

        Returns:
            vmsize     (str):  value of vm_size

        """
        return self._vm_size

    @vm_size.setter
    def vm_size(self, value):
        """
        Sets the live vmsize

        Args:
            value   (str):  value for vmsize
        """
        self._vm_size = value

    @property
    def network_interface(self):
        """
        Returns the live network inerface

        Returns:
            network_interface     (bool):  True / False to enable or disable live recovery

        """
        return self._network_interface

    @network_interface.setter
    def network_interface(self, value):
        """
        Sets the live network inerface
        Args:
            value  (str): value for network_interface
        """
        self._network_interface = value

    @property
    def security_group(self):
        """
        Returns the live security_group

        """
        return self._security_group

    @security_group.setter
    def security_group(self, value):
        """
        Sets value of security_group
        Args:
            value   (str):  value for security_group
        """
        self._security_group = value

    def _set_azure_vm_restore_options(self):
        """
                Sets the VM restore options for each VM to be restored

                Raises:
                    Exception:
                        if there is any exception in setting the restore options
        """

        try:
            _vm_list = []
            for _vm in self._vms:
                _vm_list.append(_vm)
                if (self.storage_account and self.resource_group) is None:
                    self.resource_group, self.storage_account = self.hvobj. \
                        compute_free_resources(_vm_list)
                if self.vm_size is None:
                    self.vm_size = self.hvobj.VMs[_vm].vm_size
                if self.security_group is None:
                    self.security_group = "--Auto Select--"
                if self.region is not None:
                    self._full_vm_restore_options[_vm]['region'] = self.region
                if self.network_interface is None:
                    self.network_interface = "--Auto Select--"

                self._full_vm_restore_options[_vm]['resource_group'] = self.resource_group
                self._full_vm_restore_options[_vm]['storage_account'] = self.storage_account
                self._full_vm_restore_options[_vm]['vm_size'] = self.vm_size
                self._full_vm_restore_options[_vm]['security_group'] = self.security_group
                self._full_vm_restore_options[_vm]['network_interface'] = self.network_interface

        except Exception as exp:
            self.log.exception("Exception occurred during setting restore options")
            raise exp

    def full_vm_restore(self):
        """
                Performs full VM restore for a Azure RM

                Raises:
                    Exception:
                        if full VM restore or validation fails

                """
        try:
            self.log.info("*" * 10 + "Performing full VM restore" + "*" * 10)

            if self._restore_proxy is None:
                self.restore_proxy = self._co_ordinator
            if self._restore_client is None:
                self.restore_client = self.hypervisor
                self.restore_destination_client = self.hvobj
            else:
                if not self.restore_destination_client:
                    self.restore_destination_client, \
                    self._restore_instance_type = self._create_hypervisor_object(
                        self.restore_client)

            self._full_vm_restore_options = dict.fromkeys(self._vms, {})
            self._set_azure_vm_restore_options()

            self.vm_restore_prefix = 'DEL'

            self.navigator.navigate_to_hypervisors()
            self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
            self.hypervisor_details_obj.open_subclient(self.subclient)
            self.vsa_sc_obj.restore()
            self.select_restore_obj.select_full_vm_restore()

            restore_job = self.full_vm_restore_obj.azure_full_vm_restore(
                self._vms,
                self.restore_proxy,
                self.restore_client,
                self.full_vm_in_place,
                self._full_vm_restore_options,
                self._create_public_ip,
                self.unconditional_overwrite,
                self.power_on_after_restore,
                self.managed_vm,
                restore_prefix=self.vm_restore_prefix
            )

            job_details = self.get_job_status(restore_job)
            if not job_details:
                raise Exception("Restore job failed. Please check the logs")

            for _vm in self._vms:
                if self.full_vm_in_place:
                    restore_vm = _vm
                else:
                    if len(_vm) > 12:
                        restore_vm = self.vm_restore_prefix + _vm[:12]
                    else:
                        restore_vm = self.vm_restore_prefix + _vm
                if not self.azure_cross_region_restore:
                    self.vm_restore_validation(_vm, restore_vm)
                if self.azure_cross_region_restore:
                    location = self.hvobj.VMs[_vm].get_vm_location(self.resource_group, restore_vm)
                    if location == ''.join(self.region.split()).lower():
                        self.log.info("Region validation successful. Location = %s" % location)
                    else:
                        self.log.info(
                            "Region validation failed because input region = %s and restored VM region = %s" % (
                                ''.join(self.region.split()).lower(), location))

        except Exception as exp:
            self.log.exception("Exception occurred during Azure Full VM restore. %s", str(exp))
            raise exp

    def attach_disk_restore(self):
        """
           Performs attach disk restore for a Azure RM

           Raises:
               Exception:
                   if attach disk restore or validation fails
        """
        _destination_vm_list = []
        _total_disk_attached_list = []
        _after_disk_list = []

        try:
            self.log.info("*" * 10 + "Performing attach disk operation" + "*" * 10)
            _destination_vm_list.append(self.agentless_vm)

            if self._restore_proxy is None:
                self.restore_proxy = self._co_ordinator
            if self._restore_client is None:
                self.restore_client = self.hypervisor
                self.restore_destination_client = self.hvobj
            else:
                if not self.restore_destination_client:
                    self.restore_destination_client, \
                    self._restore_instance_type = self._create_hypervisor_object(
                        self.restore_client)

            _destination_resource_group, _storage_account = self.restore_destination_client. \
                compute_free_resources(_destination_vm_list)

            _destination_vm_disk_list = self.restore_destination_client.VMs[self.agentless_vm]. \
                get_data_disk_info()

            for vm_name in self._vms:
                _disks = []
                _disks_to_attached = self.hvobj.VMs[vm_name].get_data_disk_info()
                if not _disks_to_attached:
                    continue
                self.navigator.navigate_to_hypervisors()
                self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
                self.hypervisor_details_obj.open_subclient(self.subclient)
                self.vsa_sc_obj.restore()
                self.select_restore_obj.select_disk_restore()

                for disk in _disks_to_attached:
                    _disks.append(disk["name"])
                    _total_disk_attached_list.append(self.vm_restore_prefix + disk["name"])

                restore_job = self.disk_level_restore_obj. \
                    azure_attach_disk(vm_name, self.restore_client, self.restore_proxy,
                                      _disks, self.vm_restore_prefix, self.agentless_vm,
                                      _destination_resource_group, _storage_account)

                job_details = self.get_job_status(restore_job)

            after_disk_info = self.restore_destination_client.VMs[self.agentless_vm]. \
                get_data_disk_info()
            for each_disk in after_disk_info:
                _after_disk_list.append(each_disk["name"])

            self.attach_disks_validation(_destination_vm_disk_list, _total_disk_attached_list,
                                         _after_disk_list)
        except Exception as exp:
            self.log.exception("Exception occurred in Attach disk operation %s", str(exp))
            raise exp

    def attach_disks_validation(self, before_dest_data_disk, attached_disk_info,
                                after_dest_disk):
        """
        Performs a attach disk  restore Validation
        Args:
            before_dest_data_disk    (list):     list of datadisks before attach disk restore

            attached_disk_info       (list):     disks attached

            after_dest_disk          (list):      list of data disks after attach disk restore

        """
        try:
            before_length = len(before_dest_data_disk)
            self.log.info("No of disks before restore", before_length)
            after_length = len(after_dest_disk)
            self.log.info("No of disks after restore", after_length)
            attached_length = len(attached_disk_info)
            self.log.info("total no of disks attached", attached_length)
            if before_length + attached_length != after_length:
                raise Exception("No of disks attached validation is successfull")

            for attached_name in attached_disk_info:
                if not attached_name in attached_disk_info:
                    raise Exception("Attach Disk Restore validation failed")
            self.log.info("Disk validation is succesfull")
        except Exception as exp:
            self.log.exception("Exception in disks validation %s", str(exp))
            raise exp


class AmazonAdminConsole(AdminConsoleVirtualServer):
    """
    Amazon class to perform Amazon related adminconsole tasks
    """

    def __init__(self, instance, driver=None, commcell=None, csdb=None):
        """
        Initialize any VMware related variables

        Args:
            instance        (object)    --  the instance class object

            driver          (object)    --  the browser driver object

            commcell        (object)    --  the commcell object

            csdb            (object)    --  the commserve DB object

        """

        super(AmazonAdminConsole, self).__init__(instance, driver, commcell, csdb)
        self.vol_restore_prefix = 'Restored_'
        self._ami = None
        self._availability_zone = None
        self._auto_select_instance_type = True
        self._ins_type = None
        self._network = None
        self._auto_select_security_grp = None
        self._security_group = None
        self._existing_instance = None

    @property
    def existing_instance(self):
        """
        Returns the existing instance name

        Returns:
            existing_instance  (basestring)   --  the existing instance name

        """
        return self._existing_instance

    @existing_instance.setter
    def existing_instance(self, value):
        """
        Sets the existing instance name

        Args:
            value   (basestring)   --  the existing instance name

        """
        self._existing_instance = value

    @property
    def security_group(self):
        """
        Returns the security group name

        Returns:
            security_group  (basestring)   --  the name of the security group

        """
        return self._security_group

    @security_group.setter
    def security_group(self, value):
        """
        Sets the security group name

        Args:
            value   (basestring)   --  the name of the security group

        """
        self._security_group = value

    @property
    def auto_select_security_grp(self):
        """
        Returns the auto_select_security_grp's value

        Returns:
            auto_select_security_grp  (bool)   --  if security group should be auto selected

        """
        return self._auto_select_security_grp

    @auto_select_security_grp.setter
    def auto_select_security_grp(self, value):
        """
        Sets the auto_select_security_grp's value

        Args:
            value   (basestring)   --  if security group should be auto selected

        """
        self._auto_select_security_grp = value

    @property
    def network(self):
        """
        Returns the network

        Returns:
            network  (basestring)   --  the name of the network

        """
        return self._network

    @network.setter
    def network(self, value):
        """
        Sets the network name

        Args:
            value   (basestring)   --  the name of the network

        """
        self._network = value

    @property
    def ins_type(self):
        """
        Returns the instance type

        Returns:
            ins_type  (basestring)   --  the instance type

        """
        return self._ins_type

    @ins_type.setter
    def ins_type(self, value):
        """
        Sets the ins_type name

        Args:
            value   (basestring)   --  the instance type

        """
        self._ins_type = value

    @property
    def auto_select_instance_type(self):
        """
        Returns the auto_select_instance_type's value

        Returns:
            auto_select_instance_type  (bool)   --  if instance type should be auto selected

        """
        return self._auto_select_instance_type

    @auto_select_instance_type.setter
    def auto_select_instance_type(self, value):
        """
        Sets the auto_select_instance_type's value

        Args:
            value   (bool)   --  if instance type should be auto selected

        """
        self._auto_select_instance_type = value

    @property
    def availability_zone(self):
        """
        Returns the availability_zone

        Returns:
            availability_zone  (basestring)   --  the name of the availability zone

        """
        return self._availability_zone

    @availability_zone.setter
    def availability_zone(self, value):
        """
        Sets the availability_zone name

        Args:
            value   (basestring)   --  the name of the availability zone

        """
        self._availability_zone = value

    @property
    def ami(self):
        """
        Returns the ami name

        Returns:
            ami  (basestring)   --  the name of the ami

        """
        return self._ami

    @ami.setter
    def ami(self, value):
        """
        Sets the ami name

        Args:
            value   (basestring)   --  the name of the ami

        """
        self._ami = value

    @property
    def vol_restore_prefix(self):
        """
        Returns the prefix to be attached to the restore VM name

        Returns:
            vm_restore_prefix  (basestring)   --  the prefix to tbe attached to the restore VM name

        """
        return self._vol_restore_prefix

    @vol_restore_prefix.setter
    def vol_restore_prefix(self, value):
        """
        Sets the prefix to be attached to the restore VM name

        Args:
            value   (basestring)    --  the prefix to be attached to the restore VM name

        """
        self._vol_restore_prefix = value

    def attach_disk_restore(self, attach_vol_to, vm_level=False):
        """
        Attach disk restore

        Args:
            attach_vol_to (basestring) : Disk has to be attached to existing instance or new volume

            vm_level      (bool)       :  if the restore should be initiated from VM level

        Raises:
            Exception:
                if the Attach disk restore or validation fails
        """
        try:
            self.log.info("*" * 10 + "Performing disk level restore" + "*" * 10)

            if self._restore_proxy is None:
                self.restore_proxy = self._co_ordinator

            if self.restore_client is None:
                self.restore_client = self.hypervisor
                self.restore_destination_client = self.hvobj
            else:
                if not self.commcell.clients.has_client(self.restore_client):
                    raise Exception("The destination client is not a client in the Commserv")
                if not self.restore_destination_client:
                    self.restore_destination_client, \
                    self._restore_instance_type = self._create_hypervisor_object(
                        self.restore_client)

            for _vm in self._vms:
                disks = self._get_disk_list(self.hvobj.VMs[_vm].disk_list)
                if not vm_level:
                    self.navigator.navigate_to_hypervisors()
                    self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
                    self.hypervisor_details_obj.open_subclient(self.subclient)
                    self.vsa_sc_obj.restore()
                else:
                    self._navigate_to_vm_restore(_vm)

                self.select_restore_obj.select_disk_restore()
                if attach_vol_to == 'Other instance':
                    if self.existing_instance is None:
                        raise Exception("Restore to Other Instance is not given by user")
                    restore_job = self.disk_level_restore_obj.aws_attach_disk_restore(
                        _vm,
                        disks,
                        attach_vol_to,
                        self.hypervisor,
                        self.restore_client,
                        self.existing_instance,
                        self.vm_restore_prefix
                    )
                else:
                    if self.ami is None:
                        raise Exception("Ami is mandatory for new instance restore")
                    inst_display_name = self.vm_restore_prefix + _vm

                    restore_job = self.disk_level_restore_obj.aws_attach_disk_restore(
                        _vm, disks,
                        attach_vol_to,
                        self.restore_client,
                        self.restore_proxy,
                        vol_prefix=self.vm_restore_prefix,
                        power_on=self.power_on_after_restore,
                        overwrite=self.unconditional_overwrite,
                        inst_display_name=inst_display_name,
                        ami=self.ami,
                        available_zone=self.availability_zone,
                        auto_select_instance_type=self.auto_select_instance_type,
                        ins_type=self.ins_type,
                        network=self.network,
                        auto_select_security_grp=self.auto_select_security_grp,
                        security_group=self.security_group)
                job_details = self.get_job_status(restore_job)
                if job_details['Status'] not in ['Completed',
                                                 "Completed w/ one or more errors"]:
                    raise Exception("Restore job failed. Please check the logs")

                if attach_vol_to == 'New instance':
                    for _vm in self._vms:
                        restore_vm = self.vm_restore_prefix + _vm
                        self.vm_restore_validation(_vm, restore_vm)

        except Exception as exp:
            self.log.exception("Exception occurred during Attach disk restore. %s", str(exp))
            raise exp


class VcloudAdminConsole(AdminConsoleVirtualServer):
    """
       Vcloud class to perform Vcloud related adminconsole tasks
       """

    def __init__(self, instance, driver=None, commcell=None, csdb=None):
        """
        Initialize any vCloud VM related variables

        Args:
            instance    (object):  object of the instance class

            driver      (object):  the browser object

            commcell    (object):  an instance of the commcell class

            csdb        (object): the cs DB object

        """
        super(VcloudAdminConsole, self).__init__(instance, driver, commcell, csdb)

        self._full_vm_restore_options = {}
        self._organization = None
        self._org_vdc = None
        self._vapp_name = None
        self._network_setting = False
        self._source_network = None
        self._destination_network = None
        self._destination_vapp = None
        self._destination_vdc = None
        self._new_vapp = False
        self._vm_restore_prefix = 0

    @property
    def organization(self):
        """
        Returns the  organization

        Returns:
            organization     (str):    value of organization

        """
        return self._organization

    @organization.setter
    def organization(self, value):
        """
        Sets the value of  organization

        Args:
            value   (str):  value for organization
        """
        self._organization = value

    @property
    def org_vdc(self):
        """
        Returns the live org_vdc value

        Returns:
            org_vdc     (str): True / False to enable or disable managed_vm
        """
        return self._org_vdc

    @org_vdc.setter
    def org_vdc(self, value):
        """
        Sets the org_vdc value

        Args:
            value     :  org_vdc
        """
        self._org_vdc = value

    @property
    def vapp_name(self):
        """
        Returns the live vapp_name
        Returns:
            storage_account     (str):  value of vapp_name

        """
        return self._vapp_name

    @vapp_name.setter
    def vapp_name(self, value):
        """
        Sets the value of  vapp_name

        Args:
            value   (str): value for _vapp_name
        """
        self._vapp_name = value

    @property
    def network_setting(self):
        """
        Returns the live _network_setting

        Returns:
            vmsize     (str):  value of vm_size

        """
        return self._network_setting

    @network_setting.setter
    def network_setting(self, value):
        """
        Sets the live _network_setting

        Args:
            value   (str):  value for vmsize
        """
        self._network_setting = value

    @property
    def source_network(self):
        """
        Returns the live source_network

        Returns:
            network_interface     (bool):  True / False to enable or disable live recovery

        """
        return self._source_network

    @source_network.setter
    def source_network(self, value):
        """
        Sets the live _source_network
        Args:
            value  (str): value for _source_network
        """
        self._source_network = value

    @property
    def destination_network(self):
        """
        Returns the live security_group

        """
        return self._destination_network

    @destination_network.setter
    def destination_network(self, value):
        """
        Sets value of _destination_network
        Args:
            value   (str):  value for _destination_network
        """
        self._destination_network = value

    @property
    def destination_vapp(self):
        """
        Returns the value of Destination Vapp

        """
        return self._destination_vapp

    @destination_vapp.setter
    def destination_vapp(self, value):
        """
        Sets value of _destination_Vapp
        Args:
            value   (str):  value for _destination_Vapp
        """
        self._destination_vapp = value

    @property
    def new_vapp(self):
        """
        Returns the value of Destination Vapp

        """
        return self._new_vapp

    @new_vapp.setter
    def new_vapp(self, value):
        """
        Sets value of _destination_Vapp
        Args:
            value   (str):  value for _destination_Vapp
        """
        self._new_vapp = value

    def _set_vcloud_vm_restore_options(self):
        """
                Sets the VM restore options for each VM to be restored

                Raises:
                    Exception:
                        if there is any exception in setting the restore options
        """

        try:
            for _vm in self._vms:
                if (self.vapp_name and self.source_network) is None:
                    self.vapp_name, self.source_network = self.hvobj. \
                        compute_free_resources(self._vms)
                if self.destination_vapp is not None:
                    if self.new_vapp:
                        self.vapp_name = str(self.vm_restore_prefix) + self.destination_vapp
                    else:
                        self.vapp_name = self.destination_vapp
                if self.destination_network is None:
                    self.destination_network = self.source_network

                self._full_vm_restore_options[_vm]['vapp_name'] = self.vapp_name
                self._full_vm_restore_options[_vm]['source_network'] = self.source_network
                self._full_vm_restore_options[_vm]['destination_network'] = self.destination_network

        except Exception as exp:
            self.log.exception("Exception occurred during setting restore options")
            raise exp

    def full_vm_restore(self, vm_level=False):
        """
                Performs full VM restore for a vCloud VM

                Raises:
                    Exception:
                        if full VM restore or validation fails

                """
        try:
            self.log.info("*" * 10 + "Performing full VM restore" + "*" * 10)
            self.hvobj.vcloud_auth_token = self.hvobj.check_for_login_validity()
            if self._restore_proxy is None:
                self.restore_proxy = self._co_ordinator
            if self._restore_client is None:
                self.restore_client = self.hypervisor
                self.restore_destination_client = self.hvobj
            else:
                if not self.restore_destination_client:
                    self.restore_destination_client, \
                    self._restore_instance_type = self._create_hypervisor_object(
                        self.restore_client)

            self._full_vm_restore_options = dict.fromkeys(self._vms, {})
            self._set_vcloud_vm_restore_options()
            if self.org_vdc is None:
                raise Exception("Org_vdc is none.")

            if vm_level:
                for _vm in self._vms:
                    self._navigate_to_vm_restore(_vm)
                    self.select_restore_obj.select_full_vm_restore()

            else:
                self.navigator.navigate_to_hypervisors()
                self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
                self.hypervisor_details_obj.open_subclient(self.subclient)
                self.vsa_sc_obj.restore()
                self.select_restore_obj.select_full_vm_restore()

            restore_job = self.full_vm_restore_obj.vcloud_full_vm_restore(
                self._vms,
                self.restore_proxy,
                self.restore_client,
                self.full_vm_in_place,
                self._full_vm_restore_options,
                self.org_vdc,
                power_on=True,
                overwrite=True,
                restore_vapp=True,
                restore_prefix=str(self.vm_restore_prefix)
            )

            job_details = self.get_job_status(restore_job)
            if not job_details:
                raise Exception("Restore job failed. Please check the logs")

            for _vm in self._vms:
                if self.full_vm_in_place:
                    restore_vm = _vm
                else:
                    restore_vm = str(self.vm_restore_prefix) + _vm
                attempt = 0
                while attempt < 3:
                    self.log.info("attempt =" + str(attempt))
                    try:
                        self.vm_restore_validation(_vm, restore_vm)
                        break
                    except:
                        attempt = attempt + 1
                        time.sleep(60)
                if attempt >= 3:
                    self.vm_restore_validation(_vm, restore_vm)
                attempt = 0
                while attempt < 3:
                    self.log.info("attempt =" + str(attempt))
                    try:
                        self.vm_restore_validation(_vm, restore_vm)
                        if self._vm_restore_prefix>0:
                            self.post_restore_clean_up(status= True)
                        break
                    except:
                        attempt = attempt + 1
                        time.sleep(60)
                if attempt >= 3:
                    self.vm_restore_validation(_vm, restore_vm)
                    if self._vm_restore_prefix > 0:
                        self.post_restore_clean_up(status=True)
            self._vm_restore_prefix += 1
        except Exception as exp:
            self.log.exception("Exception occurred during full VM restore. %s", str(exp))
            raise exp


class OracleCloudInfrastructureAdminConsole(AdminConsoleVirtualServer):
    """
    Oracle Cloud Infrastructure class to perform functions specific to Oracle Cloud Infrastructure.
    """

    def __init__(self, instance, driver=None, commcell=None, csdb=None):
        super(OracleCloudInfrastructureAdminConsole, self).__init__(
            instance, driver, commcell, csdb)
        self.oci_subnet_name = None
        self.oci_shape_1 = None

    @property
    def oci_shape(self):
        """ set shape"""
        return self.oci_shape_1

    @oci_shape.setter
    def oci_shape(self, value):
        """ sets shape """
        self.oci_shape_1 = value

    @property
    def restore_location(self):
        """ restore location """
        return self._restore_location

    @restore_location.setter
    def restore_location(self, value):
        self._restore_location = value

    @property
    def oci_subnet_name_prop(self):
        """ oci subnet name """
        return self.oci_subnet_name

    @oci_subnet_name_prop.setter
    def oci_subnet_name_prop(self, value):
        self.oci_subnet_name = value

    @property
    def oci_shape(self):
        """ set shape"""
        return self.oci_shape2

    @oci_shape.setter
    def oci_shape(self, value):
        """ sets shape """
        self.oci_shape2 = value

    def full_vm_restore(self, vm_level=False):
        """
        Full VM restore for subclient.
        Args:
            vm_level: Enable vm_level restore

        Returns:

        """
        try:
            if self._restore_proxy is None:
                self.restore_proxy = self._co_ordinator
            if self.restore_client is None:
                self.restore_client = self.hypervisor

            if vm_level:
                for _vm in self._vms:
                    try:
                        self._navigate_to_vm_restore(_vm)
                        self.select_restore_obj.select_full_vm_restore()
                    except Exception as exp:
                        self.log.info(exp)
            else:
                try:
                    self.hypervisor_ac_obj.navigate_to_hypervisors()
                    self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
                    self.hypervisor_details_obj.open_subclient(self.subclient)
                    self.vsa_sc_obj.restore()
                    self.select_restore_obj.select_full_vm_restore()
                except Exception as exp:
                    raise Exception(exp)
            vm_obj = self.hvobj.VMs[list(self._vms)[0]]
            restore_job = self.full_vm_restore_obj.oci_full_vm_restore(
                self._vms,
                str(self.restore_proxy),
                str(self.restore_client),
                self.power_on_after_restore,
                self.full_vm_in_place,
                vm_obj,
                restore_prefix=self.vm_restore_prefix
            )
            job_details = self.get_job_status(restore_job)
            if not job_details:
                raise Exception("Restore job failed. Please check the logs")
            self.restore_destination_client = self.hvobj
            for vm_name in self._vms:
                if vm_name[-3:] == "_DN":
                    vm_name = vm_name[:-3]
                try:
                    self.vm_restore_validation(vm_name, self.vm_restore_prefix + vm_name)
                except Exception as exp:
                    self.log.info(f"VM restore validation failed for VM: {vm_name}")
                    raise exp

        except Exception as exp:
            self.log.exception(
                "Exception occurred during OCI Full VM restore. %s", str(exp))
            raise exp

    def validate_live_sync(self, replication_group_name=None):
        """
                Validate Live Sync function, validates the live sync operation.
                Args:
                    replication_group_name (basestring)   --   Name of the replication group,
                    if not set, a replication group is automatically configured.
                Raises:
                    Exception:
                        If the validation fails
        """
        vm_backupgroup = self.subclient
        vm_instance = self.hvobj.VMs[list(self.hvobj.VMs)[0]]
        replication_target_name = "Automation_Replication_Target"
        created_replication_group = False
        dict_vals = {
            'hypervisor': self._hypervisor,
            'replication_group_name': "Automation_Replication_Group",
            'vm_backupgroup': vm_backupgroup,
            'replication_target': replication_target_name,
            'proxy': self._co_ordinator.machine_name,
            'compartment': vm_instance.compartment_name,
            'datastore': vm_instance.datastore,
            'vcn': vm_instance.vcn_name,
            'shape': vm_instance.instance_size,
            'vm_name': vm_instance.vm_name,
            'subnet_name': self.oci_subnet_name if self.oci_subnet_name else None
        }
        self.replication_group_obj.navigate_to_replication_groups()
        if not replication_group_name:
            self.replication_group_obj.configure_ls_replication_group()
            self.configure_replication_group_obj.select_live_sync()
            self.configure_vsa_replication_group_obj.configure_vsa_replication_group(dict_vals)
            replication_group_name = "Automation_Replication_Group"
            created_replication_group = True
        self.replication_group_obj.access_group(replication_group_name)
        replication_job_id = self.replication_group_details_obj.get_replication_job()
        time.sleep(30)
        try:
            self.log.info(f"Getting status for the job {replication_job_id}")
            job_details = self.get_job_status(replication_job_id)
            if job_details['Status'] not in ["Completed", "Completed w/ one or more errors"]:
                raise Exception("Replication job failed. Please check the logs")
        except Exception as exp:
            self.log.exception(
                "Exception occurred in getting the job status: %s", str(exp))
            raise exp
        self.replication_group_obj.navigate_to_replication_groups()
        self.replication_group_obj.access_group(replication_group_name)
        sync_status = self.replication_group_details_obj.get_sync_status()
        assert self.validate_status(sync_status, "in sync"), "Sync status is not 'In Sync'"
        self.restore_destination_client = self.hvobj
        source_vms, destination_vms = \
            self.replication_group_details_obj.get_source_and_destination()
        for vm_name in self.hvobj.VMs.keys():
            if vm_name in source_vms or vm_name in destination_vms:
                self.hvobj.VMs[vm_name].machine.close()
        for vm_index in range(len(source_vms)):
            vm_restore_name = destination_vms[vm_index]
            source_vm_name = source_vms[vm_index]
            if vm_restore_name[-3:] == "_DN":
                vm_restore_name = vm_restore_name[:-3]
            if source_vm_name[-3:] == "_DN":
                source_vm_name = source_vm_name[:-3]
            try:
                self.vm_restore_validation(source_vm_name, vm_restore_name)
            except Exception as exp:
                self.log.info("VM restore validation failed for VM: {}".format(source_vms[vm_index]))
                raise exp
        for vm_name in self.hvobj.VMs.keys():
            if vm_name in destination_vms:
                self.hvobj.VMs[vm_name].vm_power_off()
        if created_replication_group:
            self.log.info("Deleting Replication Group")
            self.replication_group_obj.navigate_to_replication_groups()
            self.replication_group_obj.delete_group(replication_group_name)
            self.recovery_target_obj.navigate_to_replication_targets()
            self.recovery_target_obj.action_delete(replication_target_name)

    def perform_unplanned_failover(self, source_vms):
        """
        Perform Unplanned failover
        Args:
            source_vms: List of source VMs to perform failover

        Returns: Job ID of failover job

        """
        job_id = self.replication_group_details_obj.unplanned_failover(source_vms)
        return job_id

    def perform_planned_failover(self, source_vms):
        """
        Perform planned failover
        Args:
            source_vms: List of source VMs to perform failover

        Returns: Job ID of failover job

        """
        job_id = self.replication_group_details_obj.planned_failover(source_vms)
        return job_id

    def validate_unplanned_failover(self, replication_group_name):
        """
         Validate Planned Failover.

          Args:
              replication_group_name (basestring)   --   Name of the replication group,
                if not set, a replication group is automatically configured.

          Raises:
              Exception:
              If the validation fails
        """
        self.replication_group_obj.navigate_to_replication_groups()
        self.replication_group_obj.access_group(replication_group_name)
        source_vms, destination_vms = \
            self.replication_group_details_obj.get_source_and_destination()
        job_id = self.perform_unplanned_failover(source_vms)
        time.sleep(15)
        failover_status = self.replication_group_details_obj.get_failover_status()
        assert self.validate_status(failover_status, "failover in progress"), "Failover status not valid"

        try:
            self.log.info("Getting status for the job %s", job_id)
            job_details = self.get_job_status(job_id)
            if not job_details:
                raise Exception("Failover job failed. Please check the logs")

        except Exception as exp:
            self.log.exception(
                "Exception occurred in getting the job status: %s", str(exp))
            raise exp
        if not self.hvobj:
            self.hvobj = self._create_hypervisor_object()[0]
        for vm_index in range(len(source_vms)):
            source_vm = source_vms[vm_index]
            destination_vm = destination_vms[vm_index]
            if source_vm[-3:] == "_DN":
                source_vm = source_vm[:-3]
            if destination_vm[-3:] == "_DN":
                destination_vm = destination_vm[:-3]
            instance_source = self.hvobj.get_vm_property(source_vm)
            instance_destination = self.hvobj.get_vm_property(destination_vm)
            assert instance_source['power_state'] == "STOPPED" and \
                   instance_destination['power_state'] == "RUNNING"
            self.log.info("Source VM is POWERED OFF and Destination VM is RUNNING")
        self.log.info("Unplanned failover validation successful")

    def validate_planned_failover(self, replication_group_name):
        """
        Validate Planned Failover
        Args:
              replication_group_name (basestring)   --   Name of the replication group

        Raises:
            Exception:
              If the validation fails
        """
        self.replication_group_obj.navigate_to_replication_groups()
        self.replication_group_obj.access_group(replication_group_name)
        recovery_target_name = self.replication_group_details_obj.get_recovery_target()
        self.recovery_target_obj.navigate_to_replication_targets()
        self.recovery_target_obj.select_replication_target(recovery_target_name)

        destination_client_name = self.recovery_target_obj.get_destination_hypervisor()

        self.replication_group_obj.navigate_to_replication_groups()
        self.replication_group_obj.access_group(replication_group_name)

        source_vms, destination_vms = \
            self.replication_group_details_obj.get_source_and_destination()
        if not self.hvobj:
            self.hvobj = self._create_hypervisor_object()[0]
        if not self.testdata_path:
            self.testdata_path = VirtualServerUtils.get_testdata_path(self.hvobj.machine)
        checksum_lst = []
        self._vms = []
        for vm_name in source_vms:
            if vm_name[-3:] == "_DN":
                vm_name = vm_name[:-3]
            if vm_name not in self.hvobj.VMs:
                self.hvobj.VMs = vm_name
                self._vms.append(vm_name)
                self.cleanup_testdata()
            self.hvobj.VMs[vm_name].machine.generate_test_data(self.testdata_path)
            checksum_lst.append(self.hvobj.VMs[vm_name].machine.get_checksum_list(self.testdata_path))
        if destination_client_name[-3:] == "_DN":
            destination_client_name = destination_client_name[:-3]
        self.destination_client = self._create_hypervisor_object(destination_client_name)[0]

        # Validating successfull completion of job
        job_id = self.perform_planned_failover(source_vms)
        time.sleep(15)
        failover_status = self.replication_group_details_obj.get_failover_status()
        assert self.validate_status(failover_status, "failover in progress"), "Failover status not valid"
        try:
            self.log.info("Getting status for the job %s", job_id)
            job_details = self.get_job_status(job_id)
            if not job_details:
                raise Exception("Failover job failed. Please check the logs")
        except Exception as exp:
            self.log.exception(
                "Exception occurred in getting the job status: %s", str(exp))
            raise exp

        # Checksum list on destination
        dest_checksum_lst = []
        for vm_name in destination_vms:
            if vm_name[-3:] == "_DN":
                vm_name = vm_name[:-3]
            self.destination_client.VMs = vm_name
            dest_vm = self.destination_client.VMs[vm_name]
            dest_vm.vm_power_on()
            dest_checksum_lst.append(dest_vm.machine.get_checksum_list(self.testdata_path))

        # Testdata validation
        assert checksum_lst == dest_checksum_lst
        self.log.info("Testdata Validation Completed Successfully.")

        # Config Validation
        for source_vm_name, dest_vm_name in zip(source_vms, destination_vms):
            if source_vm_name[-3:] == "_DN":
                source_vm_name = source_vm_name[:-3]
            if dest_vm_name[-3:] == "_DN":
                dest_vm_name = dest_vm_name[:-3]
            source_vm = self.hvobj.VMs[source_vm_name]
            dest_vm = self.destination_client.VMs[dest_vm_name]
            _source = self.VmValidation(source_vm)
            _dest = self.VmValidation(dest_vm)
            self.log.info("Powering off source and destination VMs")
            source_vm.vm_power_off()
            dest_vm.vm_power_off()

            assert _source == _dest, "Failed while config validation"
        self.log.info("Config Validation Successfull")

    def validate_failback(self, replication_group_name):
        """
        Validate Failback.

         Args:
              replication_group_name (basestring)   --   Name of the replication group

         Raises:
             Exception:
                If the validation fails
        """
        self.replication_group_obj.navigate_to_replication_groups()
        self.replication_group_obj.access_group(replication_group_name)
        source_vms, destination_vms = \
            self.replication_group_details_obj.get_source_and_destination()

        failover_status = self.replication_group_details_obj.get_failover_status()
        assert self.validate_status(failover_status, "failover complete"), "Failover status not valid"

        job_id = self.replication_group_details_obj.failback(source_vms)
        try:
            self.log.info("Getting status for the job %s", job_id)
            job_details = self.get_job_status(job_id)
            if not job_details:
                raise Exception("Failback job failed. Please check the logs")

        except Exception as exp:
            self.log.exception(
                "Exception occurred in getting the job status: %s", str(exp))
            raise exp
        if not self.hvobj:
            self.hvobj = self._create_hypervisor_object()[0]
        for vm_index in range(len(source_vms)):
            source_vm = source_vms[vm_index]
            destination_vm = destination_vms[vm_index]
            if source_vm[-3:] == "_DN":
                source_vm = source_vm[:-3]
            if destination_vm[-3:] == "_DN":
                destination_vm = destination_vm[:-3]
            instance_source = self.hvobj.get_vm_property(source_vm)
            instance_destination = self.hvobj.get_vm_property(destination_vm)
            assert instance_source['power_state'] == "RUNNING" and \
                   instance_destination['power_state'] == "STOPPED"
            self.log.info("Source VM is RUNNING and Destination VM is POWERED OFF")
        self.log.info("Failback validation successful")

        self.replication_group_obj.navigate_to_replication_groups()
        self.replication_group_obj.access_group(replication_group_name)
        failover_status = self.replication_group_details_obj.get_failover_status()

        assert self.validate_status(failover_status, ""), "Failover status not valid"
        self.log.info("Failback validation successfull")

    def validate_status(self, status_list, value):
        """
        Validates the status

        Args:
            status_list: a list
            value: a value

        Returns: boolean

        """
        for status in status_list:
            if status.lower() != value:
                return False
        return True

    def _compute_credentials(self, client_name):
        """
        Compute Credentials function, this has been overriden because the default
        function does not account for ocid, etc

        Args:
            client_name (basestring)   --   the name of the client whose credentials are
                                            to be obtained

        Raises:
            Exception:
                if compute credentials fails to get the credentials from the database


        """
        try:
            row_begin = '5'
            row_end = '15'
            _query = "select ac.*\
                                  from (select Row_Number() Over ( Order By id ) as num, \
                                  * from app_Instanceprop where componentNameid =( \
                                  select TOP 1 instance  from APP_Application where clientId= ( \
                                  Select TOP 1 id from App_Client where name = '%s') \
                                   and appTypeId = '106'))ac \
                                  where ac.num between %s and %s" % (client_name, row_begin, row_end)

            self.csdb.execute(_query)
            _results = self.csdb.fetch_all_rows()
            if not _results:
                raise Exception("An exception occurred getting server details")

            for item in _results:
                if 'Oracle Cloud Infrastructure Tenancy Id' in item:
                    self.oci_tenancy_id = item[5].strip()
                elif 'Oracle Cloud Infrastructure User Id' in item:
                    self.oci_user_id = item[5].strip()
                elif 'Oracle Cloud Infrastructure Finger Print' in item:
                    self.oci_finger_print = item[5].strip()
                elif 'Oracle Cloud Infrastructure Private Key Password' in item:
                    self.oci_private_key_password = \
                        cvhelper.format_string(self.commcell, item[5].strip())
                elif 'Oracle Cloud Infrastructure Region Name' in item:
                    self.oci_region_name = item[5].strip()

        except Exception as err:
            self.log.exception(
                "An Exception occurred in getting credentials for Compute Credentials  %s", err)
            raise err


class GoogleCloudAdminConsole(AdminConsoleVirtualServer):
    """
       Google Cloud class to perform functions specific to Google Cloud.
    """

    def __init__(self, instance, driver=None, commcell=None, csdb=None):
        super(GoogleCloudAdminConsole, self).__init__(
            instance, driver, commcell, csdb)

    def full_vm_restore(self):
        """
                                Performs full VM restore for a Google Cloud VM

                                Raises:
                                    Exception:
                                        if full VM restore or validation fails

                                """

        try:
            self.log.info("Restoring the Google Cloud Instance")
            if self._restore_proxy is None:
                self.restore_proxy = self._co_ordinator

            if self.restore_client is None:
                self.restore_client = self.hypervisor
                self.restore_destination_client = self.hvobj

            self.navigator.navigate_to_hypervisors()
            self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
            self.hypervisor_details_obj.open_subclient(self.subclient)
            self.vsa_sc_obj.restore()
            self.select_restore_obj.select_full_vm_restore()
            self.vm_restore_prefix = self.vm_restore_prefix.lower().replace("_", "")

            restore_job = self.full_vm_restore_obj.google_cloud_full_vm_restore(self.vms, self.vm_restore_prefix,
                                                                                self.full_vm_in_place,
                                                                                proxy=self.restore_proxy,
                                                                                power_on=self.power_on_after_restore,
                                                                                overwrite_instance=
                                                                                self.unconditional_overwrite)
            job_details = self.get_job_status(restore_job)
            for _vm in self._vms:
                restore_vm = self.vm_restore_prefix + _vm
                self.vm_restore_validation(_vm, restore_vm)

        except Exception as exp:
            self.log.exception("Exception occurred during Google Cloud VM restore. %s", str(exp))
            raise exp

    def guest_files_restore(self, in_place=False, vm_level=False):
        """
        File level restore to source VM

        Args:
            in_place    (bool):     if the files should be restored to the source path

            vm_level    (bool):     if the restore should be initiated from VM level

        Raises:
            Exception:
                if the file level restore fails

        """
        try:
            if self._restore_proxy is None:
                self.restore_proxy = self._co_ordinator

            for _vm in self._vms:
                self.agentless_vm = _vm

                for _drive, _folder in self.hvobj.VMs[_vm].drive_list.items():
                    if in_place:
                        restore_path = _folder
                        validate_path = self.hvobj.VMs[self.agentless_vm].machine.join_path(
                            restore_path, self.backup_type.name, "TestData", self.pid)
                    else:
                        restore_path = self.hvobj.VMs[self.agentless_vm].machine.join_path(
                            _folder, "AdminConsole", self.agentless_vm, _drive)
                        validate_path = self.hvobj.VMs[self.agentless_vm].machine.join_path(
                            restore_path, self.backup_type.name, "TestData", self.pid)

                    if self.hvobj.VMs[_vm].machine.check_directory_exists(restore_path):
                        if restore_path != _folder:
                            self.hvobj.VMs[_vm].machine.remove_directory(restore_path)

                    if self.ci_enabled:
                        self.navigator.navigate_to_virtual_machines()
                        self.virtual_machines_obj.open_vm(_vm)
                        self.vm_details_obj.vm_search_content(self.backup_folder_name)
                        self.vsa_search_obj.search_for_content(
                            file_name=self.backup_folder_name, contains=_drive,
                            include_folders=True)
                        self.vsa_search_obj.select_data_type("Folder")
                        self.vsa_search_obj.validate_contains(self.backup_folder_name,
                                                              name=True)
                        self.vsa_search_obj.validate_contains(_drive, name=False,
                                                              folder=True)
                    else:
                        if not vm_level:
                            self.navigator.navigate_to_hypervisors()
                            self.hypervisor_ac_obj.select_hypervisor(self.hypervisor)
                            self.hypervisor_details_obj.open_subclient(self.subclient)
                            self.vsa_sc_obj.restore()
                        else:
                            self._navigate_to_vm_restore(_vm)
                        if self.run_aux:
                            self.select_restore_obj.select_source_copy(self.aux_copy)
                        elif self.backup_method.lower() == "snap" and self.snap_restore:
                            self.select_restore_obj.select_source_copy(
                                self.storage_policy.snap_copy
                            )
                        elif self.backup_method.lower() == "snap" and not self.snap_restore:
                            self.select_restore_obj.select_source_copy("primary")
                        self.select_restore_obj.select_guest_files()
                        self.select_restore_obj.latest_backups()
                        if _folder == "/":
                            self.restore_vol_obj.select_volume(_vm, _folder)
                        else:
                            self.restore_vol_obj.select_volume(_vm, _drive)

                    restore_job = self.restore_files_obj.submit_google_cloud_vm_restore(
                        [self.backup_type.name.upper()], self.agentless_vm,
                        self.restore_proxy,
                        self.hvobj.VMs[_vm].machine.username,
                        self.hvobj.VMs[_vm].machine.password,
                        restore_path
                    )
                    self.get_job_status(restore_job)
                    self.fs_testdata_validation(self._restore_proxy, validate_path)
        except Exception as exp:
            self.log.exception("Exception occurred during guest files restore. %s", str(exp))
            raise exp
