# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be used to run
basic operations from the test case on Users page.

To begin, create an instance of UserMain for users test case.

Functions:

add_new_user                    --                Calls method from base file to add new
                                                    user in Admin Console

delete_user                     --                Calls method to delete users created in
                                                    Admin Console

edit_local_user_details         --                Calls method to edit user details

validate_user                   --                Validates if the user details are retained
                                                    correctly against usr inputs

__set_default_values_if_none    --                Organizes user input for comparison against
                                                    displayed values

logout_of_admin_console         --                Method to logout of admin console
"""

import ast

from AutomationUtils import logger

from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.AdminConsolePages.Users import Users
from Web.AdminConsole.AdminConsolePages.UserDetails import UserDetails

from cvpysdk.security.user import Users as UsersAPI


class UserMain:
    """
    Helper file to provide arguments and handle function call to base files
    """

    def __init__(self, admin_console, commcell):
        """
        Initialize method for UserMain

        Args:
            admin_page   (object)    --  AdminPage class object
            commcell    (object)    --  CommCell object
        """
        self.__admin_console = admin_console
        self.__navigator = admin_console.navigator
        self.__table = Table(self.__admin_console)
        self.__users = Users(self.__admin_console)
        self.__user_details = UserDetails(self.__admin_console)
        self.__users_api = UsersAPI(commcell)

        self.log = logger.get_log()

        self._user_name = 'User_53652'
        self._new_user_name = None
        self._email = 'xyz@commvault.com'
        self._user_groups = []
        self._full_name = 'User_53652'
        self._system_password = False
        self._password = '######'
        self._admin_password = None
        self._external_provider = None
        self._invite_user = False
        self._user_enabled = True
        self._plan = None
        self._old_user_name = None
        self.__user_list = []
        self.__external_user_list = []
        self._owner_transfer = 'admin'

    @property
    def user_name(self):
        """ Get user_name"""
        return self._user_name

    @user_name.setter
    def user_name(self, value):
        """ Set user_name"""
        self._user_name = value

    @property
    def new_user_name(self):
        """ Get new user name for editing """
        return self._new_user_name

    @new_user_name.setter
    def new_user_name(self, value):
        """ Set new_user_name for editing"""
        self._new_user_name = value

    @property
    def email(self):
        """ Get email"""
        return self._email

    @email.setter
    def email(self, value):
        """ Set email"""
        self._email = value

    @property
    def user_groups(self):
        """ Get user_groups"""
        return self._user_groups

    @user_groups.setter
    def user_groups(self, value):
        """ Set user_groups"""
        self._user_groups = value

    @property
    def full_name(self):
        """ Get full_name"""
        return self._full_name

    @full_name.setter
    def full_name(self, value):
        """ Set full_name"""
        self._full_name = value

    @property
    def system_password(self):
        """ Get system_password"""
        return self._system_password

    @system_password.setter
    def system_password(self, value):
        """ Set system_password"""
        self._system_password = value

    @property
    def password(self):
        """ Get password"""
        return self._password

    @password.setter
    def password(self, value):
        """ Set password"""
        self._password = value

    @property
    def admin_password(self):
        """ Get admin_password"""
        return self._admin_password

    @admin_password.setter
    def admin_password(self, value):
        """ Set admin_password"""
        self._admin_password = value

    @property
    def external_provider(self):
        """ Get external_provider"""
        return self._external_provider

    @external_provider.setter
    def external_provider(self, value):
        """ Set external_provider"""
        self._external_provider = value

    @property
    def invite_user(self):
        """ Get invite_user"""
        return self._invite_user

    @invite_user.setter
    def invite_user(self, value):
        """ Set invite_user"""
        self._invite_user = value

    @property
    def user_enabled(self):
        """ Get user_enabled"""
        return self._user_enabled

    @user_enabled.setter
    def user_enabled(self, value):
        """ Set user_enabled"""
        self._user_enabled = value

    @property
    def plan(self):
        """ Get plan"""
        return self._plan

    @plan.setter
    def plan(self, value):
        """ Set plan"""
        self._plan = value

    def add_new_local_user(self):
        """calls add user method from Users page"""

        self.__navigator.navigate_to_users()
        self.__users.add_local_user(self.email,
                                    self.user_name,
                                    self.full_name,
                                    self.user_groups,
                                    self.system_password,
                                    self.password,
                                    self.invite_user)
        self.__user_list.append(self.user_name)

    def add_new_external_user(self):
        """calls add user method from Users page"""

        self.__navigator.navigate_to_users()
        self.__users.add_external_user(self.external_provider,
                                       self.user_name,
                                       self.email,
                                       self.user_groups)
        self.__external_user_list.append(self.external_provider+"\\"+self.user_name)

    def delete_user(self):
        """Calls method to delete the specified user"""

        if self.__user_list:
            self.__navigator.navigate_to_users()
            for user_name in self.__user_list:
                self.__users.delete_user(user_name,
                                         self._owner_transfer)
        if self.__external_user_list:
            for user_name in self.__external_user_list:
                self.__users_api.refresh()
                self.__users_api.delete(user_name, self._owner_transfer)
                self.log.info('User : %s is deleted successfully', user_name)

    def edit_local_user_details(self):
        """
        Calls method to edit user details
        """
        self.__navigator.navigate_to_users()
        self.__table.access_link(self.user_name)
        self.__user_details.edit_user(self.email,
                                      self.full_name,
                                      self.new_user_name,
                                      self.user_groups,
                                      self.plan,
                                      self.user_enabled,
                                      self.password,
                                      self.admin_password)
        if self.user_name != self.new_user_name:
            self.__user_list.append(self.new_user_name)
            if self.user_name in self.__user_list:
                self.__user_list.remove(self.user_name)
            self._user_name = self.new_user_name

    def validate_user(self):
        """ Method to validate if the user details are displayed correctly """

        self.__navigator.navigate_to_users()
        self.__table.access_link(self.user_name)
        displayed_val = self.__user_details.get_user_details()

        validate_key_dict = self.__set_default_values_if_none()

        validate_key_dict = ast.literal_eval(validate_key_dict)
        for key_dict, val_dict in validate_key_dict.items():
            if isinstance(val_dict, list):
                self.log.info('Entity given_val "{0}"'.format(val_dict))
                if set(displayed_val[key_dict]) == set(validate_key_dict[key_dict]):
                    self.log.info(
                        "{0} displayed for {1} matches with {2} given" .format(
                            displayed_val[key_dict], key_dict, validate_key_dict[key_dict]))
                else:
                    exp = "{0} displayed for {1} does not match with {2} given " .format(
                        displayed_val[key_dict], key_dict, validate_key_dict[key_dict])
                    raise Exception(exp)
            elif isinstance(val_dict, str):
                self.log.info('Entity given_val "{0}"'.format(val_dict))
                if displayed_val[key_dict] == validate_key_dict[key_dict]:
                    self.log.info(
                        "{0} displayed for {1} matches with {2} given" .format(
                            displayed_val[key_dict], key_dict, validate_key_dict[key_dict]))
                else:
                    exp = "{0} displayed for {1} does not match with {2} given "\
                        .format(displayed_val[key_dict], key_dict, validate_key_dict[key_dict])
                    raise Exception(exp)
            else:
                self.log.info('Entity given_val :{0}'.format(val_dict))
                for item in val_dict.items():
                    d_val = displayed_val[key_dict][item]
                    key_val = validate_key_dict[key_dict][item]
                    if d_val == key_val:
                        self.log.info("{0} values match" .format(item))
                    else:
                        exp = "{0} displayed for {1} does not match with {2} given " \
                            .format(d_val, item, key_val)
                        raise Exception(exp)

    def __set_default_values_if_none(self):
        """this function sets default values to the parameters provided for user creation
            for comparison against displayed values"""
        if self.user_enabled:
            enabled = 'ON'
        else:
            enabled = 'OFF'

        if self.full_name:
            full_name = self.full_name
        else:
            full_name = ''

        group_list = ''
        list_count = 0
        if self.user_groups:
            for group in self.user_groups:
                list_count += 1
                if list_count == 1:
                    group_list = group_list + group
                else:
                    group_list = group_list + ", " + group
        else:
            group_list = 'No group is found'

        if not self.plan:
            self.plan = 'No plan'

        validate_key_dict = """{'Name':'""" + str(full_name) + """',
                            'Email': '""" + str(self.email) + """',
                            'Enabled': '""" + str(enabled) + """',
                            'Group': '"""+str(group_list)+"""',
                            'Plan': '"""+str(self.plan)+"""'}"""

        return validate_key_dict
