# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be used to run
basic operations on user groups page from test case.

To begin, create an instance of UserGroupMain for user group test case.

Functions:

add_new_user_group()             --     Calls method from base file to add new user in Admin Console

delete_user_group()              --     Calls method to delete users created in Admin Console

edit_user_group_details()        --     Calls method to edit user details

remove_users_from_usergroup()    --     Method to remove users from user group

validate_user_group()            --     Validates if the user details are retained correctly against
                                        user inputs

logout_of_admin_console()        --     Method to logout of admin console
"""

import ast

from AutomationUtils import logger
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.AdminConsolePages.UserGroups import UserGroups
from Web.AdminConsole.AdminConsolePages.UserGroupDetails import UserGroupDetails


class UserGroupMain(object):
    """
    Helper file to provide arguments and handle function call to main file
    """
    def __init__(self, admin_console):
        """
        Initialize method for UserGroupMain
        """
        self.__admin_console = admin_console
        self.__navigator = admin_console.navigator
        self.__user_groups = UserGroups(self.__admin_console)
        self.__user_group_details = UserGroupDetails(self.__admin_console)
        self.__table = Table(self.__admin_console)

        self.log = logger.get_log()

        self._group_name = 'Group_53712'
        self._description = 'Group_53712'
        self._quota = True
        self._group_enabled = True
        self._laptop_admins = False
        self._quota_limit = 50
        self._plan = None
        self._user_list = []
        self._owner_transfer = 'master'
        self._edit_flag = False
        self._old_group_name = None

    @property
    def group_name(self):
        """ Get group_name"""
        return self._group_name

    @group_name.setter
    def group_name(self, value):
        """ Set group_name"""
        self._group_name = value

    @property
    def description(self):
        """ Get description"""
        return self._description

    @description.setter
    def description(self, value):
        """ Set description"""
        self._description = value

    @property
    def quota(self):
        """ Get quota"""
        return self._quota

    @quota.setter
    def quota(self, value):
        """ Set quota"""
        self._quota = value

    @property
    def group_enabled(self):
        """ Get group_enabled"""
        return self._group_enabled

    @group_enabled.setter
    def group_enabled(self, value):
        """ Set group_enabled"""
        self._group_enabled = value

    @property
    def laptop_admins(self):
        """ Get laptop_admins"""
        return self._laptop_admins

    @laptop_admins.setter
    def laptop_admins(self, value):
        """ Set laptop_admins"""
        self._laptop_admins = value

    @property
    def quota_limit(self):
        """ Get quota_limit"""
        return self._quota_limit

    @quota_limit.setter
    def quota_limit(self, value):
        """ Set quota_limit"""
        self._quota_limit = value

    @property
    def plan(self):
        """ Get plan"""
        return self._plan

    @plan.setter
    def plan(self, value):
        """ Set plan"""
        self._plan = value

    @property
    def user_list(self):
        """ Get user_list"""
        return self._user_list

    @user_list.setter
    def user_list(self, value):
        """ Set user_list"""
        self._user_list = value

    def add_new_user_group(self):
        """calls add user method from Users page"""

        self.__navigator.navigate_to_user_groups()
        self.__user_groups.add_user_group(self.group_name,
                                          self.description,
                                          self.quota,
                                          self.quota_limit)
        self._old_group_name = self.group_name

    def delete_user_group(self):
        """Calls method to delete designated user group"""

        self.__navigator.navigate_to_user_groups()
        self.__user_groups.delete_user_group(self.group_name,
                                             self._owner_transfer)

    def edit_user_group_details(self):
        """
        Calls method to edit user group details
        """

        self.__navigator.navigate_to_user_groups()
        self.__table.access_link(self._old_group_name)
        self.__user_group_details.edit_user_group(self.group_name,
                                                  self.description,
                                                  self.plan,
                                                  self.quota,
                                                  self.group_enabled,
                                                  self.laptop_admins,
                                                  self.quota_limit)
        self.__user_group_details.add_users_to_group(self.user_list)
        self._edit_flag = True

    def remove_users_from_user_group(self):
        """
         Calls method to remove users from a user group
        """
        self.__navigator.navigate_to_user_groups()
        self.__table.access_link(self._group_name)
        self.__user_group_details.remove_users_from_group(self.user_list)

    def validate_user_group(self):
        """validates if the user group details are displayed correctly"""

        self.__navigator.navigate_to_user_groups()
        self.__table.access_link(self.group_name)
        displayed_val = self.__user_group_details.get_user_group_details()
        user_list = []
        user_details = self.__user_group_details.list_users()
        for user in user_details:
            user_list.append(user['User name'])
        displayed_val.update({'Users': user_list})

        if not self._edit_flag:
            validate_key_dict = self.__set_default_values_if_none(get_validate_key_dict='Pre-edit')
        else:
            validate_key_dict = self.__set_default_values_if_none(get_validate_key_dict='Post-edit')

        validate_key_dict = ast.literal_eval(validate_key_dict)
        for key_dict, val_dict in validate_key_dict.items():
            if isinstance(val_dict, list):
                self.log.info('Entity given_val "{0}"'.format(val_dict))
                if set(displayed_val[key_dict]) == set(validate_key_dict[key_dict]):
                    self.log.info(
                        "{0} displayed for {1} matches with {2} given" .format(
                            displayed_val[key_dict], key_dict, validate_key_dict[key_dict]))
                else:
                    exp = "{0} displayed for {1} does not match with {2} given " .format(
                        displayed_val[key_dict], key_dict, validate_key_dict[key_dict])
                    raise Exception(exp)
            elif isinstance(val_dict, str):
                self.log.info('Entity given_val "{0}"'.format(val_dict))
                if displayed_val[key_dict] == validate_key_dict[key_dict]:
                    self.log.info(
                        "{0} displayed for {1} matches with {2} given" .format(
                            displayed_val[key_dict], key_dict, validate_key_dict[key_dict]))
                else:
                    exp = "{0} displayed for {1} does not match with {2} given " \
                        .format(displayed_val[key_dict], key_dict, validate_key_dict[key_dict])
                    raise Exception(exp)
            else:
                self.log.info('Entity given_val :{0}'.format(val_dict))
                for item in val_dict.items():
                    d_val = displayed_val[key_dict][item]
                    key_val = validate_key_dict[key_dict][item]
                    if d_val == key_val:
                        self.log.info("{0} values match" .format(item))
                    else:
                        exp = "{0} displayed for {1} does not match with {2} given " \
                            .format(d_val, item, key_val)
                        raise Exception(exp)

    def __set_default_values_if_none(self, get_validate_key_dict):
        """ This function sets default values to the parameters provided for user group creation
            for comparison against displayed values"""

        if self.group_enabled:
            enabled = '✓'
        else:
            enabled = '✘'

        if self.laptop_admins:
            laptop_admins = '✓'
        else:
            laptop_admins = '✘'

        if get_validate_key_dict == 'Pre-edit':
            validate_key_dict = """{'Group name':'""" + str(self.group_name) + """',
            'Group description': '""" + str(self.description) + """',
            'Group enabled': '""" + str(enabled) + """',
            'Laptop admins': '"""+str(laptop_admins)+"""',
            'Quota limit': '"""+str(self.quota_limit)+" GB"+"""',
            'Plan': 'No plan',
            'Users':"""+str(self.user_list)+"""}"""

        elif get_validate_key_dict == 'Post-edit':
            validate_key_dict = """{'Group name':'""" + str(self.group_name) + """',
            'Group description': '""" + str(self.description) + """',
            'Group enabled': '""" + str(enabled) + """',
            'Laptop admins': '"""+str(laptop_admins)+"""',
            'Plan': '"""+str(self.plan)+"""',
            'Users':"""+str(self.user_list)+"""}"""

        return validate_key_dict
