# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations related to Storage in AdminConsole
StorageMain : This class provides methods for storage related operations

StorageMain
===========

    add_disk_storage()              --     To add a new disk storage

    list_disk_storage               --     Get the of all the disk storage in the form of a list

    delete_disk_storage()           --     Deletes the disk storage with the given name

    disk_storage_info()             --     Returns the details of given disk storage

    add_disk_backup_location()      --     To add a new backup location to an existing disk storage

    delete_disk_backup_location()   --     Deletes the backup location on disk storage

    list_disk_backup_locations()    --     Get the of all the backup location disk storage in the form of a list

    add_media_agent_disk_storage()  --     Add media agent to backup location on disk storage

    encrypt_disk_storage()          --     To encrypt the disk storage on the selected

    list_disk_storage_associated_plans() -- Get all the associated plans to the disk storage in the form of a list

    add_cloud_storage()              --     To add a new cloud storage

    list_cloud_storage               --     Get all the cloud storage in the form of a list

    delete_cloud_storage()           --     Deletes the cloud storage with the given name

    cloud_storage_info()             --     Returns the details of given cloud storage

    add_cloud_container()            --     To add a new container to an existing cloud storage

    delete_cloud_container()         --     Deletes the container on cloud storage

    list_cloud_containers()          --     Get all the container cloud storage in the form of a list

    add_media_agent_cloud_storage()  --     Add media agent to container on cloud storage

    encrypt_cloud_storage()          --     To encrypt the cloud storage on the selected

    list_cloud_storage_associated_plans() -- Get all the associated plans to the cloud storage in the form of a list

"""

from AutomationUtils import logger

from Web.AdminConsole.Storage.DiskStorage import DiskStorage
from Web.AdminConsole.Storage.DiskStorageDetails import DiskStorageDetails
from Web.AdminConsole.Storage.CloudStorage import CloudStorage
from Web.AdminConsole.Storage.CloudStorageDetails import CloudStorageDetails


class StorageMain:
    """ Admin console helper for Storage related pages """

    def __init__(self, admin_console):
        """
        Helper for storage related files

        Args:
            admin_console   (AdminConsole)    --  AdminConsole class object
        """
        self.__admin_console = admin_console
        self.__navigator = admin_console.navigator
        self.__disk = DiskStorage(self.__admin_console)
        self.__disk_details = DiskStorageDetails(self.__admin_console)
        self.__cloud = CloudStorage(self.__admin_console)
        self.__cloud_details = CloudStorageDetails(self.__admin_console)
        self.log = logger.get_log()

    def add_disk_storage(self, disk_storage_name, media_agent, backup_location, saved_credential_name=None,
                         username=None, password=None, deduplication_db_location=None):
        """
        To add a new disk storage

        Args:
            disk_storage_name (str)     -- Name of the disk storage to be created

            media_agent     (str)       -- Media agent to create storage on

            saved_credential_name (str) -- saved credential name created using credential manager

            username        (str)       -- username for the network path

            password        (str)       -- password for the network path

            backup_location (str)       -- local/network path for the backup

            deduplication_db_location (str) -- local path for the deduplication db

        **Note** MediaAgent should be installed prior, for creating a new backup location for storage.
                To use saved credentials for network path it should be created prior using credential manager,
        """
        self.__navigator.navigate_to_disk_storage()
        self.__disk.add_disk_storage(disk_storage_name, media_agent, backup_location, saved_credential_name, username,
                                     password, deduplication_db_location)
        self.log.info('Successfully added disk storage: %s', disk_storage_name)

    def list_disk_storage(self):
        """Get the of all the disk storage in the form of a list

            Returns:
                disk_storage_list    (list)  --  all disk storage
        """
        self.__navigator.navigate_to_disk_storage()
        return self.__disk.list_disk_storage()

    def delete_disk_storage(self, disk_storage):
        """
        Deletes the disk storage with the given name

        Args:
            disk_storage (str) -- name of the disk storage to be removed
        """
        self.__navigator.navigate_to_disk_storage()
        self.__disk.delete_disk_storage(disk_storage)
        self.log.info('Successfully deleted disk storage: %s', disk_storage)

    def disk_storage_info(self, disk_storage):
        """
        Returns the details of given disk storage

            Args:
                disk_storage    (str)       -- Name of the disk storage to get details
            Returns:
                info    (dict)  -- details of disk storage
        """
        self.__navigator.navigate_to_disk_storage()
        self.__disk.access_disk_storage(disk_storage)
        return self.__disk_details.disk_storage_info()

    def add_disk_backup_location(self, disk_storage, media_agent, backup_location, saved_credential_name=None,
                                 username=None, password=None):
        """
        To add a new backup location to an existing disk storage

        Args:
            disk_storage    (str)       -- Name of the disk storage to add new backup location

            media_agent     (str)       -- Media agent to create storage on

            saved_credential_name (str) -- saved credential name created using credential manager

            username        (str)       -- username for the network path

            password        (str)       -- password for the network path

            backup_location (str)       -- local/network path for the backup

        **Note** MediaAgent should be installed prior, for creating a backup location for storage.
                To use saved credentials for network path it should be created prior using credential manager,
        """
        self.__navigator.navigate_to_disk_storage()
        self.__disk.access_disk_storage(disk_storage)
        self.__disk_details.add_backup_location(media_agent, backup_location, saved_credential_name, username, password)
        self.log.info('Successfully added backup location: %s', backup_location)

    def delete_disk_backup_location(self, disk_storage, backup_location):
        """
        Deletes the backup location on disk storage

        Args:
            disk_storage    (str)  --   name of the disk storage to delete backup location

            backup_location (str)   --  name of the backup location to delete
        """
        self.__navigator.navigate_to_disk_storage()
        self.__disk.access_disk_storage(disk_storage)
        self.__disk_details.delete_backup_location(backup_location)
        self.log.info('Successfully deleted backup location: %s', disk_storage)

    def list_disk_backup_locations(self, disk_storage):
        """
        Get the of all the backup location disk storage in the form of a list

            Args:
                disk_storage    (str)       -- Name of the disk storage to get all the backup location
            Returns:
                    backup_location_list    (list)  --  all backup locations on disk
        """
        self.__navigator.navigate_to_disk_storage()
        self.__disk.access_disk_storage(disk_storage)
        return self.__disk_details.list_backup_locations()

    def add_media_agent_disk_storage(self, disk_storage, backup_location, media_agent_list):
        """
        Add media agent to backup location on disk storage

            Args:
                disk_storage       (str)  --  name of the disk storage to add media agent

                backup_location   (str)   --  backup location on which given media agent will be added

                media_agent_list  (list)  --  list of media agents to be added
        """
        self.__navigator.navigate_to_disk_storage()
        self.__disk.access_disk_storage(disk_storage)
        self.__disk_details.add_media_agent(backup_location, media_agent_list)
        self.log.info('Successfully added media agents: %s', media_agent_list)

    def list_disk_storage_associated_plans(self, disk_storage):
        """
        Get all the associated plans to the disk storage in the form of a list

            Args:
                disk_storage    (str)       -- Name of the disk storage to get all the associated plans
            Returns:
                    associated_plan_list (list)  --  all backup locations on disk
        """
        self.__navigator.navigate_to_disk_storage()
        self.__disk.access_disk_storage(disk_storage)
        return self.__disk_details.list_associated_plans()

    def encrypt_disk_storage(self, disk_storage, cipher=None, key_length=None, key_management_server=None):
        """
        To encrypt the disk storage on the selected

        Args:
            disk_storage   (str)   -- Name of the disk storage to be encrypted

            cipher      (str)   -- Encryption method to be used

            key_length  (str)   -- Key length for the chosen cipher

            key_management_server   (str)   --  Key management server for the storage pool
        """
        self.__navigator.navigate_to_disk_storage()
        self.__disk.access_disk_storage(disk_storage)
        self.__disk_details.encrypt_storage(cipher, key_length, key_management_server)
        self.log.info('Successfully encrypted the disk storage: %s', disk_storage)

    def add_cloud_storage(self, cloud_storage_name, media_agent, cloud_type, server_host, container, storage_class=None,
                          saved_credential_name=None, username=None, password=None, deduplication_db_location=None):
        """
        To add a new cloud storage

        Args:
            cloud_storage_name (str)    -- Name of the cloud storage to be created

            media_agent     (str)       -- Media agent to create storage on

            cloud_type      (str)       -- type of the cloud storage server

            server_host     (str)       -- cloud server host name

            container       (str)       -- container to be associated with the storage

            storage_class   (str)       --  storage class to be associated with the container

            saved_credential_name (str) -- saved credential name created using credential manager

            username        (str)       -- username for the network path

            password        (str)       -- password for the network path

            deduplication_db_location (str) -- local path for the deduplication db

        **Note** MediaAgent should be installed prior, for creating a new storage,
                To use saved credentials it should be created prior using credential manager.
        """
        self.__navigator.navigate_to_cloud_storage()
        self.__cloud.add_cloud_storage(cloud_storage_name, media_agent, cloud_type, server_host, container,
                                       storage_class, saved_credential_name, username, password,
                                       deduplication_db_location)
        self.log.info('Successfully added cloud storage: %s', cloud_storage_name)

    def list_cloud_storage(self):
        """Get the of all the cloud storage in the form of a list

            Returns:
                cloud_storage_list    (list)  --  all cloud storage
        """
        self.__navigator.navigate_to_cloud_storage()
        return self.__cloud.list_cloud_storage()

    def delete_cloud_storage(self, cloud_storage):
        """
        Deletes the cloud storage with the given name

        Args:
            cloud_storage (str) -- name of the cloud storage to be removed
        """
        self.__navigator.navigate_to_cloud_storage()
        self.__cloud.action_delete(cloud_storage)
        self.log.info('Successfully deleted cloud storage: %s', cloud_storage)

    def cloud_storage_info(self, cloud_storage):
        """
        Returns the details of given cloud storage

            Args:
                cloud_storage    (str)       -- Name of the cloud storage to get details
            Returns:
                info    (dict)  -- details of cloud storage
        """
        self.__navigator.navigate_to_cloud_storage()
        self.__cloud.select_cloud_storage(cloud_storage)
        return self.__cloud_details.cloud_storage_info()

    def add_cloud_container(self, cloud_storage, media_agent, server_host, container, storage_class=None,
                            saved_credential_name=None, username=None, password=None):
        """
        To add a new container to an existing cloud storage

        Args:
            cloud_storage       (str)  --  name of the cloud storage to add container

            media_agent     (str)       -- Media agent to create storage on

            server_host     (str)       -- cloud server host name

            container       (str)       -- container to be associated with the storage

            storage_class   (str)       --  storage class to be associated with the container

            saved_credential_name (str) -- saved credential name created using credential manager

            username        (str)       -- username for the network path

            password        (str)       -- password for the network path

        **Note** MediaAgent should be installed prior, for creating a storage,
                To use saved credentials it should be created prior using credential manager.
        """
        self.__navigator.navigate_to_cloud_storage()
        self.__cloud.select_cloud_storage(cloud_storage)
        self.__cloud_details.add_container(media_agent, server_host, container, storage_class, saved_credential_name,
                                           username, password)
        self.log.info('Successfully added container: %s', container)

    def delete_cloud_container(self, cloud_storage, container):
        """
        Deletes the container on cloud storage

        Args:
            cloud_storage    (str)  --   name of the cloud storage to delete container

            container (str)   --  name of the container to delete
        """
        self.__navigator.navigate_to_cloud_storage()
        self.__cloud.select_cloud_storage(cloud_storage)
        self.__cloud_details.delete_container(container)
        self.log.info('Successfully deleted container: %s', cloud_storage)

    def list_cloud_containers(self, cloud_storage):
        """
        Get the of all the container cloud storage in the form of a list

            Args:
                cloud_storage    (str)       -- Name of the cloud storage to get all the container
            Returns:
                    container_list    (list)  --  all containers on cloud
        """
        self.__navigator.navigate_to_cloud_storage()
        self.__cloud.select_cloud_storage(cloud_storage)
        return self.__cloud_details.list_all_containers()

    def add_media_agent_cloud_storage(self, cloud_storage, container, media_agent_list):
        """
        Add media agent to container on cloud storage

            Args:
                cloud_storage       (str)  --  name of the cloud storage to add media agent

                container   (str)   --  container on which given media agent will be added

                media_agent_list  (list)  --  list of media agents to be added
        """
        self.__navigator.navigate_to_cloud_storage()
        self.__cloud.select_cloud_storage(cloud_storage)
        self.__cloud_details.add_media_agent(container, media_agent_list)
        self.log.info('Successfully added media agents: %s', media_agent_list)

    def list_cloud_storage_associated_plans(self, cloud_storage):
        """
        Get all the associated plans to the cloud storage in the form of a list

            Args:
                cloud_storage    (str)       -- Name of the cloud storage to get all the associated plans
            Returns:
                    associated_plan_list (list)  --  all containers on cloud
        """
        self.__navigator.navigate_to_cloud_storage()
        self.__cloud.select_cloud_storage(cloud_storage)
        return self.__cloud_details.list_associated_plans()

    def encrypt_cloud_storage(self, cloud_storage, cipher=None, key_length=None, key_management_server=None):
        """
        To encrypt the cloud storage on the selected

        Args:
            cloud_storage   (str)   -- Name of the cloud storage to be encrypted

            cipher      (str)   -- Encryption method to be used

            key_length  (str)   -- Key length for the chosen cipher

            key_management_server   (str)   --  Key management server for the storage
        """
        self.__navigator.navigate_to_cloud_storage()
        self.__cloud.select_cloud_storage(cloud_storage)
        self.__cloud_details.encrypt_storage(cipher, key_length, key_management_server)
        self.log.info('Successfully encrypted the cloud storage: %s', cloud_storage)
