# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be used to run
basic operations on plans or subscriptions page.

To begin, create an instance of PlanMain for plans test case.

To initialize the class variables, pass the instance object to the appropriate
definition of AdminConsoleInfo

Call the required definition using the instance object.

delete_plans                    --                delete plans from admin console

validate_plans                   --               validate plans in admin console

add_plan                        --                add plans in the admin console

"""
import time
from _collections import OrderedDict
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.AdminConsolePages.Plans import Plans
from Web.AdminConsole.AdminConsolePages.PlanDetails import PlanDetails


class PlanMain:
    """ Helper file to provide arguments and handle function call to main file """

    def __init__(self, admin_console):
        """
        Initialize method for PlanMain
        """
        self.__admin_console = admin_console
        self.__navigator = self.__admin_console.navigator
        self.__driver = self.__admin_console.driver
        self.__plans = Plans(self.__admin_console)
        self.__plan_details = PlanDetails(self.__admin_console)
        self.__table = Table(self.__admin_console)
        self.log = self.__admin_console.log

        self._plan_name = None
        self._laptop_edited = False
        self._server_edited = False
        self._derived_plan = False
        self._backup_data = {'file_system': ["Mac", "Unix", "Windows"],
                             'content_backup': ["Content Library", "Documents", "Desktop",
                                                "Office", "Music", "Pictures", "Videos"],
                             'content_library': ['Content Library', 'Image',
                                                 'Office', 'System',
                                                 'Temporary Files (Windows)', 'Text',
                                                 'Thumbnail Supported', 'Video',
                                                 'Virtual Machine'],
                             'custom_content': '',
                             'exclude_folder': ['Documents'],
                             'exclude_folder_library': ['Audio', 'Disk Images', 'Executable',
                                                        'Email Files', 'Image', 'Office',
                                                        'System', 'Thumbnail Supported',
                                                        'Video', 'Temporary Files (Mac)'],
                             'exclude_folder_custom_content': ''}
        self._allow_override = {"Storage_pool": "Override required",
                                "RPO": "Override optional",
                                "Folders_to_backup": "Override not allowed",
                                "Retention": None}
        self._system_state = True
        self._full_backup = False
        self._vss = False
        self._allowed_features = {'Edge Drive': 'OFF',
                                  'audit_drive_operations': True,
                                  'notification_for_shares': True,
                                  'edge_drive_quota': '200',
                                  'DLP': 'ON',
                                  'Archiving': 'ON'}
        self._archiving_rules = {"start_clean": "40", "stop_clean": "90",
                                 "file_access_time": "85", "file_modify_time": None,
                                 "file_create_time": "2", "archive_file_size": None,
                                 "max_file_size": None, "archive_read_only": True,
                                 "replace_file": None, "delete_file": None}
        self._file_system_quota = '250'
        self._storage = {'pri_storage': None,
                         'pri_ret_period': '20',
                         'sec_storage': None,
                         'sec_ret_period': '45',
                         'ret_unit': 'Day(s)'}
        self._throttle_send = 'No limit'
        self._throttle_receive = 'No limit'
        self._alerts = {"Backup": "No backup for last 4 days",
                        "Jobfail": "Restore Job failed",
                        "edge_drive_quota": "Edge drive quota alert",
                        "file_system_quota": "File system quota alert"}
        self._backup_day = dict.fromkeys(['1', '2', '3'], 1)
        self._backup_duration = dict.fromkeys(['2', '3', '4', '5'], 1)
        self._backup_window = "Monday through Wednesday : 12am-4am"
        self._snap_recovery_points = '12'
        self._rpo_hours = '5'
        self._edit_plan_dict = {'throttle_send': '1000', 'throttle_receive': '1000',
                                'file_system_quota': '250', 'rpo_hours': '30',
                                'additional_storage': {'storage_name': 'Store',
                                                       'storage_pool': 'Secondary',
                                                       'ret_period': '10'},
                                'allowed_features': {"Edge Drive": "ON",
                                                     'audit_drive_operations': False,
                                                     'notification_for_shares': False,
                                                     'edge_drive_quota': '250',
                                                     "DLP": "OFF",
                                                     "Archiving": "OFF"},
                                "region": None,
                                'edit_storage': None,
                                'override_restrictions': {"Storage_pool": "Override optional",
                                                          "RPO": "Override not allowed",
                                                          "Folders_to_backup": "Override optional"}}
        self._edit_server_plan_dict = {'override_restrictions': {"Storage_pool": "Override optional",
                                                                 "RPO": "Override not allowed",
                                                                 "Folders_to_backup": "Override optional"},
                                       'rpo_hours': '5',
                                       'additional_storage': {'storage_name': 'Store',
                                                              'storage_pool': 'StorageA',
                                                              'ret_period': '10'},
                                       'edit_storage': None
                                       }
        self._user_usergroup_association = None
        self._user_usergroup_de_association = None
        self._content_selected = None
        self._retention = {'min_retention': {'value': 'Indefinitely', 'unit': None},
                           'deleted_item_retention': {'value': '5', 'unit': 'day(s)'},
                           'file_version_retention': {'duration': None,
                                                      'versions': None,
                                                      'rules': {'days': '4',
                                                                'weeks': '5',
                                                                'months': '6'}}}
        self._snapshot_options = {'snap_recovery_points': '8',
                                  'Enable_backup_copy': 'ON',
                                  'sla_hours': '11',
                                  'sla_minutes': '22'}

        self._database_options = {'sla_hours': '11',
                                  'sla_minutes': '22'}
        self._media_agent = None

    @property
    def plan_name(self):
        """ Get plan name"""
        return self._plan_name

    @plan_name.setter
    def plan_name(self, value):
        """ Set plan name"""
        self._plan_name = value

    @property
    def server_edited(self):
        """ Get server edited value """
        return self._server_edited

    @server_edited.setter
    def server_edited(self, value):
        """ Set server edited value """
        self._server_edited = value

    @property
    def laptop_edited(self):
        """ Get laptop edited value """
        return self._laptop_edited

    @laptop_edited.setter
    def laptop_edited(self, value):
        """ Set lapotp edited value """
        self._laptop_edited = value

    @property
    def derived_plan(self):
        """ Get derived  plan value """
        return self._derived_plan

    @derived_plan.setter
    def derived_plan(self, value):
        """ Set derived plan value """
        self._derived_plan = value

    @property
    def media_agent(self):
        """ Get media_agent"""
        return self._media_agent

    @media_agent.setter
    def media_agent(self, value):
        """ Set media_agent"""
        self._media_agent = value

    @property
    def backup_data(self):
        """Get backup data values"""
        return self._backup_data

    @backup_data.setter
    def backup_data(self, value):
        """Set backup data values"""
        self._backup_data = value

    @property
    def allow_override(self):
        """Get allow override values"""
        return self._allow_override

    @allow_override.setter
    def allow_override(self, value):
        """Set allow override values"""
        self._allow_override = value

    @property
    def system_state(self):
        """Get system_state value"""
        return self._system_state

    @system_state.setter
    def system_state(self, value):
        """Set system_state value"""
        self._system_state = value

    @property
    def full_backup(self):
        """Get full_backup value"""
        return self._full_backup

    @full_backup.setter
    def full_backup(self, value):
        """Set full_backup value"""
        self._full_backup = value

    @property
    def vss(self):
        """Get vss value"""
        return self._vss

    @vss.setter
    def vss(self, value):
        """Set vss value"""
        self._vss = value

    @property
    def allowed_features(self):
        """Get allowed_features value"""
        return self._allowed_features

    @allowed_features.setter
    def allowed_features(self, value):
        """Set allowed_features value"""
        self._allowed_features = value

    @property
    def archiving_rules(self):
        """Get archiving_rules value"""
        return self._archiving_rules

    @archiving_rules.setter
    def archiving_rules(self, value):
        """Set archiving_rules value"""
        self._archiving_rules = value

    @property
    def file_system_quota(self):
        """Get file_system_quota value"""
        return self._file_system_quota

    @file_system_quota.setter
    def file_system_quota(self, value):
        """Set file_system_quota value"""
        self._file_system_quota = value

    @property
    def storage(self):
        """Get primary and secondary storage attributes"""
        return self._storage

    @storage.setter
    def storage(self, value):
        """Set primary and secondary storage attributes"""
        self._storage = value

    @property
    def throttle_send(self):
        """Get throttle_send value"""
        return self._throttle_send

    @throttle_send.setter
    def throttle_send(self, value):
        """Set throttle_send value"""
        self._throttle_send = value

    @property
    def throttle_receive(self):
        """Get throttle_receive value"""
        return self._throttle_receive

    @throttle_receive.setter
    def throttle_receive(self, value):
        """Set throttle_receive value"""
        self._throttle_receive = value

    @property
    def alerts(self):
        """Get alerts values"""
        return self._alerts

    @alerts.setter
    def alerts(self, value):
        """Set alerts values"""
        self._alerts = value

    @property
    def backup_day(self):
        """Get backup_day values"""
        return self._backup_day

    @backup_day.setter
    def backup_day(self, value):
        """Set backup_day values"""
        self._backup_day = value

    @property
    def backup_duration(self):
        """Get backup_duration values"""
        return self._backup_duration

    @backup_duration.setter
    def backup_duration(self, value):
        """Set backup_duration values"""
        self._backup_duration = value

    @property
    def snap_recovery_points(self):
        """Get snap_recovery_points value"""
        return self._snap_recovery_points

    @snap_recovery_points.setter
    def snap_recovery_points(self, value):
        """Set snap_recovery_points value"""
        self._snap_recovery_points = value

    @property
    def rpo_hours(self):
        """Get rpo_hours value"""
        return self._rpo_hours

    @rpo_hours.setter
    def rpo_hours(self, value):
        """Set rpo_hours value"""
        self._rpo_hours = value

    @property
    def edit_plan_dict(self):
        """Get edit_plan_dict value"""
        return self._edit_plan_dict

    @edit_plan_dict.setter
    def edit_plan_dict(self, value):
        """Set edit_plan_dict value"""
        self._edit_plan_dict = value

    @property
    def edit_server_plan_dict(self):
        """Get edit_plan_dict value"""
        return self._edit_server_plan_dict

    @edit_server_plan_dict.setter
    def edit_server_plan_dict(self, value):
        """Set edit_plan_dict value"""
        self._edit_server_plan_dict = value

    @property
    def user_usergroup_association(self):
        """Get user_usergroup_association values"""
        return self._user_usergroup_association

    @user_usergroup_association.setter
    def user_usergroup_association(self, value):
        """Set user_usergroup_association values"""
        self._user_usergroup_association = value

    @property
    def user_usergroup_de_association(self):
        """Get user_usergroup_de_association values"""
        return self._user_usergroup_de_association

    @user_usergroup_de_association.setter
    def user_usergroup_de_association(self, value):
        """Set user_usergroup_de_association values"""
        self._user_usergroup_de_association = value

    @property
    def retention(self):
        """Get retention values"""
        return self._retention

    @retention.setter
    def retention(self, value):
        """Set retention values"""
        self._retention = value

    @property
    def snapshot_options(self):
        """Get snapshot_options values"""
        return self._snapshot_options

    @snapshot_options.setter
    def snapshot_options(self, value):
        """Set snapshot_options values"""
        self._snapshot_options = value

    @property
    def database_options(self):
        """Get database_options values"""
        return self._database_options

    @database_options.setter
    def database_options(self, value):
        """Set database_options values"""
        self._database_options = value

    def add_plan(self):
        """calls create plan functions from Plans file based on input
            and generates plans in Admin Console"""

        self.__navigator.navigate_to_plan()
        for key, value in self.plan_name.items():
            if key == "server_plan":
                if value is not None:
                    self.__plans.create_server_plan(
                        value,
                        self.storage,
                        self.rpo_hours,
                        self.allow_override,
                        self.backup_day,
                        self.backup_duration,
                        self.backup_data,
                        self.snapshot_options,
                        self.database_options)
                    time.sleep(3)
                    self.__navigator.navigate_to_plan()
            elif key == "laptop_plan":
                if value is not None:
                    self.__plans.create_laptop_plan(
                        value,
                        self.allowed_features,
                        self.media_agent,
                        self.archiving_rules,
                        self.backup_data,
                        self.file_system_quota,
                        self.storage,
                        self.rpo_hours,
                        self.retention,
                        self.throttle_send,
                        self.throttle_receive,
                        self.alerts,
                        self.allow_override)
                    time.sleep(3)
                    self.__navigator.navigate_to_plan()

    def validate_plans(self):
        """validates the designated plan in admin console plans page"""

        self.validate_plan_details()
        for key, value in self.plan_name.items():
            if key == "laptop_plan":
                self.__navigator.navigate_to_server_groups()
                self.__table.search_for(value + " clients")
                if self.__admin_console.check_if_entity_exists("link", "{0} clients".format(value)):
                    self.log.info("Server group created for plan named {0} is "
                                  "validated successfully.".format(value))
                self.__navigator.navigate_to_plan()
                break

    def delete_plans(self):
        """deletes the designated plan"""

        self.__navigator.navigate_to_plan()
        for value in self.plan_name.values():
            self.__plans.delete_plan(value)

    def validate_plan_details(self):
        """validates if the plan details are displayed correctly"""

        for key, value in self.plan_name.items():
            self.__navigator.navigate_to_plan()
            self.__plans.select_plan(value)
            displayed_val = self.__plan_details.plan_info(key)
            validation_dict = self.__set_default_values_if_none(key, value)
            self.log.info("Displayed values: ", displayed_val)
            self.log.info("Validation values: ", validation_dict)

            for key_dict, val_dict in validation_dict.items():
                if isinstance(val_dict, list):
                    self.log.info('Entity given_val "{0}"'.format(val_dict))
                    if set(displayed_val[key_dict]) == set(validation_dict[key_dict]):
                        self.log.info(
                            "{0} displayed for {1} matches with {2} given".format(
                                displayed_val[key_dict], key, validation_dict[key_dict]))
                    else:
                        exp = "{0} displayed for {1} does not match with {2} given ".format(
                            displayed_val[key_dict], key, validation_dict[key_dict])
                        raise Exception(exp)
                elif isinstance(val_dict, str):
                    self.log.info('Entity given_val "{0}"'.format(val_dict))
                    if displayed_val[key_dict] == validation_dict[key_dict]:
                        self.log.info(
                            "{0} displayed for {1} matches with {2} given".format(
                                displayed_val[key_dict], key_dict, validation_dict[key_dict]))
                    else:
                        exp = "{0} displayed for {1} does not match with {2} given " \
                            .format(displayed_val[key_dict], key_dict, validation_dict[key_dict])
                        raise Exception(exp)
                else:
                    self.log.info('Entity given_val :{0}'.format(val_dict))
                    for item, value_dict in val_dict.items():
                        d_val = displayed_val[key_dict][item]
                        key_val = validation_dict[key_dict][item]
                        if d_val == key_val:
                            self.log.info("{0} values match".format(item))
                        else:
                            exp = "{0} displayed for {1} does not match with {2} given " \
                                .format(d_val, item, key_val)
                            raise Exception(exp)

    def get_val_dict(self, plan_type, plan_name):
        """ gets val dict """
        return self.__set_default_values_if_none(plan_type, plan_name)

    def __set_default_values_if_none(self, plan_type, plan_name):
        """ this function sets default values to the parameters provided for plan creation """

        default_comp_values = {
            "No. of users": "0",
            'User/Group': "Role",
            "master": "Plan Creator Role",
            "No. of devices": "0",
            "secondary_schedule": plan_name + " aux copy",
            "associated_entities": []}

        self._content_selected = self.__get_selected_backup_content()

        override = {}
        if not self.server_edited:
            if self.allow_override['Storage_pool']:
                override.update({'Storage pool': self.allow_override['Storage_pool']})
            if self.allow_override['RPO']:
                override.update({'RPO': self.allow_override['RPO']})
            if self.allow_override['Folders_to_backup']:
                override.update({'Folders to backup': self.allow_override['Folders_to_backup']})
            if self.allow_override['Retention']:
                override.update({'Retention': self.allow_override['Retention']})
        else:
            override.update({'Storage pool': self.edit_server_plan_dict['override_restrictions']['Storage_pool']})
            override.update({'RPO': self.edit_server_plan_dict['override_restrictions']['RPO']})
            override.update({'Folders to backup':
                            self.edit_server_plan_dict['override_restrictions']['Folders_to_backup']})

        retention_tile = {}

        if self.retention['deleted_item_retention']['value'] == 'Indefinitely':
            retention_tile.update({'Deleted item retention': self.retention['deleted_item_retention']['value']})
        else:
            retention_tile.update({'Deleted item retention': self.retention['deleted_item_retention']['value']
                                                             + " " + self.retention['deleted_item_retention']['unit']})

        if self.retention['file_version_retention']['versions']:
            retention_tile.update({'File versions': self.retention['file_version_retention']['versions']})
        elif self.retention['file_version_retention']['duration']:
            retention_tile.update({'File versions': self.retention['file_version_retention']['duration']['value'] + " "
                                  + self.retention['file_version_retention']['duration']['unit']})
        else:
            if self.retention['file_version_retention']['rules']['days'] and \
                    not self.retention['file_version_retention']['rules']['weeks'] and \
                    not self.retention['file_version_retention']['rules']['months']:
                retention_tile.update({'File versions': 'Daily versions for ' +
                                                        self.retention['file_version_retention']['rules'][
                                                            'days'] + ' day(s)'})
            elif self.retention['file_version_retention']['rules']['weeks'] and \
                    not self.retention['file_version_retention']['rules']['days'] and \
                    not self.retention['file_version_retention']['rules']['months']:
                retention_tile.update({'File versions': 'Weekly versions for ' +
                                                        self.retention['file_version_retention']['rules'][
                                                            'weeks'] + ' week(s)'})
            elif self.retention['file_version_retention']['rules']['months'] and \
                    not self.retention['file_version_retention']['rules']['days'] and \
                    not self.retention['file_version_retention']['rules']['weeks']:
                retention_tile.update({'File versions': 'Weekly versions for ' +
                                                        self.retention['file_version_retention']['rules'][
                                                            'weeks'] + ' month(s)'})
            elif self.retention['file_version_retention']['rules']['days'] and \
                    self.retention['file_version_retention']['rules']['weeks'] and \
                    not self.retention['file_version_retention']['rules']['months']:
                retention_tile.update({'File versions': 'Daily versions for ' +
                                                        self.retention['file_version_retention']['rules'][
                                                            'days'] + ' day(s)\nWeekly versions for ' +
                                                        self.retention['file_version_retention']['rules']['weeks'] +
                                                        ' week(s)'})
            elif self.retention['file_version_retention']['rules']['days'] and \
                    not self.retention['file_version_retention']['rules']['weeks'] and \
                    self.retention['file_version_retention']['rules']['months']:
                retention_tile.update({'File versions': 'Weekly versions for ' +
                                                        self.retention['file_version_retention']['rules'][
                                                            'days'] + ' day(s)\nMonthly versions for ' +
                                                        self.retention['file_version_retention']['rules']['months'] +
                                                        ' month(s)'})
            elif not self.retention['file_version_retention']['rules']['days'] and \
                    self.retention['file_version_retention']['rules']['weeks'] and \
                    self.retention['file_version_retention']['rules']['months']:
                retention_tile.update({'File versions': 'Weekly versions for ' +
                                                        self.retention['file_version_retention']['rules'][
                                                            'weeks'] + ' week(s)\nMonthly versions for ' +
                                                        self.retention['file_version_retention']['rules'][
                                                            'months'] + ' month(s)'})
            elif self.retention['file_version_retention']['rules']['days'] and \
                    self.retention['file_version_retention']['rules']['weeks'] and \
                    self.retention['file_version_retention']['rules']['months']:
                retention_tile.update({'File versions': 'Daily versions for ' +
                                                        self.retention['file_version_retention']['rules'][
                                                            'days'] + ' day(s)\nWeekly versions for ' +
                                                        self.retention['file_version_retention']['rules'][
                                                            'weeks'] + ' week(s)\nMonthly versions for ' +
                                                        self.retention['file_version_retention']['rules'][
                                                            'months'] + ' month(s)'})

        def ret_value(ret_period):
            if int(ret_period) % 30 == 0:
                if int(ret_period)//30 == 1:
                    retention_val = str(int(ret_period)//30) + ' Month'
                else:
                    retention_val = str(int(ret_period)//30) + ' Months'
            elif int(ret_period) % 7 == 0:
                if int(ret_period) // 7 == 1:
                    retention_val = str(int(ret_period)//7) + ' Week'
                else:
                    retention_val = str(int(ret_period)//7) + ' Weeks'
            else:
                retention_val = ret_period + ' Days'
            return retention_val

        pri_ret_val = ret_value(self.storage['pri_ret_period'])
        if self.storage['sec_ret_period']:
            sec_ret_val = ret_value(self.storage['sec_ret_period'])

        if self.edit_plan_dict['edit_storage'] or self.edit_server_plan_dict['edit_storage']:
            if self.storage.get('sec_storage'):
                storage_text = {self.edit_server_plan_dict['edit_storage']['new_storage_name']:
                                [self.storage['pri_storage'], pri_ret_val],
                                'Secondary': [self.storage['sec_storage'], sec_ret_val]}
            else:
                storage_text = {self.edit_server_plan_dict['edit_storage']['new_storage_name']:
                                [self.storage['pri_storage'], pri_ret_val],
                                f"{plan_name} snap copy": [self.storage['pri_storage'], '10 Recovery points']
                                }
        else:
            if self.storage.get('sec_storage'):
                storage_text = {'Primary': [self.storage['pri_storage'], pri_ret_val],
                                'Secondary': [self.storage['sec_storage'], sec_ret_val]}
            else:
                storage_text = {'Primary': [self.storage['pri_storage'], pri_ret_val],
                                f"{plan_name} snap copy": [self.storage['pri_storage'], '1 Month']}

        backup_copy_rpo = self.snapshot_options['sla_hours'] + " hour(s) and " + \
                          self.snapshot_options['sla_minutes'] + " minute(s)"

        log_backup_rpo = self.database_options['sla_hours'] + " hour(s) and " + \
                         self.database_options['sla_minutes'] + " minute(s)"

        alert_list = []
        if self.alerts['Backup']:
            alert_list.append(self.alerts['Backup'])
        if self.alerts['Jobfail']:
            alert_list.append(self.alerts['Jobfail'])
        if self.alerts['edge_drive_quota']:
            alert_list.append(self.alerts['edge_drive_quota'])
        if self.alerts['file_system_quota']:
            alert_list.append(self.alerts['file_system_quota'])

        if plan_type == "server_plan":

            validation_dict = {'PlanName': plan_name,
                               'Security': {'master': [[default_comp_values['master']]]},
                               'RPO': {'Backup frequency': "Runs every 5 Day(s) at 9:00 PM",
                                       'Add full backup': 'OFF',
                                       'Backup window': self._backup_window,
                                       'Full backup window': self._backup_window},
                               'Override restrictions': override,
                               'Backup destinations': storage_text,
                               'Snapshot options': {'Backup Copy': self.snapshot_options['Enable_backup_copy'],
                                                    'Backup copy frequency (in HH:MM)': backup_copy_rpo},
                               'Database options': {'Log backup RPO': log_backup_rpo},
                               'Backup content': self._content_selected}

            return validation_dict

        elif plan_type == "laptop_plan":

            if not self.laptop_edited:
                allowed_features = self.allowed_features
                allowed_features.pop("audit_drive_operations")
                allowed_features.pop("notification_for_shares")
                allowed_features.pop("edge_drive_quota")

                validation_dict = {
                    'PlanName': plan_name,
                    'General': {'No. of users': default_comp_values['No. of users'],
                                'No. of laptops': default_comp_values['No. of devices']},
                    'Allowed features': allowed_features,
                    'Security': {'master': [[default_comp_values["master"]]]},
                    'RPO': {'Backup frequency': 'Runs every ' + self.rpo_hours + ' Hour(s)'},
                    'Override restrictions': override,
                    'Backup destinations': [self.storage['pri_storage'], self.storage['sec_storage']],
                    'Retention': retention_tile,
                    'Alerts': alert_list,
                    'Associated users and user groups': [],
                    'Options': {'File system quota': self.file_system_quota + ' GB'},
                    'Backup content': self._content_selected}

                return validation_dict

            else:
                no_of_users = self.__get_no_of_users()
                allowed_features = dict(self.edit_plan_dict['allowed_features'])
                allowed_features.pop("audit_drive_operations")
                allowed_features.pop("notification_for_shares")
                allowed_features.pop("edge_drive_quota")

                backup_dest = [self.storage['pri_storage']]
                if self.storage['sec_storage']:
                    backup_dest.append(self.storage['sec_storage'])
                if not self.edit_plan_dict['region']:
                    if self.edit_plan_dict['edit_storage']:
                        if self.edit_plan_dict['edit_storage']['pri_storage']:
                            backup_dest[0] = self.edit_plan_dict['edit_storage']['pri_storage']
                        if self.edit_plan_dict['edit_storage']['sec_storage']:
                            backup_dest[1] = self.edit_plan_dict['edit_storage']['sec_storage']
                    if self.edit_plan_dict['additional_storage']:
                        backup_dest.append(self.edit_plan_dict['additional_storage']['storage_pool'])
                else:
                    storage_list = backup_dest
                    if self.edit_plan_dict['edit_storage']:
                        if self.edit_plan_dict['edit_storage']['pri_storage']:
                            storage_list[0] = self.edit_plan_dict['edit_storage']['pri_storage']
                        if self.edit_plan_dict['edit_storage']['sec_storage']:
                            storage_list[1] = self.edit_plan_dict['edit_storage']['sec_storage']
                    if self.edit_plan_dict['additional_storage']:
                        storage_list.append(self.edit_plan_dict['additional_storage']['storage_pool'])
                    backup_dest = {self.edit_plan_dict['region']: storage_list}

                validation_dict = {
                    'PlanName': plan_name,
                    'General': {'No. of users': str(no_of_users),
                                'No. of laptops': default_comp_values['No. of devices']},
                    'Allowed features': allowed_features,
                    'Network resources': {'Throttle send': self.edit_plan_dict['throttle_send'] + ' Kbps',
                                          'Throttle receive': self.edit_plan_dict['throttle_receive'] + ' Kbps'},
                    'Security': {'master': [[default_comp_values["master"]]]},
                    'RPO': {'Backup frequency': 'Runs every ' + self.edit_plan_dict['rpo_hours'] + ' Hour(s)'},
                    'Override restrictions':
                        {'Storage pool': self.edit_plan_dict['override_restrictions']['Storage_pool'],
                         'RPO': self.edit_plan_dict['override_restrictions']['RPO'],
                         'Folders to backup': self.edit_plan_dict['override_restrictions']['Folders_to_backup'],
                         'Retention': self.edit_plan_dict['override_restrictions']['Retention']},
                    'Backup destinations': backup_dest,
                    'Retention': retention_tile,
                    'Alerts': alert_list,
                    'Associated users and user groups': self.user_usergroup_association,
                    'Options': {'File system quota': self.edit_plan_dict['file_system_quota'] + ' GB'},
                    'Backup content': self._content_selected}
                return validation_dict

    def edit_laptop_plan(self):
        """ This function gives call to functions for editing plan tiles """

        for value in self.plan_name.values():
            self.__navigator.navigate_to_plan()
            self.__plans.select_plan(value)
            if self.edit_plan_dict['additional_storage']:
                self.__plan_details.edit_plan_storage_pool(self.edit_plan_dict['additional_storage'],
                                                           self.edit_plan_dict['edit_storage'],
                                                           self.edit_plan_dict['region'])
            if self.alerts:
                self.__plan_details.edit_plan_alerts(self.alerts)

            if self.edit_plan_dict['throttle_send'] or self.edit_plan_dict['throttle_receive']:
                self.__plan_details.edit_plan_network_resources(self.edit_plan_dict['throttle_send'],
                                                                self.edit_plan_dict['throttle_receive'])
            if self.edit_plan_dict['rpo_hours']:
                self.__plan_details.edit_plan_rpo_hours(self.edit_plan_dict['rpo_hours'])
            if self.edit_plan_dict['file_system_quota']:
                self.__plan_details.edit_plan_options(self.edit_plan_dict['file_system_quota'])
            if self.edit_plan_dict['backup_data']:
                new_data = self.edit_plan_dict['backup_data']
                self.__plan_details.edit_plan_backup_content(new_data['file_system'],
                                                             new_data['content_backup'],
                                                             new_data['content_library'],
                                                             str(new_data['custom_content']),
                                                             new_data['exclude_folder'],
                                                             new_data['exclude_folder_library'],
                                                             str(new_data['exclude_folder_custom_content']))
            if self.user_usergroup_association:
                self.__plan_details.edit_plan_associate_users_and_groups(self.user_usergroup_association)
            if self.edit_plan_dict['allowed_features']:
                self.__plan_details.edit_plan_features(self.edit_plan_dict['allowed_features'],
                                                       self.archiving_rules)
            if self.retention:
                self.__plan_details.edit_plan_retention(self.retention)
            if self.edit_plan_dict['override_restrictions']:
                self.__plan_details.edit_plan_override_restrictions(self.edit_plan_dict['override_restrictions'],
                                                                    override_laptop_plan=True)
        self._laptop_edited = True

    def edit_server_plan(self):
        """ This function gives call to functions for editing plan tiles """

        for key, value in self.plan_name.items():
            self.__navigator.navigate_to_plan()
            self.__plans.select_plan(value)
            self.__plan_details.edit_plan_rpo_hours(self.edit_server_plan_dict['rpo_hours'])
            self.__plan_details.edit_server_plan_storage_pool(additional_storage={},
                                                              edit_storage=self.edit_server_plan_dict['edit_storage'])
            self.__plan_details.edit_plan_override_restrictions(self.edit_server_plan_dict['override_restrictions'])
            self.__plan_details.edit_snapshot_options({'hours': '11', 'minutes': '22'}, '10')
            self.__plan_details.edit_database_options({'hours': '11', 'minutes': '22'})
            self.__plan_details.edit_plan_backup_content(self.backup_data['file_system'],
                                                         self.backup_data['content_backup'],
                                                         self.backup_data['content_library'],
                                                         str(self.backup_data['custom_content']),
                                                         self.backup_data['exclude_folder'],
                                                         self.backup_data['exclude_folder_library'],
                                                         str(self.backup_data['exclude_folder_custom_content']))
        self._server_edited = True

    def remove_associations(self):
        """ This function gives call to another function for removing Users/UserGroups associated to a plan """
        for key, value in self.plan_name.items():
            self.__navigator.navigate_to_plan()
            self.__plans.select_plan(value)
            self.__plan_details.remove_associated_users_and_groups(self.user_usergroup_de_association)

    def __get_selected_backup_content(self):
        """ This function gives value of backup content selected for laptop and filesystem plan
        based on user input from test case """
        # divyanshu
        if not self.backup_data:
            self._content_selected = {
                'Windows': 'Desktop, Documents, Office, Pictures, Image, MigrationAssistant',
                'Mac': 'Desktop, Documents, Office, Pictures, Image, MigrationAssistant',
                'Unix': 'Desktop, Documents, Office, Pictures, Image'
            }
            return self._content_selected
        if 'Library' in self.backup_data['content_backup']:
            if self.backup_data['custom_content']:
                self._content_selected = {'Windows': 'Archives, Audio, Disk images, Email files, Executable, Image, '
                                                     'Office, Scripts, System, Temporary files (linux), '
                                                     'Temporary files (mac), Temporary files (windows), Text, '
                                                     'Thumbnail supported, Video, Virtual machine, Desktop,'
                                                     'Documents, Music, Pictures, Videos, ' +
                                                     str(self.backup_data['custom_content']) + '',
                                          'Mac': 'Archives, Audio, Disk images, Email files, Executable, Image, '
                                                 'Office, Scripts, System, Temporary files (linux), '
                                                 'Temporary files (mac), Temporary files (windows), Text, '
                                                 'Thumbnail supported, Video, Virtual machine, Desktop, '
                                                 'Documents, Music, Pictures, Videos, ' +
                                                 str(self.backup_data['custom_content']) + '',
                                          'Unix': 'Archives, Audio, Disk images, Email files, Executable, Image, '
                                                  'Office, Scripts, System, Temporary files (linux), '
                                                  'Temporary files (mac), Temporary files (windows), Text, '
                                                  'Thumbnail supported, Video, Virtual machine, ' +
                                                  str(self.backup_data['custom_content'])}
            else:
                self._content_selected = {
                    'Windows': 'Archives, Audio, Disk images, Email files, Executable, Image,'
                               'Office, Scripts, System, Temporary files (linux), '
                               'Temporary files (mac), Temporary files (windows), Text, '
                               'Thumbnail supported, Video, Virtual machine, Desktop, '
                               'Documents, Music, Pictures, Videos',
                    'Mac': 'Archives, Audio, Disk images, Email files, Executable, Image, Office,'
                           ' Scripts, System, Temporary files (linux), Temporary files (mac), '
                           'Temporary files (windows), Text, Thumbnail supported, Video, '
                           'Virtual machine, Desktop,Documents, Music, Pictures, Videos',
                    'Unix': 'Archives, Audio, Disk images, Email files, Executable, Image, '
                            'Office, Scripts, System, Temporary files (linux), '
                            'Temporary files (mac), Temporary files (windows), Text, '
                            'Thumbnail supported, Video, Virtual machine, Desktop, '
                            'Documents, Music, Pictures, Videos'}

        elif 'Content Library' in self.backup_data['content_backup']:
            if 'Content Library' in self.backup_data['content_library']:
                self.backup_data['content_backup'] = sorted(self.backup_data['content_backup'])
                if self.backup_data['custom_content']:
                    content = ['Archives', 'Audio', 'Disk images', 'Email files', 'Executable',
                               'Image', 'Office', 'Scripts', 'System', 'Temporary files (linux)',
                               'Temporary files (mac)', 'Temporary files (windows)',
                               'Text', 'Thumbnail supported', 'Video', 'Virtual machine'] \
                              + self.backup_data['content_backup']
                    content.append(self.backup_data['custom_content'])
                else:
                    content = ['Archives', 'Audio', 'Disk images', 'Email files', 'Executable',
                               'Image', 'Office', 'Scripts', 'System', 'Temporary files (linux)',
                               'Temporary files (mac)', 'Temporary files (windows)',
                               'Text', 'Thumbnail supported', 'Video', 'Virtual machine'] + \
                              self.backup_data['content_backup']
                content.remove('Content Library')
                content = list(OrderedDict.fromkeys(content))
                final_content = ', '.join(content)
                self._content_selected = {'Windows': final_content, 'Mac': final_content, 'Unix': final_content}
            else:
                self.backup_data['content_backup'] = sorted(self.backup_data['content_backup'])
                self.backup_data['content_library'] = sorted(self.backup_data['content_library'])
                if self.backup_data['custom_content']:
                    content = self.backup_data['content_library'] + self.backup_data['content_backup']
                    content.append(str(self.backup_data['custom_content']))
                else:
                    content = self.backup_data['content_library'] + self.backup_data['content_backup']
                content.remove('Content Library')
                content = list(OrderedDict.fromkeys(content))
                final_content = ', '.join(content)
                self._content_selected = {'Windows': final_content, 'Mac': final_content, 'Unix': final_content}
        else:
            self.backup_data['content_backup'] = sorted(self.backup_data['content_backup'])
            if self.backup_data['custom_content']:
                content = self.backup_data['content_backup']
                content.append(str(self.backup_data['custom_content']))
            else:
                content = self.backup_data['content_backup']
            content = list(OrderedDict.fromkeys(content))
            final_content = ', '.join(content)
            self._content_selected = {'Windows': final_content, 'Mac': final_content, 'Unix': final_content}
        return self._content_selected

    def __get_no_of_users(self):

        list_user_groups = []
        no_of_users = 0

        self.__navigator.navigate_to_users()
        for value in self.user_usergroup_association:
            self.__table.search_for(value)
            if not self.__admin_console.check_if_entity_exists("link", value):
                list_user_groups.append(value)
            elif self.__admin_console.check_if_entity_exists("link", value):
                no_of_users += 1

        if not list_user_groups:
            for value in list_user_groups:
                self.__navigator.navigate_to_user_groups()
                self.__table.access_link(value)
                user_grid = self.__driver.find_element_by_xpath("//div[@class='ui-grid-canvas']")
                users_user_group = user_grid.find_elements_by_xpath(".//div[@class='ui-grid-row ng-scope']")
                no_of_users += len(users_user_group)
        return no_of_users

    def add_laptop_derivedplan(self):
        """method to create laptop derived plan """

        self.__navigator.navigate_to_plan()
        self.__admin_console.wait_for_completion()
        self.__plans.select_plan(self.plan_name['laptop_plan'])
        self.__plans.click_create_derived_plan_button()
        self.__plans.create_laptop_derived_plan(self.derived_plan,
                                                self.backup_data,
                                                self.storage,
                                                self.rpo_hours,
                                                self.retention)
        time.sleep(3)
        self.__navigator.navigate_to_plan()

    def validate_derivedplan_overriding(self, validation_dict):

        """
        Method to validate derive plan

        Args:
            validation_dict (str): Name of the plan to be deleted

        Raises:
            Exception:
                -- if fails to delete plan
        """

        self.log.info("validation of derived plan overriding started{0}".format(self._derived_plan))
        self.__navigator.navigate_to_plan()
        self.__admin_console.wait_for_completion()
        self.__plans.select_plan(self._derived_plan)
        displayed_val = self.__plan_details.plan_info('laptop_plan')
        for key_dict, val_dict in validation_dict.items():
            if isinstance(val_dict, list):
                self.log.info('Entity given_val "{0}"'.format(val_dict))
                if set(displayed_val[key_dict]) == set(validation_dict[key_dict]):
                    self.log.info(
                        "Override displayed value {0} matches with {1} given".format(
                            displayed_val[key_dict], validation_dict[key_dict]))
                else:
                    exp = "Override displayed value [{0}] does not match with given value [{1}]".format(
                        displayed_val[key_dict], validation_dict[key_dict])
                    raise Exception(exp)
            elif isinstance(val_dict, str):
                self.log.info('Entity given_val "{0}"'.format(val_dict))
                if displayed_val[key_dict] == validation_dict[key_dict]:
                    self.log.info(
                        "Override displayed value {0} matches with {1} given".format(
                            displayed_val[key_dict], validation_dict[key_dict]))
                else:
                    exp = "Override displayed value [{0}] does not match with given value [{1}]" \
                        .format(displayed_val[key_dict], validation_dict[key_dict])
                    raise Exception(exp)
            else:
                self.log.info('Entity given_val :{0}'.format(val_dict))
                for item, value_dict in val_dict.items():
                    d_val = displayed_val[key_dict][item]
                    key_val = validation_dict[key_dict][item]
                    if d_val == key_val:
                        self.log.info("{0} values match".format(item))
                    else:
                        exp = "Override displayed value [{0}] does not match with given value [1}]"\
                            .format(d_val, key_val)
                        raise Exception(exp)
