# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# -------------------------------------------------------------------------

"""
This module provides the function or operations that can be used to run
basic operations on Operation Window page.

Classes:

    OperationWindowMain()

Functions:

    create_operation_rule            :  Add a configuration to the Operation Window page

    modify_operation_rule            :  Modifies/Edits the Added Operation Rule

    del_operation_rule               :  Deletes the Back up rule added by create_operation_rule method
                                        after validation

    validate_operation_rule          :  Validates the Backup rule added by create_operation_rule method
"""

from datetime import datetime

from AutomationUtils import logger
from Web.AdminConsole.AdminConsolePages.OperationWindow import OperationWindow
from Web.AdminConsole.Components.table import Table


class OperationWindowMain:
    """Helper file to provide arguments and handle function call to main file"""

    def __init__(self, admin_console):
        """ Initialize method for BackupRuleMain """

        self.__admin_console = admin_console
        self.__navigator = admin_console.navigator
        self.__operation_window = OperationWindow(self.__admin_console)
        self.__table = Table(self.__admin_console)

        self.log = logger.get_log()
        self._operation_rule_name = None
        self._old_operation_rule_name = None
        self._start_date = None
        self._end_date = None
        self._operation_day = dict.fromkeys(['Sunday', 'Monday', 'Wednesday', 'Friday'], 1)
        self._operation_time = dict.fromkeys(['0', '1', '2', '3', '4', '14',
                                              '15', '16', '17', '18'], 1)
        self._original_operation_time = "12:00 am to 5:00 am, 2:00 pm to 7:00 pm"
        self._modified_operation_time = "6:00 am to 2:00 pm, 9:00 pm to 11:59 pm"
        self._company = None
        self._backup_operations = {'Full': '1',
                                   'Incremental and Differential': '0',
                                   'Synthetic Full': '1'}
        self.edit_config = 0

    @property
    def operation_rule_name(self):
        """ Get operation rule name"""
        return self._operation_rule_name

    @operation_rule_name.setter
    def operation_rule_name(self, value):
        """ Set operation rule name"""
        self._operation_rule_name = value
        if self._old_operation_rule_name is None:
            self._old_operation_rule_name = value

    @property
    def start_date(self):
        """ Get start date"""
        return self._start_date

    @start_date.setter
    def start_date(self, value):
        """ Set start date"""
        self._start_date = value

    @property
    def end_date(self):
        """ Get end date"""
        return self._end_date

    @end_date.setter
    def end_date(self, value):
        """ Set end date"""
        self._end_date = value

    @property
    def operation_day(self):
        """ Get operation_day value"""
        return self._operation_day

    @operation_day.setter
    def operation_day(self, value):
        """ Set operation_day value"""
        self._operation_day = value

    @property
    def operation_time(self):
        """ Get operation_time value"""
        return self._operation_time

    @operation_time.setter
    def operation_time(self, value):
        """ Set operation_time value"""
        self._operation_time = value

    @property
    def company(self):
        """ Get company value"""
        return self._company

    @company.setter
    def company(self, value):
        """ Set company value"""
        self._company = value

    @property
    def backup_operations(self):
        """ Get backup_operations value"""
        return self._backup_operations

    @backup_operations.setter
    def backup_operations(self, value):
        """ Set backup_operations value"""
        self._backup_operations = value

    def create_operation_rule(self):
        """Calls the function to create Operation Rule"""

        self.__navigator.navigate_to_operation_window()
        self.__operation_window.add_operation_rule(
            self.operation_rule_name,
            self.start_date,
            self.end_date,
            self.operation_day,
            self.operation_time,
            self.backup_operations,
            self.company)
        self._start_date = [self.start_date]
        self._end_date = [self.end_date]

    def modify_operation_rule(self):
        """Calls the function to modify/edit Operation Rule"""

        self.__navigator.navigate_to_operation_window()
        for start_date_val in self.start_date:
            for end_date_val in self.end_date:
                self.edit_config += 1
                if self.edit_config == 1:
                    self.__operation_window.edit_operation_rule(
                        self._old_operation_rule_name,
                        self.operation_rule_name,
                        self.company,
                        self.backup_operations,
                        start_date_val,
                        end_date_val,
                        self.operation_day,
                        self.operation_time)
                else:
                    self.__operation_window.edit_operation_rule(
                        self.operation_rule_name,
                        self.operation_rule_name,
                        self.company,
                        self.backup_operations,
                        start_date_val,
                        end_date_val,
                        self.operation_day,
                        self.operation_time)

    def del_operation_rule(self):
        """Calls the function to delete Operation Rule"""

        self.__navigator.navigate_to_operation_window()
        self.__operation_window.delete_operation_rule(self.operation_rule_name)

    def validate_operation_rule(self):
        """Validates operation Rule, if values are retained correctly or not"""

        self.__navigator.navigate_to_operation_window()
        operation_rule_details = self.__operation_window.fetch_operation_rule_details(self.operation_rule_name)
        operation_days_list = []
        operation_days = ''
        counter = 0
        for key, value in self.operation_day.items():
            if value == 1:
                operation_days_list.append(key)

        list_len = len(operation_days_list)

        for day in operation_days_list:
            counter += 1
            if counter == list_len:
                operation_days = operation_days + str(day)
            else:
                operation_days = operation_days + str(day) + ", "

        if not operation_rule_details['Name'] == [self.operation_rule_name]:
            exp = "Operation Rule name given %s does not match with displayed %s" \
                  % ([self.operation_rule_name], operation_rule_details['Name'])
            raise Exception(exp)
        self.log.info("Operation rule name given %s matched with %s displayed "
                      % ([self.operation_rule_name], operation_rule_details['Name']))

        if not operation_rule_details['Days'] == [operation_days]:
            exp = "Operation day values given %s does not match with displayed %s values" \
                  % ([operation_days], operation_rule_details['Days'])
            raise Exception(exp)
        self.log.info("Operation day values %s given matched with %s displayed"
                      % ([operation_days], operation_rule_details['Days']))

        if self.edit_config == 0:
            if not operation_rule_details[
                       'Do not run between the following time intervals'] == [self._original_operation_time]:
                exp = "Operation time given %s, does not match with %s displayed" % \
                      ([self._original_operation_time],
                       operation_rule_details['Do not run between the following time intervals'])
                raise Exception(exp)
            self.log.info("Operation time values given %s matched with %s displayed" %
                          ([self._original_operation_time],
                           operation_rule_details['Do not run between the following time intervals']))
        else:
            if not operation_rule_details[
                       'Do not run between the following time intervals'] == [self._modified_operation_time]:
                exp = "Operation time given %s, does not match with %s displayed" % \
                      ([self._modified_operation_time],
                       operation_rule_details['Do not run between the following time intervals'])
                raise Exception(exp)
            self.log.info("Operation time values given %s matched with %s displayed" %
                          ([self._modified_operation_time],
                           operation_rule_details['Do not run between the following time intervals']))

        if self.start_date[0] and self.end_date[0]:
            start = datetime.strftime(datetime.strptime(self.start_date[0], '%m/%d/%Y'), '%b %d, %Y')
            end = datetime.strftime(datetime.strptime(self.end_date[0], '%m/%d/%Y'), '%b %d, %Y')
            date_string = str(start) + " to " + str(end)
            if not operation_rule_details['Dates'] == [date_string]:
                exp = "start date and end date given %s do not match with values %s displayed" % \
                      ([date_string], operation_rule_details['Dates'])
                raise Exception(exp)
        elif self.start_date[0]:
            start = datetime.strftime(datetime.strptime(self.start_date[0], '%m/%d/%Y'), '%b %d, %Y')
            date_string = str(start) + " to Not set"
            if not operation_rule_details['Dates'] == [date_string]:
                exp = "start date and end date given %s do not match with values %s displayed" % \
                      ([date_string], operation_rule_details['Dates'])
                raise Exception(exp)
        elif self.end_date[0]:
            end = datetime.strftime(datetime.strptime(self.end_date[0], '%m/%d/%Y'), '%b %d, %Y')
            date_string = "Not set to " + str(end)
            if not operation_rule_details['Dates'] == [date_string]:
                exp = "start date and end date given %s do not match with values %s displayed" % \
                      ([date_string], operation_rule_details['Dates'])
                raise Exception(exp)
        else:
            date_string = "Not set"
            if not operation_rule_details['Dates'] == [date_string]:
                exp = "start date and end date given %s do not match with values %s displayed" % \
                      ([date_string], operation_rule_details['Dates'])
                raise Exception(exp)
        self.log.info("start date and end date given %s matched with values %s displayed" %
                      ([date_string], operation_rule_details['Dates']))

        if self.company:
            if not operation_rule_details['Company'] == [self.company]:
                exp = "Company name given %s does not match with company name %s displayed" % \
                      (self.company, operation_rule_details['Company'])
                raise Exception(exp)
            self.log.info("Company name given %s matched with %s displayed" %
                          (self.company, operation_rule_details['Company']))
        else:
            if not operation_rule_details['Company'] == ['CommCell']:
                exp = "Company name given %s does not match with company name %s displayed" % \
                      (['CommCell'], operation_rule_details['Company'])
                raise Exception(exp)
            self.log.info("Company name given %s matched with %s displayed" %
                          (['CommCell'], operation_rule_details['Company']))

        self.log.info("Operation rule validated successfully, all values were correctly retained")
