# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""This Module provides methods to run different template for NAS Operations

Class : NASTemplate

Functions:
    cleanup()           : To cleanup entities created during execution

    create_entities()   : To create required entities like Plans, Subclient etc

    verify_backup()     : To backup nas client

    verify_restore()     : To restore nas client

    get_test_data_path()  : Getting test data path

    add_test_data()      : Add test data for nas subclient

    get_path_from_content() : Add path from content

    validate_restored_content() : to validate restore data

    compare_data()   : to compare backed up data and restored data

    nastemplate()     : Template for NAS Test Cases

"""
import random
import string
from datetime import date

from Web.AdminConsole.Components.dialog import ModalDialog
from AutomationUtils.machine import Machine
from AutomationUtils.options_selector import OptionsSelector
from Web.AdminConsole.AdminConsolePages.Plans import Plans
from Web.AdminConsole.Components.panel import Backup, PanelInfo
from Web.AdminConsole.FileServerPages.file_servers import FileServers
from Web.AdminConsole.FileServerPages.fsagent import FsSubclient
from Web.AdminConsole.FileServerPages.fssubclientdetails import FsSubclientDetails
from Web.AdminConsole.Components.table import Table
from Web.Common.exceptions import CVTestStepFailure
from Web.Common.page_object import TestStep


class NASTemplate(object):
    """ NasTemplate Class for NAS Cases"""
    test_step = TestStep()

    def __init__(self, testcase, admin_console):

        """Initializing the Test case file"""
        self.__admin_console = admin_console
        self.__navigator = admin_console.navigator
        self.tcinputs = testcase.tcinputs
        self.commcell = testcase.commcell
        self.testcase = testcase
        self.__driver = admin_console.driver
        self.__modal_dialog = ModalDialog(self.__admin_console)
        self.plan_obj = Plans(self.__admin_console)
        self.fs_servers = FileServers(self.__admin_console)
        self.fs_subclient = FsSubclient(self.__admin_console)
        self.fs_scdetails = FsSubclientDetails(self.__admin_console)
        self._snappanel = PanelInfo(self.__admin_console, 'Snap')
        self.table = Table(self.__admin_console)
        self.log = testcase.log
        self._storagepool_name = {'pri_storage': self.tcinputs['StoragePoolName']}
        _random_number = random.randint(0, 100)
        self._plan_name = "CC_AutoPlan_{0}_{1}".format(self.testcase.id, _random_number)
        self._subclient_name = "CC_AutoSC_{0}_{1}".format(self.testcase.id, _random_number)
        self.server_name = self.tcinputs['ServerName']
        self._backupset_name = 'defaultBackupSet'
        self.out_restore_path = None
        self.local_machine = Machine()
        self.subclient_content = self.tcinputs["SubclientContent"].split(',')
        self.domainUser = self.tcinputs["domainUsername"]
        self.domainPassword = self.tcinputs["domainPassword"]
        self.agent_name = self.tcinputs['Agent']
        self.dest_filer = self.tcinputs["DestinationFiler"]
        self.filer_restore_path = self.tcinputs["FilerRestoreLocation"]
        self.windows_client = self.tcinputs['WindowsDestinationClient']
        self.unix_client = self.tcinputs['UnixDestinationClient']
        windows_client_obj = self.commcell.clients.get(self.windows_client)
        self.windows_restore_machine = Machine(windows_client_obj)
        unix_client_obj = self.commcell.clients.get(self.unix_client)
        self.unix_restore_machine = Machine(unix_client_obj)
        self.options_selector = OptionsSelector(self.commcell)
        self.enable_snap = testcase.enable_snap
        self.engine_name = self.tcinputs['ArrayVendor']
        self.__admin_console._load_properties(self)
        self.test_data_paths = []

    @property
    def storagepool_name(self):
        """Return Storage Pool Name"""
        return self._storagepool_name

    @storagepool_name.setter
    def storagepool_name(self, value):
        """Set Storage Pool name"""
        self._storagepool_name = value

    @property
    def plan_name(self):
        """Return Plan Name"""
        return self._plan_name

    @plan_name.setter
    def plan_name(self, value):
        """Set Plan name"""
        self._plan_name = value

    @property
    def subclient_name(self):
        """Return Subclient Name"""
        return self._subclient_name

    @subclient_name.setter
    def subclient_name(self, value):
        """Set Subclient Name"""
        self._subclient_name = value

    def wait_for_job_completion(self, jobid):
        """Waits for Backup or Restore Job to complete"""
        job_obj = self.commcell.job_controller.get(jobid)
        return job_obj.wait_for_completion()

    @test_step
    def cleanup(self):
        """To perform cleanup operation"""
        try:
            self.__navigator.navigate_to_file_servers()
            self.fs_servers.access_server(self.server_name)
            self.__admin_console.wait_for_completion()
            self.table.access_link(self.agent_name)
            self.fs_subclient.delete_subclient(self._backupset_name, subclient_name=self.subclient_name)
            self.__admin_console.wait_for_completion()
            self.__navigator.navigate_to_file_servers()
            self.fs_servers.access_server(self.server_name)
            self.__admin_console.select_hyperlink(self.__admin_console.props['label.releaseLicense'])
            self.__admin_console.checkbox_select('check0')
            self.__modal_dialog.click_submit()
            self.__admin_console.wait_for_completion()
            self.__navigator.navigate_to_file_servers()
            self.fs_servers.retire_server(server_name=self.server_name)
            self.__admin_console.wait_for_completion()
            self.__admin_console.refresh_page()
            self.__navigator.navigate_to_file_servers()
            self.fs_servers.delete_client(server_name=self.server_name)
            self.__admin_console.wait_for_completion()
            self.__navigator.navigate_to_plan()
            self.plan_obj.delete_plan(self.plan_name)

        except Exception as exp:
            raise CVTestStepFailure(f'Cleanup entities failed with error : {exp}')

    @test_step
    def create_entities(self):
        """To create required entities for test case"""
        try:
            # To create a new plan
            self.log.info("Adding a new plan: %s", self.plan_name)
            self.__navigator.navigate_to_plan()
            self.plan_obj.create_server_plan(plan_name=self.plan_name,
                                             storage=self.storagepool_name
                                             )
            self.log.info("successfully created plan: %s", self.plan_name)

            # To add a new client
            self.log.info("Adding a new client %s", self.server_name)
            self.__navigator.navigate_to_file_servers()
            ndmp, cifs, nfs = None, None, None
            if self.agent_name.lower() == 'ndmp':
                ndmp = True
            elif self.agent_name.lower() == 'cifs':
                cifs = True
            else:
                nfs = True
            self.fs_servers.add_nas_client(server_name=self.server_name,
                                           ndmp_ida_=ndmp,
                                           cifs_ida=cifs,
                                           nfs_ida=nfs,
                                           ndmp_username=self.tcinputs['CIFSShareUser'],
                                           ndmp_password=self.tcinputs['CIFSSharePassword'],
                                           ndmp_port=10000,
                                           accessnode_ma=None)
            self.log.info("Created a new nas server %s", self.server_name)

            # To add a new Subclient
            self.log.info("Adding a new subclient %s", self.subclient_name)
            self.__navigator.navigate_to_file_servers()
            self.fs_servers.access_server(self.server_name)
            self.table.access_link(self.agent_name)
            self.fs_subclient.add_nas_subclient(subclient_name=self.subclient_name,
                                                subclient_content=self.subclient_content,
                                                nas_plan=self.plan_name)
            self.log.info("Created a new subclient %s", self.subclient_name)
            if self.enable_snap:
                self.__navigator.navigate_to_file_servers()
                self.fs_servers.access_server(self.server_name)
                self._snappanel.enable_toggle(self.__admin_console.props['label.snapEnabled'])
                self.table.access_link(self.agent_name)
                self.fs_subclient.access_subclient(self._backupset_name, subclient_name=self.subclient_name)
                self.fs_scdetails.enable_snapshot_engine(enable_snapshot=True,
                                                         engine_name=self.engine_name)
        except Exception as exp:
            raise CVTestStepFailure(f'Create entities failed with error : {exp}')

    @test_step
    def verify_backup(self, backup_type):
        """Verify Backup"""
        try:
            self.__navigator.navigate_to_file_servers()
            self.fs_servers.access_server(self.tcinputs['ServerName'])
            self.table.access_link(self.agent_name)
            jobid = self.fs_subclient.backup_subclient(
                backupset_name=self._backupset_name, subclient_name=self.subclient_name,
                backup_type=backup_type)
            job_status = self.wait_for_job_completion(jobid)
            if not job_status:
                exp = "{0} Job ID {1} didn't succeed".format(backup_type, jobid)
                raise Exception(exp)
            return jobid
        except Exception as exp:
            raise CVTestStepFailure(f'Backup operation failed : {exp}')

    @test_step
    def verify_restore(self, dest_client=None, restore_path=None):
        """Restores a NAS Subclient from subclient level and verifies restore job completion

        Args:
            restore_path:str: restore path of the destination  (optional)
            dest_client: str: destination client       (optional)

        """

        try:
            self.__navigator.navigate_to_file_servers()
            self.fs_servers.access_server(self.tcinputs['ServerName'])
            self.table.access_link(self.agent_name)
            selected_content = []
            for path in self.subclient_content:
                temp_path = path.strip('/').split('/')
                selected_content.append('/'.join(temp_path[:-1]))
            rjobid = self.fs_subclient.restore_subclient(backupset_name="defaultBackupSet",
                                                         subclient_name=self.subclient_name,
                                                         dest_client=dest_client,
                                                         diff_os=True,
                                                         restore_path=restore_path,
                                                         selected_files=selected_content)
            rjob_status = self.wait_for_job_completion(rjobid)
            if not rjob_status:
                exp = "Restore Job ID {0} didn't succeed".format(rjobid)
                raise Exception(exp)

        except Exception as exp:
            raise CVTestStepFailure(f'Restore operation failed : {exp}')

    @test_step
    def restore_from_client(self, dest_client=None, restore_path=None):
        """Restores a NAS Subclient from Client level and verifies restore job completion

        Args:
            restore_path:str: restore path of the destination  (optional)
            dest_client: str: destination client       (optional)

        """
        try:
            self.__navigator.navigate_to_file_servers()
            selected_content = []
            for path in self.subclient_content:
                temp_path = path.strip('/').split('/')
                selected_content.append('/'.join(temp_path[:-1]))
            rjobid = self.fs_servers.restore_subclient(client_name=self.server_name,
                                                       dest_client=dest_client,
                                                       restore_path=restore_path,
                                                       subclient_name=self.subclient_name,
                                                       selected_files=selected_content)
            rjob_status = self.wait_for_job_completion(rjobid)
            if not rjob_status:
                exp = "Restore Job ID {0} didn't succeed".format(rjobid)
                raise Exception(exp)

        except Exception as exp:
            raise CVTestStepFailure(f'Restore operation failed : {exp}')

    @test_step
    def restore_from_recovery_point(self, recovery_time, dest_client=None, restore_path=None):
        """Restores from a recovery point and verifies restore job completion

        Args:
            recovery_time:str: basestring: the backup date in 01-September-1960 format
            restore_path:str: restore path of the destination  (optional)
            dest_client: str: destination client       (optional)

        """
        try:
            self.__navigator.navigate_to_file_servers()
            self.fs_servers.access_server(self.tcinputs['ServerName'])
            self.table.access_link(self.agent_name)
            self.fs_subclient.access_subclient(self._backupset_name, subclient_name=self.subclient_name)
            selected_content = []
            for path in self.subclient_content:
                temp_path = path.strip('/').split('/')
                selected_content.append('/'.join(temp_path[:-1]))
            rjobid = self.fs_scdetails.restore_recovery_points(recovery_time=recovery_time,
                                                               dest_client=dest_client,
                                                               restore_path=restore_path,
                                                               selected_files=selected_content)
            rjob_status = self.wait_for_job_completion(rjobid)
            if not rjob_status:
                exp = "Restore Job ID {0} didn't succeed".format(rjobid)
                raise Exception(exp)

        except Exception as exp:
            raise CVTestStepFailure(f'Restore operation failed : {exp}')

    @test_step
    def restore_by_job(self, jobid, dest_client=None, restore_path=None):
        """Restores a NAS Subclient from subclient level and verifies restore job completion

        Args:
            restore_path:str: restore path of the destination  (optional)
            dest_client: str: destination client       (optional)

        """
        try:
            self.__navigator.navigate_to_file_servers()
            self.fs_servers.access_server(self.tcinputs['ServerName'])
            self.table.access_link(self.agent_name)
            selected_content = []
            for path in self.subclient_content:
                temp_path = path.strip('/').split('/')
                selected_content.append('/'.join(temp_path[:-1]))
            rjobid = self.fs_subclient.restore_subclient_by_job(backupset_name="defaultBackupSet",
                                                                subclient_name=self.subclient_name,
                                                                job_id=jobid,
                                                                dest_client=dest_client,
                                                                restore_path=restore_path,
                                                                selected_files=selected_content)
            rjob_status = self.wait_for_job_completion(rjobid)
            if not rjob_status:
                exp = "Restore Job ID {0} didn't succeed".format(rjobid)
                raise Exception(exp)

        except Exception as exp:
            raise CVTestStepFailure(f'Restore operation failed : {exp}')

    def get_test_data_path(self, size):
        """Returns the test data path on this machine

            Args:
                size            (int)       --  size of test data that is to be generated

            Returns:
                str     -   path where test data is generated

            Raises:
                Exception:
                    if failed to generate test data on controller machine
        """
        drives_dict = self.local_machine.get_storage_details()
        if self.local_machine.os_info == "WINDOWS":
            for drive in drives_dict.keys():
                if not isinstance(drives_dict[drive], dict):
                    continue

                if float(drives_dict[drive]['available']) >= size:
                    return drive + ":\\" + ''.join(random.choices(string.ascii_uppercase, k=7))
        elif self.local_machine.os_info == "UNIX":
            if float(drives_dict['available']) >= size:
                return "/" + ''.join(random.choices(string.ascii_uppercase, k=7))

        raise Exception("Failed to get test data path")

    @test_step
    def add_test_data(self):
        """
                Adds test under every path in subclient content

                Returns:
                    list    --  paths where test data is generated
        """
        list_paths = {}
        for item in self.subclient_content:
            path, vol = self.get_path_from_content(self.server_name, item)
            test_data_size = 10
            test_data_path = self.get_test_data_path(test_data_size)
            test_data_folder = test_data_path.split(self.local_machine.os_sep)[-1]
            self.local_machine.generate_test_data(test_data_path, file_size=test_data_size)

            self.log.info("Generated Test Data at path: " + test_data_path)
            self.local_machine.copy_folder_to_network_share(test_data_path, path,
                                                            self.domainUser, self.domainPassword)
            self.log.info("Copying test data to: " + path)
            list_paths[path + '\\' + test_data_folder] = self.local_machine.get_folder_hash(test_data_path)
            self.log.info(f"Removing locally generated test data {test_data_path}")
            self.local_machine.remove_directory(test_data_path)

        return list_paths

    def get_path_from_content(self, server_name, content):
        """Returns the Share path and volume name

            Args:
                server_name (str)  -- server name
                content     (str)   --  volume path from the subclient content

            Returns:
                (str, str)  -   cifs share path and path of the volume

        """
        if '/' in content:
            content = content.strip("/").split("/")
        else:
            content = content.strip("\\").split("\\")
        volume_name = "\\".join(content[1:])
        vserver_ip = server_name
        return r"\\{0}\{1}".format(vserver_ip, volume_name), volume_name

    @test_step
    def validate_filer_restored_content(self, restored_filer_name, filer_restore_location):
        """Validates the restored content
                Args:
                    restored_filer_name: str: name of the filer
                    filer_restore_location: str: restore path on the filer
            Raises:
                Exception:
                    if failed to validate restored content

        """
        for content in self.test_data_paths.keys():
            src_path, volume_name = self.get_path_from_content(self.tcinputs["ServerName"], content)
            dest_path, restore_vol = self.get_path_from_content(restored_filer_name, filer_restore_location)
            dest_restore_path = dest_path + "\\" + volume_name
            self.log.info("Destination Restore Path:{0}".format(dest_restore_path))

            src_hash = self.test_data_paths[content]

            dest_mount_drive = self.local_machine.mount_network_path(dest_restore_path,
                                                                     self.domainUser, self.domainPassword)
            dest_hash = self.local_machine._get_folder_hash(dest_mount_drive)
            self.log.info(f"Unmounting {dest_restore_path} at {dest_mount_drive}")
            self.local_machine.unmount_drive(dest_mount_drive)
            if src_hash == dest_hash:
                self.log.info("Restore validation Success.")
            else:
                raise Exception("Restore validation failed. Please check logs for more details.")

    @test_step
    def validate_inplace_restored_content(self, test_data):
        """Validates the restored content
                Args:
                    test_data : dict: local {test_data path : data_hash}
            Raises:
                Exception:
                    if failed to validate restored content

        """
        for i in test_data.keys():
            mount_drive = self.local_machine.mount_network_path(i,
                                                                self.domainUser, self.domainPassword)
            dest_restored_hash = self.local_machine._get_folder_hash(mount_drive)
            self.log.info(f"Unmounting {i} at {mount_drive}")
            self.local_machine.unmount_drive(mount_drive)
            if dest_restored_hash == test_data[i]:
                self.log.info("Restore Validation Success.")
            else:
                raise Exception("Restore validation failed. Please check logs for more details.")

    @test_step
    def validate_os_restored_content(self, restore_client, restore_location, os_type='windows'):
        """Validates the restored content in windows/Unix client

            Args:
                restore_client      (object)    --  machine class object for  client
                                                                where the content was restored
                restore_location (str) -- destination restore location path
                os_type (str) -- 'windows' or 'unix' OS

            Raises:
                Exception:
                    if failed to validate restored content

        """
        diff = True
        for content in self.test_data_paths:
            src_path, volume_name = self.get_path_from_content(self.tcinputs["ServerName"], content)
            self.log.info("source path:{0} ,volume path: {1}".format(src_path, volume_name))
            restore_path = restore_location + "\\" + volume_name
            if os_type == 'unix':
                volume_name = volume_name.replace('\\', '/')
                restore_path = restore_location + "/" + volume_name
            self.log.info("Destination Restore Path:{0}".format(restore_path))
            diff = self.compare_data(src_path, restore_client, restore_path)
        if not diff:
            self.log.error(
                "Restore validation failed.")
            raise Exception(
                "Restore validation failed. Please check logs for more details."
            )

        self.log.info("Successfully validated restored content")

    def compare_data(self, source_content, destination_machine, restore_path):
        """Compares the two directories on different machines
            Arguments:
                source_content: str: data path at source
                destination_machine: object: destination client machine object
                restore_path: str: restore data path at  dest restored client

            Returns:
                 bool: True : if hash check matches
                       False: if hash match fails
        """
        mount_drive = self.local_machine.mount_network_path(source_content,
                                                            self.domainUser, self.domainPassword)
        source_hash = self.local_machine._get_folder_hash(mount_drive)
        self.local_machine.unmount_drive(mount_drive)
        dest_hash = destination_machine._get_folder_hash(restore_path)
        result = bool(source_hash == dest_hash)
        if not result:
            self.log.info("Data at two paths do not match")
            return False
        self.log.info('Comparison successful , data at both paths is identical')
        return True

    def NasTemplate(self):
        """Main function for test case execution"""

        self.create_entities()
        self.log.info("Add Test Data")
        self.test_data_paths = self.add_test_data()
        # Run Full backup and Inplace Restore
        self.verify_backup(Backup.BackupType.FULL)
        self.verify_restore()
        self.validate_inplace_restored_content(self.test_data_paths)
        # Run INCR Backup and Outplace filer Restore
        self.log.info("Add Test Data")
        self.test_data_paths = self.add_test_data()
        self.verify_backup(Backup.BackupType.INCR)
        self.verify_restore(dest_client=self.dest_filer, restore_path=self.filer_restore_path)
        self.validate_filer_restored_content(self.dest_filer, self.filer_restore_path)
        # Run DIFF backup and Outplace Restore to Windows Client
        self.log.info("Add Test Data")
        self.test_data_paths = self.add_test_data()
        self.verify_backup(Backup.BackupType.DIFF)
        dir_name = self.options_selector._get_restore_dir_name()
        windows_restore_path = f"C:\\{dir_name}"
        self.out_restore_path = windows_restore_path
        self.log.info(f"Creating restore path at destination : {windows_restore_path}")
        self.windows_restore_machine.create_directory(windows_restore_path)
        self.verify_restore(dest_client=self.windows_client, restore_path=self.out_restore_path)
        self.validate_os_restored_content(self.windows_restore_machine, self.out_restore_path)
        self.log.info(f"Removing windows restored directory: {windows_restore_path}")
        self.windows_restore_machine.remove_directory(windows_restore_path)
        # Run FULL backup and Outplace Restore to Unix Client
        self.log.info("Add Test Data")
        self.test_data_paths = self.add_test_data()
        self.verify_backup(Backup.BackupType.FULL)
        dir_name = self.options_selector._get_restore_dir_name()
        unix_restore_path = f"/root/Desktop/{dir_name}"
        self.out_restore_path = f"root/Desktop/{dir_name}"
        self.log.info(f"Creating restore path at destination : {unix_restore_path}")
        self.unix_restore_machine.create_directory(unix_restore_path)
        self.verify_restore(dest_client=self.unix_client, restore_path=self.out_restore_path)
        self.validate_os_restored_content(self.unix_restore_machine, unix_restore_path, os_type='unix')
        self.log.info(f"Removing Unix restored directory: {unix_restore_path}")
        self.unix_restore_machine.remove_directory(unix_restore_path)
        self.cleanup()

    def NasTemplate2(self):
        """Main function for test case execution
            Implementing restores from different points"""
        self.create_entities()
        self.log.info("Add Test Data")
        self.test_data_paths = self.add_test_data()
        # Run Full backup and Inplace Restore
        self.verify_backup(Backup.BackupType.FULL)
        self.verify_restore()
        self.validate_inplace_restored_content(self.test_data_paths)
        # Run INCR Backup and Outplace filer Restore
        self.log.info("Add Test Data")
        self.test_data_paths = self.add_test_data()
        self.verify_backup(Backup.BackupType.INCR)
        self.restore_from_client(dest_client=self.dest_filer, restore_path=self.filer_restore_path)
        self.validate_filer_restored_content(self.dest_filer, self.filer_restore_path)
        # Run DIFF backup and Outplace Windows Restore
        self.log.info("Add Test Data")
        self.test_data_paths = self.add_test_data()
        backup_jobid = self.verify_backup(Backup.BackupType.DIFF)
        dir_name = self.options_selector._get_restore_dir_name()
        windows_restore_path = f"C:\\{dir_name}"
        self.out_restore_path = windows_restore_path
        self.log.info(f"Creating restore path at destination : {windows_restore_path}")
        self.windows_restore_machine.create_directory(windows_restore_path)
        self.restore_by_job(backup_jobid, dest_client=self.windows_client, restore_path=self.out_restore_path)
        self.validate_os_restored_content(self.windows_restore_machine, self.out_restore_path)
        self.log.info(f"Removing windows restored directory: {windows_restore_path}")
        self.windows_restore_machine.remove_directory(windows_restore_path)
        # Run FULL backup and Outplace Unix Restore
        self.log.info("Add Test Data")
        self.test_data_paths = self.add_test_data()
        self.verify_backup(Backup.BackupType.FULL)
        recovery_point = date.today().strftime("%d-%B-%Y")
        dir_name = self.options_selector._get_restore_dir_name()
        unix_restore_path = f"root/Desktop/{dir_name}"
        self.out_restore_path = f"root/Desktop/{dir_name}"
        self.log.info(f"Creating restore path at destination : {unix_restore_path}")
        self.unix_restore_machine.create_directory(unix_restore_path)
        self.restore_from_recovery_point(recovery_time=recovery_point,
                                         dest_client=self.unix_client, restore_path=self.out_restore_path)
        self.validate_os_restored_content(self.unix_restore_machine, unix_restore_path, os_type='unix')
        self.log.info(f"Removing windows restored directory: {unix_restore_path}")
        self.unix_restore_machine.remove_directory(unix_restore_path)
        self.cleanup()
