# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations related to Laptop in AdminConsole
LaptopMain : This class provides methods for Laptop related operations

Class:

    LaptopMain()

Functions:

get_client_name_from_hostname     : To get the client name from the host name

__validate_install                  : To validate install on the client machine

check_if_laptop_added               : Method to check whether laptop appears in the Solutions->Laptop page

add_new_windows_laptop              : calls method to add windows laptop from laptops page

add_new_mac_aptop()                 : Method to add mac laptop to laptops page

deactivate_laptop()                 : Method to deactivate laptop from laptops page

update_software()                   : Method to update software from laptops page

check_readiness()                   : Method to perform check readiness

send_logs()                         : Method to perform action send logs

retire_client()                     : Method to retire the client

activate_laptop_byuser()            : Method to activate laptop by enduser

activate_laptop_byplan()            : Method to activate laptop by plan

perform_backup_now()                : call the method to perform backup

validate_backup_after_deactivation() : validation of the backup after deactivation

covert_size_to_mb()                  : convert given the size to MB

auto_trigger_backupjob()             : Auto trigger backup job when subclient content added

get_backup_details()                 : Read the current backup details from adminconsole

validate_laptop_summaryinfo()        : Validates the laptop summary info of the laptop page

set_validation_values()              : Set the validation values for laptop

validate_laptop_details()            : Validates the values displayed for client against values provided as input

subclient_restore()                  : validate laptop restore

"""

import datetime
import math
import inspect
import ntpath
import time
from AutomationUtils import logger
from AutomationUtils.idautils import CommonUtils
from AutomationUtils.options_selector import OptionsSelector
from Server.JobManager.jobmanager_helper import JobManager
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Laptop.Laptops import Laptops
from Web.AdminConsole.Laptop.LaptopDetails import LaptopDetails
from Web.AdminConsole.Components.panel import Backup

from Web.AdminConsole.AdminConsolePages.Jobs import Jobs

from Install.install_validator import InstallValidator

class LaptopMain:
    """Admin console helper for Laptop operations"""

    def __init__(self, admin_page, commcell, csdb=None):
        """
        Helper for Laptop related files

        Args:
            admin_page  (object)   --  AdminPage class object

        """
        self.commcell = commcell
        self.csdb = csdb
        self.__admin_console = admin_page
        self.__driver = admin_page.driver
        self.__table = Table(admin_page)
        self.__laptops = Laptops(admin_page)
        self.__jobs = Jobs(self.__admin_console)
        self.__navigator = admin_page.navigator
        self.__backup = Backup(admin_page)
        self.__ida_utils = CommonUtils(commcell)
        self.__utility = OptionsSelector(commcell)
        self.__job_manager = JobManager(commcell=commcell)
        self.__laptop_details = LaptopDetails(admin_page, commcell=commcell)
        self.log = logger.get_log()

        self._validate = None
        self._host_name = None
        self._client_name = None
        self._user_name = None
        self._password = None
        self._confirm_password = None
        self._activation_user = None
        self._activation_plan = None
        self._machine_object = None
        self._source_dir = None
        self._subclient_obj = None
        self._activation_status = 'Configured'


    @property
    def host_name(self):
        """ Get host_name"""
        return self._host_name

    @host_name.setter
    def host_name(self, value):
        """ Set host_name"""
        self._host_name = value

    @property
    def client_name(self):
        """ Get client_name"""
        return self._client_name

    @client_name.setter
    def client_name(self, value):
        """ Set client_name"""
        self._client_name = value

    @property
    def user_name(self):
        """ Get user name """
        return self._user_name

    @user_name.setter
    def user_name(self, value):
        """ Set user_name """
        self._user_name = value

    @property
    def password(self):
        """ Get password"""
        return self._password

    @password.setter
    def password(self, value):
        """ Set password"""
        self._password = value

    @property
    def confirm_password(self):
        """ Get confirm_password"""
        return self._password

    @confirm_password.setter
    def confirm_password(self, value):
        """ Set confirm_password"""
        self._confirm_password = value

    @property
    def activation_user(self):
        """ Get activate_user"""
        return self._activation_user

    @activation_user.setter
    def activation_user(self, value):
        """ Set activate_user"""
        self._activation_user = value

    @property
    def activation_plan(self):
        """ Get activation_plan"""
        return self._activation_plan

    @activation_plan.setter
    def activation_plan(self, value):
        """ Set activation_plan"""
        self._activation_plan = value

    @property
    def machine_object(self):
        """ Get machine_object"""
        return self._machine_object

    @machine_object.setter
    def machine_object(self, value):
        """ Set machine_object"""
        self._machine_object = value

    @property
    def source_dir(self):
        """ Get source data directory"""
        return self._source_dir

    @source_dir.setter
    def source_dir(self, value):
        """ Set source data directory"""
        self._source_dir = value


    def __validate_install(self, client_name):
        """ To validate install on the client machine"""
        self._validate = InstallValidator(client_name, self)
        self._validate.validate_sp_version()
        self._validate.validate_baseline()
        self._validate.validate_services()

    def validate_client_from_adminconsole(self, actvation=True):
        """
           From adminconsole make sure you get a "x" [deconfigured] in front of the client

        """
        client_status = self.__laptops.get_client_configuration_status(self._client_name)
        if not actvation:
            if not client_status == 'Deconfigured':
                exp = "Client {0} status is not showing correctly after deactivation"\
                    .format(self._client_name)
                self.log.exception(exp)
                raise Exception(exp)
            self.log.info('Deactivation validation succeeded from adminconsole')
        else:
            if not client_status == 'Configured':
                exp = "Client {0} status is not showing correctly after activation"\
                    .format(self._client_name)
                self.log.exception(exp)
                raise Exception(exp)
            self.log.info('Activation validation succeeded from adminconsole')


    def check_if_laptop_added(self):
        """ Method to check whether laptop appears in the Solutions->Laptop page """
        self.__navigator.navigate_to_devices()
        if not self.__table.is_entity_present_in_column("Name", self._client_name):
            exp = "Laptop with Given client name {0} does not exist"\
                .format(self._client_name)
            self.log.exception(exp)
            raise Exception(exp)

    def add_new_windows_laptop(self):
        """calls method to add windows laptop from laptops page"""
        self.__navigator.navigate_to_devices()
        job_id = self.__laptops.add_windows_laptop(self.host_name,
                                                   self.user_name,
                                                   self.password,
                                                   self.confirm_password)
        self.log.info("Job %s is started to add new WINDOWS laptop", job_id)
        job_details = self.__jobs.job_completion(job_id)
        if not job_details['Status'] == 'Completed':
            exp = "Add windows laptop job {0} is not completed successfully for machine {1}"\
                    .format(job_id, self.host_name)
            self.log.exception(exp)
            raise Exception(job_details)
        self.check_if_laptop_added()
        self.log.info('New windows laptop is added successfully')

    def add_new_mac_laptop(self):
        """calls method to add mac laptop from laptops page"""
        self.__navigator.navigate_to_devices()
        job_id = self.__laptops.add_mac_laptop(self.host_name,
                                               self.user_name,
                                               self.password,
                                               self.confirm_password)
        self.log.info("Job %s is started to add new MAC laptop", job_id)
        job_details = self.__jobs.job_completion(job_id)
        if not job_details['Status'] == 'Completed':
            exp = "Add Mac laptop job {0} is not completed successfully for machine {1}"\
                    .format(job_id, self.host_name)
            self.log.exception(exp)
            raise Exception(job_details)
        self.check_if_laptop_added()
        self.log.info('New MAC laptop is added successfully')

    def deactivate_laptop(self):
        """calls method to deactivate laptop from laptops page"""
        self.__navigator.navigate_to_devices()
        self.__laptops.deactivate_laptop(self._client_name)
        self.log.info("Client {0} deactivated successfully".format(self._client_name))

    def update_software(self, reboot=False):
        """calls method to update software from laptops page"""
        self.__navigator.navigate_to_devices()
        job_id = self.__laptops.action_update_software(self._client_name, reboot=reboot)
        self.log.info("Job %s is started for update software", job_id)
        job_details = self.__jobs.job_completion(job_id)
        if not job_details['Status'] == 'Completed':
            exp = "Update Software job {0} is not completed for client {1}"\
                    .format(job_id, self._client_name)
            self.log.exception(exp)
            raise Exception(job_details)
        self.__validate_install(self._client_name)
        self.log.info("Successfully updated the client {0}".format(self._client_name))

    def check_readiness(self):
        """calls method to perform check readiness"""
        self.__navigator.navigate_to_devices()
        self.__laptops.action_check_readiness(self._client_name)
        self.log.info("Check readiness completed successfully for client {0}"\
                      .format(self._client_name))

    def retire_client(self):
        """calls method to retire the client"""
        self.__navigator.navigate_to_devices()
        job_id = self.__laptops.action_retire_client(self._client_name)
        self.log.info("Job %s is started to retire the client", job_id)
        job_details = self.__jobs.job_completion(job_id)
        if not job_details['Status'] == 'Completed':
            exp = "Retire client job {0} is not completed for client {1}"\
                    .format(job_id, self._client_name)
            self.log.exception(exp)
            raise Exception(job_details)
        self.log.info("Retire client operation completed Successfully for the client {0}"\
                        .format(self._client_name))

    def activate_laptop_byuser(self):
        """calls method to activate the laptop by enduser"""
        self.__navigator.navigate_to_devices()
        self.__laptops.activate_laptop_byuser(self._client_name)

    def activate_laptop_byplan(self, plan_name=None):
        """calls method to activate the laptop by plan"""
        self.__navigator.navigate_to_devices()
        self.__laptops.activate_laptop_byplan(self._client_name, plan_name)

    def perform_backup_now(self, suspend_resume=False):
        """method to perform backup """
        self.__navigator.navigate_to_devices()
        self.__driver.refresh()

        job_id = self.__laptops.subclient_backup_now(self._client_name, self.__backup.BackupType.INCR)
        if suspend_resume:
            self.log.info("*****Verification of Suspend and resume backup for the client [{0}]*****"\
                      .format(self._client_name))

            jobobj = self.commcell.job_controller.get(job_id=job_id)
            self.__navigator.navigate_to_jobs()
            self.__jobs.suspend_job(job_id)
            self.__admin_console.click_button('Yes')
            #-----------Wait for backup job to move into suspend from "suspend pending" --------#
            self.__jobs.access_job_by_id(job_id)
            self.__job_manager.job = jobobj
            self.__job_manager.wait_for_state('suspended')
            job_details = self.__jobs.job_details()
            if not job_details['Status'] == 'Suspended':
                exp = "Job [{0}] is not suspended for client [{1}]" .format(job_id, self.client_name)
                self.log.exception(exp)
                raise Exception(job_details)
            self.log.info("Job [{0}] suspended Successfully for the client [{1}]"\
                          .format(job_id, self.client_name))
            self.__jobs.resume_job(job_id) #resume the backup job
            time.sleep(60)# wait for the job to move into job history
        job_details = self.__jobs.job_completion(job_id)
        if not job_details['Status'] == 'Completed':
            exp = "Backup job [{0}] is not completed for client [{1}]"\
                    .format(job_id, self._client_name)
            self.log.exception(exp)
            raise Exception(job_details)
        self.log.info("***** Backup job completed Successfully for the client {0} *****"\
                      .format(self._client_name))
        return job_id

    def validate_backup_after_deactivation(self):
        """call the method to validate backup after deactivation"""
        self.__navigator.navigate_to_devices()
        # verify "INCREMENTAL" backup job for deactivated client
        try:
            job_id = self.__laptops.subclient_backup_now(self._client_name, self.__backup.BackupType.INCR)
        except Exception as err:
            self.log.info("As Expected ! Unable to trigger INCREMENTAL job for deactivated client")
        else:
            if job_id:
                exp = "INCREMENTAL job {0} triggered for deactivated client {1}"\
                        .format(job_id, self._client_name)
                self.log.exception(exp)
                raise Exception(exp)

        # verify FULL backup job for deactivated client
        self.__laptops.cancel_the_form()
        self.__navigator.navigate_to_devices()

        try:
            job_id = self.__laptops.subclient_backup_now(self._client_name, self.__backup.BackupType.FULL)
        except Exception as err:
            self.log.info("As Expected ! Unable to trigger FULL job for deactivated client")
        else:
            if job_id:
                exp = "FULL job {0} triggered for deactivated client {1}"\
                        .format(job_id, self._client_name)
                self.log.exception(exp)
                raise Exception(exp)

        # verify SYNTHETIC_FULL backup job for deactivated client
        self.__laptops.cancel_the_form()
        self.__navigator.navigate_to_devices()
        
        try:
            job_id = self.__laptops.subclient_backup_now(self._client_name, self.__backup.BackupType.FULL)
        except Exception as err:
            self.log.info("As Expected ! Unable to trigger SYNTHETIC_FULL job for deactivated client")
        else:
            if job_id:
                exp = "SYNTHETIC_FULL {0} triggered for deactivated client {1}"\
                        .format(job_id, self._client_name)
                self.log.exception(exp)
                raise Exception(exp)
        
    @staticmethod
    def covert_size_to_mb(backup_size, backup_units):
        "convert given the size to MB"

        if backup_units == 'KB':
            new_backup_size = float(backup_size) /1024
        elif backup_units == 'MB':
            new_backup_size = float(backup_size)
        elif backup_units == 'GB':
            new_backup_size = float(backup_size)*1024
        else:
            new_backup_size = float(backup_size) /(1024*1024)
        return new_backup_size

    def auto_trigger_backupjob(self, options=None):
        """ Auto trigger backup job when subclient content added"""

        # Add new content with required size to default subclient of this client and run incremental backup
        #DEFAULT_FILE_SIZE = 2500 KB
        self.log.info("Add new content with required size and run the incremental job on this client")
        self._subclient_obj = self.__ida_utils.get_subclient(self._client_name)
        self._source_dir = self.__utility.create_directory(self._machine_object)
        self.log.info("Adding directory to subclient content with: {0}".format(self._source_dir))
        if not options==None:
            self._subclient_obj.content += [self.__utility.create_test_data(self._machine_object, self._source_dir, options=options)]
        else:
            self._subclient_obj.content += [self.__utility.create_test_data(self._machine_object, self._source_dir)]

        #----------- Wait for auto triggered backups-------#
        self.log.info("Wait for auto triggered backup job after content added")
        jobs = self.__job_manager.get_filtered_jobs(
            self._client_name,
            time_limit=30,
            retry_interval=5,
            backup_level='Incremental',
            current_state='Running'
            )
        jobid = jobs[1][0]
        jobobj = self.commcell.job_controller.get(job_id=jobid)
        return jobobj, jobid

    def __get_backup_details(self, displayed_val):
        """ Retrieve client backup details from the client"""

        self.log.info("Read the current Application  size from adminconsole")
        total_backup = displayed_val

        backupsize_and_time = {}
        backup_size_unit = total_backup.split()
        backup_size = backup_size_unit[0]
        backup_units = backup_size_unit[1]
        #------- convert the backup size into "MB" --------#
        current_total_backup = self.covert_size_to_mb(backup_size, backup_units)

        jobobj, _joid = self.auto_trigger_backupjob()
        #----------- Read the folder size which intern returns in "MB"---------#
        backup_content_size = int(self._machine_object.get_folder_size(self._source_dir))
        backupsize_and_time['Application size'] = current_total_backup + backup_content_size

        #----------- read the last backup time and size from above triggered job --------#
        full_job_details = jobobj.details
        sub_dict = full_job_details['jobDetail']['detailInfo']
        backupsize_and_time['Last backup time'] = sub_dict.get('startTime')
        backupsize_and_time['Last Backup Size'] = int(sub_dict.get('sizeOfApplication'))
        backupsize_and_time['Content'] = self._subclient_obj.content
        return  backupsize_and_time

    def validate_laptop_summaryinfo(self):
        """validates the laptop summary info of the laptop page"""

        temp_info_dict = self.__laptops.laptop_summary_info(self._client_name)
        backup_info = self.__get_backup_details(temp_info_dict['Application size'][0])
        total_backup_size = backup_info['Application size']
        backup_time = datetime.datetime.fromtimestamp(int(backup_info['Last backup time']))
        last_backup_time = backup_time.strftime('%b %#d, %#I:%M %p')

        validation_dict = {'Name':[self._client_name],
                           'Status':[self._activation_status],
                           'Application size':[total_backup_size],
                           'Last backup':[last_backup_time],
                           'Owners':[self._activation_user],
                           'Plans':[self._activation_plan]}
        #----------- get the latest info from adminconsole after backup--------#
        # Go back tolaptop listing page for the values to be updated from mongoDB cache
        # But Don't refresh the page for updated values
        #----------------------------------------------------------------------#
        self.__table.access_link(self._client_name)
        summary_dict = self.__laptops.laptop_summary_info(self._client_name)

        #----------- validation of the summary info from laptop page--------#
        for dict_key, dict_val in validation_dict.items():
            self.log.info('validation values "{0}"'.format(dict_val))
            if dict_key == 'Application size':
                #------- convert the current Application size into "MB" --------#
                total_backupsize_before_convertion = summary_dict[dict_key]
                temp_sizeunit_list = total_backupsize_before_convertion[0].split()
                backup_size = temp_sizeunit_list[0]
                backup_units = temp_sizeunit_list[1]
                total_backupsize_after_convertion = self.covert_size_to_mb(backup_size, backup_units)

                validation_val = dict_val[0]
                if total_backupsize_after_convertion >= validation_val:
                    self.log.info("'Application size' {0} updated with correct value after backup" .format(
                        total_backupsize_after_convertion))
                else:
                    exp = "'Application size' {0} does not updated correctly".format(
                        total_backupsize_after_convertion)
                    raise Exception(exp)
            else:
                if dict_val == summary_dict[dict_key]:
                    self.log.info("displayed value {0} matches with given value {1}".format(
                        summary_dict[dict_key], dict_val))

                else:
                    exp = "displayed value {0} does not match with given value {1}".format(
                        summary_dict[dict_key], dict_val)
                    raise Exception(exp)

    def set_validation_values(self, displayed_val):
        """Set the validation values for laptop """

        #-------------------------------------------------------------------------------------
        #validation of client properties#
        #    Last Backup Time  - Compare with Start Time of the last backup job
        #    Last Backup Size  - Compare with Size of Application of the last backup job
        #    Application size - Get the existing Application size from adminconsole +
        #                        add the specified amount of data (For Ex: 2 mb) + run incr backup.
        #                        Application size must be updated with new value
        #    Last Seen online - X - Get Last seen online time from Adminconsole page
        #                       Restart services on on client
        #                       Go to laptop page and comeback but [Do not Refresh the Adminconsole Page]
        #                       Y- Get the updated time from Adminconsole page
        #                       Compare Y>X
        #    #---Last seen online time updates for every 24 hours from CCR tables,
        #    #--- to update the value immediately restart the CVD on client machine ------#
        #-------------------------------------------------------------------------------------

        #-----------get the last backup size from last backup job and convert into MBS--------#
        backup_info = self.__get_backup_details(displayed_val['Summary']['Application size'])
        total_backupsize = backup_info['Application size']
        #-----get the Last backup size from last job convert to adminconsole display format--------#
        self.log.info("get the last backup size from last backup job ")
        last_backup_size_before_convertion = backup_info['Last Backup Size']
        last_backupsize_after_conversion = self.covert_size_to_mb(last_backup_size_before_convertion, 'bytes')
        i = int(math.floor(math.log(last_backupsize_after_conversion, 1024)))
        power = math.pow(1024, i)
        last_backupsize = round(last_backupsize_after_conversion / power, 2)

        #-----get the last backup time from last backup job and convert to adminconsole display format--------#
        backup_time = datetime.datetime.fromtimestamp(int(backup_info['Last backup time']))
        last_backup_time = backup_time.strftime('%b %#d, %#I:%M:%S %p')
        #-----------get the client last seen online time --------#
        self.log.info("get the laptop last seen online info from the laptop details page and restart the services")
        last_online_time = displayed_val['Summary']['Last seen online']
        #---Restart the client services ---#
        self.__ida_utils.restart_services([self._client_name])

        #-----------Create the dictionary with default values from laptop details page--------#
        default_values = {"Schedules": ['Incremental Automatic schedule',
                                        'Space Reclamation'],
                          "Content":['Desktop',
                                     'Documents',
                                     'Office',
                                     'Pictures',
                                     'Image',
                                     'MigrationAssistant'],
                          "Region":'Not set\nEdit'}


        validation_dict = {
            'Summary': {'Host Name': self._host_name,
                        'Last backup time': last_backup_time,
                        'Last backup size': last_backupsize,
                        'Application size': total_backupsize,
                        'Last seen online': last_online_time,
                        'Region': default_values["Region"]},
            'Security': [self._activation_user],
            'Schedules':default_values["Schedules"],
            'Content':default_values["Content"], #backup_info['Content'],
            'Plan': self._activation_plan,
        }

        for each_item in  backup_info['Content']:
            validation_dict['Content'].append(each_item)

        return validation_dict


    def validate_laptop_details(self):
        """
        Validates the values displayed for client against values provided as input
        """

        #----------- get the current  info from adminconsole after backup--------#
        current_displayed_val = self.__laptop_details.laptop_info(self._client_name)

        #----------- set the validation values based on current displayed values--------#
        validation_dict = self.set_validation_values(current_displayed_val)
        self.log.info("validation values: {0}".format(validation_dict))

        #----------- get the latest info from adminconsole after backup--------#

        displayed_val = self.__laptop_details.laptop_info(self._client_name)

        for key, value in displayed_val.items():
            if key == 'Content':
                for index, each_item in enumerate(value):
                    if ' MigrationAssistant\nUser defined' in each_item:
                        new_list = each_item.split('\n')
                        del displayed_val['Content'][index]
                        for eac_data in new_list:
                            if not eac_data == 'User defined':
                                displayed_val['Content'].append(eac_data)

                    elif 'Inherited from plan\nDesktop ' in each_item:
                        displayed_val['Content'][index] = 'Desktop'


        for key, value in displayed_val.items():
            if key == 'Content':
                for index, each_item in enumerate(value):
                    displayed_val['Content'][index] = each_item.strip()


        for key, value in validation_dict.items():
            #----------validation of fields with list of values---------#
            if isinstance(value, list):
                self.log.info('Entity given val %s', value)
                if  displayed_val[key] != "None" and validation_dict[key] != "None":
                    count = 0
                    validation_dict[key] = sorted(validation_dict[key])
                    max_val = max(len(displayed_val[key]), len(validation_dict[key]))
                    for  each_summary in sorted(displayed_val[key]):
                        if count < max_val:
                            if str(each_summary).strip() == validation_dict[key][count]:
                                self.log.info("{0} displayed for {1} matches"
                                              .format(each_summary, key))
                            else:
                                exp = "{0} displayed for {1} does not match \
                                with {2}".format(key, each_summary, validation_dict[key][count])
                                self.log.exception(exp)
                                raise Exception(exp)
                        else:
                            break
                        count += 1
            #----------validation of string fields ex:Plan  ---------#
            elif isinstance(value, str):
                if displayed_val[key] == validation_dict[key]:
                    self.log.info("{0} displayed for {1} matches with {2} given"
                                  .format(displayed_val[key], key, validation_dict[key]))
                else:
                    exp = "{0} displayed for {1} does not match with {2} given ".format(
                        displayed_val[key], key, validation_dict[key])
                    self.log.exception(exp)
                    raise Exception(exp)

            else:
                #----------validation of fields from Summary info---------#
                self.log.info('Entity given val :{0}'.format(value))
                for summary_key, summary_value in value.items():
                    if summary_key == 'Application size':
                        #------- convert the current Application size into "MB" --------#
                        total_backupsize_before_convertion = displayed_val[key][summary_key]
                        temp_sizeunit_list = total_backupsize_before_convertion.split()
                        backup_size = temp_sizeunit_list[0]
                        backup_units = temp_sizeunit_list[1]
                        total_backupsize_after_convertion = self.covert_size_to_mb(backup_size, backup_units)
                        if total_backupsize_after_convertion >= summary_value:
                            self.log.info("'Application size' {0} updated with correct value after backup" .format(
                                total_backupsize_after_convertion))
                        else:
                            exp = "'Application size' {0} does not updated correctly".format(
                                total_backupsize_after_convertion)
                            raise Exception(exp)

                    elif summary_key == 'Last seen online':
                        disp_val = displayed_val[key][summary_key]
                        key_val = validation_dict[key][summary_key]
                        if disp_val > key_val:
                            self.log.info("Last seen online time updated correctly")
                        else:
                            exp = "Last seen online time {0} not updated correctly ".format(summary_value)
                            self.log.exception(exp)
                            raise Exception(exp)

                    elif summary_key == 'Last backup size':
                        disp_val = displayed_val[key][summary_key]
                        key_val = validation_dict[key][summary_key]
                        #------- convert the Last backup size into "MB" --------#
                        temp_sizeunit_list = disp_val.split()
                        backup_size = temp_sizeunit_list[0]
                        backup_units = temp_sizeunit_list[1]
                        last_backup_size_after_conversion = self.covert_size_to_mb(backup_size, backup_units)
                        i = int(math.floor(math.log(last_backup_size_after_conversion, 1024)))
                        power = math.pow(1024, i)
                        last_backup_size = round(last_backup_size_after_conversion / power, 2)
                        if last_backup_size == key_val:
                            self.log.info("'{0}' displayed value '{1}' match with validation value '{2}'"
                                          .format(summary_key, last_backup_size, key_val))
                        else:
                            exp = "Last backup size {0} not updated correctly ".format(displayed_val[key][summary_key])
                            self.log.exception(exp)
                            raise Exception(exp)

                    else:
                        disp_val = displayed_val[key][summary_key]
                        key_val = validation_dict[key][summary_key]
                        if disp_val == key_val:
                            self.log.info("'{0}' displayed value '{1}' match with validation value '{2}'"
                                          .format(summary_key, disp_val, key_val))
                        else:
                            exp = "'{0}' displayed for '{1}' does not match with validation value '{2}' given".format(
                                summary_key, disp_val, key_val)
                            self.log.exception(exp)
                            raise Exception(exp)

    def subclient_restore(self, backup_jobid=None, tmp_path=None, cleanup=True, **kwargs):
        """validate laptop restore"""
        try:
            if tmp_path is None:
                dest_path = self.__utility.create_directory(self._machine_object)
            compare_source = self._source_dir
            compare_destination = dest_path
            if not self._machine_object.os_info == 'WINDOWS':
                self._machine_object.change_file_permissions(dest_path, 775)

            self.log.info(
                """
                Starting restore with source:[{0}],
                destination:[{1}],
                With restore option as : [{2}]
                """.format(self._source_dir, dest_path, kwargs.get('restore_type'))
                )
            #------- RESTORE FROM LAST BACKUP JOB--------#
            if kwargs.get('restore_type') == 'restore_from_job':
                restore_job_id = self.__laptops.restore_from_job(backup_jobid, self._source_dir, dest_path)
            #------- RESTORE FROM LAPTOP DETAILS PAGE--------#
            elif kwargs.get('restore_type') == 'restore_from_detailspage':
                restore_job_id = self.__laptops.restore_from_details_page(self._client_name, self._source_dir, dest_path)
            elif kwargs.get('restore_type') == 'restore_from_actions':
            #------- RESTORE FROM ACTIONS--------#
                restore_job_id = self.__laptops.restore_from_actions(self._client_name, self._source_dir, dest_path)

            job_details = self.__jobs.job_completion(restore_job_id)
            if job_details['Status'] == "Completed":
                self.log.info("Restore job [{0}] completed successfully".format(restore_job_id))
            else:
                raise Exception("Restore job [{0}] did not complete successfully".format(restore_job_id))

            #------- Validation for restored content --------#
            self.log.info("""Executing backed up content validation:
                        Source: [{0}], and
                        Destination [{1}]""".format(compare_source, compare_destination))

            result, diff_output = self._machine_object.compare_meta_data(compare_source, compare_destination)

            self.log.info("Performing meta data comparison on source and destination")
            if not result:
                self.log.error("Meta data comparison failed")
                self.log.error("Diff output: \n{0}".format(diff_output))
                raise Exception("Meta data comparison failed")
            self.log.info("Meta data comparison successful")

            self.log.info("Performing checksum comparison on source and destination")
            result, diff_output = self._machine_object.compare_checksum(compare_source, compare_destination)
            if not result:
                self.log.error("Checksum comparison failed")
                self.log.error("Diff output: \n{0}".format(diff_output))
                raise Exception("Checksum comparison failed")
            self.log.info("Checksum comparison successful")

        except Exception as excp:
            raise Exception("\n [{0}] [{1}]".format(inspect.stack()[0][3], str(excp)))
        finally:
            if cleanup:
                self.__utility.remove_directory(self._machine_object, dest_path)

