# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be used to run FSO
AdminConsole Automation test cases.

Class:

    FSO()

Functions:

    create_sqlite_db_connection()    --   Establishes connection to the
                                        SQLite Database at the given file path
    db_get_orphan_files_count()     -- Get the count of orphan files from FSO db.
    get_fso_file_count_db()         -- Get Total file count from FSO db
    get_fso_dir_count_db()          -- Get Total dir count frm FSO db
    get_fso_data_size_db()          -- Get total file size from FSO db
    get_file_types_db()             -- Get count of distinct file types from FSO db.
    get_owner_count_db()            -- Get count of distinct owners from FSO db
    duplicate_dashboard_data_db()   -- Get duplicate dashboard data from FSO db
    get_db_column_data()            -- Get data for a particular column from FSO db
    get_file_security_dashboard_info() -- Get files security dashboard info from FSO db
    review_size_distribution_dashboard() -- Review size distribution dashboard in FSO
    review_file_duplicates_dashboard()  -- Review file duplicates dashboard in fso
    review_fso_file_ownership_dashboard() -- Review file ownership dashboard in FSO
    review_fso_file_security_dashboard() -- Review file security dashboard in FSO
    fetch_fso_files_db()                -- Fetch given number filepath values from FSO db
    fso_cleanup()                       -- Delete FSO data source
    analyze_client_details()            -- Analyze Client Details Page
    analyze_client_quickview()          -- Analyze CLient Quick view panel
    verify_fso_time_data                -- Verify FSO time info
    get_fso_time_metadata               -- Get fso time info from DB
    fso_file_count_formatting           -- Perform FSO dashboard file count formatting






"""

import sqlite3
import ast
import re
import calendar
from datetime import datetime, timedelta
from AutomationUtils import logger
from Web.AdminConsole.Components.table import Table
from dynamicindex.utils.activateutils import ActivateUtils
from Web.AdminConsole.GovernanceAppsPages.FileStorageOptimization import FileStorageOptimization
from Web.AdminConsole.GovernanceAppsPages.InventoryManagerDetails import InventoryManagerDetails


class FSO:
    """FSO Helper Class"""

    def __init__(self, admin_console, commcell=None, csdb=None):
        """
        FSO helper class provides functions to perform DB operations
        to FSO test DB and perform verify operations on FSO
        Args:
            admin_console : Admin Console Object
            commcell : Commcell Object
            csdb : CSDB Object
        """
        self.__admin_console = admin_console
        self.log = logger.get_log()
        self.commcell = commcell
        self.csdb = csdb
        self.sqlitedb = None
        self.fso_obj = FileStorageOptimization(self.__admin_console)
        self.activate_utils = ActivateUtils()
        self.__table = Table(self.__admin_console)
        self._data_source_name = None
        self.backup_file_path = None
        self.__inventory_manager_details = InventoryManagerDetails(self.__admin_console)
        self.data_size_mapping = {
            'KB': 1, 'MB': 2, 'GB': 3, 'TB': 4
        }
        self.data_size_mapper_filter = {
            'k': 1, 'M': 2, 'G': 3, 'T': 4
        }

    @property
    def data_source_name(self):
        """
        Returns the data source name
        Return:
            (str): Name of FSO data source
        """
        return self._data_source_name

    @data_source_name.setter
    def data_source_name(self, value):
        """
        Sets the  data source name for FSO client
        Args:
            value (str): Name of FSO data source
        """
        self._data_source_name = value

    def create_sqlite_db_connection(self, sql_database_path):
        """
        Connect to given sql database path
        Args:
            sql_database_path (str): path to database
        """
        self.sqlitedb = sqlite3.connect(sql_database_path)
        self.sqlitedb.row_factory = sqlite3.Row

    def db_get_orphan_files_count(self):
        """
        Get Orphan file count from database
        Returns:
            (int): Count of orphan files from the database
        """
        query = '''
        select COUNT(*) as ORPHAN_COUNT from fso_metadata where FILE_OWNER = 'ORPHAN\\ORPHAN'
        '''
        return self.sqlitedb.execute(query).fetchall()[0]['ORPHAN_COUNT']

    def get_fso_time_metadata(self):
        """
        Get AccessTime, CreateTime, Modified Time info
        for fso data source
            Return:
                (list) [sqlite3.Row,sqlite3.Row]
        """
        query = '''
        select FILEPATH,CREATED_TIME,ACCESS_TIME,MODIFIED_TIME,FILE_SIZE from fso_metadata
        '''
        return self.sqlitedb.execute(query).fetchall()

    def get_fso_file_count_db(self):
        """
        Get number of files count from FSO DB
        Returns:
            (int): No of files in FSO DB
        """
        get_file_count_query = '''
        SELECT COUNT(DISTINCT(FILEPATH)) as File_Count from fso_metadata
        '''
        return \
            self.sqlitedb.execute(get_file_count_query).fetchall()[0]['File_Count']

    def get_fso_dir_count_db(self):
        """
        Get no of directory in FSO db
        Return:
            (int): No of directory in FSO DB
        """
        get_dir_count_query = '''
        SELECT DISTINCT(TOTAL_DIR_COUNT) as Dir_Count from fso_metadata
        '''
        return \
            self.sqlitedb.execute(get_dir_count_query).fetchall()[0]['Dir_Count']

    def get_fso_data_size_db(self):
        """
        Get the sum of file size in bytes
        Returns:
            (int) : Sum of all files size
        """
        get_file_size = '''
        SELECT SUM(FILE_SIZE) as File_Size from fso_metadata
        '''
        return \
            self.sqlitedb.execute(get_file_size).fetchall()[0]['File_Size']

    def get_file_types_db(self):
        """
        Get the count of no of file types
        Return:
            (int) :  Count of number of file types
        """
        get__file_types_count = '''
        SELECT COUNT(DISTINCT(MIME_TYPE)) as Mime_Type_Count from fso_metadata
        '''
        return \
            self.sqlitedb.execute(get__file_types_count).fetchall()[0]['Mime_Type_Count']

    def get_owner_count_db(self, crawl_type='Live'):
        """
        Get total no of owners count
        Args:
            crawl_type (str): FSO Crawl type Live/Backup
        Returns:
            (int): Count of owner related to fso file data
        """
        if crawl_type.lower() == 'backup':
            get_owner_count = "SELECT COUNT(DISTINCT(FILE_OWNER)) as Owner_Count from\
             fso_metadata where FILE_OWNER NOT LIKE 'ORPHAN\\ORPHAN'"
        else:
            get_owner_count = '''
            SELECT COUNT(DISTINCT(FILE_OWNER)) as Owner_Count from fso_metadata
            '''
        return \
            self.sqlitedb.execute(get_owner_count).fetchall()[0]['Owner_Count']

    def duplicate_dashboard_data_db(self):
        """
        Get ['FILENAME', 'MODIFIED_TIME', 'FILE_SIZE', 'File_Count', 'Total_Size']
        information for duplicate files in fso db
        Returns:
            [{},{},....]: List of dictionary in the below format
            [{Entity}:Value,....]
        """
        duplicate_dashboard_query = '''
        Select FILENAME,MODIFIED_TIME,FILE_SIZE,
        COUNT(FILENAME) as File_Count,sum(FILE_SIZE) as Total_Size
        from fso_metadata GROUP BY FILENAME,MODIFIED_TIME,FILE_SIZE HAVING COUNT(*) > 1
        '''
        data = self.sqlitedb.execute(duplicate_dashboard_query).fetchall()
        return [dict(item) for item in data]

    def get_db_column_data(self, column_name, columns_alias=None):
        """
        Get the list of items present in mentioned db field
        Args:
            columns_alias (str): Alias for Column name
            column_name (str): Name of column to fetch
        Returns:
            (list): List of column data
        """
        query = f'''
        SELECT {column_name} from fso_metadata
        '''
        if not column_name:
            column_name = columns_alias
        data = self.sqlitedb.execute(query).fetchall()
        return [item[column_name] for item in data]

    def get_file_security_dashboard_info(self):
        """ security dasboard info
        Get FSO File
        """
        permission_dict = {
            'Full Control': set(),
            'Modify Access': set(),
            'Write Access': set(),
            'List Access': set()
        }

        query = '''
        Select PARENT_DIR_PERMISSION, FILE_PERMISSION from fso_metadata
        '''
        data = self.sqlitedb.execute(query).fetchall()
        for item in data:
            dir_perm = ast.literal_eval(item['PARENT_DIR_PERMISSION'])
            file_perm = ast.literal_eval(item['FILE_PERMISSION'])
            for key, val in dir_perm.items():
                if val.__contains__('F'):
                    permission_dict['Full Control'].add(key)
                if val.__contains__('M'):
                    permission_dict['Modify Access'].add(key)
                if len(val.intersection({'W', 'GW', 'GA'})) > 0:
                    permission_dict['Write Access'].add(key)
                if len(val.intersection({'R', 'GR', 'GA', 'RD', 'RX', 'Rc'})) > 0:
                    permission_dict['List Access'].add(key)
            for key, val in file_perm.items():
                if val.__contains__('F'):
                    permission_dict['Full Control'].add(key)
                if val.__contains__('M'):
                    permission_dict['Modify Access'].add(key)
                if len(val.intersection({'W', 'GW', 'GA'})) > 0:
                    permission_dict['Write Access'].add(key)

        permission_dict['Modify Access'] = \
            permission_dict['Full Control'] | permission_dict['Modify Access']
        permission_dict['Write Access'] = \
            permission_dict['Write Access'] | permission_dict['Modify Access']
        permission_dict['List Access'] = \
            permission_dict['List Access'] | permission_dict['Full Control'] | permission_dict['Modify Access']
        return permission_dict

    def fso_file_count_formatting(self, ui_count, db_count):
        """
        Perform FSO dashboards file count formatting
            Args:
                ui_count (str): Count from FSO dashboard
                db_count (int): Count from DB
            Return: Tuple (ui_count, db_count): Formatted UI and DB count
        """
        ui_count = ui_count.replace(',', '')
        temp_list = ui_count.split()
        if len(temp_list) == 2:
            db_count = round(db_count / pow(1000, self.data_size_mapper_filter[temp_list[1]]), 2)
            ui_count = temp_list[0]
        return float(ui_count), float(db_count)

    def review_size_distribution_dashboard(self, crawl_type='Live'):
        """
        Review FSO Size distribution dashboard
        Args:
            crawl_type (str): Type of FSO crawl Live/Backup
        """
        error_list = list()
        folder_count_offset = 0
        if crawl_type == 'Backup' and self.backup_file_path is not None:
            folder_count_offset = len(self.backup_file_path.split('\\'))
        self.fso_obj.select_fso_dashboard("Size distribution")
        ui_count = self.fso_obj.fso_dashboard_entity_count("Files")
        db_count = self.get_fso_file_count_db()
        ui_count, db_count = self.fso_file_count_formatting(ui_count, db_count)
        if ui_count != db_count:
            error_list.append('FILES_COUNT_MISMATCH: {DB: %s,UI: %s}' % (db_count, ui_count))

        ui_count = self.fso_obj.fso_dashboard_entity_count("Folders")
        db_count = self.get_fso_dir_count_db()
        ui_count, db_count = self.fso_file_count_formatting(ui_count, db_count)
        if ui_count != (db_count + folder_count_offset):
            error_list.append("FOLDER_COUNT_MISMATCH: {DB: %s,UI: %s}"
                              % (db_count + folder_count_offset, ui_count))
        data_size_ui = self.fso_obj.fso_dashboard_entity_count("Size").split(' ')
        data_size_db = self.get_fso_data_size_db()
        data_size_db = round(data_size_db / pow(1024, self.data_size_mapping.get(
            data_size_ui[1]
        )), 2)
        if data_size_db != float(data_size_ui[0]):
            error_list.append("FILE_SIZE_MISMATCH: {DB: %s,UI: %s}" % (data_size_db, float(data_size_ui[0])))
        ui_count = int(self.fso_obj.fso_dashboard_entity_count('File Types'))
        db_count = self.get_file_types_db() + 1
        if db_count != ui_count:
            error_list.append('FILE_TYPE_COUNT_MISMATCH: {DB: %s, UI: %s}' % (db_count, ui_count))

        owner_count = self.get_owner_count_db(crawl_type=crawl_type)
        owner_count_ui = int(self.fso_obj.fso_dashboard_entity_count('Owners'))
        if owner_count != owner_count_ui:
            error_list.append('FILE_OWNERS_COUNT_MISMATCH: {DB: %s,UI: %s}' % (owner_count, owner_count_ui))
        if len(error_list) != 0:
            raise Exception(
                f"Following Errors Occurred in Size distribution dashboard page: {error_list} "
            )

    def review_file_duplicates_dashboard(self):
        """
        Review File duplicates dashboard in FSO
        """
        self.fso_obj.select_fso_dashboard("File duplicates")
        error_list = []
        ui_count = self.fso_obj.fso_dashboard_entity_count("Total Files")
        db_count = self.get_fso_file_count_db()
        ui_count, db_count = self.fso_file_count_formatting(ui_count, db_count)
        if ui_count != db_count:
            error_list.append('FILES_COUNT_MISMATCH: {DB: %s,UI: %s}' % (db_count, ui_count))
        data_size_ui = self.fso_obj.fso_dashboard_entity_count(
            "Size of Files"
        ).split(' ')
        data_size_db = self.get_fso_data_size_db()
        data_size_db = round(data_size_db / pow(1024, self.data_size_mapping.get(
            data_size_ui[1]
        )), 2)
        if data_size_db != float(data_size_ui[0]):
            error_list.append("FILE_SIZE_MISMATCH: {DB: %s, UI: %s}" % (data_size_db, float(data_size_ui[0])))
        duplicate_dashboard_data = self.duplicate_dashboard_data_db()
        duplicate_files_count = self.fso_obj.get_duplicate_file_count()
        db_duplicate_files_count = len(duplicate_dashboard_data)
        duplicate_files_count, db_duplicate_files_count = self.fso_file_count_formatting(
            duplicate_files_count, db_duplicate_files_count
        )
        duplicate_file_size_db = 0
        for items in duplicate_dashboard_data:
            duplicate_file_size_db = duplicate_file_size_db + items['FILE_SIZE']
        duplicate_file_size_ui = self.fso_obj.get_duplicate_file_size().split()
        duplicate_file_size_db = round(duplicate_file_size_db / pow(
            1024, self.data_size_mapping.get(duplicate_file_size_ui[1])
        ), 2)
        if duplicate_files_count != db_duplicate_files_count:
            error_list.append("DUPLICATE_FILES_COUNT_MISMATCH: {DB: %s,UI: %s}"
                              % (db_duplicate_files_count, duplicate_files_count))
        if float(duplicate_file_size_ui[0]) != duplicate_file_size_db:
            error_list.append("DUPLICATE_FILE_SIZE_MISMATCH: {DB: %s,UI: %s}" %
                              (duplicate_file_size_db, float(duplicate_file_size_ui[0])))
        if len(error_list) != 0:
            raise Exception(
                f"Following Errors Occurred in FSO duplicates dashboard page: {error_list} "
            )

    def review_fso_file_ownership_dashboard(self, crawl_type="Live"):
        """
        Review File Ownership dashboard
        Args:
            crawl_type (str): FSO crawl Type
        """
        self.fso_obj.select_fso_dashboard("File ownership")
        error_list = []
        ui_count = self.fso_obj.fso_dashboard_entity_count("Files")
        db_count = self.get_fso_file_count_db()
        ui_count, db_count = self.fso_file_count_formatting(ui_count, db_count)
        if ui_count != db_count:
            error_list.append('FILES_COUNT_MISMATCH: {DB: %s, UI: %s}' % (db_count, ui_count))
        data_size_ui = self.fso_obj.fso_dashboard_entity_count("Size").split(' ')
        data_size_db = self.get_fso_data_size_db()
        data_size_db = round(data_size_db / pow(1024, self.data_size_mapping.get(
            data_size_ui[1]
        )), 2)
        if data_size_db != float(data_size_ui[0]):
            error_list.append("FILE_SIZE_MISMATCH: {DB: %s,UI: %s}"
                              % (data_size_db, float(data_size_ui[0])))

        ui_count = int(self.fso_obj.fso_dashboard_entity_count('Owners'))
        db_count = self.get_owner_count_db(crawl_type=crawl_type)
        if db_count != ui_count:
            error_list.append('FILE_OWNERS_COUNT_MISMATCH: {DB: %s, UI: %s}' % (db_count, ui_count))

        ui_count = self.fso_obj.fso_dashboard_entity_count("Orphan Files")
        db_count = self.db_get_orphan_files_count()
        ui_count, db_count = self.fso_file_count_formatting(ui_count, db_count)
        if db_count != ui_count:
            error_list.append("ORPHAN_FILES_COUNT_MISMATCH:{DB: %s,UI: %s}" % (db_count, ui_count))
        if len(error_list) != 0:
            raise Exception(
                f"Following Errors Occurred in FSO Ownership dashboard page: {error_list} "
            )

    def review_fso_file_security_dashboard(self):
        """
        Review file security dashboard

        """
        error_list = []
        self.fso_obj.select_fso_dashboard("File security")
        db_permission_dict = self.get_file_security_dashboard_info()
        db_count = len(db_permission_dict['Full Control'])
        ui_count = self.fso_obj.get_file_security_dashboard_user_count('Full Control')
        if db_count != ui_count:
            error_list.append("FULL_ACCESS_USER_COUNT_DOESNT_MATCH: {DB: %s,UI: %s}" %(db_count, ui_count))
        db_count = len(db_permission_dict['Modify Access'])
        ui_count = self.fso_obj.get_file_security_dashboard_user_count('Modify Access')
        if db_count != ui_count:
            error_list.append("MODIFY_ACCESS_USER_COUNT_DOESNT_MATCH: {DB: %s,UI: %s}" %(db_count, ui_count))
        db_count = len(db_permission_dict['Write Access'])
        ui_count = self.fso_obj.get_file_security_dashboard_user_count('Write Access')
        if db_count != ui_count:
            error_list.append("WRITE_ACCESS_USER_COUNT_DOESNT_MATCH: {DB: %s,UI: %s}" %(db_count, ui_count))
        db_count = len(db_permission_dict['List Access'])
        ui_count = self.fso_obj.get_file_security_dashboard_user_count('List Access')
        if db_count != ui_count:
            error_list.append("LIST_ACCESS_USER_COUNT_MISMATCH: {DB: %s,UI: %s}" %(db_count, ui_count))
        if len(error_list) > 0:
            raise Exception(
                f"Following error occurred in File Security Dashboard: {str(error_list)}"
            )

    def fetch_fso_files_db(self, count):
        """
        Get given number of files from FSO database
        Args:
            count  (int): Count of no of files to fetch
        Returns:
            (list): List of filepaths
        """
        query = f"""
        Select FILEPATH from fso_metadata LIMIT {count} 
        """
        data = self.sqlitedb.execute(query).fetchall()
        return [item['FILEPATH'] for item in data]

    def get_duplicate_file_count_db(self, file_name):
        """
        Get file count for a given file
        Args:
            file_name (str): Name of file
        Returns (int): Count of occurrences for a file
        """
        data = self.duplicate_dashboard_data_db()
        count = 1
        for item in data:
            if item['FILENAME'].__eq__(file_name):
                count = item['File_Count']
                break
        return count

    def fso_cleanup(self, client_name, datasource_name, dir_path=None,
                    pseudo_client_name=None):
        """
        Delete FSO data source
        Args:
            client_name (Str): Name of FSO client
            datasource_name (str): Name of FSO data source
            dir_path (str): UNC path of dir to delete its contents
            pseudo_client_name (str): Name of pseudo client to delete from Commcell
        """
        self.__admin_console.navigator.navigate_to_governance_apps()
        self.__inventory_manager_details.select_file_storage_optimization()
        if self.fso_obj.check_if_client_exists(client_name):
            self.fso_obj.select_details_action(client_name)
            if self.fso_obj.check_if_datasource_exists(datasource_name):
                self.fso_obj.delete_fso_datasource(datasource_name)
        if dir_path is not None:
            self.activate_utils.delete_old_data(dir_path)
        if pseudo_client_name is not None:
            all_clients_names = self.commcell.clients.all_clients.keys()
            for c_name in all_clients_names:
                if re.search(pseudo_client_name, c_name, re.IGNORECASE):
                    self.log.info('Deleting client: %s' % c_name)
                    try:
                        self.commcell.clients.delete(c_name)
                    except Exception as excp:
                        self.log.info('Unable to delete client: "%s" with '
                                      'reason: "%s". Continuing anyway.' % (c_name, excp))

    def analyze_client_details(self, client_name, data_source_name,
                               docs_count, plan_name, is_backed_up=False):
        """
        Analyze details page for a FSO client
        Args:
             client_name (str): Name of FSO client
             data_source_name (str): FSO client data source name
             docs_count (int): Count of docs present for FSO data source
             plan_name (str): Associated plan name for FSO client
             is_backed_up (bool): Is associated DS backed up or live
        """
        error_list = []
        folder_count_offset = 0
        if is_backed_up and self.backup_file_path is not None:
            folder_count_offset = len(self.backup_file_path.split('\\'))
        if self.fso_obj.check_if_client_exists(client_name):
            self.fso_obj.select_details_action(client_name)
        table_data = self.__table.get_table_data()
        docs_dict = dict(zip(table_data["Data source"], table_data["Docs"]))
        crawl_type_dict = dict(zip(table_data["Data source"], table_data["Type"]))
        ui_count, db_count = self.fso_file_count_formatting(
            docs_dict[data_source_name], docs_count)

        plan_dict = dict(zip(table_data["Data source"], table_data["Plan"]))
        if docs_dict.__contains__(data_source_name):
            if ui_count != (db_count + folder_count_offset):
                error_list.append("Docs count in details page does not match:{DB: %s, UI:%s}"
                                  % (db_count+folder_count_offset, ui_count))
            if not plan_name.__eq__(plan_dict[data_source_name]):
                error_list.append('Plan name does not match on details page or missing')
            if is_backed_up:
                if not str(crawl_type_dict[data_source_name]). \
                        __eq__("Backup"):
                    error_list.append("Data Source Type for FSO client is not backup as expected")
            else:
                if not (str(crawl_type_dict[data_source_name])). \
                        __eq__("Source"):
                    error_list.append("Data source type for FSO client is not Source as Expected")
        else:
            raise Exception("%s not found in details page for %s " % (data_source_name, client_name))
        if len(error_list) > 0:
            raise Exception(str(error_list))
        self.log.info("Verified details page without exceptions for FSO client %s" % client_name)

    def analyze_client_quickview(self, client_name):
        """
        Analyze quick view panel for FSO client
        Args:
            client_name (str) : Name of FSO client
        """
        if self.fso_obj.check_if_client_exists(client_name):
            self.fso_obj.select_quickview_action(client_name)
        else:
            raise Exception(f"fso client {client_name} not found")
        if self.data_source_name not in self.fso_obj.get_quickview_datasource_list():
            raise Exception("Data Source not found in FSO client quick view")
        self.fso_obj.close_quickview_panel()

    def verify_fso_time_data(self):
        """
        Verify Access Time, Create Time, Modified Time for FSO data source
            Returns (True/False): Depending on whether time info is correct
        """
        time_ui_dict = self.fso_obj.get_fso_time_info()
        temp_list = self.get_fso_time_metadata()
        db_dict = {
            "CreatedTime": {1: 0, 2: 0, 3: 0, 4: 0, 5: 0, 6: 0},
            "AccessTime": {1: 0, 2: 0, 3: 0, 4: 0, 5: 0, 6: 0},
            "ModifiedTime": {1: 0, 2: 0, 3: 0, 4: 0, 5: 0, 6: 0}
        }
        now_time = datetime.now()
        current_year = now_time.year
        leap_timestamp = datetime.strptime(f'28/02/{current_year} 00:00:00', '%d/%m/%Y %H:%M:%S')
        current_year_offset = timedelta()
        if now_time > leap_timestamp and calendar.isleap(current_year):
            if now_time.day == 29 and now_time.month == 2:
                current_year_offset = now_time - leap_timestamp
            else:
                current_year_offset = timedelta(days=1)
        # Get timedelta difference due to leap years between passed year and current year
        timedelta_offset = lambda year: current_year_offset if year == current_year else \
            (timedelta(days=1) if calendar.isleap(year) else timedelta()) + timedelta_offset(year + 1)

        # Get timedelta object for i number of years in the past from current year
        tdl_value = lambda i: now_time - timedelta(365 * i) - timedelta_offset(current_year - i)
        tdl_dict = {
            1: tdl_value(1),
            2: tdl_value(2),
            3: tdl_value(3),
            4: tdl_value(4),
            5: tdl_value(5),
        }
        # Map passed time object to correct year span i.e between
        # {0-1 years:1, 1-2 years:2, 2-3 years:3, 3-4 years:4, 4-5 years:5, 5+ years:6]
        year_mapper = lambda tdl: 1 if tdl_dict[1] < tdl <= now_time else \
            2 if tdl_dict[2] < tdl <= tdl_dict[1] else \
                3 if tdl_dict[3] < tdl <= tdl_dict[2] else \
                    4 if tdl_dict[4] < tdl <= tdl_dict[3] else \
                        5 if tdl_dict[5] < tdl <= tdl_dict[4] else 6
        for item in temp_list:
            format_time = '%B %d, %Y %I:%M:%S %p'
            c_time_key = year_mapper(datetime.strptime(item['CREATED_TIME'], format_time))
            a_time_key = year_mapper(datetime.strptime(item['ACCESS_TIME'], format_time))
            m_time_key = year_mapper(datetime.strptime(item['MODIFIED_TIME'], format_time))
            size = item['FILE_SIZE']
            db_dict['CreatedTime'][c_time_key] = db_dict['CreatedTime'][c_time_key] + size
            db_dict['AccessTime'][a_time_key] = db_dict['AccessTime'][a_time_key] + size
            db_dict['ModifiedTime'][m_time_key] = db_dict['ModifiedTime'][m_time_key] + size
        data_size_mapper = \
            lambda fl, y: self.data_size_mapper_filter[time_ui_dict[fl][y].split()[1]]
        for key, value in db_dict.items():
            for key1, value1 in value.items():
                temp = 1
                if time_ui_dict[key][key1] != '0':
                    temp = data_size_mapper(key, key1)
                db_dict[key][key1] = round(value1 / pow(1000, temp), 2)
        for key, value in time_ui_dict.items():
            for key1, value1 in value.items():
                if value1.__ne__('0'):
                    time_ui_dict[key][key1] = float(value1.split()[0])
                else:
                    time_ui_dict[key][key1] = float('0')
        self.log.info('DB Time Dict %s' % str(db_dict))
        self.log.info('UI Time Dict %s' % str(time_ui_dict))
        return db_dict == time_ui_dict
