# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be used to run
basic operations on companies or subscriptions page.

Class:

    CompanyMain()

Functions:

    add_new_company()               :   Calls method add_company from companies class

    delete_existing_company()       :   Calls methods to de-associate plans and delete company
                                        from CompanyDetails & Company class

    validate_company()               :   Validates the values displayed for company against
                                        values provided as input

    __set_default_values_if_none()  :   provides defaults for values provided to compare
                                        with values displayed

    edit_company_details()          :   Edits company tiles from company details page

    edit_plans()                    :   Method to edit plans associated to the company

    add_test_user()                 :   Method to add users to given company
"""

import datetime

from AutomationUtils import logger
from Web.AdminConsole.AdminConsolePages.Companies import Companies
from Web.AdminConsole.AdminConsolePages.CompanyDetails import CompanyDetails
from Web.AdminConsole.AdminConsolePages.Users import Users
from Web.AdminConsole.Components.table import Table


class CompanyMain(object):
    """
        Helper for companies/subscriptions page
    """

    def __init__(self, admin_console, csdb=None):
        """
            Initializes the company helper module

            Args:
                admin_page  (object)   --  AdminPage class object

                csdb        (Object)   --  CommServe database object
        """

        self.csdb = csdb
        self.__admin_console = admin_console
        self.__navigator = admin_console.navigator
        self.__companies = Companies(self.__admin_console)
        self.__company_details = CompanyDetails(self.__admin_console)
        self.__users = Users(self.__admin_console)
        self.__table = Table(self.__admin_console)

        self.log = logger.get_log()
        self._company_name = "TestCompany"+" "+str(datetime.datetime.today())
        self._email = "xyz@commvault.com"
        self._contact_name = "xyz"
        self._enable_custom_domain = True
        self._plans = None
        self._company_alias = self._company_name.translate({ord(c): None for c in ":,-.' '"})
        self._smtp = "######"
        self._primary_site = self._company_name+"."+self._smtp
        self._mail_template = "Automatic"
        self._edited = False
        self._secondary_sites = "defgh.com, ghijk.com"
        self._sender_name = 'abc'
        self._sender_email = 'abc@commvault.com'
        self._general_settings = {'company_alias': 'Test Alias',
                                  'smtp': 'comm.com',
                                  'authcode': 'ON',
                                  'UPN': 'ON',
                                  '2-factor': 'ON',
                                  'reseller_mode': 'ON',
                                  'data_encryption': 'ON',
                                  'auto_discover_applications': 'ON',
                                  'infra_type': 'Own storage',
                                  'supported_solutions': ['File server',
                                                          'Archiving',
                                                          'Cloud apps',
                                                          'Sharepoint',
                                                          'Active directory',
                                                          'Office 365']}
        self._server_default_plan = None
        self._laptop_default_plan = None
        self._operators = {'add': {self._company_alias+r'\Tenant Operator': 'Tenant Operator'},
                           'remove': None}
        self._file_exceptions = {'windows_path': [r'C:\TestFolder', r'C:\TestFolder1'],
                                 'unix_path': ['root/TestFolder', 'root/TestFolder1']}
        self._active_directory = None
        self._job_start_time = 'System default'
        self._company_filters = 'ON'
        self._supported_solutions = ['File server',
                                     'Virtualization',
                                     'Laptop',
                                     'Databases',
                                     'Exchange',
                                     'Office 365']

    @property
    def company_name(self):
        """ Get Company name"""
        return self._company_name

    @company_name.setter
    def company_name(self, value):
        """ Set Company name"""
        self._company_name = value

    @property
    def email(self):
        """ Get Company email"""
        return self._email

    @email.setter
    def email(self, value):
        """ Set Company name"""
        self._email = value

    @property
    def contact_name(self):
        """ Get Company contact name"""
        return self._contact_name

    @contact_name.setter
    def contact_name(self, value):
        """ Set Company contact name"""
        self._contact_name = value

    @property
    def plans(self):
        """ Get plans"""
        return self._plans

    @plans.setter
    def plans(self, value):
        """ Set plans"""
        self._plans = value

    @property
    def company_alias(self):
        """ Get Company alias"""
        return self._company_alias

    @company_alias.setter
    def company_alias(self, value):
        """ Set Company alias"""
        self._company_alias = value

    @property
    def smtp(self):
        """ Get SMTP"""
        return self._smtp

    @smtp.setter
    def smtp(self, value):
        """ Set SMTP"""
        self._smtp = value

    @property
    def primary_site(self):
        """ Get primary domain"""
        return self._primary_site

    @primary_site.setter
    def primary_site(self, value):
        """ Set primary domain"""
        self._primary_site = value

    @property
    def mail_template(self):
        """ Get primary domain"""
        return self._mail_template

    @mail_template.setter
    def mail_template(self, value):
        """ Set primary domain"""
        self._mail_template = value

    @property
    def enable_custom_domain(self):
        """ Get flag for custom domain"""
        return self._enable_custom_domain

    @enable_custom_domain.setter
    def enable_custom_domain(self, value):
        """ Set flag for custom domain"""
        self._enable_custom_domain = value

    @property
    def secondary_sites(self):
        """ Get values for secondary domains"""
        return self._secondary_sites

    @secondary_sites.setter
    def secondary_sites(self, value):
        """ Set values for secondary domains"""
        self._secondary_sites = value

    @property
    def sender_name(self):
        """ Get value for sender name"""
        return self._sender_name

    @sender_name.setter
    def sender_name(self, value):
        """ Set value for sender name"""
        self._sender_name = value

    @property
    def sender_email(self):
        """ Get value for sender name"""
        return self._sender_email

    @sender_email.setter
    def sender_email(self, value):
        """ Set value for sender name"""
        self._sender_email = value

    @property
    def general_settings(self):
        """ Get values for general settings"""
        return self._general_settings

    @general_settings.setter
    def general_settings(self, value):
        """ Set values for general settings"""
        self._general_settings = value

    @property
    def server_default_plan(self):
        """ Get value for default server plan for company"""
        return self._server_default_plan

    @server_default_plan.setter
    def server_default_plan(self, value):
        """ Set value for default server plan for company"""
        self._server_default_plan = value

    @property
    def laptop_default_plan(self):
        """ Get value for default laptop plan for company"""
        return self._laptop_default_plan

    @laptop_default_plan.setter
    def laptop_default_plan(self, value):
        """ Set value for default laptop plan for company"""
        self._laptop_default_plan = value

    @property
    def operators(self):
        """ Get operators for company"""
        return self._operators

    @operators.setter
    def operators(self, value):
        """ Set operators for company"""
        self._operators = value

    @property
    def file_exceptions(self):
        """ Get file_exceptions for company"""
        return self._file_exceptions

    @file_exceptions.setter
    def file_exceptions(self, value):
        """ Set file_exceptions for company"""
        self._file_exceptions = value

    @property
    def active_directory(self):
        """ Get active_directory values for company"""
        return self._active_directory

    @active_directory.setter
    def active_directory(self, value):
        """ Set active_directory values for company"""
        self._active_directory = value

    @property
    def company_filters(self):
        """ Get company_filters for file exceptions"""
        return self._company_filters

    @company_filters.setter
    def company_filters(self, value):
        """ Set company_filters for file exceptions"""
        self._company_filters = value

    @property
    def job_start_time(self):
        """ Get job_start_time value"""
        return self._job_start_time

    @job_start_time.setter
    def job_start_time(self, value):
        """ Set job_start_time value"""
        self._job_start_time = value

    @property
    def supported_solutions(self):
        """ Get supported_solutions value"""
        return self._supported_solutions

    @supported_solutions.setter
    def supported_solutions(self, value):
        """ Set supported_solutions value"""
        self._supported_solutions = value

    def add_new_company(self):
        """ Method to call base methods to navigate and create a company """

        self.__navigator.navigate_to_companies()
        self.__companies.add_company(self.company_name,
                                     self.email,
                                     self.contact_name,
                                     self.plans,
                                     self.company_alias,
                                     self.smtp,
                                     self.mail_template,
                                     self.primary_site)

    def delete_existing_company(self):
        """ Method to call base methods to perform delete operation for given company """

        self.__navigator.navigate_to_companies()
        self.__table.access_link(self.company_name)
        self.__company_details.de_associate_plans_from_company()
        self.__navigator.navigate_to_companies()
        self.__companies.deactivate_and_delete_company(self.company_name)

    def validate_company(self):
        """ Method to validate displayed values against given input values """
        self.__navigator.navigate_to_companies()
        self.__table.access_link(self.company_name)
        displayed_val = self.__company_details.company_info()

        if self._edited:
            validation_dict = self.__set_default_values_if_none("Post-edit")
        else:
            validation_dict = self.__set_default_values_if_none("Pre-edit")

        for key, value in validation_dict.items():
            if isinstance(value, list):
                self.log.info('Entity given val %s', value)
                if (key == 'Plans' and
                        displayed_val[key] != "None" and
                        validation_dict[key] != "None"):
                    count = 0
                    validation_dict[key] = sorted(validation_dict[key])
                    max_val = max(len(displayed_val[key]), len(validation_dict[key]))
                    for plan in sorted(displayed_val[key]):
                        if count < max_val:
                            plan_name, sep, rest = plan.partition('Default')
                            if str(plan_name).strip() == validation_dict[key][count]:
                                self.log.info("{0} displayed for {1} matches"
                                              .format(plan, key))
                            else:
                                exp = "{0} displayed for {1} does not match \
                                with {2}".format(key, plan, validation_dict[key][count])
                                self.log.exception(exp)
                                raise Exception(exp)
                        else:
                            break
                        count += 1
                elif (key == 'Operators'and
                      displayed_val[key] != "None" and
                      validation_dict[key] != "None"):
                    count = 0
                    validation_dict[key] = sorted(validation_dict[key])
                    max_val = max(len(displayed_val[key]), len(validation_dict[key]))
                    for operator in sorted(displayed_val[key]):
                        if count < max_val:
                            if '(' in operator:
                                operator = operator.rsplit('(', 1)[1].rsplit(')', 1)[0]

                            if str(operator).strip() == validation_dict[key][count]:
                                self.log.info("{0} displayed for {1} matches"
                                              .format(operator, key))
                            else:
                                exp = "{0} displayed for {1} does not match with {2}"\
                                    .format(key, operator, validation_dict[key][count])
                                self.log.exception(exp)
                                raise Exception(exp)
                        else:
                            break
                        count += 1
                else:
                    if set(displayed_val[key]) == set(validation_dict[key]):
                        self.log.info("{0} displayed for {1} matches with {2} given"
                                      .format(displayed_val[key], key, validation_dict[key]))
                    else:
                        exp = "{0} displayed for {1} does not match with {2} given".format(
                            displayed_val[key], key, validation_dict[key])
                        self.log.exception(exp)
                        raise Exception(exp)

            elif isinstance(value, str):
                if displayed_val[key] == validation_dict[key]:
                    self.log.info("{0} displayed for {1} matches with {2} given"
                                  .format(displayed_val[key], key, validation_dict[key]))
                else:
                    exp = "{0} displayed for {1} does not match with {2} given ".format(
                        displayed_val[key], key, validation_dict[key])
                    self.log.exception(exp)
                    raise Exception(exp)

            else:
                self.log.info('Entity given val :{0}'.format(value))
                for item, value_dict in value.items():
                    d_val = displayed_val[key][item]
                    key_val = validation_dict[key][item]
                    if d_val == key_val:
                        self.log.info("{0} values match" .format(item))
                    else:
                        exp = "{0} displayed for {1} does not match with {2} given".format(
                            d_val, item, key_val)
                        self.log.exception(exp)
                        raise Exception(exp)

    def __set_default_values_if_none(self, get_validation_dict):
        """ Method to create dictionary with given inputs for comparison with displayed values """

        default_comp_values = {"Requires authcode for installation": "OFF",
                               "Use UPN instead of e-mail": "OFF",
                               "Enable two factor authentication": "OFF",
                               "Secondary domain name": "Not set",
                               "Navigation preferences": ['Navigation customization'],
                               "Email settings": [],
                               "External authentication": [],
                               "data_encryption": "ON",
                               "infra_type": "Rented storage",
                               "file_exceptions": []}

        _query = "select attrVal from App_CompanyProp where componentNameId=\
            (select id from UMDSProviders where serviceType=5 and domainName=\
            '{0}') and attrName='Creation Time'".format(self.company_alias)
        self.csdb.execute(_query)
        unix_company_time = self.csdb.fetch_one_row()
        creation_time = datetime.datetime.fromtimestamp(int(unix_company_time[0]))
        comp_creation_time = creation_time.strftime('%b %#d, %#I:%M:%S %p')

        file_exception_windows = ""
        file_exception_unix = ""
        i = 0
        for exception in self.file_exceptions['windows_path']:
            i += 1
            if i == 1:
                file_exception_windows = file_exception_windows + str(exception)
            else:
                file_exception_windows = file_exception_windows + '\n' + str(exception)

        i = 0
        for exception in self.file_exceptions['unix_path']:
            i += 1
            if i == 1:
                file_exception_unix = file_exception_unix + str(exception)
            else:
                file_exception_unix = file_exception_unix + '\n' + str(exception)

        if self.primary_site:
            primary_site = self.primary_site
        else:
            primary_site = 'Not set'

        if self.secondary_sites:
            secondary_sites = self.secondary_sites
        else:
            secondary_sites = 'Not set'

        if get_validation_dict == "Pre-edit":
            validation_dict = {
                'CompanyName': str(self.company_name),
                'Contacts': {'Contact name': str(self.contact_name)},
                'Plans': self.plans,
                'Email settings': default_comp_values["Email settings"],
                'General': {'Company created on': comp_creation_time,
                            'Company alias': self.company_alias,
                            'Associated SMTP': self.smtp,
                            'Requires authcode for installation':
                                default_comp_values["Requires authcode for installation"],
                            'Use UPN instead of e-mail': default_comp_values["Use UPN instead of e-mail"],
                            'Enable two factor authentication':
                                default_comp_values["Enable two factor authentication"],
                            'Allow owners to enable data encryption': default_comp_values['data_encryption'],
                            'Infrastructure type': default_comp_values['infra_type'],
                            'Supported solutions': self.supported_solutions},
                'Navigation preferences': default_comp_values["Navigation preferences"],
                'Sites': {'Primary site name': primary_site,
                          'Secondary site name': default_comp_values['Secondary domain name']},
                'Operators': self.operators['add'],
                'External authentication': default_comp_values['External authentication'],
                'File exceptions': default_comp_values['file_exceptions']
            }
        else:
            validation_dict = {
                'CompanyName': self.company_name,
                'Contacts': {'Contact name': self.contact_name},
                'Plans': self.plans,
                'Email settings': {'Sender name': self.sender_name,
                                   'Sender email': self.sender_email},
                'General': {'Company created on': comp_creation_time,
                            'Company alias': self.company_alias,
                            'Associated SMTP': self.general_settings['smtp'],
                            'Requires authcode for installation': self.general_settings['authcode'],
                            'Use UPN instead of e-mail': self.general_settings['UPN'],
                            'Enable two factor authentication': self.general_settings['2-factor'],
                            'Allow owners to enable data encryption': self.general_settings['data_encryption'],
                            'Infrastructure type': self.general_settings['infra_type'],
                            'Supported solutions': self.general_settings['supported_solutions']},
                'Navigation preferences': default_comp_values["Navigation preferences"],
                'Sites': {'Primary site name': primary_site,
                          'Secondary site name': secondary_sites},
                'Operators': self.operators['add'],
                'External authentication': [self.active_directory['netbios_name']],
                'File exceptions': {'Use company filters on all subclients': self.company_filters,
                                    'Windows': file_exception_windows,
                                    'Unix': file_exception_unix}
            }
        return validation_dict

    def edit_company_details(self):
        """ Method to call base methods for editing company properties from company details page """

        secondary_sites = self.secondary_sites.split(',')
        contact_names = self.contact_name.split(',')
        self.add_test_user()
        self.__navigator.navigate_to_companies()
        self.__table.access_link(self.company_name)
        self.__company_details.edit_general_settings(self.general_settings)
        self.__company_details.edit_contacts(contact_names)
        self.__company_details.edit_sender_email(self.sender_name, self.sender_email)
        self.__company_details.edit_external_authentication(self.active_directory['dir_type'],
                                                            self.active_directory['netbios_name'],
                                                            self.active_directory['username'],
                                                            self.active_directory['password'],
                                                            self.active_directory['domain_name'],
                                                            self.active_directory['proxy_client'])
        self.__company_details.edit_sites(self.primary_site, secondary_sites)
        self.__company_details.edit_company_plans(self.plans,
                                                  self.server_default_plan,
                                                  self.laptop_default_plan)
        self.__company_details.edit_company_file_exceptions(self.file_exceptions)
        self.__company_details.edit_company_operators(self.operators)
        self._edited = True

    def edit_plans(self):
        """Method to edit plans associated to a company from company details page"""

        self.__navigator.navigate_to_companies()
        self.__table.access_link(self.company_name)
        self.__company_details.edit_company_plans(self.plans,
                                                  self.server_default_plan,
                                                  self.laptop_default_plan)
        self._edited = True

    def add_test_user(self):
        """Method to add test user for verification"""

        self.__navigator.navigate_to_users()
        contacts = [str(i) for i in self.contact_name.split(',')]
        for contact in contacts:
            self.__users.add_local_user(email=contact+'@commvault.com',
                                        groups=[self.company_alias + r'\Tenant Admin'],
                                        username=contact,
                                        name=contact,
                                        password="######")
