# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the common functions or operations that can be
used by the complete AdminConsole Automation Package


Classes:

    AdminConsoleBase() ---> object()


AdminConsoleBase   --  acts as the base class for all the other classes
                            such as AdminPage, login_page, and others

    Functions:

    __init__()                      --  initialize an instance of the AdminConsoleHelper class

    fill_form_by_id()               --  fill the values in a textfield specified by the id

    search_for()                    --  Opens the address specified in the url in the browser

    navigate()                      --  opens the url in browser

    get_driver_obj()                --  returns the webdriver object

    current_url()                   --  returns the currently opened url in the browser

    check_if_entity_exists()        --  checks whether an entity exists on the webpage or not

    wait_for_completion()           --  wait for the page to load

    select_value_from_dropdown()    --  Selects the value from the dropdown

    error_out_screenshot()          --  save the webpage screenshot

    cv_table_next_button_exists()   --  check if *next* button exists

    cv_table_click_next_button()    --  click on *next* button

    cv_table_click_first_button()   --  click on *first* button

    mouseover_and_click()           --  move mouse over and click on an element

    is_element_present()            --  Checks if the given element is present in the given parent
                                            tag

    scroll_into_view()              -- Scrolls to a particular element of the page

    label_getvalue                    -- Fetch the value corresponding to given label.

    tile_select_hyperlink           -- This selects the hyperlink at tile label level.

                                       hyperlink within tile wont be selected.

    cvselect_from_dropdown          -- Selects the value from the dropdown.Use this for cv multi
                                       select option

    select_radio                    -- Selects radio button given the value.

    submit_form                     -- Clicks save Button on open forms.

    toggle_enable                   -- Enables the toggle bar if disabled

    toggle_disable                  -- Disables the toggle bar if enabled

    enable_toggle                   -- Method to enable given toggle using index of toggle

    disable_toggle                  -- Method to disable given toggle using index of toggle

    toggle_edit                     -- Selects edit button next to toggle

    select_hyperlink                -- Selects hyperlink in the given page.

    click_by_xpath                  -- Clicks the element with the given path

    checkbox_select                 -- Selects all the checkbox that matches the IDs
                                       in the values lists

    checkbox_deselect               -- Deselects all the checkbox that matches the IDs
                                       in the values lists
    select_destination_host()       -- Selects a ESX host as destination host or snap mount ESX

    get_notification()              -- Gets the Job Text on submitted job

    get_jobid_from_popup()          -- Gets the job id from pop up

    click_yes_button()              -- Clicks on yes button

    check_error_message()           -- Checks if there is any error message

    get_error_message()             -- get error message if any on the page

    search_vm()                     -- Searches for the VM while selecting content or
                                       during restore

    get_locale_name()               -- Gets the name of the locale thats currently selected

    change_language()               -- Changes the language to the given one

    select_for_restore()            -- Selects files and folders for restore

    recovery_point_restore()        -- Restored data from the time selected in recovery points

    select_overview_tab()           -- Selects the overview tab

    select_configuration_tab()      -- Selects the configuration tab

    select_distribution_tab()       -- Selects the distribution tab

    expand_options()                -- Expands the advanced options or other chevron elements

    snapshot_engine()               -- Sets the snapshot engine for the subclient

    backup_for_specific_date()      -- It will open the backed up contents from a specific date

    backup_for_date_range()         -- Opens the backed up content from a specific date range

    refresh_page()                  -- Refreshes the page

    open_new_tab()                  -- To open a new tab

    switch_to_latest_tab()          -- To switch to the latest tab

    close_current_tab()             -- To close the current tab

    check_radio_button()            -- Check if radio button exists

    access_sub_menu()               -- Method to click on the item under a dropdown
    in database page

"""
import inspect
import re
import time
import os
from time import sleep

from selenium.common.exceptions import (
    TimeoutException,
    NoSuchElementException,
    StaleElementReferenceException
)

from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as ec
from selenium.webdriver.common.keys import Keys
from selenium.webdriver.common.by import By
from selenium.webdriver.common.action_chains import ActionChains
from selenium.webdriver.support.ui import Select
from AutomationUtils import constants
from AutomationUtils import logger

from Web.AdminConsole.Helper.adminconsoleconstants import LocaleFile
from Web.Common.page_object import (
    WebAction,
    PageService
)
from Web.Common.exceptions import (
    CVWebAutomationException
)


class AdminConsoleBase(object):
    """ Base class for all the Admin Console pages """

    def __init__(self, driver):
        self.driver = driver
        self.url = self.current_url()
        self.log = logger.get_log()
        self.cb_flag = False
        self.ext = None
        self.loaded_locale_files = []
        self.props = {}
        self.locale_ids = {}

    @WebAction(log=False, delay=1)
    def __wait_for_loader(self, wait_time):
        """
        Waits for the loader on the page to load completely so that all items on page are visible
        """
        WebDriverWait(self.driver, wait_time).until(
            ec.invisibility_of_element_located((By.XPATH, "//span[@class = 'grid-spinner']")))

    def __check_sort(self, select):
        """
        Checks if the dropdown items are in sorted order

        Args:
            select (basestring): Select values from element search

        Raises:
            Exception:
                If the elements are not in sorted order

        """
        self.log.info("Checks if options are in sorted order")
        unsorted_list = []
        global_list = ['Not Connected', 'Select MediaAgent', 'Select a vendor type',
                       'Create MediaAgent', 'Select datastore', 'Select resource pool']

        for option in select.options:
            if option.get_attribute('label') not in global_list:
                unsorted_list.append(option.get_attribute('label').lower())
        sorted_list = sorted(unsorted_list)
        self.log.info("Unsorted List : %s", str(unsorted_list))
        self.log.info("Sorted List : %s", str(sorted_list))

        if not sorted_list == unsorted_list:
            raise Exception("The dropdown elements are not in a sorted order")

    def _load_properties(self, class_obj, unique=False):
        """
        Loads the properties of the current class

        Args:
            class_obj   (object):   instance of the class whose locale properties has to be loaded
            unique      (bool):  set to True to if key is unique to the class. default is False

        """
        if not self.ext:
            self.get_locale_name()
        class_list = []
        files_list = ''
        # Get the parent classes of the given class
        base_classes = inspect.getmro(class_obj.__class__)
        for base in base_classes:
            if base.__name__ in ['object', 'AdminConsoleBase']:
                continue
            class_list.append(base.__name__)
        class_list = list(set(class_list))

        for locale_class in class_list:
            try:
                files_list += LocaleFile[locale_class].value + ','
            except KeyError:
                pass

        locale_files = [x.strip() for x in files_list.rsplit(',', 1)[0].split(",")]
        unique_dict = dict()
        for locale_file in locale_files:
            if unique or locale_file not in self.loaded_locale_files:
                if not unique:
                    self.loaded_locale_files.append(locale_file)
                file_name = locale_file + self.ext
                file_path = os.path.join(constants.ADMINCONSOLE_DIRECTORY, 'Localization', file_name)
                if not os.path.exists(file_path):
                    continue
                prop_file = open(file_path, 'r', encoding='UTF-8')
                for line in prop_file.readlines():
                    try:
                        line = line.strip()
                        if line.isspace() or line.startswith("#") or line == '':
                            continue
                        key, value = [x.strip() for x in line.split("=", 1)]
                        if unique:
                            unique_dict[key] = value
                        else:
                            self.props[key] = value
                    except Exception as exp:
                        self.log.exception("Exception occurred while getting the locale properties. "
                                           "%s", str(exp))
                prop_file.close()
        if unique_dict:
            self.props[class_obj.__class__.__name__] = unique_dict

    @WebAction()
    def __click_menu(self, name):
        """ Method to click on the menu next to the tab on a page """
        self.driver.find_element_by_xpath(
            f"//span[@data-ng-bind='action.title' and text()='{name}']").click()

    @WebAction()
    def __click_main_bar_dropdown(self, col_id=None):
        """ Method to click on the dropdown menu next to a tab """
        if not col_id:
            self.driver.find_element_by_xpath(
                "//a[contains(@class, 'uib-dropdown-toggle main-tab-menu-toggle')]").click()
        else:
            self.driver.find_element_by_xpath(
                f"//label[@class='col-xs-{col_id} col-md-3 col-lg-3 cv-main-bar-dropdown-menu menu "
                f"ng-scope dropdown']//a[@class='uib-dropdown-toggle main-tab-menu-toggle dropdown-toggle']"
                f"//*[local-name()='svg']").click()

    @WebAction()
    def click_by_id(self, id):
        """ Method to click on the dropdown menu next to a tab """
        self.driver.find_element_by_xpath(f'//*[@id="{id}"]').click()
        self.wait_for_completion()

    @WebAction()
    def __click_access_sub_menu(self, name):
        """Method to click on the access sub menu if visible"""
        elems = self.driver.find_elements_by_xpath(
            f"//div[@class='panel-body' and normalize-space()='{name}']")
        for elem in elems:
            if elem.is_displayed():
                elem.click()
                break

    @WebAction()
    def __click_on_action(self, action_name):
        """
        clicks on actions of class cv-main-bar-action
        Args:
            action_name:  text used for action
        """
        self.driver.find_element_by_xpath(
            f"//a[@class='cv-main-bar-action' and contains(text(),'{action_name}')]"
        ).click()

    @WebAction()
    def __get_page_actions_list(self):
        """Reads data from page actions menu"""
        if self.check_if_entity_exists("xpath", "//div[contains(@class,'cv-main-bar-dropdown-menu')]"):
            self.driver.find_element_by_xpath("//div[contains(@class,'cv-main-bar-dropdown-menu')]").click()
            self.wait_for_completion()
        return self.driver.find_elements_by_xpath("//*[@id='moreActions']/li//span | //*[@id='moreActions']/li//hr")

    @WebAction()
    def __expand_locale_selection_drop_down(self):
        """ Method to expand language selection drop down on home screen """
        username = self.driver.find_element_by_xpath("//div[@id='user-account-toggle']/a/span[2]")
        username.click()
        time.sleep(2)
        language_dropdown = self.driver.find_element_by_xpath(
            "//a[@id='user-header-lang']")
        language_dropdown.click()

    @WebAction()
    def __get_selected_locale(self):
        """
        Method to get selected locale

        Returns:
            selected_locale (str) : Current selected locale in admin console
        """
        xp = "//li[@id='user-header-dropdown_mn_active']//li[contains(@class,'selected')]/a"
        selected_locale = self.driver.find_element_by_xpath(xp).get_attribute('id')
        #  to remove the locale selection dropdown
        global_search = self.driver.find_element_by_xpath("//input[@id='nav-search-field']")
        global_search.click()
        return selected_locale

    def __get_locale_ids(self):
        """ Method to get locale Ids """
        file_path = os.path.join(constants.ADMINCONSOLE_DIRECTORY, 'Localization',
                                 "HeaderLanguages.properties")
        lang_file = open(file_path, 'r', encoding='UTF-8')
        for line in lang_file.readlines():
            line = line.strip()
            line = [x.strip() for x in line.split("=")]
            self.locale_ids[line[1].lower()] = line[0].lower()
        lang_file.close()

    def __select_language(self, lang_id):
        """
        Method to select localization language of Admin Console

        Args:
            lang_id (str) : language id for language to be selected
        """
        self.driver.find_element_by_id(lang_id).click()
        self.wait_for_completion()

    @WebAction()
    def __click_on_toggle(self, label):
        """
        Clicks on toggle element

        Args:
            label(basestring) : value of toggle label if label to be clicked on
        """
        xpath_toggle = f"//*[contains(text(),'{label}')]/preceding-sibling::toggle-control"
        self.driver.find_element_by_xpath(xpath_toggle).click()

    @WebAction()
    def __get_all_languages(self):
        """
        Returns the list of all language's locale name

        Returns:
            (list)  -- List of all languages

        """
        languages = []

        for language in self.driver.find_elements_by_xpath("//li[@id='user-header-dropdown_mn_active']//li"):
            languages.append(language.text)

        return languages

    @WebAction(log=False, delay=1)
    def __wait_for_load_line(self, wait_time):
        """
        Waits for the loader for two times on the page to load completely so that all items on page are visible
        This method first waits for load line to be invisible, after which it again looks to see if another load line
        is visible. The second check is added to fit cases where load line is spawned right after the
        finish of the previous load line
        """
        waiter = WebDriverWait(self.driver, wait_time / 2, poll_frequency=2)
        try:
            waiter.until(ec.invisibility_of_element_located((By.XPATH, "//div[@id='loading-bar']")))
        except TimeoutException:
            pass
        try:
            waiter.until(ec.invisibility_of_element_located((By.XPATH, "//div[@id='loading-bar']")))
        except TimeoutException as excep:
            raise CVWebAutomationException(excep)

    @WebAction()
    def switch_to_react_frame(self):
        """Move to React frame"""
        frame = self.driver.find_element_by_name("react-frame")
        self.driver.switch_to_frame(frame)

    @WebAction()
    def navigate(self, url):
        """Opens the address specified in the url in the browser.

        Args:
            url (basestring): the URL to navigate to
        """
        try:
            self.driver.get(url)
            self.wait_for_completion()
        except Exception as exp:
            raise exp

    def current_url(self):
        """Returns the currently opened url in the browser."""
        return self.driver.current_url

    def check_if_entity_exists(self, entity_name, entity_value):
        """Check if a particular element exists or not
        :param entity_name      (str)   --  the entity attribute to check for presence
        :param entity_value     (str)   --  the entity to be checked
        :return:
            True    --  If the entity is available
            False   --  If the entity is not available
        """
        try:
            if entity_name == "link":
                self.driver.find_element_by_link_text(entity_value)
                return True
            elif entity_name == "id":
                self.driver.find_element_by_id(entity_value)
                return True
            elif entity_name == "css":
                self.driver.find_element_by_css_selector(entity_value)
                return True
            elif entity_name == "xpath":
                self.driver.find_element_by_xpath(entity_value)
                return True
            elif entity_name == "name":
                self.driver.find_element_by_name(entity_value)
                return True
            elif entity_name == "class":
                self.driver.find_element_by_class_name(entity_value)
                return True
        except NoSuchElementException:
            return False

    @PageService()
    def wait_for_completion(self, wait_time=300):
        """Waits for the page load to complete"""
        self.__wait_for_loader(wait_time)
        try:
            self.__wait_for_load_line(wait_time)
        except Exception as exp:
            self.log.info("Retrying for the loader to finish")
            self.__wait_for_load_line(wait_time)

    @PageService()
    def select_value_from_dropdown(self, select_id, value, attribute=False, check_sort=True,
                                   search=False):
        """ Selects the value from the drop down

            Args:

                select_id       (str) -- the ID or name of the dropdown element

                value           (str) -- the value to be chosen from the dropdown
                                         in case of multi select drop down
                                          value will be list of string.

                attribute       (str) -- true, if the value of the option should be used

                check_sort      (bool) -- true if dropdown has to be checked for sorted order

                search          (bool):  true if the value contains more text than actual
                    Eg: value to be selected is Datastore1
                        value displayed in UI is Datastore1 (Free: 5GB, total: 10GB)

            Raises:
                Exception:
                    if failed to choose the value from dropdown
        """
        try:
            if self.check_if_entity_exists("id", select_id) and \
                    self.driver.find_element_by_id(select_id).tag_name.lower() == "select":
                select = Select(self.driver.find_element_by_id(select_id))

            elif self.check_if_entity_exists("name", select_id) and \
                    self.driver.find_element_by_name(select_id).tag_name.lower() == "select":
                select = Select(self.driver.find_element_by_name(select_id))

            elif self.check_if_entity_exists("xpath",
                                             "//label[contains(text(), '" + select_id + "')]"
                                                                                        "/select"):
                select = Select(self.driver.find_element_by_xpath(
                    "//label[contains(text(), '" + select_id + "')]/select"))

            else:
                raise CVWebAutomationException(f"There is no dropdown with the given name or ID {select_id}")

            if self.cb_flag:
                if check_sort:
                    self.__check_sort(select)

            flag = 1

            if search:
                if not isinstance(value, str):
                    raise CVWebAutomationException("Expected string for value to select")
                pattern = re.compile(value, re.IGNORECASE)
                if attribute:
                    sel = "value"
                else:
                    sel = "label"
                for option in select.options:
                    attr_value = option.get_attribute(sel)
                    if pattern.search(attr_value):
                        option.click()
                        self.wait_for_completion()
                        flag = 0
                        break

            elif isinstance(value, str):
                for option in select.options:
                    if attribute:
                        element_value = option.get_attribute('value')
                    else:
                        element_value = option.get_attribute('label')
                    if value.lower() == element_value.lower() or value.lower() == element_value[:len(element_value) - 3].lower():
                        option.click()
                        self.wait_for_completion()
                        flag = 0
                        break

            else:
                raise CVWebAutomationException("Expected type string for argument:value ")

            if flag != 0:
                raise CVWebAutomationException("The specified option was not present in the list")

        except Exception as exp:
            raise CVWebAutomationException("Exception occurred while selecting an option from the dropdown."
                            " {0}".format(str(exp)))

    @WebAction()
    def cv_table_next_button_exists(self):
        """
        Checks if there is a button to navigate to the next page.
        """
        try:
            return bool(self.driver.find_element_by_xpath(
                "//button[@ng-disabled='cantPageForward()']").is_displayed())
        except Exception:
            return False

    @WebAction()
    def cv_table_click_next_button(self):
        """
        Clicks on the button to navigate to the next page.
        """
        self.driver.find_element_by_xpath(
            "//button[@ng-disabled='cantPageForward()']").click()
        self.wait_for_completion()

    @WebAction()
    def cv_table_click_first_button(self):
        """
        Clicks on the button to navigate to the first page.
        """
        self.driver.find_element_by_xpath(
            "//button[@ng-click='pageFirstPageClick()']").click()
        self.wait_for_completion()

    @PageService()
    def mouseover_and_click(self, mouse_move_over, mouse_click):
        """
        Pass MouseMoveOver and MouseClick Element.

        Args:
            mouse_move_over (WebElement) -- mouse move over element

            mouse_click (WebElement)    -- mouse click element
        """
        hover = ActionChains(self.driver).move_to_element(mouse_move_over)
        hover.perform()
        sleep(2)
        mouse_click.click()
        self.wait_for_completion()

    def is_element_present(self, locator, root_tag=None):
        """ Checks if the given element is present in the given parent tag

            Args:
                locator     (str) -- the xpath of the element to be located
                root_tag    (str) -- the parent tag under which the element is to be located
                    default:    None
            Returns:
                True    --  If the element is present
                False   --  If the element is not present
        """
        try:
            if not root_tag:
                self.driver.find_element_by_xpath(locator)
            else:
                root_tag.find_element_by_xpath(locator)
        except Exception:
            return False
        return True

    @WebAction()
    def close_popup(self):
        """
        Checks for pop up and closes it if it is present
        """
        if self.check_if_entity_exists("id", 'customStartupMessage_button_#3974'):
            self.driver.find_element_by_id('customStartupMessage_button_#3974').click()
            self.wait_for_completion()

    @WebAction()
    def access_tile(self, id):
        """ Method to click on the given tile """
        self.driver.find_element_by_id(id).click()
        self.wait_for_completion()

    @WebAction()
    def access_tab(self, tab_header):
        """ Method to click on the given tile """
        self.driver.find_element_by_xpath(
            f"//span[@ng-bind='tab.title' and text()='{tab_header}']").click()
        self.wait_for_completion()

    @WebAction()
    def scroll_into_view(self, element_name):
        """
        Scrolls the element into view
        """
        elem = ""
        if self.check_if_entity_exists("xpath", element_name):
            elem = self.driver.find_element_by_xpath(element_name)

        elif self.check_if_entity_exists("id", element_name):
            elem = self.driver.find_element_by_id(element_name)
        if elem:
            self.driver.execute_script("arguments[0].scrollIntoView();", elem)

    @WebAction()
    def check_toggle_status(self, label):
        """
        Checks the status of toggle

        Args:
            label          (basestring):   Label corresponding to the toggle option.

        Returns:
            Status
        """
        return 'enabled' in self.driver.find_element_by_xpath(
            f"//*[contains(text(),'{label}')]/preceding-sibling::toggle-control/div").get_attribute('class')

    @PageService()
    def toggle_enable(self, label):
        """
        Enables the toggle bar if disabled,
        Args:
            label          (basestring):   Label corresponding to the toggle option.

        Note : Please do not use this method, as we are deprecating this, please use method enable_toggle instead
        """
        if not self.check_toggle_status(label):
            self.__click_on_toggle(label)
            self.wait_for_completion()

    @PageService()
    def toggle_disable(self, label):
        """
        Disables the toggle bar if enabled

        Args:
            label          (basestring):   Label corresponding to the toggle option.

        Note : Please do not use this method, as we are deprecating this, please use method disable_toggle instead
        """
        if self.check_toggle_status(label):
            self.__click_on_toggle(label)
            self.wait_for_completion()

    @WebAction()
    def __return_toggle(self, cv_toggle):
        """
        Method to return list of toggles present on page

        Returns:
                toggle_list (list) : List of toggles present on page
        """
        if not cv_toggle:
            toggle_control_list = self.driver.find_elements_by_xpath("//toggle-control")
            return toggle_control_list
        else:
            toggle_cv_list = self.driver.find_elements_by_xpath("//div[contains(@class, 'cv-toggle-wrapper')]")
            return toggle_cv_list

    @WebAction()
    def __is_toggle_enabled(self, toggle=None, toggle_id=None, cv_toggle=False):
        """ Method to check toggle status """
        if toggle and not cv_toggle:
            return 'enabled' in toggle.find_element_by_xpath("./div").get_attribute('class')
        elif cv_toggle:
            if toggle_id:
                toggle = self.driver.find_element_by_id(toggle_id)
            return 'isOn' in toggle.find_element_by_xpath("./div").get_attribute('class')
        else:
            return 'enabled' in self.driver.find_element_by_xpath(
                f"*//toggle-control[@id='{toggle_id}']/div").get_attribute('class')

    @WebAction()
    def __click_toggle(self, toggle=None, toggle_id=None):
        """ Method to click on toggle button """
        if toggle:
            toggle.click()
        else:
            self.driver.find_element_by_id(toggle_id).click()

    @PageService()
    def enable_toggle(self, index=None, toggle_id=None, cv_toggle=False):
        """
        Method to Enable toggle

        Args:
            index (int) :   index corresponding to the toggle option (Ex.- 0,1....so on)
            toggle_id      (basestring):   Id corresponding to the toggle-control tag
            cv_toggle       (bool) : Whether cv toggle element is present or not
        """
        if index is not None:
            toggle_list = self.__return_toggle(cv_toggle)
            if not self.__is_toggle_enabled(toggle=toggle_list[index], cv_toggle=cv_toggle):
                self.__click_toggle(toggle_list[index])
                self.wait_for_completion()
        elif toggle_id:
            if not self.__is_toggle_enabled(toggle_id=toggle_id, cv_toggle=cv_toggle):
                self.__click_toggle(toggle_id=toggle_id)
                self.wait_for_completion()

    @PageService()
    def disable_toggle(self, index=None, toggle_id=None, cv_toggle=False):
        """
        Method to Disable toggle

        Args:
            index (int) :   index corresponding to the toggle option (Ex.- 0,1....so on)
            toggle_id      (basestring):   Id corresponding to the toggle-control tag
            cv_toggle       (bool) : Whether cv toggle element is present or not
        """
        if index is not None:
            toggle_list = self.__return_toggle(cv_toggle)
            if self.__is_toggle_enabled(toggle=toggle_list[index], cv_toggle=cv_toggle):
                self.__click_toggle(toggle_list[index])
                self.wait_for_completion()
        else:
            if self.__is_toggle_enabled(toggle_id=toggle_id, cv_toggle=cv_toggle):
                self.__click_toggle(toggle_id=toggle_id)
                self.wait_for_completion()

    @WebAction()
    def toggle_edit(self, label):
        """
        Selects edit button next to toggle.

        Args:
            label   (basestring):   Label corresponding to the toggle option.

        """
        xpath_toggle_enable = "//ul/li/span[contains(text(),'"\
                              + label + "')]/../span/toggle-control/div[@class = " \
                                        "'cv-material-toggle cv-toggle enabled']/.."
        if self.check_if_entity_exists("xpath", xpath_toggle_enable):
            xpath_toggle_edit = xpath_toggle_enable + "/../button"
            if self.check_if_entity_exists("xpath", xpath_toggle_edit):
                self.driver.find_element_by_xpath(xpath_toggle_edit).click()

    @WebAction()
    def submit_form(self, wait=True, form_name=None):
        """
        Clicks save Button on open forms.
        Since it searches by type , should work on all forms.

        Raises:
            Exception:
                If the button is not found
        """
        if form_name:
            form_elem = self.driver.find_element_by_name(form_name)
            form_elem.find_element_by_xpath(".//button[@type='submit']").click()
        else:
            elems = self.driver.find_elements_by_xpath("//button[@type='submit']")
            for elem in elems:
                if elem.is_displayed():
                    elem.click()
                    break
        if wait:
            self.wait_for_completion()

    @WebAction()
    def cancel_form(self, form_name=None):
        """
        Clicks Cancel Button on open forms.

        Raises:
            Exception:
                If the button is not found
        """
        if form_name:
            form_elem = self.driver.find_element_by_name(form_name)
            form_elem.find_element_by_xpath(".//button[contains(.,'Cancel')]").click()
        else:
            self.driver.find_element_by_xpath("//button[contains(.,'Cancel')]").click()
            self.wait_for_completion()

    @WebAction()
    def button_next(self):
        """
        Clicks the Next button on forms

        Raises:
            Exception:
                If the button is not found
        """
        self.driver.find_element_by_xpath("//button[contains(.,'" + self.props['Next'] +
                                          "')]").click()
        self.wait_for_completion()

    @WebAction()
    def select_radio(self, id):
        """
        Selects radio button given the value.

        Args:
            id  (str)   -- ID of the radio button.

        """
        xpath_radio_from_value = "//input[@type = 'radio'][@id='" + id + "']"
        self.driver.find_element_by_xpath(xpath_radio_from_value).click()

    @WebAction()
    def label_getvalue(self, label):
        """
        Fetch the value corresponding to given label.

        Args:
            label   (str)   -- Name of the label.

        Raises:
            Exception:
                If the tile is not found

        """
        xpath_label = "//span[contains(text(), '" + label + "')]"
        xpath_value = xpath_label + "/../span[2]"
        xpath2_value = xpath_label + "/../a"
        if self.check_if_entity_exists("xpath", xpath_label):
            return self.driver.find_element_by_xpath(xpath_value).text
        elif self.check_if_entity_exists("xpath", xpath2_value):
            return self.driver.find_element_by_xpath(xpath2_value).text
        else:
            raise NoSuchElementException("Label :[{}] not found".format(label))

    def tile_select_hyperlink(self, tile_name, link_text):
        """
        This selects the hyperlink at tile label level. hyperlink within tile wont be selected.

        Args:

            tile_name   (str)   -- name of the tile on which operation needs to be performed.

            link_text   (str)   -- hyperlink name.

        """
        xpath_load_tile = "//cv-tile-component//span[contains(., '" + tile_name + "')]"
        xpath1 = xpath_load_tile + "/..//a[contains(.,'" + link_text + "')]"
        xpath2 = xpath_load_tile + "/../..//a[contains(.,'" + link_text + "')]"
        if self.check_if_entity_exists("xpath", xpath1):
            xpath_tile_hyperlink = xpath1
        elif self.check_if_entity_exists("xpath", xpath2):
            xpath_tile_hyperlink = xpath2
        else:
            raise Exception(
                "Could not locate Hyperlink:[{}] in  tile :[{}] not found".format(
                    link_text, tile_name))
        self.driver.find_element_by_xpath(xpath_tile_hyperlink).click()
        self.wait_for_completion()

    def cvselect_from_dropdown(self, label, value=None, select_all=False):
        """ Selects the value from the dropdown.Use this for cv multi select option


            Args:
                label       (str) -- Label name for the dropdown element

                value       (str) -- the value to be chosen from the dropdown

                select_all  (bool)-- To select all the values in the dropdown
                                        deafult: False


            Raises:

                Exception:

                    if dropdown cannot be loaded

                    if value not present in drop down list

                    if value is not string or list
        """
        if self.check_if_entity_exists("xpath", "//label[contains(text(),'" + label + "')]"):
            label_xpath = "//label[contains(text(),'" + label + "')]"
        elif self.check_if_entity_exists("xpath", "//label/span[contains(text(),'"
                                         + label + "')]"):
            label_xpath = "//label/span[contains(text(),'" + label + "')]"
        else:
            raise Exception("There is no label with the given text")

        if self.check_if_entity_exists("xpath", label_xpath + "/..//isteven-multi-select"):
            xpath_select_cvdropdown = label_xpath + "/..//isteven-multi-select"
        elif self.check_if_entity_exists("xpath", label_xpath +
                                         "/../div/cv-client-picker/isteven-multi-select"):
            xpath_select_cvdropdown = label_xpath + "/../div/cv-client-picker/isteven-multi-select"
        else:
            label_for = self.driver.find_element_by_xpath(label_xpath).get_attribute("for")
            xpath_select_cvdropdown = "//isteven-multi-select[@id='" + label_for + "']"

        xpath_select_cvdropdown_search_ok = xpath_select_cvdropdown + \
            "//div[@class='line-searchOk']"
        xpath_select_cvdropdown_options = xpath_select_cvdropdown + \
            "//div[@class='checkBoxContainer']/div"

        if self.check_if_entity_exists("xpath", xpath_select_cvdropdown):
            self.driver.find_element_by_xpath(xpath_select_cvdropdown).click()
            self.wait_for_completion()
        else:
            raise Exception("There is no dropdown with the given name or ID")

        if select_all:
            if self.check_if_entity_exists('xpath',
                                           "//button[@ng-bind-html='lang.selectAll']"):
                self.driver.find_element_by_xpath(
                    "//button[@ng-bind-html='lang.selectAll']").click()
                self.wait_for_completion()
                self.driver.find_element_by_xpath(xpath_select_cvdropdown_search_ok).click()
                self.wait_for_completion()
                return
            else:
                raise Exception("could not locate select all button")

        if isinstance(value, str):
            """
            To verify sorting , test on scale setup , if we can get all list.
            for option in xpath_select_cvdropdown_options:
                option = option + "/div/label/span"
                if value.lower() == option.text().lower():
                    option.click()
                    break
            """
            value = [value]

        value = [x.lower() for x in value]
        for item in value:
            xpath_select_cvdropdown_search = xpath_select_cvdropdown + \
                "//div[@class='line-search']/input"

            if self.check_if_entity_exists("xpath", xpath_select_cvdropdown_search):
                self.driver.find_element_by_xpath(xpath_select_cvdropdown_search).clear()
                self.driver.find_element_by_xpath(xpath_select_cvdropdown_search).send_keys(item)

            options = self.driver.find_elements_by_xpath(xpath_select_cvdropdown_options)
            for option in options:
                option_name = option.find_element_by_xpath("./div/label/span")
                if option_name.text.strip().lower() == item:
                    class_value = option.get_attribute('class')
                    if 'selected' not in class_value:
                        option_name.click()
                        self.wait_for_completion()
                    break

            if self.check_if_entity_exists("xpath", xpath_select_cvdropdown_search):
                self.driver.find_element_by_xpath("//button[@class='clearButton']").click()
                self.wait_for_completion()

        if self.check_if_entity_exists("xpath", xpath_select_cvdropdown):
            self.driver.find_element_by_xpath(xpath_select_cvdropdown).click()
            self.wait_for_completion()

    @WebAction()
    def checkbox_select(self, checkbox_id):
        """
        Selects checkbox that matches the ID
        Args:
            checkbox_id  (str)  -- id of the checkbox from dev or input tag
        """
        xp = f"//*[@id = '{checkbox_id}']"
        chkbox = self.driver.find_element_by_xpath(xp)
        if (chkbox.tag_name == 'input' and not chkbox.is_selected()) or chkbox.get_attribute(
                "data-state") == 'unchecked' \
                or 'partial-selection' in chkbox.get_attribute('class'):
            xpath = (f"//*[@id = '{checkbox_id}']/following-sibling::label | "
                     f"//*[@id = '{checkbox_id}']/following-sibling::span")
            self.driver.find_element_by_xpath(xpath).click()

    @WebAction()
    def checkbox_deselect(self, checkbox_id):
        """
        Deselects checkbox that matches the ID
        Args:
            checkbox_id  (str)  -- id of the checkbox from dev or input tag
        """
        xp = f"//*[@id = '{checkbox_id}']"
        chkbox = self.driver.find_element_by_xpath(xp)
        if chkbox.is_selected() or chkbox.get_attribute("data-state") == 'checked':
            xpath = (f"//*[@id = '{checkbox_id}']/following-sibling::label | "
                     f"//*[@id = '{checkbox_id}']/following-sibling::span")
            self.driver.find_element_by_xpath(xpath).click()

    def refresh_page(self):
        """Refreshes the contents in the browser tab"""
        self.driver.refresh()
        self.wait_for_completion()

    @WebAction()
    def select_hyperlink(self, link_text, index=0):
        """
        Selects hyperlink in the given page. Try with <a> if it fails try with span beneath.
        Add more logic for it to work in all cases.

        Args:

            link_text   (str)   --  Link name as displayed in the webpage.

            index       (int)   --  Index in case multiple links exist, default is 0.

        Returns:

        """
        for _ in range(5):
            try:
                xp = f'//a[text() = "{link_text}"]'
                if self.check_if_entity_exists("xpath", xp):
                    links = self.driver.find_elements_by_xpath(xp)
                    for link in links:
                        if link.is_displayed():
                            link.click()
                            break
                else:
                    xp = f'//a[contains(text(),"{link_text}")]'
                    self.driver.find_elements_by_xpath(xp)[index].click()
                self.wait_for_completion(1000)
                return
            except StaleElementReferenceException:
                continue

    @WebAction()
    def click_by_xpath(self, xpath):
        """Clicks on element with the given xpath"""
        self.driver.find_element_by_xpath(xpath).click()
        self.wait_for_completion()

    @WebAction()
    def access_page_action_menu(self, link_text):
        """
        Access page access menu with div having class page-action-item
        Args:
            link_text(str): Link text

        """
        xp = f"//div[contains(@class,'page-action-item')]//*[text()='{link_text}']"
        self.driver.find_element_by_xpath(xpath=xp).click()
        self.wait_for_completion()

    @WebAction()
    def access_page_action_menu_by_class(self, class_name):
        """
        Access page access menu with div having class page-action-item
        Args:
            class_name(str): class name

        """
        xp = f"//div[contains(@class,'page-action-item')]//*[@class='{class_name}']"
        self.driver.find_element_by_xpath(xpath=xp).click()
        self.wait_for_completion()

    def select_destination_host(self, host, contains_host_name=False):
        """
        Selects a ESX host as destination host or snap mount ESX

        Args:
            host        (basestring):   the esx host to choose as destination or snap mount

            contains_host_name    (bool):         if the host match should be partial

        """

        def expand_buttons(button_elements, contains_text=False):
            """
            Expands the non-collapsed buttons

            Args:
                button_elements  (list):   the list of buttons to expand

                contains_text    (bool):   if the host match should be partial

            Returns:
                0  :  if the given host is selected

                1  :  if the given host could not be selected

            """
            inter_buttons = []

            for elem in button_elements:
                elem.click()
                self.wait_for_completion()
                flag = 1
                child_elements = elem.find_elements_by_xpath("./../../div[2]/div")
                for child in child_elements:
                    host_elem = child.find_element_by_xpath("./div[1]/span")
                    if contains_text:
                        if host.strip() in host_elem.text.strip():
                            if 'selected' not in host_elem.find_element_by_xpath("./../../div"
                                                                                 ).get_attribute('class'):
                                host_elem.click()
                                self.wait_for_completion()
                            flag = 0
                            break
                    else:
                        if host_elem.text.strip() == host:
                            if 'selected' not in host_elem.find_element_by_xpath("./../../div"
                                                                                 ).get_attribute('class'):
                                host_elem.click()
                                self.wait_for_completion()
                            flag = 0
                            break
                if flag == 1:
                    inter_buttons.append(elem)
                else:
                    break
            for item in inter_buttons:
                button_elements.remove(item)
            if not button_elements:
                return 1
            return 0

        if self.check_if_entity_exists("xpath", "//div[@class='browseAndSelectVM']//"
                                                "div[@class='ng-scope selected']"):
            xpath = "//div[@class='browseAndSelectVM']//div[@class='ng-scope selected']/span"

            span = self.driver.find_element_by_xpath(xpath)
            if span.text.strip() == host:
                return

        buttons = self.driver.find_elements_by_xpath("//div[@class='"
                                                     "browseAndSelectVM']//button["
                                                     "@class='collapsed']")
        while buttons:
            ret = expand_buttons(buttons, contains_host_name)
            if ret == 0:
                buttons = []
            else:
                buttons = self.driver.find_elements_by_xpath("//div[@class='"
                                                             "browseAndSelectVM']"
                                                             "//button[@class='"
                                                             "collapsed']")
                if not buttons:
                    raise Exception("The ESX was not found")

    @WebAction()
    def add_security_associations(self, userroles):
        """
        Adding associations to the agent.

        Args:
            userroles (dict) -- a dictionary, with username and roles we want to associate
                                    with the subclients

            Sample dict :
                {
                    'User1': 'Master',
                    'User2': 'View'
                }
        Raises:
            Exception:
                If is no option to edit the security associations"
        """
        self.log.info("Adding security associations for iDA and subclients")
        if self.check_if_entity_exists("xpath",
                                       "//cv-tile-component[@data-title='Security']/div/div[3]/"
                                       "div/div[2]/div[2]/a"):
            self.driver.find_element_by_xpath("//cv-tile-component[@data-title='Security']/"
                                              "div/div[3]/div/div[2]/div[2]/a").click()
            self.wait_for_completion()
            for user in userroles.keys():
                self.driver.find_element_by_xpath(
                    "//div[@id='s2id_userList']/a/span[1]").click()
                self.wait_for_completion()
                self.driver.find_element_by_xpath(
                    "//div[@class='select2-search']/input").send_keys(user)
                self.wait_for_completion()
                elements = self.driver.find_elements_by_xpath(
                    "//div[@id='select2-drop']/ul[@role='listbox']/li")
                for elem in elements:
                    ugname = elem.find_element_by_xpath("./div/span").text
                    if ugname.split('(')[1].split(')')[0] == user or \
                            ugname.split(' (')[0] == user:
                        elem.find_element_by_xpath("./div/span").click()
                        self.wait_for_completion()
                        break
                self.select_value_from_dropdown("adduserId", userroles[user])
                self.driver.find_element_by_xpath(
                    "//div[@id='subclient-content-tab']/div[1]/button").click()
                self.wait_for_completion()
                self.check_error_message()
            self.driver.find_element_by_xpath(
                "//body/div[1]/div/div/div[2]/div[2]/button[2]").click()
            self.wait_for_completion()
            self.check_error_message()
        else:
            exp = "There is no option to edit the security associations"
            self.log.error(exp)
            raise Exception(exp)

    @WebAction(delay=0)
    def __read_notification_text(self, wait_time):
        """
        Reads the notification text

        Args:
            wait_time   (int)   -- time to wait for the popup

        Returns:
            notification_text (basestring): the notification string
        """
        try:
            WebDriverWait(self.driver, wait_time).until(ec.presence_of_element_located((
                By.XPATH, "//div[@class='growl']/div/div/div")))
            notification_text = self.driver.find_element_by_xpath(
                "//div[@class='growl']/div/div/div").text
            if not notification_text:
                WebDriverWait(self.driver, wait_time).until(ec.presence_of_element_located((
                    By.XPATH, "//div[@class='growl']/div/div/div")))
                notification_text = self.driver.find_element_by_xpath(
                    "//div[@class='growl']/div/div/div").text
            return notification_text
        except TimeoutException:
            return ""

    @PageService()
    def get_notification(self, wait_time=60):
        """
        Gets the notification text

        Args:
            wait_time   (int)   -- time to wait for the popup
                                    default: 60

        Returns:

            notification_text (basestring): the notification string

        """
        return self.__read_notification_text(wait_time)

    @PageService()
    def get_jobid_from_popup(self, wait_time=60):
        """
        Gets the job id from pop up

        Args:
            backup  (bool): if the job ID for backup has to be obtained

        Returns:
            job_id (int):  the job id for the submitted request

        """
        job_text = self.get_notification(wait_time)
        if not job_text:  # retry for job id
            job_text = self.get_notification(wait_time)
            if not job_text:
                raise CVWebAutomationException("No notification is popped up to extract job id")
        job_id = re.findall(r'\d+', job_text)[0]
        self.log.info("Job %s has started", str(job_id))
        return job_id

    def get_error_message(self):
        """
        get if there is any error message

        Args:

        Returns:
            error message string
            empty string when there is no error message
        """
        exp = ""
        xpath_list = ["//span[contains(@class,'performActionMessage')]",
                      "//*[contains(@class, 'serverMessage')]",
                      "//span[contains(@class. 'help-block')]",
                      "//div[@class='growl-item alert ng-scope alert-error alert-danger']",
                      "//span[contains(@class, 'error')]",
                      "//div[contains(@class,'vw-login-message error-box error')]"
                      ]
        for xpath in xpath_list:
            if xpath == "//div[@class=" \
                        "'growl-item alert ng-scope alert-error alert-danger']":
                xpath = xpath + "/div/div/div"
            if self.check_if_entity_exists("xpath", xpath):
                exp = self.driver.find_element_by_xpath(xpath).text
                break
        return exp

    def check_error_message(self, raise_error=True):
        """
        Checks if there is any error message

        Args:
            raise_error (bool): should error be raised or not

        Raises:
            Exception:
                if there is an error message after submitting the request

        """
        exp = self.get_error_message()

        if exp != "":
            self.log.error(exp)

        if exp != "" and raise_error:
            raise CVWebAutomationException(exp)

    def search_vm(self, vm_name):
        """
        Searches for a VM in the vcenter

        Args:
            vm_name  (basestring)    :   the name of the VM to be searched

        Raises:
            Exception:
                if the given VM is not found in the vcenter

        """

        try:
            self.log.info("Searching for VM %s", str(vm_name))
            if self.check_if_entity_exists('xpath', '//input[@placeholder="Search VMs"]'):
                search = self.driver.find_element_by_xpath('//input[@placeholder="Search VMs"]')
            else:
                raise NoSuchElementException("There is no option to search vms")
            search.clear()
            search.send_keys(vm_name)
            search.send_keys(Keys.ENTER)
            self.wait_for_completion()

            if self.check_if_entity_exists("xpath", "//span[@title='{0}']".format(vm_name)):
                self.driver.find_element_by_xpath("//span[@title='{0}']".format(vm_name)).click()
            else:
                raise Exception("There is no VM named {0}.".format(vm_name))
            self.log.info("Selected the VM %s", str(vm_name))
        except Exception as exp:
            raise Exception("Exception occurred while searching and selecting VM. {0}".format(
                str(exp)
            ))

    def cv_single_select(self, label, value):
        """
        Selects an element from the dropdown

        Args:
            label   (basestring):   the label corresponding to the dropdown
            value   (basestring):   the value to be selected in the dropdown

        Raises:
            Exception:
                if the value passed is not a string or
                if the dropdown is not present or
                if the plan is not present in the dropdown

        """
        if not isinstance(value, str):
            raise Exception("The dropdown will accept only single value. Please pass a string")

        if self.check_if_entity_exists("xpath", "//label[contains(text(),'" + label + "')]"):
            label_xpath = "//label[contains(text(),'" + label + "')]"
        elif self.check_if_entity_exists("xpath", "//label/span[contains(text(),'" + label + "')]"
                                         ):
            label_xpath = "//label/span[contains(text(),'" + label + "')]"
        else:
            raise Exception("There is no dropdown with the given name or ID")

        select_xpath = label_xpath + "//isteven-multi-select"
        if not self.check_if_entity_exists("xpath", select_xpath):
            select_xpath = label_xpath + "/..//isteven-multi-select"
            if not self.check_if_entity_exists("xpath", select_xpath):
                select_xpath = label_xpath + "/../..//isteven-multi-select"

        try:
            self.driver.find_element_by_xpath(select_xpath).click()
            self.wait_for_completion()
        except Exception:
            raise Exception('Drop down cannot be found by cv_single_select, please check the logs')

        # To use search box if exists
        search = f"{select_xpath}//div[@class='line-search']/input"
        if self.check_if_entity_exists('xpath', search):
            self.driver.find_element_by_xpath(search).clear()
            self.driver.find_element_by_xpath(search).send_keys(value)
            self.wait_for_completion()

        select_option_xpath = select_xpath + "//div[@class='checkBoxContainer']"
        if self.check_if_entity_exists("xpath", select_option_xpath):
            span_xpath = select_option_xpath + "//span[contains(text(),'" + value + "')]"
            div_xpath = select_option_xpath + "//div[contains(text(),'" + value + "')]"

            if self.check_if_entity_exists('xpath', span_xpath):
                checkbox_sibling = span_xpath + "/preceding-sibling::input"
                if not self.driver.find_element_by_xpath(checkbox_sibling).is_selected():
                    self.driver.find_element_by_xpath(span_xpath).click()
                    self.wait_for_completion()
                else:
                    self.driver.find_element_by_xpath(select_xpath).click()
                    self.wait_for_completion()

            elif self.check_if_entity_exists('xpath', div_xpath):
                self.driver.find_element_by_xpath(div_xpath).click()
                self.wait_for_completion()

        else:
            raise Exception("The given value is not present")

    def date_picker(self,
                    time_value,
                    time_id=None,
                    pick_time_only=False):
        """
        Picks the time in the date or time picker

        Args:
            time_value   (dict):        the time to be set as range during the browse

                Sample dict:    {   'year':     2017,
                                    'month':    december,
                                    'date':     31,
                                    'hours':    09,
                                    'mins':     19,
                                    'session':  'AM'
                                }

            time_id      (basestring):  from / to to be chosen as date range

            pick_time_only (bool):     Picks only the time if True

        """
        if time_id:
            # Used during browse to select from and to time
            year_month = f"//div[@id='{time_id}']/div//tr[1]/th[2]/button"
            date_xpath = f"//div[@id='{time_id}']/div//button/span[contains(text()" \
                         f",'{time_value['date']}') and not(contains(@class,'text-muted'))]"
            hours_xpath = f"//div[@id='{time_id}']//td[contains(@class," \
                          "'form-group uib-time hours')]/input"
            mins_xpath = f"//div[@id='{time_id}']//td[contains(@class," \
                         "'form-group uib-time minutes')]/input"
            session_xpath = f"//div[@id='{time_id}']//td[@class='uib-time am-pm']/button"
        else:
            # Used during schedule creation
            year_month = "//tr[1]/th[2]/button"
            date_xpath = f"//button/span[contains(text(),'{time_value['date']}')" \
                         " and not(contains(@class,'text-muted'))]"
            hours_xpath = "//td[contains(@class, 'form-group uib-time hours')]/input"
            mins_xpath = "//td[contains(@class, 'form-group uib-time minutes')]/input"
            session_xpath = "//td[@class='uib-time am-pm']/button"

        if not pick_time_only:
            self.driver.find_element_by_xpath(year_month).click()
            self.wait_for_completion()

            if time_value.get('year'):
                self.driver.find_element_by_xpath(year_month).click()
                self.wait_for_completion()
                self.driver.find_element_by_xpath(
                    "//button/span[contains(text(),'" + time_value['year'] + "')]").click()
                self.wait_for_completion()
            self.driver.find_element_by_xpath(
                "//button/span[contains(text(),'" + time_value['month'].capitalize() + "')]").click()
            self.wait_for_completion()
            self.driver.find_element_by_xpath(date_xpath).click()
            self.wait_for_completion()

        if time_value.get('hours'):
            self.driver.find_element_by_xpath(hours_xpath).clear()
            self.driver.find_element_by_xpath(mins_xpath).clear()
            self.driver.find_element_by_xpath(mins_xpath).send_keys(time_value['mins'])
            self.driver.find_element_by_xpath(hours_xpath).send_keys(time_value['hours'])

        if time_value.get('session'):
            sess = self.driver.find_element_by_xpath(session_xpath)
            if not time_value['session'] == sess.text:
                sess.click()

    @WebAction()
    def click_button_using_text(self, value):
        """ Method to click on a button """
        buttons = self.driver.find_elements_by_xpath(
            f"//button[contains(.,'{value}')]")
        for button in buttons:
            if button.is_displayed():
                button.click()
                break

    @PageService()
    def click_button(self, value):
        """
        Clicks on button by text value

        Args:
            value (str) : text of the button to be clicked

        Returns:
            None
        """
        self.click_button_using_text(value)
        self.wait_for_completion()

    @PageService
    def get_page_actions_list(self, group_by=False):
        """Gets visible page actions from drop down menu"""
        flatten_grid_list = []
        nested_grid_list = []
        group_list = []
        page_actions_list = self.__get_page_actions_list()
        page_actions_list = [action.text for action in page_actions_list]
        if group_by:
            for action in page_actions_list:
                if action == '':
                    nested_grid_list += [group_list]
                    group_list = []
                else:
                    group_list += [action]
            nested_grid_list += [group_list]
            return nested_grid_list
        else:
            for action in page_actions_list:
                if action != '':
                    flatten_grid_list += [action]
            return flatten_grid_list

    @PageService()
    def get_locale_name(self):
        """
        Returns the currently selected language's locale name in admin console

        Returns:
            locale    (basestring):   the language in action

        """
        self.__expand_locale_selection_drop_down()
        selected_locale = self.__get_selected_locale()

        if selected_locale in ["en_US", "en"]:
            selected_locale = "en"
            self.ext = ".properties"
        else:
            self.ext = "_" + selected_locale + ".UTF8"
        return selected_locale

    @PageService()
    def get_all_languages(self):
        """
        Returns the list of all language's locale name

        Returns:
            (list)  -- List of all languages

        """
        self.__expand_locale_selection_drop_down()
        return self.__get_all_languages()

    @PageService()
    def check_for_locale_errors(self):
        """To check for locale errors on the page"""
        try:
            self.driver.find_element_by_xpath(
                '//span[contains(text(), "??")] | //div[contains(text(), "??")]')
        except NoSuchElementException:
            pass
        else:
            raise CVWebAutomationException(f'Locale errors found on {self.driver.current_url} page')

    def change_language(self, language, page_obj):
        """
        Change the localization language of admin console

        Args:
            page_obj    (object):       the instance of the class

            language    (basestring):   the language to be changed to in English

        """

        if not self.locale_ids:
            self.__get_locale_ids()
        if self.locale_ids.get(language.lower()):
            lang_id = self.locale_ids[language.lower()]
        else:
            raise Exception("There is no language with the given name. Please check your input")

        self.__expand_locale_selection_drop_down()
        self.__select_language(lang_id)
        self.get_locale_name()
        self.props = {}
        self._load_properties(page_obj)

    def select_for_restore(self, file_folders, all_files=False, select_one=False):
        """
        Selects files, folders for restore

        Args:
            file_folders (list):    the list of files and folders to select for restore

            all_files   (bool):     select all the files shown for restore / download

            select_one  (bool):     select first file in the list for restore /download

        Raises:
            Exception:
                if at least one of the files / folders could not be selected

        """
        self.log.info("Selecting items to submit for restore")
        if all_files:
            elem = self.driver.find_element_by_xpath("//div[@class='ui-grid-header-cell-wrapper']"
                                                     "/div/div/div/div/div")
            if 'selected' not in elem.get_attribute('class'):
                elem.click()
                self.wait_for_completion()
            return

        selected = []
        while True:
            elements = self.driver.find_elements_by_xpath(
                "//div[1]/div[2]/div[2]/div[@class='ui-grid-canvas']/div")
            index = 1
            flag = []
            for elem in elements:
                f_elem = elem.find_element_by_xpath("./div/div[1]").text
                for file in file_folders:
                    if f_elem == file:
                        flag.append(index)
                        selected.append(f_elem)
                    else:
                        continue
                index = index + 1
            for flg in flag:
                flg = str(flg)
                self.driver.find_element_by_xpath(
                    "//div[1]/div[1]/div[2]/div[@class='ui-grid-canvas']/div[" +
                    flg +
                    "]/div/div/div/div/div").click()
                file_folders = list(set(file_folders) - set(selected))
                if select_one:
                    file_folder = []
                    break
            if self.cv_table_next_button_exists():
                if self.driver.find_element_by_xpath(
                        "//button[@ng-disabled='cantPageForward()']").is_enabled():
                    self.cv_table_click_next_button()
                    continue
                else:
                    break
            else:
                break
        if file_folders:
            raise Exception("Could not find the items " + str(file_folders))

    @PageService()
    def recovery_point_restore(self, recovery_time=None):
        """
        Restores the VM from the backup history

        Args:
            recovery_time     (basestring):   the backup date in 01-September-1960 format

        """
        try:
            self.select_overview_tab()
            if recovery_time:
                calender = {}
                calender['date'], calender['month'], calender['year'] = recovery_time.split("-")
                self.date_picker(calender)
            self.tile_select_hyperlink("Recovery point", "Restore")
        except Exception as exp:
            raise CVWebAutomationException(
                "Either recovery point was not given or the current date does not have any backups. ", exp)

    @WebAction(delay=0)
    def fill_form_by_id(self, element_id, value):
        """
        Fill the value in a text field with id element id.

        Args:
            element_id (basestring) -- the ID attribute of the element to be filled
            value (basestring)      -- the value to be filled in the element

        Raises:
            Exception:
                If there is no element with given name/id

        """
        element = self.driver.find_element_by_id(element_id)
        element.clear()
        element.send_keys(value)
        self.wait_for_completion()

    @WebAction()
    def fill_form_by_name(self, name, value):
        """
        Fill the value in a text field with id element id.

        Args:
            name (basestring) - name attribute value of the element
            value (basestring)      -- the value to be filled in the element

        Raises:
            Exception:
                If there is no element with given name/id

        """
        element = self.driver.find_element_by_name(name)
        element.clear()
        element.send_keys(value)
        self.wait_for_completion()

    @PageService()
    def select_overview_tab(self):
        """
        Selects the overview tab

        Raises:
            NoSuchElementException:
                if the Overview tab is not present
        """
        self.access_tab(self.props['label.tab.overview'])

    @PageService()
    def select_configuration_tab(self):
        """
        Selects the configuration tab

        Raises:
            NoSuchElementException:
                if the configuration tab is not present
        """
        self.access_tab(self.props['label.nav.configuration'])

    def expand_options(self, label):
        """
        Expands the advanced options or other chevron options

        Raises:
            NoSuchElementException:
                if the element is not found

        """
        xpath = "//div[contains(text(),'" + label + "')]/i"
        if self.check_if_entity_exists("xpath", xpath):
            class_value = self.driver.find_element_by_xpath(xpath).get_attribute("class")
            if 'ion-chevron-right' in class_value:
                self.driver.find_element_by_xpath(xpath).click()
                self.wait_for_completion()
        else:
            raise NoSuchElementException("No chevron found with the given label %s", label)

    @WebAction()
    def check_radio_button(self, label):
        """
        checks if radio button given the value exists.

        Args:
            label  (str)   -- Label text of the radio button.

        Returns:
            True or False
        """
        xpath_radio_from_value = f"//input[@type='radio']/../../label[contains(.,'{label}')]"
        return self.check_if_entity_exists("xpath", xpath_radio_from_value)

    @WebAction(delay=0)
    def expand_accordion(self, label):
        """Clicks the heading of an accordion"""
        xp = f"//span[contains(text(),'{label}')]/..//i[contains(@class,'glyphicon-chevron-right')]"
        expand_icon = self.driver.find_elements_by_xpath(xp)
        if expand_icon:
            expand_icon[0].click()

    @PageService()
    def access_menu(self, name):
        """
        Click menu item
        Args:
            name: localized menu text
        """
        self.__click_menu(name)
        self.wait_for_completion()

    @PageService()
    def access_menu_from_dropdown(self, name):
        """ Method to click on the menu under a dropdown on a page
        Args:
            name: localized menu text
        """
        self.__click_main_bar_dropdown()
        self.wait_for_completion()
        self.access_menu(name)

    @PageService()
    def access_sub_menu(self, name):
        """ Method to click on the item under a dropdown in database page
        Args:
            name: localized sub menu text
        """
        self.__click_access_sub_menu(name)
        self.wait_for_completion()

    @PageService()
    def access_action(self, action_name):
        """
        Access action in page
        Args:
            action_name: text used for action
        """
        self.__click_on_action(action_name)
        self.wait_for_completion()

    @WebAction()
    def __click_breadcrumb_link(self, link_text):
        """
        Method to click on breadcrumb link using given text

        Args:
            link_text (str)  -- Text on the breadcrumb link
        """
        xp = f"//a/div[contains(text(),'{link_text}')]"
        self.driver.find_element_by_xpath(xp).click()

    @PageService()
    def select_breadcrumb_link_using_text(self, link_text):
        """
        Selects breadcrumb link containing the given text
        Args:
            link_text (str)  -- Text on the breadcrumb link
        """
        self.__click_breadcrumb_link(link_text)
        self.wait_for_completion()

    @WebAction()
    def access_tab_action(self, name):
        """ Method to click on the item under a dropdown in acess tab
        Args:
            name: localized sub menu text
        """
        self.driver.find_element_by_xpath(
            f"//a[@class='tabAction'][contains(text(),'{name}')]").click()

    @WebAction()
    def access_tab_from_dropdown(self, name, col_id=None):
        """Method to click on the the access tab drop down
        Args:
            name       (str): Name of the access tab to click
            col_id      (int): Column id to click
        """
        if col_id:
            self.__click_main_bar_dropdown(col_id)
        else:
            self.__click_main_bar_dropdown()
        self.wait_for_completion()
        self.access_tab_action(name)
