# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the methods that can be done of the Sensitive Data Analysis
Project details page.


Classes:

    SensitiveDataAnalysisProjectDetails() ---> SensitiveDataAnalysis() --->
    GovernanceApps() ---> object()


SensitiveDataAnalysisProjectDetails  --  This class contains all the methods for
    action in Sensitive Data Analysis Project details page and is inherited by other
    classes to perform GDPR related actions

    Functions:

    select_add_data_source()          --  Selects the data source type to be added
    get_data_sources()     -       -- Returns all the data sources
    wait_for_data_source_status_completion() -- Waits for the data source scan status
                                                                            completion
    select_data_source() -- Selects a given data source
    select_overview()                    -- Clicks on the Overview link
    select_details()                    -- Clicks on the Details link
"""

import re
import time

from Web.AdminConsole.GovernanceAppsPages.SensitiveDataAnalysis import SensitiveDataAnalysis
from Web.Common.page_object import WebAction
from Web.AdminConsole.Components.panel import PanelInfo
from Web.AdminConsole.Components.table import Table


class SensitiveDataAnalysisProjectDetails(SensitiveDataAnalysis):
    """
     This class contains all the methods for action in Sensitive Data Analysis
     Project details page
    """

    def __init__(self, admin_console):
        """
        Args:
            admin_console (AdminConsole): adminconsole base object
        """
        super().__init__(admin_console)
        self.__admin_console = admin_console
        self.driver = self.__admin_console.driver
        self.__admin_console._load_properties(self)
        self.log = self.__admin_console.log
        self.__table = Table(self.__admin_console)
        self.__panelinfo = PanelInfo(self.__admin_console)

    @WebAction()
    def select_add_data_source(self, data_source_type='File system'):
        """
            Selects the File system data source to be addedpylint

            Args:
                data_source_type (str) - Type of the data source to be selected
                Values:
                    Default - "File system"

            Raise:
                Exception if invalid data source type provided
        """
        data_source_list = [
            self.__admin_console.props['label.datasource.file'],
            self.__admin_console.props['label.datasource.onedrive'],
            self.__admin_console.props['label.datasource.exchange'],
            self.__admin_console.props['label.datasource.sharepoint'],
            self.__admin_console.props['label.datasource.database'],
            self.__admin_console.props['label.datasource.gmail'],
            self.__admin_console.props['label.datasource.googledrive']
        ]
        self.log.info('Clicking on Add button')
        self.__admin_console.access_tile('ADD_DATA_SOURCE')
        self.log.info('Selecting %s data source option' % data_source_type)
        if data_source_type in data_source_list:
            self.__panelinfo.open_hyperlink_on_tile(
                data_source_type)
        else:
            raise Exception("Invalid data source type: %s" % data_source_type)
        self.__admin_console.wait_for_completion()

    @WebAction()
    def get_number_files(self, data_source_name):
        """
                Returns number of files in data sources

                    Return:
                        Number of files

                """
        number_files = str(
            self.driver.find_element_by_xpath(
                "//a[text()='%s']/../../div[4]/span" %
                data_source_name).text)
        return number_files

    @WebAction()
    def get_data_sources(self):
        """
        Returns all the list of data sources

            Return:
                List of data sources

        """
        data_sources_list = []
        rows = self.driver.find_elements_by_xpath(
            "//div[@class='ui-grid-canvas']/div")
        for row_id in range(1, len(rows) + 1):
            data_sources_list.append(str(self.driver.find_element_by_xpath
                                         ("//div[@class='ui-grid-canvas']\
                                         /div[%d]/div/div[1]/a" % row_id).text))
        self.log.info('List of data source names obtained are: %s'
                      % data_sources_list)
        return data_sources_list

    @WebAction()
    def wait_for_data_source_status_completion(
            self, data_source_name, timeout=30):
        """
        Waits for the data source scan status completion

            Args:
                data_source_name (str)  - Name of the data source
                timeout     (int)   --  minutes
                    default: 20

            Returns:
                bool  - boolean specifying if the data source scan had finished or not
                    True    -   if the data source scan had finished successfully

                    False   -   if the data source scan was not completed within the
                                                                            timeout

        """
        status = False
        start_time = int(time.time())
        current_time = start_time
        completion_time = start_time + timeout * 60

        while completion_time > current_time:
            self.log.info("Refreshing the page")
            self.driver.refresh()
            self.__admin_console.wait_for_completion()
            self.log.info(
                'Obtaining the data source status of: %s' % data_source_name)
            current_status = self.__table.get_column_data('Status')[0]
            self.log.info(
                'Data source status obtained is: %s' %
                current_status)
            if re.search("FINISHED", current_status, re.IGNORECASE):
                status = True
                break
            elif re.search(self.__admin_console.props['label.taskDetail.status.Completed'],\
                            current_status, re.IGNORECASE):
                status = True
                break
            elif re.search("COMPLETED WITH ERRORS", current_status, re.IGNORECASE):
                status = True
                break
            self.log.info("Refreshing the page")
            self.driver.refresh()
            self.__admin_console.wait_for_completion()
            time.sleep(30)
            current_time = int(time.time())
        return status

    @WebAction()
    def select_data_source(self, data_source_name, refresh=True, refresh_count=0):
        """
        Selects the given data source

            Args:
                data_source_name  - Data Source name

            Raise:
                Exception if data source not found
        """
        self.log.info("Selecting Data Source: %s" % data_source_name)
        self.driver.find_element_by_xpath(
            "//a[text()='%s']" %
            data_source_name).click()
        self.__admin_console.wait_for_completion()
        if refresh:
            for count in range(refresh_count):
                self.log.info("Refreshing the page %s/%s" % (count, refresh_count))
                self.driver.refresh()
                self.__admin_console.wait_for_completion()
                time.sleep(30)

    @WebAction()
    def select_overview(self):
        """
        Clicks on the Overview link
        """
        self.driver.find_element_by_xpath(
            "//a[contains(@class,'bc-item') and text()='%s']"\
            %self.__admin_console.props['label.overview']).click()
        self.__admin_console.wait_for_completion()

    @WebAction()
    def select_details(self):
        """
        Clicks on the Details link
        """
        self.driver.find_element_by_xpath(
            "//a[contains(@class,'bc-item') and text()='%s']"\
            %self.__admin_console.props['label.details']).click()
        self.__admin_console.wait_for_completion()
