# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the methods that can be done of the Sensitive Data Analysis
project page.

Classes:

    SensitiveDataAnalysis() ---> GovernanceApps() ---> object()


SensitiveDataAnalysis  --  This class contains all the methods for action in
    Sensitive Data Analysis Project page and is inherited by other classes to
    perform GDPR related actions

    Functions:

    add_project()          --  adds a project
    search_for_project()    -- Searches for a given project
    navigate_to_project_details()    -- Navigates to project details page
    delete_project()              -- Deletes a Project

"""
from Web.AdminConsole.Components.table import Table
from Web.Common.page_object import (
    WebAction,
    PageService
)
from Web.AdminConsole.GovernanceAppsPages.GovernanceApps import GovernanceApps
from Web.AdminConsole.Components.panel import DropDown
from Web.AdminConsole.Components.dialog import ModalDialog


class SensitiveDataAnalysis(GovernanceApps):
    """
     This class contains all the methods for action in Sensitive Data Analysis page
    """

    def __init__(self, admin_console):
        """
        Args:
            admin_console (AdminConsole): adminconsole base object
        """
        super().__init__(admin_console)
        self.__admin_console = admin_console
        self.driver = self.__admin_console.driver
        self.__admin_console._load_properties(self)
        self.log = self.__admin_console.log
        self.__table = Table(self.__admin_console)
        self.__dropdown = DropDown(self.__admin_console)
        self.__modal_dialog = ModalDialog(self.__admin_console)

    @WebAction()
    def add_project(self, project_name, plan_name, inventory_name):
        """
        Adds a project

            Args:
                project_name (str)  - Project name to be added
                plan_name (str)  - Plan name to be selected
                inventory_name (str)  - Inventory name to be selected

            Raise:
                Exception if project addition failed
        """
        self.log.info("Clicking on Add project")
        self.driver.find_element_by_xpath(
            "//a[contains(text(),'%s')]"\
            %self.__admin_console.props['label.gdpr.addProject']).click()
        self.__admin_console.wait_for_completion()
        self.log.info("Entering Project name")
        self.__admin_console.fill_form_by_id("name", project_name)
        self.log.info("Selecting Plan: %s" % plan_name)
        self.__dropdown.select_drop_down_values(0, [plan_name])
        self.log.info("Selecting Inventory: %s" % inventory_name)
        self.__dropdown.select_drop_down_values(1, [inventory_name])
        self.log.info("Clicking on Save")
        self.__admin_console.submit_form()
        if self.__admin_console.check_if_entity_exists(
                "xpath", "//*[@class='serverMessage error']"):
            raise Exception(self.driver.find_element_by_xpath(
                "//*[@class='serverMessage error']").text)

    @WebAction()
    def search_for_project(self, project_name):
        """
        Searches for a given project

            Args:
                project_name (str)  - Project name to be searched for

            Returns True/False based on the presence of the Project
        """
        return self.__table.is_entity_present_in_column(
            self.__admin_console.props['label.name'], project_name)

    @PageService()
    def navigate_to_project_details(self, project_name):
        """
        Navigates to project details page

            Args:
                project_name (str)  - project name details to be navigated

        """
        self.__table.access_context_action_item(
            project_name, self.__admin_console.props['label.details'])

    @PageService()
    def delete_project(self, project_name):
        """
        Deletes a project

            Args:
                project_name (str)  - Project name to be deleted

            Raise:
                Exception if project deletion failed
        """
        self.__table.access_context_action_item(
            project_name, self.__admin_console.props['label.delete'])
        self.__modal_dialog.click_submit()
        self.__admin_console.check_error_message()
