# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the methods that can be done of the request Manager page.


Classes:

    ReviewRequest() ---> GovernanceApps() ---> AdminPage() ---> login_page --->
    AdminConsoleBase() ---> object()


ReviewRequest  --  This class contains methods for reviewing actions in Request
    Manager page and is inherited by other classes to perform GDPR related actions

    Functions:

    review_approve_request() --- Review Approve Request Action
    check_job_status()       --- Check job information for GDPR task approval
"""
from Web.AdminConsole.AdminConsolePages.Jobs import Jobs
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.GovernanceAppsPages.RequestManager import RequestManager
from Web.Common.exceptions import CVWebAutomationException
from Web.Common.page_object import (
    WebAction,
    PageService
)


class ReviewRequest:
    """
        Review request based on entity and associated sensitive file
    """

    def __init__(self, admin_console):
        self._admin_console = admin_console
        self._admin_console._load_properties(self)
        self.__table = Table(admin_console)
        self.__job = Jobs(admin_console)
        self.table_map = {
            self._admin_console.props['label.datasource.file']:
                {"FilePath": "File path", "FileName": "Name"},
            self._admin_console.props['label.datasource.onedrive']:
                {"FilePath": "Name", "FileName": "Name"}
        }

    @WebAction()
    def _get_facet_count(self, facet_type, facet_name):
        """
                Get reviewed item count
                    Args:
                    Returns count string
                """
        xp = "//label[contains(text(),'{}')]/following::span[text()='{}']/following::span[1]". \
            format(facet_type, facet_name)
        count = str(self._admin_console.driver.find_element_by_xpath(xp).text)
        return count

    @WebAction()
    def _get_first_file_path(self, data_source_type):
        """
                                Get first file path in table
                                    Args:
                                        data_source_type(str): Data Source Type
                                    Returns (String) name
                                """
        name = self.__table.get_column_data(
            self.table_map[data_source_type]["FilePath"]
        ).pop(0)
        return name

    @WebAction()
    def _get_first_file_name(self, data_source_type):
        """
                                Get first file name in table
                                    Args:
                                        data_source_type(str): Data Source Type
                                    Returns String name
                                """
        name = self.__table.get_column_data(
            self.table_map[data_source_type]["FileName"]
        ).pop(0)
        return name

    @WebAction()
    def _request_approval(self):
        """
        Request approval for request
        """
        self._admin_console.click_button("Request approval")

    @PageService()
    def _review_file_with_comment(self, comment):
        """
        Review with comment
        :param comment: Comment
        """
        self._admin_console.driver.find_element_by_id("declineReasonText").clear()
        self._admin_console.driver.find_element_by_id("declineReasonText").send_keys(comment)
        self._admin_console.click_button_using_text(
            self._admin_console.props['label.taskpreview.accept']
        )
        self._admin_console.wait_for_completion()
        self._admin_console.click_button_using_text("Yes")
        self._admin_console.wait_for_completion()

    @PageService()
    def review_approve_request(self, request_name, filepath_to_verify,
                               datasource_type="File system"):
        """
               Navigates to request manager details page
                    :param request_name (str)  - request name details to be navigated
                    :param filepath_to_verify(str) - file to verify
                    :param datasource_type(str)- Data Source Type To review
                    :raise if files are not marked reviewed or request status is incorrectly set
               """
        _request = RequestManager(self._admin_console)
        _request.select_request_by_name(request_name)
        reason = "This is automated review"
        not_reviewed = self._get_facet_count(
            self._admin_console.props['label.taskdetail.reviewStatus'],
            self._admin_console.props['label.taskdetail.notReviewed']
        )
        not_reviewed = not_reviewed[not_reviewed.index('(') + len('('): not_reviewed.index(')')]
        self._admin_console.log.info("Not Reviewed [%s]", not_reviewed)
        filepath = self._get_first_file_path(datasource_type)
        if filepath_to_verify == filepath:
            self._admin_console.log.info("File path matched")
        else:
            self._admin_console.log.error(
                "File path did not match. Source [%s], destination [%s]", filepath_to_verify,
                filepath)
        filename = self._get_first_file_name(datasource_type)
        self._admin_console.driver.find_element_by_link_text(filename).click()
        self._admin_console.wait_for_completion()
        self._review_file_with_comment(reason)
        _request.search_for_request(request_name)
        status = _request.get_status()
        if "Review completed" not in status:
            raise CVWebAutomationException("Incorrect status [%s]", status)
        _request.select_request_by_name(request_name)
        reviewed = self._get_facet_count(
            self._admin_console.props['label.taskdetail.reviewStatus'],
            self._admin_console.props['label.taskdetail.reviewed']
        )
        not_reviewed = self._get_facet_count(
            self._admin_console.props['label.taskdetail.reviewStatus'],
            self._admin_console.props['label.taskdetail.notReviewed']
        )
        reviewed = reviewed[reviewed.index('(') + len('('): reviewed.index(')')]
        not_reviewed = not_reviewed[not_reviewed.index('(') + len('('): not_reviewed.index(')')]
        self._admin_console.log.info("Reviewed [%s], Not Reviewed [%s]", reviewed, not_reviewed)
        if int(not_reviewed) != 0:
            self._admin_console.log.error(
                'Review failed, after review, number of not reviewed items [%s]',
                not_reviewed)
            raise CVWebAutomationException(
                'Review failed, after review, number of not reviewed items [%s]',
                not_reviewed)
        self._request_approval()

    @PageService()
    def fetch_request_job(self):
        """
        Check job information for GDPR task approval
        :return: job id (list)
        """
        self.__job.access_active_jobs()
        self.__table.apply_filter_over_column_selection(
            self._admin_console.props['label.taskDetail.operation'],
            'GDPR task approval'
        )
        _id = self.__table.get_column_data("Job Id")
        return _id
