# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the methods that can be done of the request Manager page.


Classes:

    RequestManager() ---> GovernanceApps() ---> AdminPage() ---> login_page --->
    AdminConsoleBase() ---> object()


RequestManager  --  This class contains all the methods for action in Request
    Manager page and is inherited by other classes to perform GDPR related actions

    Functions:

    add_request()          --  adds an request
    search_for_request()    -- Searches for an request
    navigate_to_request_details()    -- Navigates to request manager details page
    delete_request() -- Deletes a specific request
    assign_reviewer_approver() -- Assign reviewer and approver to a request
"""
import time

from Web.AdminConsole.Components.panel import ModalPanel, DropDown
from Web.AdminConsole.Components.table import Table
from Web.Common.exceptions import CVWebAutomationException
from Web.Common.page_object import (
    WebAction,
    PageService
)


class RequestManager:
    """
     This class contains all the methods for action in request Manager page
    """

    create = None
    delete = None
    configure = None
    constants = None

    def __init__(self, admin_console):
        self.__admin_console = admin_console
        self.__table = Table(admin_console)
        self.__admin_console._load_properties(self)
        self.create = _CreateRequest(self.__admin_console)
        self.delete = _DeleteRequest(self.__admin_console)
        self.configure = _ConfigureRequest(self.__admin_console)
        self.constants = _RequestManagerConstants()

    @PageService()
    def search_for_request(self, request_name):
        """
        Searches for an request

            Args:
                request_name (str)  - request name to be searched for

            Returns True/False based on the presence of the request
        """
        __flag = False
        self.__table.apply_filter_over_column("Name", request_name)
        if self.__admin_console.check_if_entity_exists("link", request_name):
            __flag = True
        return __flag

    @WebAction()
    def click_request_action(self, request_name):
        """
        Clicks on an request action button
            Args:
                request_name (str)  - request name action to be clicked
        """
        self.__table.access_link(request_name)

    def select_request_by_name(self, request_name):
        """
               Select request by request_name
                   Args:
                       request_name (str)  - request name to be deleted
               """
        self.__table.access_link_by_column("Name", request_name)

    def get_status(self):
        """
        Get status of request
        :return: status
        """
        return self.__table.get_column_data(
            self.__admin_console.props['label.status']
        )


class _RequestManagerConstants:
    """
    __RequestManagerConstants are constants required for request operations
    """
    DELETE = "Delete"
    EXPORT = "Export"
    entities_list = [
        'Credit card number',
        'US Social Security number',
        'Email']
    approval_url_suffix = "webconsole/forms/?tab=1&filter=true"


class _DeleteRequest:
    """
        Delete a request from request manager
    """

    def __init__(self, admin_console):
        self._admin_console = admin_console
        self.__table = Table(admin_console)

    @PageService()
    def delete_request(self, request_name):
        """
        Delete a request
        :param request_name: Request Name
        """
        _request = RequestManager(self._admin_console)
        if _request.search_for_request(request_name):
            self.__table.access_context_action_item(
                request_name, self._admin_console.props['label.action.delete']
            )
            self._admin_console.click_button('Yes')
            self._admin_console.check_error_message()


class _CreateRequest:
    """
            Create a request using request manager
    """

    def __init__(self, admin_console):
        self._admin_console = admin_console
        self.__table = Table(admin_console)
        self.__panel = ModalPanel(admin_console)
        self.__dropdown = DropDown(admin_console)

    def _click_add_request(self):
        """
        Click add request
        """
        self._admin_console.select_hyperlink(
            self._admin_console.props['label.taskmanager.add']
        )

    @WebAction()
    def _select_delete_from_backup(self, enable_option=False):
        """
        Enable delete from backup
        :param enable_option: True to enable
        """
        if enable_option:
            self._admin_console.checkbox_select(
                self._admin_console.props['label.deleteFromBackup']
            )

    @WebAction()
    def _enable_redaction(self, enable_option=False):
        """
        Enable redaction
        :param enable_option: True to enable
        """
        if enable_option:
            self._admin_console.enable_toggle(0)

    @WebAction()
    def _enable_document_chaining(self, enable_option=False):
        """
        Enable document chaining
        :param enable_option: True to enable
        """
        if enable_option:
            self._admin_console.enable_toggle(1)

    @WebAction()
    def _select_request_type(self, request_type):
        """
        Select request type
        :param request_type: Request Type
        """
        if request_type == _RequestManagerConstants.DELETE:
            xpath = "//input[@type = 'radio' and ./../text()='{}']".format(
                _RequestManagerConstants.DELETE)
            self._admin_console.driver.find_element_by_xpath(xpath).click()

        elif request_type == _RequestManagerConstants.EXPORT:
            xpath = "//input[@type = 'radio' and ./../text()='{}']".format(
                _RequestManagerConstants.EXPORT)
            self._admin_console.driver.find_element_by_xpath(xpath).click()

    @WebAction()
    def _search_select_entity(self, entity_type_list):
        """
         Search and select an entity
         :param entity_type_list (list): list of entities to select
        """
        self.__dropdown.select_drop_down_values(0, entity_type_list)

    @WebAction()
    def _set_input_by_name(self, locator, value):
        """
        Type inputs
        Args:
            locator (str) - locator
            value(str) - value
            """
        self._admin_console.fill_form_by_id(locator, value)

    @PageService()
    def navigate_to_request_details(self, request_name):
        """
        Navigates to request manager details page

            Args:
                request_name (str)  - request name details to be navigated
                :raise If request is not found

        """
        _request = RequestManager(self._admin_console)
        if _request.search_for_request(request_name):
            _request.click_request_action(request_name)
            self.__table.access_action_item(
                request_name, self._admin_console.props['label.taskDetail.detail'])
            self._admin_console.wait_for_completion()
        else:
            raise CVWebAutomationException(
                "The request {0} is not present".format(request_name))

    @PageService()
    def add_request(self, request_name, requester, entity_type, entity, request_type):
        """
        Adds a request
            :param request_name    (str)   --  request name to be added
            :param requester    (str)  -- requester
            :param entity_type    (str)  -- entity type e.g. credit card
            :param entity    (str)  -- value of entity
            :param request_type    (str)  -- request type e.g. export or delete
            :return _flag (bool) -- Request creation fails or passes
            Raise:
                CVWebAutomationException if request addition failed

        """
        _flag = False
        _delete = _DeleteRequest(self._admin_console)
        _request = RequestManager(self._admin_console)
        if _request.search_for_request(request_name):
            _delete.delete_request(request_name)
        else:
            self._admin_console.log.info("Going to create new request")
        self._click_add_request()
        self._admin_console.wait_for_completion()
        self._set_input_by_name("taskname", request_name)
        self._select_request_type(request_type)
        self._select_delete_from_backup()
        self._enable_redaction()
        self._enable_document_chaining()
        self._set_input_by_name("userEmail", requester)
        self._search_select_entity([entity_type])
        self._set_input_by_name('entity_email', entity)
        self._admin_console.wait_for_completion()
        self.__panel.submit()
        self._admin_console.wait_for_completion()
        self._admin_console.check_error_message()
        if _request.search_for_request(request_name):
            _status = _request.get_status()
            if 'Request created' in _status:
                _flag = True
        return _flag


class _ConfigureRequest:
    """
        Configure project, reviewer and approver association
    """

    def __init__(self, admin_console):
        self._admin_console = admin_console
        self.__table = Table(admin_console)
        self.__panel = ModalPanel(admin_console)
        self.__dropdown = DropDown(admin_console)

    @WebAction()
    def _set_reviewer(self, reviewer):
        """Set reviewer"""
        self._admin_console.driver.find_element_by_xpath(
            "//h4[contains(text(),'Assign reviewers')]/"
            "following::label/following::input").send_keys(reviewer)

    @WebAction()
    def _set_approver(self, approver):
        """
        Set approvers
        """
        self._admin_console.driver.find_element_by_xpath(
            "//h4[contains(text(),'Assign approvers')]/"
            "following::label/following::input").send_keys(approver)

    @WebAction()
    def _select_approver(self):
        """
        Select approver
        """
        self._admin_console.select_hyperlink(
            self._admin_console.props['label.taskmanager.approvers']
        )

    @WebAction()
    def _select_reviewer(self):
        """
        Select reviewer
        """
        self._admin_console.select_hyperlink(
            self._admin_console.props['label.taskmanager.reviewers']
        )

    @WebAction()
    def _select_from_suggestion(self, wait_to_load=90):
        """
        Select from suggestion
        wait_wait_to_load : as suggestions needs LDAP query, setting wait_to_load to handle delays
        """
        time.sleep(wait_to_load)
        self._admin_console.driver.find_element_by_xpath(
            "//ul[@class='select2-results']/li").click()

    @WebAction()
    def _click_request_manager(self):
        """
        Click request manager link
        """
        self._admin_console.driver.find_element_by_xpath(
            "//div[contains(text(),'Request manager')]").click()

    @PageService()
    def assign_reviewer_approver(self, request_name, approver, reviewer, project_name):
        """
                Searches for an request
                    Args:
                        :param request_name (str)  - request name to be searched for
                        :param approver (str)  - approver
                        :param reviewer (str)  - reviewer
                        :param project_name    (str)  -- project name to be selected
                    Returns True/False based on the presence of the request
                """
        _flag = False
        _request = RequestManager(self._admin_console)
        _request.select_request_by_name(request_name)
        self._admin_console.wait_for_completion()
        self.__dropdown.select_drop_down_values(0, [project_name])
        self._admin_console.log.info("Selecting Project [%s]", project_name)
        self._select_reviewer()
        self._set_reviewer(reviewer)
        self._admin_console.wait_for_completion()
        self._select_from_suggestion()
        self._select_approver()
        self._set_approver(approver)
        self._admin_console.wait_for_completion()
        self._select_from_suggestion()
        self._admin_console.wait_for_completion()
        self.__panel.submit()
        self._admin_console.wait_for_completion()
        self._click_request_manager()
        self._admin_console.check_error_message()
        if _request.search_for_request(request_name):
            _status = _request.get_status()
            if 'Request configured' in _status:
                _flag = True
        return _flag
