# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the methods that can be done of the Inventory Manager
details page.


Classes:

    InventoryManagerDetails() ---> InventoryManager() ---> GovernanceApps() ---> object()


InventoryManagerDetails  --  This class contains all the methods for action in
    Inventory Manager details page and is inherited by other classes to perform
    GDPR related actions

    Functions:

    _select_add_asset_type()          --  Selects the asset type to be added
    get_assets()     -       -- Returns all the assets from the inventory details page
    add_asset_name_server()      -- Adds name server asset from Inventory details page
    wait_for_asset_status_completion()   -- Waits for the asset scan status completion
    select_overview()                    -- Clicks on the Overview link
    select_details()                    -- Clicks on the Details link
    __convert_list_to_string()         -- Convert list to a comma separated string
    __check_schedule()                 -- Checks for schedule
    check_if_schedule_is_assigned()    -- Checks if there is any existing schedule associated
    remove_schedule()                  -- Removes any schedule associated
    select_add_or_edit_schedule()      -- Selects either add or edit schedule
    add_edit_schedule()                -- Add or Edit the schedule
    __one_time_schedule()              -- Adds a one time schedule
    __daily_schedule()                 -- Adds daily schedule
    __weekly_schedule()                -- Adds a weekly schedule
    __monthly_schedule()               -- Adds a monthly schedule
    get_last_collection_time()         -- Returns the last collection time for a given asset
    __get_exceptions_toggle_check_element() -- Gets Exceptions Toggle Check WebElement
    __get_exceptions_toggle_element()  -- Gets Exceptions Toggle WebElement
    is_exceptions_toggle_enabled()     -- Checks if exceptions toggle is enabled
    enable_exceptions_toggle()         -- Enables the exceptions toggle bar if disabled
    disable_exceptions_toggle()        -- Disables the exceptions toggle bar if enabled
    get_schedule_frequency()           -- Returns the current selected schedule frequency
    add_exception()                    -- Adds an exception to the current schedule

"""

import re
import time
from Web.Common.page_object import WebAction, PageService
from Web.AdminConsole.Components.panel import PanelInfo, DropDown
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.GovernanceAppsPages.InventoryManager import InventoryManager

class InventoryManagerDetails(InventoryManager):
    """
     This class contains all the methods for action in Inventory Manager details page
    """

    def __init__(self, admin_console):
        """
        Args:
            admin_console (AdminConsole): adminconsole base object
        """
        super().__init__(admin_console)
        self.__admin_console = admin_console
        self.driver = self.__admin_console.driver
        self.__admin_console._load_properties(self)
        self.log = self.__admin_console.log
        self.__panelinfo = PanelInfo(self.__admin_console)
        self.__dropdown = DropDown(self.__admin_console)
        self.__modal_dialog = ModalDialog(self.__admin_console)
        self.__table = Table(self.__admin_console)
        self.panel_details = None
        self.schedule_options = None

    @WebAction()
    def _select_add_asset_type(self, asset_type):
        """
            Selects the asset type to be added

            Args:
                asset_type(str)  - Type of the asset to be selected
                Values:
                    "Name server",
                    "File server"

            Raise:
                Exception if invalid asset type provided
        """
        
        self.log.info('Clicking on Add button')
        self.__panelinfo.open_hyperlink_on_tile(self.__admin_console.props['label.assets.add'])
        self.__admin_console.wait_for_completion()
        self.log.info('Selecting %s asset option' % asset_type)
        if re.search("Name server", str(asset_type), re.IGNORECASE):
            self.__panelinfo.open_hyperlink_on_tile(
                self.__admin_console.props['label.assets.nameserver'])
        elif re.search("File server", str(asset_type), re.IGNORECASE):
            self.__panelinfo.open_hyperlink_on_tile(
                self.__admin_console.props['label.assets.fileserver'])
        else:
            raise Exception("Invalid asset type: %s" % asset_type)
        self.__admin_console.wait_for_completion()

    @WebAction()
    def get_assets(self):
        """
        Returns all the list of assets

            Return:
                List of assets

        """
        assets_list = []
        rows = self.driver.find_elements_by_xpath(
            "//div[@class='ui-grid-canvas']/div")
        for row_id in range(1, len(rows) + 1):
            assets_list.append(str(self.driver.find_element_by_xpath
                                   ("//div[@class='ui-grid-canvas']\
                                   /div[%d]/div/div[1]/a" % row_id).text))
        self.log.info('List of asset names obtained are: %s'
                      % assets_list)
        return assets_list

    @WebAction()
    def add_asset_name_server(self, name_server):
        """
        Adds name server asset

            Args:
                name_server (str)  - Name of the domain server

            Raise:
                Exception if inventory creation failed
        """
        self._select_add_asset_type(self.__admin_console.props['label.assets.nameserver'])
        self.log.info("Selecting name servers")
        self.driver.find_element_by_xpath(
            "//form[@name='addNameServerAssetForm']//*/span/button").click()
        self.__admin_console.wait_for_completion()
        total_name_servers = len(self.driver.find_elements_by_xpath(
            '//*[@id="nameServer"]//div[@class="checkBoxContainer"]/div'))
        name_server_obtained = False
        for name_server_id in range(1, total_name_servers + 1):
            name_server_found = self.driver.find_element_by_xpath(
                '//*[@id="nameServer"]//div[@class="checkBoxContainer"]\
                /div[%d]/div/label/span' %name_server_id).text
            if re.search(name_server, name_server_found, re.IGNORECASE):
                name_server_obtained = True
                break
        if name_server_obtained:
            self.log.info("Selecting the asset: %s" % name_server)
            self.driver.find_element_by_xpath(
                '//*[@id="nameServer"]//div[@class="checkBoxContainer"]\
                /div[%d]/div/label/span' %name_server_id).click()
            self.__admin_console.wait_for_completion()
            self.log.info("Selecting name servers")
            self.driver.find_element_by_xpath(
                "//form[@name='addNameServerAssetForm']//*/span/button").click()
            self.__admin_console.wait_for_completion()
            self.log.info("Clicking on Save after the selection of the asset")
            self.__admin_console.submit_form()
            if self.__admin_console.check_if_entity_exists(
                    "xpath", "//*[@class='serverMessage error']"):
                raise Exception(self.driver.find_element_by_xpath(
                    "//*[@class='serverMessage error']").text)
        else:
            raise Exception("Nameserver: '%s' not found" % name_server)

    @WebAction()
    def wait_for_asset_status_completion(self, asset_name, timeout=20):
        """
        Waits for the asset scan status completion

            Args:
                asset_name (str)  - Name of the asset
                timeout     (int)   --  minutes
                    default: 20

            Returns:
                bool    -   boolean specifying if the asset scan had finished or not
                    True    -   if the asset scan had finished successfully

                    False   -   if the asset scan was not completed within the timeout

        """
        status = False
        start_time = int(time.time())
        current_time = start_time
        completion_time = start_time + timeout * 60

        while completion_time > current_time:
            self.log.info("Refreshing the page")
            self.driver.refresh()
            self.__admin_console.wait_for_completion()
            self.log.info('Obtaining the asset status of: %s' % asset_name)
            assets = self.__table.get_column_data(self.__admin_console.props['label.name'])
            assets_status = self.__table.get_column_data(
                self.__admin_console.props['label.status'])
            current_status = dict(zip(assets, assets_status))[asset_name]
            self.log.info('Asset status obtained is: %s' % current_status)
            if re.search(self.__admin_console.props['label.taskDetail.status.Completed'],\
                          current_status, re.IGNORECASE):
                status = True
                break
            time.sleep(30)
            current_time = int(time.time())
        return status

    @WebAction()
    def select_overview(self):
        """
        Clicks on the Overview link
        """
        self.driver.find_element_by_xpath(
            "//a[contains(@class,'bc-item') and text()='%s']"\
            %self.__admin_console.props['label.overview']).click()
        self.__admin_console.wait_for_completion()

    @WebAction()
    def select_details(self):
        """
        Clicks on the Details link
        """
        self.driver.find_element_by_xpath(
            "//a[contains(@class,'bc-item') and text()='%s']"\
            %self.__admin_console.props['label.details']).click()
        self.__admin_console.wait_for_completion()

    @staticmethod
    def __convert_list_to_string(values_list):
        '''
        Convert list to a comma separated string
            Args:
                values_list (List)  - List of string values to be converted
            Example:
                values_list = ['Second','Third']
                returns:
                    'Second,Third'
        '''
        return_text = ''
        for value in values_list:
            return_text += f",{value}"
        return return_text[1:]

    @PageService()
    def __check_schedule(self):
        '''
        Checks for schedule
        Returns True if present, False if not
        '''
        string_pattern = f" at {self.schedule_options['hours']}:{self.schedule_options['mins']} "+\
        f"{self.schedule_options['session']}"
        if self.schedule_options['frequency'] == 'One time':
            string_pattern = f"{self.schedule_options['month'][:3]} "+\
            f"{self.schedule_options['date']} {self.schedule_options['year']}"+string_pattern
        elif self.schedule_options['frequency'] == 'Daily':
            if int(self.schedule_options['repeat']) == 1:
                string_pattern = "Every day"+string_pattern
            else:
                string_pattern = f"Every {self.schedule_options['repeat']} days"+string_pattern
        elif self.schedule_options['frequency'] == 'Weekly':
            days_text = self.__convert_list_to_string(self.schedule_options['days'])
            if int(self.schedule_options['repeat']) == 1:
                string_pattern = f"Every week on {days_text}"+string_pattern
            else:
                string_pattern = f"Every {self.schedule_options['repeat']} weeks on {days_text}"+\
                string_pattern
        elif self.schedule_options['frequency'] == 'Monthly':
            if int(self.schedule_options['repeat']) == 1:
                temp_str = "Every month"
            else:
                temp_str = f"Every {self.schedule_options['repeat']} months"
            string_pattern = temp_str+f" on day {self.schedule_options['onday']}"+string_pattern
        if 'exceptionweeks' in self.schedule_options:
            string_pattern += '\n.*\nexcept\nOn '
            string_pattern += self.__convert_list_to_string(
                self.schedule_options['exceptionweeks'])
            string_pattern += ' '
            string_pattern += self.__convert_list_to_string(
                self.schedule_options['exceptiondays'])
        elif 'exceptiondays' in self.schedule_options:
            string_pattern += '\n.*\nexcept\nOn days: '
            string_pattern += self.__convert_list_to_string(
                self.schedule_options['exceptiondays'])
        self.log.info(f"Matching with string pattern: '{string_pattern}'")
        if re.search(string_pattern,\
                     self.panel_details[self.__admin_console.props['label.schedule']],\
                     re.IGNORECASE):
            return True
        return False

    @PageService()
    def check_if_schedule_is_assigned(self, default_schedule=False, schedule_options=None):
        """
        Checks if there is any existing schedule associated
            Returns: True if schedule is assigned
            Args:
                default (Bool)  - To check for default schedule
                schedule_options (dict)  -- schedule options to check for
                Usage:
                    * Sample dict for one time schedule
                        {
                            'frequency': 'One time',
                            'year': '2020',
                            'month': 'January',
                            'date': '17',
                            'hours': '03',
                            'mins': '45',
                            'session': 'PM'
                        }
                    * Sample dict for daily schedule
                        {
                            'frequency': 'Daily',
                            'repeat': '1',
                            'hours': '03',
                            'mins': '45',
                            'session': 'PM'
                        }
                    * Sample dict for weekly schedule
                        {
                            'frequency': 'Weekly',
                            'repeat': '1',
                            'days': ['Monday','Wednesday'],
                            'hours': '03',
                            'mins': '45',
                            'session': 'PM'
                        }
                    * Sample dict for monthly schedule
                        {
                            'frequency': 'Monthly',
                            'repeat': '1',
                            'onday': '31',
                            'hours': '03',
                            'mins': '45',
                            'session': 'PM'
                        }
                        or
                        {
                            'frequency': 'Monthly',
                            'repeat': '1',
                            'customweeks': 'First',
                            'customdays': 'Saturday',
                            'hours': '03',
                            'mins': '45',
                            'session': 'PM'
                        }
        """
        self.panel_details = self.__panelinfo.get_details()
        if schedule_options:
            return self.__check_schedule()
        if default_schedule:
            if not re.search('Every day at 12:00 AM',\
                             self.panel_details[self.__admin_console.props['label.schedule']],\
                             re.IGNORECASE):
                return False
        else:
            if re.search(self.__admin_console.props['info.notAssigned'],\
                         self.panel_details[self.__admin_console.props['label.schedule']]):
                return False
        return True

    @PageService()
    def remove_schedule(self):
        """
        Removes any schedule associated
        """
        self.__panelinfo.edit_tile_entity(self.__admin_console.props['label.schedule'])
        self.__admin_console.click_button(self.__admin_console.props['label.delete'])
        self.__modal_dialog.click_submit()
        self.__admin_console.check_error_message()

    @PageService()
    def select_add_or_edit_schedule(self):
        """
        Selects either add or edit schedule based on whatever is displayed
        """
        if self.__panelinfo.check_if_hyperlink_exists_on_tile(
                self.__admin_console.props['label.editLaunchesModal']):
            self.__panelinfo.edit_tile_entity(self.__admin_console.props['label.schedule'])
        else:
            self.__panelinfo.open_hyperlink_on_tile(self.__admin_console.props['label.add'])

    @PageService()
    def add_edit_schedule(self, schedule_options):
        """
        Add or Edit the schedule
            Args:
                schedule_options (dict)  -- schedule options to create or edit a schedule
                Usage:
                    * Sample dict for one time schedule
                        {
                            'frequency': 'One time',
                            'year': '2020',
                            'month': 'January',
                            'date': '17',
                            'hours': '03',
                            'mins': '45',
                            'session': 'PM'
                        }
                    * Sample dict for daily schedule
                        {
                            'frequency': 'Daily',
                            'repeat': '1',
                            'hours': '03',
                            'mins': '45',
                            'session': 'PM'
                        }
                    * Sample dict for weekly schedule
                        {
                            'frequency': 'Weekly',
                            'repeat': '1',
                            'days': ['Monday','Wednesday'],
                            'hours': '03',
                            'mins': '45',
                            'session': 'PM'
                        }
                    * Sample dict for monthly schedule
                        {
                            'frequency': 'Monthly',
                            'repeat': '1',
                            'onday': '31',
                            'hours': '03',
                            'mins': '45',
                            'session': 'PM'
                        }
                        or
                        {
                            'frequency': 'Monthly',
                            'repeat': '1',
                            'customweeks': 'First',
                            'customdays': 'Saturday',
                            'hours': '03',
                            'mins': '45',
                            'session': 'PM'
                        }
                    * Sample dict for schedule exception options
                        {
                            'exceptiondays': ['1','22']
                        }
                        or
                        {
                            'exceptionweeks': ['First','Last'],
                            'exceptiondays': ['Monday','Saturday']
                        }
        """
        self.schedule_options = schedule_options
        self.select_add_or_edit_schedule()
        if self.schedule_options['frequency'] == 'One time':
            self.__one_time_schedule()
        if self.schedule_options['frequency'] == 'Daily':
            self.__daily_schedule()
        elif self.schedule_options['frequency'] == 'Weekly':
            self.__weekly_schedule()
        elif self.schedule_options['frequency'] == 'Monthly':
            self.__monthly_schedule()
        if 'exceptiondays' in self.schedule_options:
            self.add_exception(self.schedule_options)
        else:
            self.__modal_dialog.click_submit()
            self.__admin_console.check_error_message()

    @PageService()
    def __one_time_schedule(self):
        '''
        Adds a one time schedule
            Args:
                schedule_options (dict)  -- schedule options to create or edit a one time schedule
                Usage:
                    * Sample dict for schedule options
                        {
                            'year': '2020',
                            'month': 'January',
                            'date': '17',
                            'hours': '03',
                            'mins': '45',
                            'session': 'PM'
                        }
        '''
        self.__dropdown.select_drop_down_values(0, [self.__admin_console.props['option.oneTime']])
        self.__admin_console.date_picker(self.schedule_options)

    @PageService()
    def __daily_schedule(self):
        '''
        Adds daily schedule
            Args:
                schedule_options (dict)  -- schedule options to create or edit a one time schedule
                Usage:
                    * Sample dict for schedule options
                        {
                            'hours': '03',
                            'mins': '45',
                            'session': 'PM',
                            'repeat': '1',
                        }
        '''
        self.__dropdown.select_drop_down_values(0, [self.__admin_console.props['option.daily']])
        self.__admin_console.date_picker(self.schedule_options, pick_time_only=True)
        self.__admin_console.fill_form_by_id('repeatFrequency', self.schedule_options['repeat'])

    @PageService()
    def __weekly_schedule(self):
        '''
        Adds a weekly schedule
            Args:
                schedule_options (dict)  -- schedule options to create or edit a weekly schedule
                Usage:
                    * Sample dict for schedule options
                        {
                            'repeat': '1',
                            'days': ['Monday','Wednesday'],
                            'hours': '03',
                            'mins': '45',
                            'session': 'PM'
                        }
        '''
        self.__dropdown.select_drop_down_values(0, [self.__admin_console.props['option.weekly']])
        self.__admin_console.date_picker(self.schedule_options, pick_time_only=True)
        self.__dropdown.select_drop_down_values(1, self.schedule_options['days'])
        self.__admin_console.fill_form_by_id('repeatFrequency', self.schedule_options['repeat'])

    @PageService()
    def __monthly_schedule(self):
        '''
        Adds a monthly schedule
            Args:
                schedule_options (dict)  -- schedule options to create or edit a monthly schedule
                Usage:
                    * Sample dict for schedule options
                        {
                            'repeat': '1',
                            'onday': '31',
                            'hours': '03',
                            'mins': '45',
                            'session': 'PM'
                        }
                        or
                        {
                            'repeat': '1',
                            'customweeks': 'First',
                            'customdays': 'Saturday',
                            'hours': '03',
                            'mins': '45',
                            'session': 'PM'
                        }
        '''
        self.__dropdown.select_drop_down_values(0, [self.__admin_console.props['option.monthly']])
        self.__admin_console.date_picker(self.schedule_options, pick_time_only=True)
        if 'onday' in self.schedule_options:
            self.__admin_console.fill_form_by_name(
                'monthlyScheduleDayOfMonth', self.schedule_options['onday'])
        else:
            self.__dropdown.select_drop_down_values(1, [self.schedule_options['customweeks']])
            self.__dropdown.select_drop_down_values(2, [self.schedule_options['customdays']])
        self.__admin_console.fill_form_by_id('repeatFrequency', self.schedule_options['repeat'])

    @PageService()
    def get_last_collection_time(self, asset_name):
        """
        Returns the last collection time(str) for a given asset
        """
        assets = self.__table.get_column_data(self.__admin_console.props['label.name'])
        last_completion_times = self.__table.get_column_data(
            self.__admin_console.props['label.lastActive'])
        return dict(zip(assets, last_completion_times))[asset_name]

    @WebAction()
    def __get_exceptions_toggle_check_element(self):
        """ Gets Exceptions Toggle Check WebElement

        Returns : toggle check WebElement

        """
        return self.driver.find_element_by_xpath(
            "//*[@id='addScheduleForm']//div[contains(@class,'cv-accordion-header')]")

    @WebAction()
    def __get_exceptions_toggle_element(self):
        """ Gets Exceptions Toggle WebElement

        Returns : toggle WebElement

        """
        return self.driver.find_element_by_xpath(
            "//*[@id='addScheduleForm']//div[contains(@class,'cv-material-toggle cv-toggle')]")

    @WebAction()
    def is_exceptions_toggle_enabled(self):
        """ Checks if exceptions toggle is enabled

        Returns (bool) : True if toggle is enabled

        """
        element = self.__get_exceptions_toggle_check_element()
        if 'expanded' in element.get_attribute('class'):
            return False
        return True

    @PageService()
    def enable_exceptions_toggle(self):
        """ Enables the exceptions toggle bar if disabled """
        if not self.is_exceptions_toggle_enabled():
            self.__get_exceptions_toggle_element().click()
            self.__admin_console.wait_for_completion()

    @PageService()
    def disable_exceptions_toggle(self):
        """ Disables the exceptions toggle bar if enabled """
        if self.is_exceptions_toggle_enabled():
            self.__get_exceptions_toggle_element().click()
            self.__admin_console.wait_for_completion()

    @WebAction()
    def get_schedule_frequency(self):
        """Returns the current selected schedule frequency"""
        return self.driver.find_element_by_xpath(
            '//*[@id="scheduleFrequency"]//div[contains(@class,"buttonLabel")]').text

    @PageService()
    def add_exception(self, schedule_options):
        '''
        Adds an exception to the current schedule
            Args:
                schedule_options (dict)  -- schedule options to create or edit a weekly schedule
                Usage:
                    * Sample dict for schedule exception options
                        {
                            'exceptiondays': ['1','22']
                        }
                        or
                        {
                            'exceptionweeks': ['First','Last'],
                            'exceptiondays': ['Monday','Saturday']
                        }
        '''
        self.schedule_options = schedule_options
        self.enable_exceptions_toggle()
        if self.get_schedule_frequency() == self.__admin_console.props['option.daily']:
            dropdown_index = 1
        elif self.get_schedule_frequency() == self.__admin_console.props['option.weekly']:
            dropdown_index = 2
        else: #Monthly
            dropdown_index = 3
        if 'exceptionweeks' in self.schedule_options:
            self.__admin_console.select_radio('weekInMonth')
            self.__dropdown.select_drop_down_values(
                dropdown_index, self.schedule_options['exceptionweeks'])
            self.__dropdown.select_drop_down_values(
                dropdown_index+1, self.schedule_options['exceptiondays'])
        else:
            self.__admin_console.select_radio('dayInMonth')
            self.__dropdown.select_drop_down_values(
                dropdown_index, self.schedule_options['exceptiondays'])
        self.__admin_console.click_button(self.__admin_console.props['label.add'])
        self.__modal_dialog.click_submit()
        self.__admin_console.check_error_message()
