# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the methods that can be done of the Inventory Manager page.


Classes:

    InventoryManager() ---> GovernanceApps() ---> object()


InventoryManager  --  This class contains all the methods for action in Inventory
    Manager page and is inherited by other classes to perform GDPR related actions

    Functions:

    add_inventory()          --  adds an inventory
    search_for_inventory()    -- Searches for an inventory
    navigate_to_inventory_details()    -- Navigates to inventory manager details page
    delete_inventory() -- Deletes a specific inventory
"""
from Web.AdminConsole.Components.table import Table
from Web.Common.page_object import (
    WebAction,
    PageService
)
from Web.AdminConsole.GovernanceAppsPages.GovernanceApps import GovernanceApps
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.AdminConsole.Components.panel import DropDown

class InventoryManager(GovernanceApps):
    """
     This class contains all the methods for action in Inventory Manager page
    """

    def __init__(self, admin_console):
        """
        Args:
            admin_console (AdminConsole): adminconsole base object
        """
        super().__init__(admin_console)
        self.__admin_console = admin_console
        self.driver = self.__admin_console.driver
        self.__admin_console._load_properties(self)
        self.log = self.__admin_console.log
        self.__table = Table(self.__admin_console)
        self.__modal_dialog = ModalDialog(self.__admin_console)
        self.__dropdown = DropDown(self.__admin_console)

    @PageService()
    def add_inventory(self, inventory_name, index_server):
        """
        Adds an inventory

            :param inventory_name    (str)   --  Inventory name to be added
            :param index_server    (str)  -- Index Server to be selected

            Raise:
                Exception if inventory addition failed
        """
        self.__admin_console.access_tile("inventoryAdd")
        self.__admin_console.wait_for_completion()
        self.__admin_console.fill_form_by_id("name", inventory_name)
        self.__dropdown.select_drop_down_values(0, [index_server])
        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @WebAction()
    def search_for_inventory(self, inventory_name):
        """
        Searches for an inventory

            Args:
                inventory_name (str)  - Inventory name to be searched for

            Returns True/False based on the presence of the Inventory
        """
        return self.__table.is_entity_present_in_column(
            self.__admin_console.props['label.name'], inventory_name)

    @PageService()
    def navigate_to_inventory_details(self, inventory_name):
        """
        Navigates to inventory manager details page

            Args:
                inventory_name (str)  - Inventory name details to be navigated

        """
        self.__table.access_context_action_item(
            inventory_name, self.__admin_console.props['label.details'])

    @PageService()
    def delete_inventory(self, inventory_name):
        """
        Deletes an inventory

            Args:
                inventory_name (str)  - Inventory name to be deleted

            Raise:
                Exception if inventory deletion failed
        """
        self.__table.access_context_action_item(
            inventory_name, self.__admin_console.props['label.delete'])
        self.__modal_dialog.click_submit()
        self.__admin_console.check_error_message()
