# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the methods that can be done of the File storage
optimization page.

FileStorageOptimization  --  This class contains all the methods for action in
    File Storage Optimization page and is inherited by other classes to
    ...

    Functions:
    add_client()                            -- Adds an FSO client
    select_fso_review_tab()                 -- Select review tab in FSO review page
    get_duplicate_file_count()              -- Get Duplicate file count from FSO duplicate dashboard
    get_duplicate_file_size()               -- Get Duplicate File size from FSO duplicate dashboard
    select_client()                         -- select fso client or associated data source
    select_data_source()                    -- select fso data source from client quick view
    fso_dashboard_entity_count()            -- get fso dashboard entity count
    select_fso_dashboard()                  -- Select given FSO dashboard
    get_quickview_entity_count()            -- Get count of entity present in quickview panel
    select_client_review_page()             -- Go to client review page
    get_file_security_dashboard_user_count() -- Get User count for a given permission in
                                                FSO security dashboard
    load_fso_dashboard()                    --  Load FSO dashboards
    fso_data_source_details_page()          -- Goto FSO datasource details page
    select_quickview_action()               -- Select Quick view action
    select_details_action()                 -- select details action
    select_delete_action()                    -- select delete action
    delete_fso_datasource()                 -- Delete fso datasource
    close_quickview_panel()                 -- close client quickview panel
    get_quickview_datasource_list()         -- Get data source list in quickview panel
    select_review_page_filter()              -- Click passed review page filter
    get_fso_time_info()                     -- Get FSO time info from UI
    get_fso_time_dict()                     -- Get fso time dict for passed filter.





"""
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import DropDown, PanelInfo
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.Common.page_object import WebAction, PageService
from Web.AdminConsole.GovernanceAppsPages.InventoryManagerDetails import InventoryManagerDetails


class FileStorageOptimization:
    """
     This class contains all the methods for action in File Storage Optimization page
    """

    def __init__(self, admin_console):
        self.__admin_console = admin_console
        self.__table = Table(self.__admin_console)
        self.__dropdown = DropDown(self.__admin_console)
        self.__panel_info = PanelInfo(self.__admin_console)
        self.__modal_dialog = ModalDialog(self.__admin_console)
        self.__admin_console._load_properties(self)
        self.__inventory_manager_details = InventoryManagerDetails(self.__admin_console)
        self.time_dict = {
            1: "0 to 1 Year",
            2: "1 to 2 Years",
            3: "2 to 3 Years",
            4: "3 to 4 Years",
            5: "4 to 5 Years",
            6: "5 Years+"
        }

    @PageService()
    def add_client(self, inventory_name, plan_name):
        """
        Adds a client
            Args:
                plan_name (str)  - Plan name to be selected
                inventory_name (str)  - Inventory name to be selected
        """
        self.__admin_console.access_tile("ADD_ANALYTICS_CLIENT")
        self.__dropdown.select_drop_down_values(0, [plan_name])
        self.__dropdown.select_drop_down_values(1, [inventory_name])
        self.__admin_console.button_next()
        self.__admin_console.check_error_message()

    @PageService()
    def select_fso_review_tab(self):
        """
        Select Review tab in FSO review page
        """
        self.__admin_console.driver.find_element_by_xpath(
            "//a[contains(@class,'nav-link') and text()='%s']"
            % self.__admin_console.props['label.review']).click()
        self.__admin_console.wait_for_completion()

    @WebAction()
    def select_review_page_filter(self, filter_name):
        """
        Select filter dropdown
            Args:
                filter_name (str): Name of filter
        """
        self.__admin_console.driver.find_element_by_xpath(
            f"//cvfilterandsearch//span[text() = '{filter_name}']"
        ).click()

    @WebAction()
    def get_fso_time_dict(self, filter_name):
        """
        Get (Access/Created/Modified) Time data for FSO data source
            Args:
                filter_name (str): Access Time/Modified Time/Created Time
            Returns: temp_dict (dict)  {
                            1:"66 k",2:"5 k",3:"11 k",4:"452 k",5:"56 k",6:"12 k"
                            }
        """
        facet_elem = self.__admin_console.driver.find_element_by_xpath(
            f"//cvfilterandsearch//label[text()= '{filter_name}']/ancestor::cvfacet"
        )
        temp_dict = {}
        for key, value in self.time_dict.items():
            temp_dict[key] = facet_elem.find_element_by_xpath(
                f"//cvfacet//span[text() = '{value}']/following::span[1]"
            ).text.strip()[1:-1]
        return temp_dict

    @WebAction()
    def get_duplicate_file_count(self):
        """
        Get duplicate file count from FSO duplicate dashboard
        Returns:
            (int): Duplicate file count
        """
        xpath = '//cvhits//div[text()="Total Duplicate Files"]/following::p[1]'
        return self.__admin_console.driver.find_element_by_xpath(xpath).text

    @WebAction()
    def get_duplicate_file_size(self):
        """
        Get Duplicate file size from FSO Duplicate dashboard
        Returns:
            (str): Duplicate file size in format ('10 KB')
        """
        xpath = \
            "//div[@id='otherInfo']//div[text()='Size of Duplicate Files']/following::p[1]"
        return self.__admin_console.driver.find_element_by_xpath(xpath).text

    @WebAction()
    def check_if_client_exists(self, client_name):
        """
        Searches for a client
            Args:
                client_name (str)  - Client name to be searched for
            Returns:
                 (bool) True/False based on the presence of the Client
        """
        return self.__table.is_entity_present_in_column(
            self.__admin_console.props['label.name'],
            client_name)

    @WebAction()
    def check_if_datasource_exists(self, datasource_name):
        """
        Searches for FSO data source
            Args:
                datasource_name (str): Name of FSO data source
            Returns:
                (bool) True/False based on presence of data source
        """
        return self.__table.is_entity_present_in_column(
            self.__admin_console.props['label.datasource'],
            datasource_name)

    @WebAction()
    def select_client(self, client_name):
        """
        Selects FSO Client or associated datasource
            Args:
                client_name (str)  - Client name to be selected
        """
        self.__table.access_link(client_name)

    @PageService()
    def select_data_source(self, data_source_name):
        """
        Selects a data source from client quick view
        Args:
            data_source_name (str): Name of DataSource to be selected
        """
        self.__panel_info.open_hyperlink_on_tile(data_source_name)
        self.__admin_console.wait_for_completion()

    @WebAction()
    def fso_dashboard_entity_count(self, entity_name):
        """
        Get FSO Dashboard entity count
        Args:
            entity_name (str): Entity name whose count to be fetched
        Returns:
            (str) Count of Entity in FSO dashboard
        """
        return self.__admin_console.driver.find_element_by_xpath(
            f'//*[@id="otherInfo"]//div[text()="{entity_name}"]/following-sibling::p'
        ).text

    @WebAction()
    def _click_dashboard_dropdown(self):
        """
        Click dashboard drop down in FSO report page
        """
        x_path = "//div[@id='customDatasourceReport']//a[@data-toggle='dropdown']"
        elem = self.__admin_console.driver.find_element_by_xpath(x_path)
        if elem.get_attribute('aria-expanded').__eq__('false'):
            elem.click()

    @PageService()
    def select_fso_dashboard(self, dashboard_name):
        """
        Select passed dashboard for FSO client
        Args:
             dashboard_name (str): Name of FSO Dashboard
        """
        base_xp = "//div[@id='customDatasourceReport']"
        self._click_dashboard_dropdown()
        self.__admin_console.driver.find_element_by_xpath(
            base_xp + f"//a[text()='{dashboard_name}']"
        ).click()
        self.__admin_console.wait_for_completion()
        self.__admin_console.check_error_message()

    @WebAction()
    def get_quickview_entity_count(self, entity_name):
        """
        Obtains the entity count  from client quickview pane
        Args:
            entity_name (str): Quickview pane entity name
        Returns:
            count (int) : Entity count
        """
        count = int(self.__admin_console.driver.find_element_by_xpath(
            f"//cv-entity-pane//span[text()='{entity_name}']/preceding::span[1]").text)
        return count

    @PageService()
    def select_client_review_page(self, client_name, data_source_name):
        """
        Select the review tab for given FSO client
            Args:
                client_name (str): FSO client name
                data_source_name (str): FSO data source name
        """
        self.select_quickview_action(client_name)
        self.select_data_source(data_source_name)
        self.select_fso_review_tab()

    @WebAction()
    def get_file_security_dashboard_user_count(self, permission_type):
        """
        Get the user count for a particular permissions in FSO
        Security dashboard
        Args:
            permission_type (str): Type of permission
        Return:
            (int) -- no of user having mentioned permission
        """
        base_xp = f"//label[text()='{permission_type}']/following::span[1]"
        count = self.__admin_console.driver.find_elements_by_xpath(base_xp)[1].text.strip()[1:-1]
        if str(count).__eq__(''):
            count = '0'
        return int(count)

    @PageService()
    def load_fso_dashboard(self):
        """
        Load FSO Dashboards
        """
        self.select_fso_dashboard(self.__admin_console.props['reports.treeSize'])
        self.__admin_console.check_error_message()
        self.select_fso_dashboard(self.__admin_console.props['reports.fileDuplicate'])
        self.__admin_console.check_error_message()
        self.select_fso_dashboard(self.__admin_console.props['reports.fileSecurity'])
        self.__admin_console.check_error_message()
        self.select_fso_dashboard(self.__admin_console.props['reports.sizeDistribution'])
        self.__admin_console.check_error_message()
        self.select_fso_dashboard(self.__admin_console.props['reports.fileOwnership'])
        self.__admin_console.check_error_message()

    @PageService()
    def fso_data_source_details_page(self, client_name, data_source_name):
        """
        Go to fso data source details page
        Args:
            client_name (str): Name of FSO client
            data_source_name (str): Name of FSO data source
        """
        self.select_details_action(client_name)
        self.select_details_action(data_source_name)

    @WebAction()
    def select_quickview_action(self, client_name):
        """
        Select Quick View for given FSO Client
            Args:
                client_name (str):- FSO client name
        """
        self.__table.access_action_item(
            client_name,
            self.__admin_console.props['label.quickView'])

    @WebAction()
    def select_details_action(self, client_name):
        """
        Select details action for given FSO Client
            Args:
                client_name (str):- FSO client name
        """
        self.__table.access_action_item(
            client_name,
            self.__admin_console.props['label.details'])

    @WebAction()
    def select_delete_action(self, client_name):
        """
        Select delete action for given FSO Client
            Args:
                client_name (str):- FSO client name
        """
        self.__table.access_action_item(
            client_name,
            self.__admin_console.props['label.delete'])

    @PageService()
    def delete_fso_datasource(self, datasource_name):
        """
        Delete given FSO data source
        Args:
            datasource_name (str): Name of FSO datasource
        """
        self.select_delete_action(datasource_name)
        self.__modal_dialog.click_submit()
        self.__admin_console.wait_for_completion()
        self.__admin_console.check_error_message()

    @WebAction()
    def close_quickview_panel(self):
        """
        Closes quick view panel for FSO client
        """
        self.__admin_console.click_button(self.__admin_console.props['action.close'])

    @WebAction()
    def get_quickview_datasource_list(self):
        """
        Get list of FSO data sources present in FSO client
        Quick view panel
        """
        return self.__table.get_column_data(self.__admin_console.props['label.datasource'])

    @PageService()
    def get_fso_time_info(self):
        """
        Get CreatedTime/AccessTime/Modified Time data for FSO datsource
            Returns: temp_dict (dict) {
                                "CreatedTime": {1:"66 k",2:"5 k",3:"11 k",4:"452 k",5:"56 k",6:"12 k"},
                                "AccessTime": {1:"66 k",2:"5 k",3:"11 k",4:"452 k",5:"56 k",6:"12 k"},
                                ModifiedTime: {1:"66 k",2:"5 k",3:"11 k",4:"452 k",5:"56 k",6:"12 k"}
                            }
        """
        temp_dict = dict()
        self.select_fso_dashboard(self.__admin_console.props['reports.sizeDistribution'])
        self.__admin_console.check_error_message()
        self.select_fso_review_tab()
        self.select_fso_dashboard(self.__admin_console.props['reports.sizeDistribution'])
        self.__admin_console.check_error_message()
        self.select_fso_review_tab()
        self.select_review_page_filter("Created Time")
        temp_dict['CreatedTime'] = self.get_fso_time_dict("Created Time")
        self.select_review_page_filter("Access Time")
        temp_dict['AccessTime'] = self.get_fso_time_dict("Access Time")
        self.select_review_page_filter("Modified Time")
        temp_dict['ModifiedTime'] = self.get_fso_time_dict("Modified Time")
        return temp_dict
