# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the methods that can be done of the File Server lookup page
which is opened while adding a Data Source in Sensitive Data Analysis Project Details
page.


Classes:

    FileServerLookup() ---> SensitiveDataAnalysisProjectDetails() --->
    SensitiveDataAnalysis() ---> GovernanceApps() ---> object()


FileServerLookup  --  This class contains all the methods for action in
     File Server lookup page page and is inherited by other classes to
    perform GDPR related actions

    Functions:

    _select_mailboxes() -- Selects the given Mailbox
    __get_advanced_settings_expansion_element() -- Gets Advanced settings bar WebElement
    is_advanced_settings_expanded() -- Checks if Advanced settings is expanded
    expand_advanced_settings() -- Expands the advanced settings bar if collapsed
    collapse_advanced_settings() -- Collapses the advanced settings bar if expanded
    _select_machine_to_analyze()  --  Search for the machine to Analyze
    add_file_server() -- Adds data source file server
    add_one_drive_server() --  Adds data source OneDrive Server
    add_exchange_server() --  Adds data source Exchange Server
    add_database_server() -- Adds data source database server
    add_gdrive_server() -- Adds data source  gdrive server

"""

import re
from Web.Common.page_object import WebAction, PageService
from Web.AdminConsole.GovernanceAppsPages.SensitiveDataAnalysisProjectDetails import\
    SensitiveDataAnalysisProjectDetails
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.browse import CVAdvancedTree
from Web.AdminConsole.Components.panel import PanelInfo, DropDown


class FileServerLookup(SensitiveDataAnalysisProjectDetails):
    """
     This class contains all the methods for action in File Server Lookup page
    """

    def __init__(self, admin_console):
        """
        Args:
            admin_console (AdminConsole): adminconsole base object
        """
        super().__init__(admin_console)
        self.__admin_console = admin_console
        self.driver = self.__admin_console.driver
        self.__admin_console._load_properties(self)
        self.log = self.__admin_console.log
        self.__table = Table(self.__admin_console)
        self.__cv_tree = CVAdvancedTree(self.__admin_console)
        self.__panelinfo = PanelInfo(self.__admin_console)
        self.__dropdown = DropDown(self.__admin_console)

    @WebAction()
    def _select_mailboxes(self, mailbox):
        """
        Check Selected Mailbox
        Args:
            mailbox (str) - Mailbox to be selected
        """

        self.__admin_console.driver.find_element_by_xpath(
            f'//div[contains(@class,"cv-kendo-grid")]//div[text()="{mailbox}"]/preceding::td[2]') \
            .click()

    @WebAction()
    def __select_fso_crawl_type(self, crawl_type='Quick'):
        """
        Select FSO crawl type for backed up DS, Full and Quick
        Args:
            crawl_type (str): FSO crawl type (Quick/Full)
        """
        self.__admin_console.driver.find_element_by_xpath(
            f"//label/span[text()='{crawl_type}']"
        ).click()

    @WebAction()
    def __get_advanced_settings_expansion_element(self):
        """ Gets Advanced settings bar WebElement

        Returns : Advanced settings bar WebElement

        """
        return self.driver.find_element_by_xpath("//div[contains(@class,'cv-accordion-header')]")

    @PageService()
    def is_advanced_settings_expanded(self):
        """ Checks if Advanced settings is expanded

        Returns (bool) : True if expanded

        """
        element = self.__get_advanced_settings_expansion_element()
        return self.__panelinfo.is_toggle_enabled(element)

    @PageService()
    def expand_advanced_settings(self):
        """ Expands the advanced settings bar if collapsed """
        if not self.is_advanced_settings_expanded():
            element = self.__get_advanced_settings_expansion_element()
            element.click()
            self.__admin_console.wait_for_completion()

    @PageService()
    def collapse_advanced_settings(self):
        """ Collapses the advanced settings bar if expanded """
        if self.is_advanced_settings_expanded():
            element = self.__get_advanced_settings_expansion_element()
            element.click()
            self.__admin_console.wait_for_completion()

    @WebAction()
    def _select_machine_to_analyze(
            self, search_name, search_category, agent_installed=False,
            data_source_type='File system'):
        """
        Search for the machine to Analyze

            Args:
                search_name (str) - Name to search for
                search_category (str) - Search Category to search for
                Values:
                    "Client name",
                    "Domain Name",
                    "Operating system",
                    "Host name",
                    "All"
                agent_installed (Bool) - Verify if Agent Installed
                data_source_type (str) - SDG Data Source Type
            Raise:
                Exception if Machine not found in the search
                Exception if Agent Installed status is shown otherwise than expected
        """
        search_category_dict = {
            "Client name":self.__admin_console.props['label.serverName'],
            "Domain Name":self.__admin_console.props['label.domainName'],
            "Operating system":self.__admin_console.props['label.operatingSystem'],
            "Host name":"Host name",
            "All":self.__admin_console.props['label.all']
            }
        self.log.info("Selecting search category")
        self.__admin_console.select_value_from_dropdown(
            "dataColumn", search_category_dict[search_category])
        self.__admin_console.wait_for_completion()
        self.log.info("Entering Search Name: %s", search_name)
        self.__admin_console.fill_form_by_id("searchInputFilterSearch", str(search_name))
        table_data = self.__table.get_table_data()
        tmp_txt = table_data[search_category_dict[search_category]][0]
        self.log.info("Host Name Obtained is: %s", tmp_txt)
        if re.search(tmp_txt, str(search_name), re.IGNORECASE):
            self.log.info("Found: %s" % search_name)
        else:
            raise Exception("Couldn't find: %s" % search_name)
        if not data_source_type == self.__admin_console.props['label.datasource.exchange']:
            tmp_txt = table_data['Status'][0]
            self.log.info("Agent Status Obtained is: %s", tmp_txt)
            if agent_installed:
                if re.search('Agent installed', tmp_txt, re.IGNORECASE) or \
                        re.search('Content indexing enabled', tmp_txt, re.IGNORECASE):
                    self.log.info("Agent installed or Content Indexing Enabled as expected")
                else:
                    raise Exception("Agent not installed")
            else:
                if re.search('Agent not installed', tmp_txt, re.IGNORECASE):
                    self.log.info("Agent not installed as expected")
                else:
                    raise Exception("Agent installed")
        self.log.info("Selecting the machine to analyze")
        self.__table.select_rows([''])
        self.__admin_console.wait_for_completion()
        self.log.info("Clicking on Next button")
        self.__admin_console.click_button(self.__admin_console.props['label.next'])

    @PageService()
    def add_file_server(self, search_name, search_category, display_name,
                        country_name, directory_path=None, user_name=None,
                        password=None, agent_installed=False, live_crawl=False,
                        backup_data_import=False, access_node=False,
                        fso_server=False, crawl_type='Quick'):
        """
        Adds data source file server

            Args:
                search_name (str) - Name to search for
                search_category (str) - Search Category to search for
                Values:
                    "Client name",
                    "Domain Name",
                    "Operating system",
                    "Host name",
                    "All"
                display_name (str) - Display name for this data source
                country_name (str) - Country name to be selected
                Values:
                    "United Kingdom",
                    "United States"
                directory_path (str) - Directory Path for Entity Extraction
                user_name (str) - Username to access the Directory Path
                password (str) - Password to access the Directory Path
                agent_installed (Bool) - Verify if Agent Installed or Content indexing enabled
                live_crawl (Bool) - Selects Live Crawl in case of Agent Installed
                backup_data_import (Bool) - Selects import from Backed up data option
                                             in case of Agent Installed
                access_node (str) - Access Node machine to crawl the data in case of
                                             Agent Not Installed
                fso_server (bool): Identifies if files server is added as FSO server
                crawl_type (str): Type of crawl operation to perform
        """
        self._select_machine_to_analyze(
            search_name, search_category, agent_installed)
        self.__admin_console.fill_form_by_id("displayName", display_name)
        self.__admin_console.select_value_from_dropdown("country", country_name)
        self.__admin_console.wait_for_completion()
        if not agent_installed:
            self.__admin_console.fill_form_by_id("directoryPath", directory_path)
            self.__admin_console.fill_form_by_id("userName", user_name)
            self.__admin_console.fill_form_by_id("password", password)
            if access_node:
                self.expand_advanced_settings()
                self.__dropdown.select_drop_down_values(
                    drop_down_id='accessNodes', values=[access_node])
        if live_crawl:
            if self.__admin_console.check_if_entity_exists("id", "localCrawl"):
                self.__admin_console.access_tile("localCrawl")
                self.__admin_console.wait_for_completion()
            self.__admin_console.fill_form_by_id("directoryPath", directory_path)
        if backup_data_import:
            self.driver.find_element_by_xpath(
                '//*[@id="importFromBackup"]').click()
            self.__admin_console.wait_for_completion()
        if fso_server:
            self.__select_fso_crawl_type(crawl_type)
        self.__admin_console.click_button(self.__admin_console.props['label.finish'])
        self.__admin_console.check_error_message()

    @PageService()
    def add_one_drive_server(self, search_name, search_category, display_name,
                             country_name, agent_installed=True, subclient_list=None):
        """
        Adds data source one drive server
        Args:
            search_name (str) - Name to search for
            search_category (str) - Search Category to search for
            Values:
                "Client name",
                "Domain Name",
                "Operating system",
                "Host name",
                "All"
            display_name (str) - Display name for this data source
            country_name (str) - Country name to be selected
            Values:
                "United Kingdom",
                "United States"
            agent_installed (bool): Verify if Agent Installed
            subclient_list (list): List of subclients

        """
        self._select_machine_to_analyze(
            search_name, search_category, agent_installed)
        self.__admin_console.fill_form_by_id("displayName", display_name)
        self.__admin_console.select_value_from_dropdown("country", country_name)
        self.__admin_console.wait_for_completion()
        if not subclient_list:
            self.__admin_console.select_radio("importFromBackup")
        else:
            self.__admin_console.select_radio("localCrawl")
            self.__admin_console.click_button(
                self.__admin_console.props['label.browse']
            )
            self.__cv_tree.select_elements(search_name.lower(), subclient_list)
            self.__admin_console.wait_for_completion()

            self.__admin_console.click_button(
                self.__admin_console.props['label.button.save']
            )
        self.__admin_console.click_button(self.__admin_console.props['label.finish'])
        self.__admin_console.check_error_message()

    @PageService()
    def add_exchange_server(self, search_name, search_category, display_name,
                            country_name, agent_installed=True, list_of_mailboxes=None):
        """
        Adds data source exchange server
        Args:
            search_name (str) - Name to search for
            search_category (str) - Search Category to search for
            Values:
                "Client name",
                "Domain Name",
                "Operating system",
                "Host name",
                "All"
            display_name (str) - Display name for this data source
            country_name (str) - Country name to be selected
            Values:
                "United Kingdom",
                "United States"
            agent_installed (Bool) - Verify if Agent Installed
            list_of_mailboxes (list) - list of mailboxes to be added
        """
        self._select_machine_to_analyze(
            search_name, search_category, agent_installed,
            data_source_type=self.__admin_console.props['label.datasource.exchange']
        )
        self.__admin_console.fill_form_by_id("displayName", display_name)
        self.__admin_console.select_value_from_dropdown("country", country_name)
        self.__admin_console.wait_for_completion()
        if not list_of_mailboxes:
            self.__admin_console.select_radio("allMailboxes")
        else:
            self.__admin_console.select_radio("selectMailboxes")
            self.__admin_console.click_button(
                self.__admin_console.props['label.browse']
            )
            for mailbox in list_of_mailboxes:
                self._select_mailboxes(mailbox)
            self.__admin_console.click_button(
                self.__admin_console.props["label.button.save"]
            )
        self.__admin_console.click_button(self.__admin_console.props['label.finish'])
        self.__admin_console.check_error_message()

    @PageService()
    def add_database_server(self, search_name, search_category, display_name,
                            instance_name, country_name, agent_installed=True):
        """
        Adds data source database server
        Args:
            search_name (str) - Server name to search for
            search_category (str) - Search Category to search for
            Values:
                "Client name",
                "Domain Name",
                "Operating system",
                "Host name",
                "All"
            display_name (str) - Display name for this data source
            instance_name (str) - DB Instance name
            country_name (str) - Country name to be selected
            Values:
                "United Kingdom",
                "United States"
            agent_installed (bool): Verify if Agent Installed

        """
        self._select_machine_to_analyze(
            search_name, search_category, agent_installed)
        self.__admin_console.fill_form_by_id("displayName", display_name)
        self.__admin_console.select_value_from_dropdown("country", country_name)
        self.__admin_console.select_value_from_dropdown("instance", instance_name)
        self.__admin_console.click_button(self.__admin_console.props['label.finish'])
        self.__admin_console.check_error_message()

    @PageService()
    def add_gdrive_server(self, search_name, search_category, display_name,
                          country_name, agent_installed=True):
        """
        Adds data source  gdrive server
        Args:
            search_name (str) - Name to search for
            search_category (str) - Search Category to search for
            Values:
                "Client name",
                "Domain Name",
                "Operating system",
                "Host name",
                "All"
            display_name (str) - Display name for this data source
            country_name (str) - Country name to be selected
            Values:
                "United Kingdom",
                "United States"
            agent_installed (bool): Verify if Agent Installed

        """
        self._select_machine_to_analyze(
            search_name, search_category, agent_installed)
        self.__admin_console.fill_form_by_id("displayName", display_name)
        self.__admin_console.select_value_from_dropdown("country", country_name)
        self.__admin_console.wait_for_completion()
        self.__admin_console.click_button(self.__admin_console.props['label.finish'])
        self.__admin_console.check_error_message()

    @PageService()
    def add_share_point_server(self, search_name, search_category, display_name,
                             country_name, agent_installed=True, backupset=None, sites=None):
        """
        Adds data source Share Point server
        Args:
            search_name (str) - Name to search for
            search_category (str) - Search Category to search for
            Values:
                "Client name",
                "Domain Name",
                "Operating system",
                "Host name",
                "All"
            display_name (str) - Display name for this data source
            country_name (str) - Country name to be selected
            Values:
                "United Kingdom",
                "United States"
            agent_installed (bool): Verify if Agent Installed
            backupset (str): name of the backupset
            sites (str): list of sites which has to selected

        """
        self._select_machine_to_analyze(
            search_name, search_category, agent_installed)
        self.__admin_console.fill_form_by_id("displayName", display_name)
        self.__admin_console.select_value_from_dropdown("country", country_name)
        self.__admin_console.wait_for_completion()
        self.__admin_console.select_value_from_dropdown("backupSet", backupset)
        self.__admin_console.wait_for_completion()
        self.__admin_console.click_button(self.__admin_console.props['label.browse'])
        self.__admin_console.wait_for_completion()
        list_of_sites = [sites]
        self.__cv_tree.select_elements_by_full_path(list_of_sites, "sharepoint")
        self.__admin_console.wait_for_completion()
        self.__admin_console.click_button(self.__admin_console.props['label.save'])
        self.__admin_console.wait_for_completion()
        self.__admin_console.click_button(self.__admin_console.props['label.finish'])
        self.__admin_console.check_error_message()
