# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the methods that can be done of the Sensitive Data Analysis
Project Data Source Review page.

Classes:

    DataSourceReview() ---> SensitiveDataAnalysisProjectDetails() --->
    SensitiveDataAnalysis() ---> GovernanceApps() ---> object()

DataSourceReview  --  This class contains all the methods for action in
    Sensitive Data Analysis Project Data Source Review page and is inherited by
    other classes to perform GDPR related actions

    Functions:
    get_data_source_name() -- Returns data source name from the admin page
    get_file_paths() -- Returns the list of file paths shown on the current page
    get_file_path_row_id() -- Returns the file path row id
    file_path_checked_status() -- Returns True/False based on the file path check box
    select_file() -- Selects the file for a file path
    search_file() -- Search based on the file and folder and for a given data source
    expand_entity() -- Expands the given entity
    get_entities() -- Returns a Dictionary for all entities with its
                        corresponding list of values
    get_file_name() -- Returns the file name
    select_next_file() -- Selects the next file to review by
                            clicking on the right arrow if exists
    close_file_preview() -- Closes the file being previewed
    close_action_modal() --  Close the review action modal
    review_delete_action() --  Review Delete Action for Given Data Source
    review_ignore_files_action() --  Review Ignore Files Actions
    review_ignore_risks_actions() -- Review Ignore Risks Action
    review_set_retention_action() -- Review Set Retention Action
    get_advance_search_output()   -- Get Advance Search Output
    get_advance_search_query()    -- Get Advance Search query formed
    do_advanced_search()          -- Perform Advance Search

"""

from Web.Common.page_object import WebAction, PageService
from Web.AdminConsole.GovernanceAppsPages.\
    SensitiveDataAnalysisProjectDetails import SensitiveDataAnalysisProjectDetails
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.AdminConsole.Components.panel import DropDown


class DataSourceReview(SensitiveDataAnalysisProjectDetails):
    """
     This class contains all the methods for action in Sensitive Data Analysis
     Project Data Source Review page
    """

    def __init__(self, admin_console):
        """
        Args:
            admin_console (AdminConsole): adminconsole base object
        """
        super().__init__(admin_console)
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self)
        self.driver = self.__admin_console.driver
        self.log = self.__admin_console.log
        self.__table = Table(self.__admin_console)
        self.__dialog = ModalDialog(self.__admin_console)
        self.__dropdown = DropDown(self.__admin_console)
        self.actions_title_map = {
            self.__admin_console.props['label.datasource.onedrive']: {
                "DATA_FIELD": self.__admin_console.props['label.taskdetail.name'],
                "DELETE": "Delete files",
                "RETENTION": self.__admin_console.props['risks.action.retention'],
                "IGNORE_RISKS": self.__admin_console.props['risks.action.ignoreRisks'],
                "IGNORE_FILES_FOLDERS": "Ignore files"
            },
            self.__admin_console.props['label.datasource.file']: {
                "DATA_FIELD": self.__admin_console.props['label.taskdetail.name'],
                "MOVE": "Move files",
                "DELETE": "Delete files"
            },
            self.__admin_console.props['label.datasource.googledrive']: {
                "DATA_FIELD": self.__admin_console.props['label.taskdetail.name'],
                "IGNORE_RISKS": self.__admin_console.props['risks.action.ignoreRisks'],
                "IGNORE_FILES_FOLDERS": "Ignore files"
            },
            self.__admin_console.props['label.datasource.exchange']: {
                "DATA_FIELD": self.__admin_console.props['label.subject'],
                "IGNORE_RISKS": self.__admin_console.props['risks.action.ignoreRisks'],
                "IGNORE_FILES_FOLDERS": "Ignore emails"
            },
            self.__admin_console.props['label.datasource.sharepoint']: {
                "DATA_FIELD": self.__admin_console.props['label.taskdetail.name'],
                "RETENTION": self.__admin_console.props['risks.action.retention'],
                "IGNORE_RISKS": self.__admin_console.props['risks.action.ignoreRisks'],
                "IGNORE_FILES_FOLDERS": "Ignore files"
            }
        }

    @WebAction()
    def get_data_source_name(self):
        """
        Returns data source name from the admin page
        """
        data_source_name = str(self.driver.find_element_by_xpath(
            '//*[@id="customDatasourceReport"]//h1/span').text)
        self.log.info("data source name obtained is: %s" % data_source_name)
        return data_source_name

    @WebAction()
    def _select_advanced_search(self):
        """
        Select Advanced search button in SDG data source
        review page
        """
        tmp_text = self.__admin_console.props['label.advancedsearch']
        self.driver.find_element_by_xpath(
            f'//*[@id="fileDetails"]//a[text()="{tmp_text}"]').click()

    @WebAction()
    def _enter_advance_search_entity_value(self, selector_name, selector_value):
        """
        Enter advance search value
        Args:
            selector_name (str) : Entity name
            selector_value (str): Entity Value
        """
        xp = "//label[text()='{0}']/following::div[1]/input"
        elem = self.__admin_console.driver.find_element_by_xpath(xp.format(selector_name))
        elem.send_keys(selector_value)

    @PageService()
    def get_file_paths(self):
        """
        Returns the list of file paths shown on the current page

            Return:
                List of file paths
        """
        self.log.info("Obtaining all the file paths shown")
        return self.__table.get_column_data('File path')

    @PageService()
    def get_advance_search_output(self, data_source='File system'):
        """
        Get output data after performing advanced search
        Args:
            data_source  (str): Data Source type of SDG
        Return :
            list Output data list
        """
        self.__table.set_pagination(500)
        self.__admin_console.wait_for_completion()
        return self.__table.get_column_data(
            self.actions_title_map[data_source]['DATA_FIELD'])

    @WebAction()
    def get_advance_search_query(self):
        """
        Get advanced search query formed
        Returns:
            (str) :- Advanced search query formed
        """
        return \
            self.__admin_console.driver.find_element_by_id('queryInput').get_property('value')

    @PageService()
    def select_file(self, file_name):
        """
        Selects the file name link
        """
        self.__table.access_link_by_column(file_name, file_name)

    @PageService()
    def do_advanced_search(self, selector_list=None, value_list=None,
                           input_query=None, data_source='File system'):
        """
        Perform Advanced search
        Args:
            selector_list (list): List of entities to be selected
            value_list    (list): List of values for respective entities given
            input_query   (str): Input Query for advanced search
            data_source   (str): Data Source Type
        """
        self._select_advanced_search()
        self.__admin_console.wait_for_completion()
        if input_query is not None:
            self.__admin_console.fill_form_by_id("queryInput", input_query)
            query = input_query
        else:
            self.__dropdown.select_drop_down_values(0, selector_list)
            for index, val in enumerate(selector_list):
                self._enter_advance_search_entity_value(val, value_list[index])
            query = self.get_advance_search_query()
        self.__dialog.click_submit()
        self.__admin_console.wait_for_completion()

        output = []
        if data_source == 'File system':
            output = self.get_file_paths()
        else:
            output = self.get_advance_search_output(data_source=data_source)
        return output, query

    @PageService()
    def search_file(self, file_name, folder_name=None, is_fso=False,
                    data_source_type="File system"):
        """
        Search based on the file and folder name
        Args:
            file_name (str): Name of File
            folder_name (str): Name of Folder
            is_fso (bool): Is search operation done is FSO
            data_source_type (str): Data Source Type
        """
        if data_source_type == self.__admin_console.props['label.datasource.onedrive']:
            search_string = '(FileName:{0})'.format(file_name)
        elif data_source_type == self.__admin_console.props['label.datasource.googledrive']:
            search_string = '(FileName:"{0}")'.format(file_name)
        elif data_source_type == self.__admin_console.props['label.datasource.exchange']:
            search_string = '(Subject:{0})'.format(file_name)
        elif data_source_type == self.__admin_console.props['label.datasource.database']:
            search_string = '(FileName:{0})'.format(file_name)
        elif data_source_type == self.__admin_console.props['label.datasource.sharepoint']:
            search_string = '(FileName:{0})'.format(file_name)
        else:
            if folder_name is None:
                search_string = \
                    f'FileName_idx:({file_name})'
            else:
                search_string = \
                    '(FolderName_idx:({0})) AND (FileName_idx:({1}))'.format(
                        folder_name, file_name)
        if is_fso:
            search_string = file_name
            self.__admin_console.fill_form_by_id(
                "searchInputFilterSearch", search_string)
        else:
            self._select_advanced_search()
            self.__admin_console.wait_for_completion()
            self.__admin_console.fill_form_by_id("queryInput", search_string)
            self.__dialog.click_submit()

    @WebAction()
    def expand_entity(self, entity_name, data_source_type="OneDrive"):
        """
        Expands the given entity

            Args:
                entity_name (str)  - Name of the entity
                data_source_type (str)  - Name of the data source
        """
        ds_db = self.__admin_console.props['label.datasource.database']
        ds_gd = self.__admin_console.props['label.datasource.googledrive']
        class_string = "modal-content"
        if data_source_type == ds_db or data_source_type == ds_gd:
            class_string = "panel-body"
        xpath = f"//div[@class='{class_string}']//span[text()='{entity_name}']\
        //following-sibling::i[contains(@class, 'down')]"
        if not self.__admin_console.check_if_entity_exists('xpath', xpath):
            self.log.info("Expanding Entity: %s" % entity_name)
            self.driver.find_element_by_xpath(
                f"//div[@class='{class_string}']//span[text()='{entity_name}']")\
                .click()
            self.__admin_console.wait_for_completion()

        else:
            self.log.info("Entity: %s already expanded" % entity_name)

    @WebAction()
    def get_entities(self, data_source_type="OneDrive"):
        """
        Returns a Dictionary for all entities with its corresponding list of values

            Args:
                data_source_type (str)  - Name of the data source

            Return (Dict):
                Dictionary with list of entity values
        """
        ds_db = self.__admin_console.props['label.datasource.database']
        ds_gd = self.__admin_console.props['label.datasource.googledrive']
        sensitivity = self.__admin_console.props['label.filter.sensitivity']
        class_string = "modal-content"
        if data_source_type == ds_db or data_source_type == ds_gd:
            class_string = "panel-body"

        base_xp = f"//div[@class='{class_string}']//div[@role='tablist']"
        if data_source_type == ds_gd:
            base_xp = f"//h4[@class='panel-title']//span[text()='{sensitivity}']//ancestor::div[@role='tablist']"
        entities_list = []
        entities_dict = {}
        elems = self.driver.find_elements_by_xpath(
            base_xp+"//div//a/span/span[1]")
        for index in range(1, len(elems) + 1):
            entities_list.append(
                self.driver.find_element_by_xpath(
                    base_xp + f"//div[{str(index)}]/div/h4/a/span/span[1]").text)
        self.log.info("Entity Names Obtained are: %s" % entities_list)
        for entity_name in entities_list:
            self.expand_entity(entity_name, data_source_type)
            self.log.info(
                "Obtaining Entity Values for Entity: %s" %
                entity_name)
            elems = self.driver.find_elements_by_xpath(
                f"//div[@class='{class_string}']//span[text()='{entity_name}']/ancestor::node()[5]\
                //div[contains(@class,'-list')]/form/div")
            entity_values_list = []
            for index in range(1, len(elems) + 1):
                entity_values_list.append(
                    self.driver.find_element_by_xpath(
                        f"//div[@class='{class_string}']//span[text()='{entity_name}']\
                        /ancestor::node()[5]//div[contains(@class,'-list')]\
                        /form/div[{str(index)}]/label").text)
            entity_values_list = sorted(entity_values_list, key=str.lower)
            entities_dict[entity_name.lower()] = entity_values_list
        self.log.info(
            "Entities Dictionary obtained is: '{0}'".format(entities_dict))
        return entities_dict

    @WebAction()
    def get_file_name(self):
        """
        Returns the file name

            Return (str):
                Name of the file
        """
        file_name = self.driver.find_element_by_xpath(
            "//div[@class='modal-content']//div/span").text
        self.log.info("File Name Obtained is: %s" % file_name)
        return file_name

    @WebAction()
    def select_next_file(self):
        """
        Selects the next file to review by clicking on the right arrow if exists

            Return (Bool):
                True/False based on the status
        """
        xpath = "//div[@class='modal-content']//button[contains(@data-ng-click,'Next')]"
        if self.__admin_console.check_if_entity_exists('xpath', xpath):
            self.log.info("Clicking on the next button")
            self.driver.find_element_by_xpath(xpath).click()
            self.__admin_console.wait_for_completion()
            return True

        self.log.info("Next button not found")
        return False

    @WebAction()
    def close_file_preview(self):
        """
        Closes the file being previewed
        """
        self.driver.find_element_by_xpath(
            "//a[@class='modal__close-btn']").click()
        self.__admin_console.wait_for_completion()

    @PageService()
    def close_action_modal(self):
        """
        Closes the Review Page Action Modal
        """
        self.__dialog.click_cancel()

    @WebAction()
    def _click_review_action(self, action_title):
        """
        Click on Review Action
        Args:
            action_title (str): Title of Review Action
        """
        xpath = f'//cv-resolve-actions//div[@title="{action_title}"]'
        self.__admin_console.driver.find_element_by_xpath(xpath).click()

    @PageService()
    def _enable_toggle(self, label):
        """
        Enable Toggle in Modal according to input label
        Args:
            label (str): Label for toggle element
        """
        toggle_status = self.__admin_console.check_toggle_status(label)
        if not toggle_status:
            self.__admin_console.toggle_enable(label)

    @WebAction()
    def _get_panel_element(self, name):
        """
        Get Panel Element in Review Page
        Args:
            name (str): Text within panel element

        Returns: Return Panel Element
        """
        elem = None
        xpath = f'//span[text()="{name}"]/ancestor::a'
        if self.__admin_console.check_if_entity_exists("xpath", xpath):
            elem = self.driver.find_element_by_xpath(xpath)
        return elem

    @PageService()
    def review_move_action(self, file_name, destination_path, username, password,
                           is_fso=False, data_source_type='File system'):
        """
        Move action for a file
        Args:
            file_name         (str)    --  Name of the file
            destination_path  (str)    --  UNC path where the file would be moved
            username          (str)    --  Username needed to access the UNC path
            password          (str)    --  Password needed to access the UNC path
            is_fso            (bool)   -- Is it a FSO Review Action
            data_source_type  (str)    --  Type of the datasource
        Returns:
            bool - The status of this operation

        """
        self.search_file(file_name, is_fso=is_fso,
                         data_source_type=data_source_type)
        self.__table.select_rows([''])
        self._click_review_action(
            self.actions_title_map[data_source_type]['MOVE']
        )
        self.__admin_console.wait_for_completion()
        self.__admin_console.fill_form_by_id(
            "destinationPath", destination_path)
        self.__admin_console.fill_form_by_id("username", username)
        self.__admin_console.fill_form_by_id("password", password)
        self.__dialog.click_submit()
        self.__admin_console.wait_for_completion()
        self.__admin_console.check_error_message()
        self.search_file(file_name, is_fso=is_fso,
                         data_source_type=data_source_type)
        review_page_file_list = self.__table.get_column_data(
            self.actions_title_map[data_source_type]['DATA_FIELD']
        )
        self.__admin_console.wait_for_completion()
        return True if review_page_file_list else False

    @PageService()
    def review_delete_action(self, file_name, delete_from_backup=False,
                             is_fso=False, data_source_type='File system'):
        """
        Review Delete Action for List of Files
        Args:
            file_name (str): Name of The File
            data_source_type (str): Data Source type
            is_fso (bool): Is it FSO delete action
            delete_from_backup (bool): To Enable Delete form Backup
        Return (Bool):
                True/False based on the status

        """
        delete_flag = True
        self.search_file(file_name, is_fso=is_fso,
                         data_source_type=data_source_type)
        self.__table.select_rows([''])
        self._click_review_action(
            self.actions_title_map[data_source_type]['DELETE']
        )
        self.__admin_console.wait_for_completion()
        self.__admin_console.select_radio("selectedFiles")
        if delete_from_backup:
            self._enable_toggle(
                self.__admin_console.props["label.deleteFromBackup"])
        self.__dialog.click_submit()
        self.__admin_console.wait_for_completion()
        self.__admin_console.check_error_message()
        self.search_file(file_name, is_fso=is_fso,
                         data_source_type=data_source_type)
        review_page_file_list = self.__table.get_column_data(
            self.actions_title_map[data_source_type]['DATA_FIELD']
        )
        self.__admin_console.wait_for_completion()
        if len(review_page_file_list) > 0:
            delete_flag = False
        return delete_flag

    @PageService()
    def review_ignore_files_action(self, file_name, data_source_type='File system'):
        """
        Review Ignore Files Action for List of Files
        Args:
            file_name (str): Name of fiLe
            data_source_type (str): Data Source Type

        Return (Bool):
                True/False based on the status

        """
        # Make Sure DisplaySensitive Files Toggle Is On
        ignore_flag = True
        self.search_file(file_name, data_source_type=data_source_type)
        self.__table.select_rows([''])
        self._click_review_action(
            self.actions_title_map[data_source_type]["IGNORE_FILES_FOLDERS"]
        )
        self.__admin_console.wait_for_completion()
        self.__admin_console.select_radio("selectedFiles")
        self.__dialog.click_submit()
        self.__admin_console.wait_for_completion()
        self.__admin_console.check_error_message()
        self.search_file(file_name, data_source_type=data_source_type)
        review_page_file_list = self.__table.get_column_data(
            self.actions_title_map[data_source_type]['DATA_FIELD']
        )
        self.__admin_console.wait_for_completion()
        if len(review_page_file_list) > 0:
            ignore_flag = False
        return ignore_flag

    @PageService()
    def review_ignore_risks_actions(self, file_name,
                                    risk_type_list, data_source_type='File system'):
        """
        Review Ignore Risks Action For files in One Drive Review Page
        Args:
            file_name (str): Name Of File
            risk_type_list (list): List of Risks To Be Ignored
            data_source_type (str): Data Source Type

        Return (Bool):
            True/False based on the status

        """
        ignore_risks_flag = True
        self.search_file(file_name, data_source_type=data_source_type)
        self.__table.select_rows([''])
        self._click_review_action(
            self.actions_title_map[data_source_type]["IGNORE_RISKS"]
        )
        self.__admin_console.wait_for_completion()
        self.__dropdown.select_drop_down_values(0, risk_type_list)
        self.__admin_console.select_radio("selectedFiles")
        self.__dialog.click_submit()
        self.__admin_console.wait_for_completion()
        self.__admin_console.check_error_message()
        self.search_file(file_name, data_source_type=data_source_type)
        self.__table.select_rows([''])
        elem = self._get_panel_element(
            self.__admin_console.props['label.taskdetail.risks'])
        self.__admin_console.wait_for_completion()
        if elem is not None:
            if not elem.get_attribute("aria-expanded"):
                elem.click()
        for risks in risk_type_list:
            elem = self._get_panel_element(risks)
            self.__admin_console.wait_for_completion()
            if elem is not None:
                ignore_risks_flag = False
                break
        self.__table.select_rows([''])
        return ignore_risks_flag

    @PageService()
    def review_set_retention_action(self, file_name,
                                    retention_months, data_source_type='File system'):
        """

        Args:
            file_name (str):File Name
            retention_months (int): no of months
            data_source_type (str): Data Source Type

        Returns:
            True/False based on the status
        """
        retention_flag = True
        self.search_file(file_name, data_source_type=data_source_type)
        self.__table.select_rows([''])
        self._click_review_action(
            self.actions_title_map[data_source_type]["RETENTION"]
        )
        self.__admin_console.wait_for_completion()
        self.__admin_console.fill_form_by_id("retention", retention_months)
        self.__admin_console.select_radio("selectedFiles")
        self.__dialog.click_submit()
        self.__admin_console.wait_for_completion()
        self.__admin_console.check_error_message()
        self.search_file(file_name, data_source_type=data_source_type)
        self.__table.select_rows([''])
        elem = self._get_panel_element(
            self.__admin_console.props['label.taskdetail.risks'])
        self.__admin_console.wait_for_completion()
        if elem is not None:
            if not elem.get_attribute("aria-expanded"):
                elem.click()
        elem = self._get_panel_element(
            self.__admin_console.props['risks.noRetention'])
        self.__admin_console.wait_for_completion()
        if elem is not None:
            retention_flag = False
        self.__table.select_rows([''])
        return retention_flag
