# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the methods that can be done of the Sensitive Data Analysis
Project Data Source discover page.


Classes:

    DataSourceDiscover() ---> SensitiveDataAnalysisProjectDetails() --->
    SensitiveDataAnalysis() ---> GovernanceApps() ---> object()


DataSourceDiscover  --  This class contains all the methods for action in
    Sensitive Data Analysis Project Data Source discover page and is inherited by
    other classes to perform GDPR related actions

    Functions:
    get_data_source_name() -- Returns data source name from the admin page
    get_total_files() -- Returns total number of files count
    get_sensitive_files() -- Returns total number of sensitive files count
    get_size() -- Returns total files size
    get_owners() -- Returns total number of owners for the files
    select_review() -- Selects the review link
    select_details() -- Clicks on the Details link
    start_data_collection_job() -- Starts the data collection job
    get_running_job_id()    --  Return running data collection job id
"""

from Web.Common.page_object import WebAction
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.GovernanceAppsPages.\
    SensitiveDataAnalysisProjectDetails import SensitiveDataAnalysisProjectDetails

class DataSourceDiscover(SensitiveDataAnalysisProjectDetails):
    """
     This class contains all the methods for action in Sensitive Data Analysis
     Project Data Source discover page
    """

    def __init__(self, admin_console):
        """
        Args:
            admin_console (AdminConsole): adminconsole base object
        """
        super().__init__(admin_console)
        self.__table = Table(admin_console)
        self.__admin_console = admin_console
        self.driver = self.__admin_console.driver
        self.__admin_console._load_properties(self)
        self.log = self.__admin_console.log

    @WebAction()
    def get_data_source_name(self):
        """
        Returns data source name from the admin page
        """
        data_source_name = str(self.driver.find_element_by_xpath(
            '//*[@id="customDatasourceReport"]/div[1]/h1/span').text)
        self.log.info("data source name obtained is: %s" % data_source_name)
        return data_source_name

    @WebAction()
    def get_total_number_after_crawl(self):
        """
        Get total number of files after crawl
        :return: total_files
        """
        total_files = self.__table.get_column_data('Docs')
        return total_files.pop(0)

    @WebAction()
    def get_total_files(self):
        """
        Returns total number of files count
        """
        count = self.driver.find_element_by_xpath(
            '//*[@id="otherInfo"]//div[text()="Total Files"]/following-sibling::p')\
            .text
        self.log.info("total files count obtained is: %s" % count)
        return int(count)

    @WebAction()
    def get_sensitive_files(self):
        """
        Returns total number of sensitive files count
        """
        count = self.driver.find_element_by_xpath(
            '//*[@id="otherInfo"]//div[text()="Sensitive Files"]\
            /following-sibling::p').text
        self.log.info("sensitive files count obtained is: %s" % count)
        return int(count)

    @WebAction()
    def get_size(self):
        """
        Returns total files size
        """
        size = self.driver.find_element_by_xpath(
            '//*[@id="otherInfo"]//div[text()="%s"]/following-sibling::p'\
            %self.__admin_console.props['label.size']).text
        self.log.info("size obtained is: %s" % size)
        return size

    @WebAction()
    def get_owners(self):
        """
        Returns total number of owners for the files
        """
        count = self.driver.find_element_by_xpath(
            '//*[@id="otherInfo"]//div[text()="Owners"]/following-sibling::p').text
        self.log.info("Owners obtained are: %s" % count)
        return int(count)

    @WebAction()
    def select_review(self):
        """
        Selects the review link
        """
        self.driver.find_element_by_xpath(
            "//a[contains(@class,'nav-link') and text()='%s']"\
            %self.__admin_console.props['label.review']).click()
        self.__admin_console.wait_for_completion()

    @WebAction()
    def select_details(self):
        """
        Clicks on the Details link
        """
        self.driver.find_element_by_xpath(
            "//span[contains(@class,'ng-binding') and text()='%s']"\
            %self.__admin_console.props['label.details']).click()
        self.__admin_console.wait_for_completion()

    @WebAction()
    def start_data_collection_job(self, job_type='incremental'):
        """
        Clicks on the Start Data collection and run full
            or incremental crawl based on job type provided
        :param job_type    (str)   --  takes input of job type
                                        'full' or 'incremental'
        """
        self.driver.find_element_by_xpath(
            "//a[contains(@class,'ng-binding') and text()='%s']"\
            %self.__admin_console.props['label.datasource.startJob']).click()
        self.__admin_console.wait_for_completion()
        if job_type == 'full':
            self.driver.find_element_by_id("full").click()
            self.__admin_console.wait_for_completion()
        self.__admin_console.click_button(self.__admin_console.props['label.datasource.startJob'])
        self.__admin_console.wait_for_completion()

    @WebAction()
    def get_running_job_id(self):
        """
        Returns running Data collection Job id
        """
        table_data = self.__table.get_table_data()
        job_id = table_data['Job Id'][0]
        self.log.info('Data source Job ID obtained is: %s' % job_id)
        return job_id
