# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the methods that can be done on ComplianceSearch page.

ComplianceSearch  --  This class contains all the methods ComplianceSearch related actions

"""
import re
import time
from dateutil.parser import parse
from selenium.webdriver.common.keys import Keys
from selenium.common.exceptions import NoSuchElementException, ElementClickInterceptedException
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.table import CVTable
from Web.AdminConsole.Components.panel import DropDown
from Web.AdminConsole.Components.browse import Browse
from Web.AdminConsole.Components.panel import ModalPanel
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.Common.page_object import (
    WebAction, PageService
)
from Web.AdminConsole.AdminConsolePages.Jobs import Jobs


class ComplianceSearch:
    """
     This class contains all the methods for action in ComplianceSearch page
    """

    def __init__(self, admin_console):
        """
        Args:
            admin_console (AdminConsole): adminconsole base object
        """
        self.__admin_console = admin_console
        self.driver = admin_console.driver
        self.__table = Table(self.__admin_console)
        self.__panel = ModalPanel(self.__admin_console)
        self.__dropdown = DropDown(self.__admin_console)
        self.__dialog = ModalDialog(self.__admin_console)
        self.__export_to_dialog = _ExportToForm(self.__admin_console)
        self.__custom_filter = CustomFilter(self.__admin_console)
        self.__cvtable = CVTable(self.__admin_console)
        self.__job = Jobs(self.__admin_console)
        self.__browse = Browse(self.__admin_console)
        self.export_jobid = None

    @WebAction()
    def __click_preference(self):
        """
        Click on Preference link
        """
        self.driver.find_element_by_xpath(
            "//span[contains(text(),'Preference')]").click()

    @WebAction(delay=1)
    def __select_export_to(self, operation):
        """
        select Export To operation

        Args:
            operation (str)  :  Type of operation- CAB/PST/PDF
        """
        element = self.driver.find_element_by_xpath(
            "//span[contains(text(),'Export to')]")
        element.click()

        operation_element = self.driver.find_element_by_xpath(
            "//span[contains(text(),'" + operation + "')]")
        operation_element.click()

    @WebAction(delay=1)
    def __select_table_checkbox(self, count):
        """
        Method to select values from a table

        Args:
            count (int)  :  Number of items to be chosen in the table, starting from zero
        """
        indx = 2
        while indx <= count:
            element = self.driver. find_element_by_xpath(
                "(//td[@id=\'cv-k-grid-td-:checkbox\']/label)"
                "[" + str(indx) + "]")
            element.click()
            indx = indx + 1

    @WebAction()
    def __enter_search_term(self, value):
        """
        enter search on grid
        Args:
                value (str)  - keyword to be searched for
        """
        search = self.driver. \
            find_element_by_xpath("//div[@class='search-toolbar']"
                                  "//input[@id='searchInput']")
        search.clear()
        search.send_keys(value)
        search.send_keys(Keys.ENTER)

    @WebAction()
    def __select_export_set(self):
        """
        select exportset link
        """
        element = self.driver. \
            find_element_by_xpath("//span[contains(text(),'Export set')]")
        element.click()

    @WebAction()
    def __locate_jobid_and_get_parent_element(self, value):
        """
        enter search keyword for filter
        Args:
                value (str)  - job id to be searched for
        Returns:
                <tr> element to get the row
        """
        jobid_element = self.driver.find_element_by_xpath(
            f"//a/span[contains(text(), '{value}')]"
        )
        row_element = jobid_element.find_element_by_xpath(".//ancestor::tr")
        return row_element

    @WebAction()
    def __is_item_available_for_download(self):
        """
        Check if item is available for download
        Returns: True if item is ready for download
                 False if item is not ready for download
        """
        try:
            element = "//*[@id=\"cv-k-grid-td-statusStr\"]" \
                      "/div/span[@title=\"Available for download\"]"
            self.driver.find_element_by_xpath(element)
        except NoSuchElementException:
            return False
        return True

    @WebAction()
    def __check_if_item_available_for_download(self, row_element):
        """
         Check if item is available for download
         Args:
                 row_element (str)  - <tr> element
         Returns:
                 True if item is ready for download
                 False if item is not ready for download
         """
        try:
            row_element.find_element_by_xpath(
                "td/div/span[@title='Available for download']"
            )
        except NoSuchElementException:
            return False
        return True

    @WebAction()
    def __click_view_job_details(self):
        """
        Click on view job details alert
        """
        try:
            element = self.driver.find_element_by_xpath(
                "//a[contains(text(),'View job details')]")
            element.click()
            return 1
        except NoSuchElementException:
            return 0

    @WebAction()
    def __click_checkbox_item(self):
        """
        Click on checkbox of exportset items
        """
        select_item = "//div[@class='modal-content']/*//td[@id='cv-k-grid-td-:checkbox']/label"
        element = self.driver.find_element_by_xpath(select_item)
        element.click()

    @WebAction()
    def __click_checkbox(self, row_element):
        """
        Click on checkbox of exportset items
        """
        row_element.find_element_by_xpath(
            "td/label[@class='k-checkbox-label k-no-text']"
        ).click()

    @WebAction()
    def __click_delete_exportset(self, id):
        """
        Click to delete exportset
        Args:
            id: Id of the link
        """
        delete_icon = "//div[@class=\"toolbar\"]//span[@class=\"k-link k-menu-link\"]/a[@id=\"" + id + "\"]"
        element = self.driver.find_element_by_xpath(delete_icon)
        element.click()

    @WebAction()
    def __perform_download_exportset(self):
        """
        Click to download exportset
        """
        download_icon = "//*[@id=\"cv-k-grid-td-actions\"]/a/span"
        element = self.driver.find_element_by_xpath(download_icon)
        element.click()

    @WebAction()
    def __download_export_set(self, row_element):
        """
         Click to download exportset
         Args:
                 row_element (str)  - <tr> element
         """
        try:
            row_element.find_element_by_xpath(
                "td[@id='cv-k-grid-td-actions']/a/span"
            ).click()
        except ElementClickInterceptedException:
            download_element = row_element.find_element_by_xpath(
                "td[@id='cv-k-grid-td-actions']/a/span"
            )
            self.__admin_console.driver.execute_script(
                "arguments[0].scrollIntoView();", download_element)
            self.__admin_console.driver.execute_script(
                "arguments[0].click();", download_element)

    @WebAction()
    def click_confirm_delete_option(self):
        """
        Click to confirm delete exportset
        """
        self.driver.find_element_by_xpath(
            "//html/body/div[1]/div/div//button[@class='btn btn-primary ng-binding']") .click()

    @WebAction()
    def __get_exportjobid(self):
        """
        Get export jobid
        Returns: Export jobid (str)
        """
        element = self.driver.find_element_by_xpath(
            "//*[@id=\"wrapper\"]"
            "/div[2]/div[2]/div/div[2]/div/div[1]/div/h1")
        job_string = element.text
        restore_jobid = (job_string.split("-")[0]).split(" ")[1].strip()
        return restore_jobid

    @WebAction()
    def __close_export_set_dialog(self):
        """
        Close exportset dialog
        """
        self.driver.find_element_by_xpath(
        "//button[@class='btn btn-default cvBusyOnAjax']").click()

    @PageService()
    def set_indexserver_preference(self, index_server):
        """Setting IndexServer
         Args:
                index_server (str)  - index_server to be set
        """
        self.__click_preference()
        self.__admin_console.wait_for_completion()
        self.__dropdown.select_drop_down_values(0, [index_server])
        self.__panel.submit()
        self.__admin_console.wait_for_completion()

    @PageService()
    def select_custodian_and_get_results(self, user_smtp_list):
        """
        Clicks on Custodian link and returns count
         Args:
                user_smtp_list (list)  - list of custodian users

         Returns number of items
        """
        self.__custom_filter.select_filter_dropdown('Custodian')
        if len(user_smtp_list) > 1:
            # multiple users, clear the previous case of single user selection
            self.__custom_filter.clear_custodian_filter()
            self.__admin_console.wait_for_completion()
            self.__custom_filter.select_filter_dropdown('Custodian')
        self.__custom_filter.select_custom_filter(user_smtp_list)
        self.__admin_console.wait_for_completion()
        count = self.__table.get_total_rows_count()
        return int(count)

    @PageService()
    def search_for_keyword_get_results(self, index_server, keyword):
        """
        Searches for a given keyword

            Args:
                keyword (str)  - keyword name to be searched for
                index_server (str)  - index_server to be set

            Returns Number of items
        """
        self.set_indexserver_preference(index_server)
        count = self.get_total_rows_count(keyword)
        return int(count)

    @PageService()
    def do_export_to(self, operation_type,
                     selected_item_count, download_set_name, file_format = None):
        """
        perform export to operations
            Args:
                operation_type (str)  - CAB/PST
                selected_item_count (str)  - Number of items to be selected and exported
                                             All- for all items
                download_set_name (str) - Name of download set name
                file_format (str) - Format of exported message (Applicable only for Export To CAB)
            Returns:
                1 - Export Job submitted successfully
                0- Error submitting Export Job
        """
        if "All" not in str(selected_item_count):
            selection_range = "Selected (" + str(selected_item_count) + ")"
            self.__select_table_checkbox(int(selected_item_count))
        else:
            selection_range = selected_item_count
        self.__select_export_to(operation_type)
        self.__admin_console.fill_form_by_id(
            "downloadSetName", download_set_name)
        self.__export_to_dialog.choose_selection_range(selection_range)
        self.__export_to_dialog.choose_file_format(operation_type, file_format)
        self.__admin_console.wait_for_completion()
        self.export_jobid = self.__admin_console.get_jobid_from_popup()
        if not self.export_jobid:
            return 0
        else:
            return 1

    @PageService()
    def get_export_job_details(self):
        """get export jobid and details upon job completion"""
        job_details = self.__job.job_completion(self.export_jobid)
        return self.export_jobid, job_details

    @PageService()
    def perfom_download(self, jobid):
        """
        perform download operation on given jobid
            Args:
                jobid (str): Export Job id
        """
        self.__select_export_set()
        self.__admin_console.wait_for_completion()
        # self.__export_to_dialog.enter_search_filter(jobid)
        # if self.__is_item_available_for_download():
        #     self.__perform_download_exportset()
        row_element = self.__locate_jobid_and_get_parent_element(jobid)
        if self.__check_if_item_available_for_download(row_element):
            self.__download_export_set(row_element)

    @PageService()
    def delete_exportset(self, jobid):
        """
        delete exportset of given jobid
            Args:
                jobid (str): Export Job id
        """
        self.__close_export_set_dialog()
        self.__select_export_set()
        self.__admin_console.wait_for_completion()
        # self.__export_to_dialog.enter_search_filter(jobid)
        # self.__admin_console.wait_for_completion()
        # if self.__is_item_available_for_download():
        #     self.__click_checkbox_item()
        row_element = self.__locate_jobid_and_get_parent_element(jobid)
        if self.__check_if_item_available_for_download(row_element):
            self.__click_checkbox(row_element)
            self.click_delete("deleteDownload")
            self.__admin_console.wait_for_completion()
            self.click_confirm_delete_option()
            self.__admin_console.wait_for_completion()
            self.__close_export_set_dialog()

    @PageService()
    def click_delete(self, id):
        """
        Click on delete link
        Args:
            id: id of link to be delete
        """
        self.__click_delete_exportset(id)

    @PageService()
    def get_total_rows_count(self, search_keyword=None):
        """
        Get the total number of items
        Args:
                search_keyword (str)  - keyword to be searched for
                                        it will return rows already present

        Returns Number of items in table
        """
        if search_keyword:
            self.__enter_search_term(search_keyword)
            self.__admin_console.wait_for_completion()
        return self.__table.get_total_rows_count()


class CustomFilter():
    """
    This class contains all functions for selecting custom filters
    """

    def __init__(self, admin_console):
        self.__admin_console = admin_console
        self.__driver = admin_console.driver

    @WebAction()
    def clear_custodian_filter(self):
        """
        Clears custodian filter
        Returns: bool - True/False
        """
        try:
            clear_button = self.__driver.find_element_by_css_selector(
                ".uib-dropdown > .k-icon")
            if clear_button:
                clear_button.click()
            return True
        except Exception:
            return False

    @WebAction()
    def select_filter_dropdown(self, filtername):
        """
        select filter
        Args: filtername (str): Type of filter to be selected

        Returns: bool - True/False
        """
        try:
            element = self.__driver.find_element_by_xpath(
                "//*[@class=\"row\"]//span[contains(text(),'" + filtername + "')]")
            if element:
                element.click()
            return True
        except Exception:
            return False

    @WebAction()
    def select_date_filter(self, values_list):
        """
        Select Date filter and enter value
        Args: values_list (list) : list of date values to be selected
                                Ex: ["Beyond one year","past year",'12/12/2019"]
                                if value in the above list matches date format (ex: 12/12/2019), then it will be entered in the
                                from date textbox in the filter. Search will be fired for date from that date to till date
                                else, it is do string comparision and select the filter value (ex: past month, past year... )

        """
        # enter date
        is_date = False
        for value in values_list:
            try:
                if parse(value, fuzzy=False):
                    is_date = True
            except ValueError:
                is_date = False

            finally:
                if is_date:
                    endtime_text_box = self.__driver.find_element_by_xpath(
                        "//input[contains(@id,'cv-kendo-endtime')]")
                    endtime_text_box.clear()
                    endtime_text_box.send_keys(value)
                    add_element = self.__driver.find_element_by_xpath(
                        "//*[@class=\"dropdown-menu gdpr-chip-menu\"]"
                        "//span[1]/a[contains(text(),'Add')]")
                    add_element.click()
                else:
                    self.select_custom_filter([value])

    @WebAction()
    def select_size_filter(self, values_list):
        """
        Select Size filter and enter value
        Args: values_list (list) : list of size values to be selected
        Ex: ["1 - 5 MB","6-20 SizeKB"]
                                if value in the above list contains "KB" string, it will be entered in the range text box.
                                ex: 6 to 20 KB
                                else, it is do string comparision and select the filter value (ex: 1-5 MB )
        """
        for value in values_list:
            if "SizeKB" in value:
                values = re.findall(r'\d+', value)
                from_text_box = self.__driver.find_element_by_xpath(
                    "//input[@id=\"from\"]")
                from_text_box.clear()
                from_text_box.send_keys(values[0])
                time.sleep(3)
                to_text_box = self.__driver.find_element_by_xpath(
                    "//input[@id=\"to\"]")
                to_text_box.clear()
                to_text_box.send_keys(values[1])
                time.sleep(3)
                add_element = self.__driver.find_element_by_xpath(
                    "//*[@class=\"dropdown-menu gdpr-chip-menu\"]"
                    "//span[3]/a[contains(text(),'Add')]")
                add_element.click()
            else:
                self.select_custom_filter([value])

    @WebAction(delay=4)
    def select_custom_filter(self, value_list):
        """
        select values in filter
        Args:
                valuelist (list)  - List of values(strings) to be selected for filtering
                                   ex:[abc@def.com, "test username",..]

        Returns: False : if value is not available in the list of facets
                else True
        """
        try:
            value_button_list = []
            for value in value_list:
                xpath = "//ul[@class=\"dropdown-menu gdpr-chip-menu\"]/li/label[contains(text(),\'" + value + "\')]"
                value_button_list.append(xpath)

            for path in value_button_list:
                if self.__driver.find_element_by_xpath(path):
                    self.__driver.find_element_by_xpath(path).click()
                    self.__admin_console.wait_for_completion()
            return True
        except NoSuchElementException:
            # self.clear_custodian_filter()
            return False

    @WebAction()
    def __enter_facet_value(self, value):
        """
        Enter values in textbox
        Args:
                value  - string to be selected for filtering
                                   ex:[abc@def.com, "test username",..]
        """
        text_box = self.__driver.find_element_by_xpath("(//input[@placeholder=\'Enter Email folder\'])")
        text_box.clear()
        text_box.send_keys(value[0:7])
        time.sleep(3)
        suggestion_element = self.__driver.find_element_by_xpath("//a[@ng-attr-title=\"{{match.label}}\"]")
        suggestion_element.click()

    @PageService()
    def apply_date_filters(self, daterange_list, filtername):
        """
        Applies date filter if it exists
         Args:
                daterange list(str)  - list of Date ranges whose value has to be applied
         :returns
                 True : if date filter is applied
                 False: if date filter is not applied
        """
        is_exists = False
        if self.select_filter_dropdown(filtername):
            self.select_date_filter(daterange_list)
            self.__admin_console.wait_for_completion()
            is_exists = True
        return is_exists

    @PageService()
    def apply_size_filters(self, sizerange_list, filtername):
        """
        Applies size filter if it exists
         Args:
                sizerange list(str)  - list of Size ranges whose value has to be applied
         :returns
                 True : if size filter is applied
                 False: if size filter is not applied
        """
        is_exists = False
        if self.select_filter_dropdown(filtername):
            self.select_size_filter(sizerange_list)
            self.__admin_console.wait_for_completion()
            is_exists = True
        return is_exists

    @PageService()
    def apply_custom_filters(self, filters_list):
        """
        Applies custom filters if they exist
         Args:
                 filterslist (dict)  - dictionary with filter name as Key
                and list of filter content as Value
         :returns
                dict: applied filters
        """
        parameters = {}
        for filtertype in filters_list:
            if self.select_filter_dropdown(filtertype):
                filter_values = filters_list[filtertype]
                parameters[filtertype] = self.select_custom_filter(filter_values)
            else:
                parameters[filtertype] = False
            self.__admin_console.wait_for_completion()
        return parameters

    @PageService()
    def enter_custom_filters(self, filtertype, value):
        """
        Enter the value in the facet dropdown textbox
        Args:
            filtertype: Type of filter
            value: value to be entered
        """
        self.select_filter_dropdown(filtertype)
        self.__enter_facet_value(value)
        self.__admin_console.wait_for_completion()


class _ExportToForm(ModalDialog):
    """
     This class contains all the actions for 'Export To' form
    """

    @WebAction()
    def choose_selection_range(self, selection_range):
        """
        Method to choose selection range
        Args:
            selection_range (str)  - value to be selected
        """
        radio_button = self._driver.find_element_by_xpath(
            "//span[contains(text(),'" + selection_range + "')]")
        radio_button.click()

    @WebAction()
    def choose_file_format(self, operation_type, file_format):
        """
        Method to choose selection range
        Args:
            operation_type (str) - CAB/ PST
             file_format (str)  - file format to be selected
        """
        if operation_type == "CAB":
            radio_button = self._driver.find_element_by_xpath(
                "//span[contains(text(),'" + file_format + "')]")
            radio_button.click()
        self.click_submit()


    @WebAction()
    def enter_search_filter(self, value):
        """
        enter search keyword for filter
        Args:
                value (str)  - keyword to be searched for
        """
        search = self._driver. find_element_by_xpath(
            "//div[@class='modal-content']//input[@id='searchInput']")
        search.clear()
        search.send_keys(value)
        search.send_keys(Keys.ENTER)
