# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the methods that can be done on Case Manager page.

Case Manager  --  This class contains all the methods for action in Case Manager page

    Functions:

    select_dropdown_input()     -- Selects the dropdown input value of the element given the id
    get_index_copy_job_id()     --  Gets the Job Id from the popup
    submit_collection_job()     --  Submits collection job
    enter_case_details()        --  Enters the details to add a case
    create_definition()	        --  Adds a new definition to a case
    get_keyword_email_count()   --  Gets the number of emails containing that word
    select_add_case()           --  Clicks on the 'Add case' link
    select_case()               --  Opens the case page of the case with given name
    open_search_tab()           --  Opens the search tab of a case
    open_job_history()          --  Opens job history of that particular case
    select_add_definition()     --  Clicks on 'Add definition'

"""

from Web.AdminConsole.Components.panel import DropDown, ModalPanel
from Web.AdminConsole.Components.table import Table, CVTable
from Web.Common.page_object import WebAction, PageService
from Web.AdminConsole.Helper.AdminConsoleBase import AdminConsoleBase
from Web.AdminConsole.GovernanceAppsPages.ComplianceSearch import ComplianceSearch


class CaseManager:
    """
    This class contains all the methods for action in Case Manager page
    """

    def __init__(self, admin_console):
        """
        Args:
            admin_console (AdminConsole): adminconsole base object

        """
        self.__admin_console = admin_console
        self.__driver = self.__admin_console.driver
        self.__admin_console._load_properties(self)
        self.view_jobs_label = self.__admin_console.props['label.viewJobs']
        self.__dropdown = DropDown(self.__admin_console)
        self.__panel = ModalPanel(self.__admin_console)
        self.__adminconsole_base = AdminConsoleBase(self.__driver)
        self.__table = Table(self.__admin_console)
        self.__cvtable = CVTable(self.__admin_console)
        self.__compl_search = ComplianceSearch(self.__admin_console)
        self.__filter_count = 0

    @WebAction()
    def __click_legal_hold(self):
        """
        Clicks on the link for Legal Hold during Case Creation
        """
        self.__driver.find_element_by_id('addLegalHold').click()

    @WebAction()
    def __enter_custodian_value(self, value):
        """
        Enters the custodian value
        Args:
            value (str): Email id of custodian

        """
        self.__driver.find_element_by_xpath(
            '//input[@ng-model="model.selectedCustodian"]'
        ).send_keys(value)

    @WebAction()
    def __add_custodian(self):
        """
        Clicks on the Add button to add the custodian
        """
        self.__driver.find_element_by_xpath(
            '//button[@data-ng-click="model.addCustodian()"]'
        ).click()

    @WebAction()
    def __select_filter_menu_list(self, value):
        """
        Selects the required value from the filter menu list
        Args:
            value (str): Value to be selected

        """
        values = self.__driver.find_elements_by_xpath(
            '//ul[@class="dropdown-menu case-filter-menu"]/li'
        )
        for option in values:
            if option.text == value:
                option.click()
                break

    @WebAction()
    def __enter_filter_text_input(self, value):
        """
        Enters the value in the field Attachment name
        Args:
            value (str): Value to be entered

        """
        self.__driver.find_elements_by_xpath(
            '//*[@data-ng-if="addedFilter.type === \'text\'"]'
        )[-1].find_element_by_xpath('.//input').send_keys(value)

    @WebAction()
    def __enter_filter_folder(self, value):
        """
        Enters and selects the values in the field 'Folder'
        Args:
            value (list): List of values to be selected and entered

        """
        self.__driver.find_elements_by_xpath(
            '//*[@data-ng-if="addedFilter.type === \'match\'"]/div/input'
        )[-1].send_keys(value[0])
        self.__driver.find_element_by_xpath(
            '//*[@data-ng-if="addedFilter.type === \'match\'"]/div/select'
        ).click()
        values = self.__driver.find_elements_by_xpath(
            '//*[@data-ng-if="addedFilter.type === \'match\'"]/div/select/option'
        )
        for option in values:
            if value[1] == option.text:
                option.click()
                break

    @WebAction()
    def __enter_filter_received_time(self, value):
        """
        Enters the values in the field 'Received Time'
        Args:
            value (List): Time to be entered

        """
        datepickers = self.__driver.find_elements_by_xpath(
            '//*[@data-role="datetimepicker"]'
        )
        for i in range(2):
            r_time = (
                value[i]['Month'] +
                '/' +
                value[i]['Day'] +
                '/' +
                value[i]['Year'] +
                ' ' +
                value[i]['Hours'] +
                ':' +
                value[i]['Minutes'] +
                ' ' +
                value[i]['Time'])
            datepickers[i].clear()
            datepickers[i].send_keys(r_time)

    @WebAction()
    def __add_filter(self, additional_filter, value):
        """
        Adds a filter to the case / definition
        Args:
            additional_filter (str):Field Name
            value (list/str):       Value to be entered / selected

        """
        self.__dropdown.select_drop_down_values(
            drop_down_id='queryFilter',
            values=[additional_filter]
        )
        if additional_filter in ['Email address', 'From', 'To', 'CC', 'BCC']:
            self.__adminconsole_base.fill_form_by_id('suggestionInput', value)
        elif additional_filter in ['Attachment name', 'Subject']:
            self.__enter_filter_text_input(value)
        elif additional_filter == 'Has attachment':
            self.__dropdown.select_drop_down_values(
                values=[value],
                drop_down_id=('selection-' + str(self.__filter_count))
            )
        elif additional_filter == 'Folder':
            self.__enter_filter_folder(value)
        elif additional_filter == 'Received time':
            self.__enter_filter_received_time(value)

    @WebAction()
    def get_index_copy_job_id(self):
        """
        Gets the Index Copy Job Id
        Returns:
            int: Job Id

        """
        return self.__adminconsole_base.get_jobid_from_popup()

    @WebAction()
    def __access_links(self, text):
        """
        Access links
        """
        self.__driver.\
            find_element_by_xpath("//span[contains(text(),'" + text + "')]")\
            .click()

    @WebAction()
    def delete_case_confirmation(self):
        """
        To provide confirmation for delete case
        """
        self.__driver.find_element_by_xpath(
            '//*[@id="reconfirmActionId"]'
        ).send_keys("erase and reuse media")
        self.__adminconsole_base.checkbox_select('deleteClientConfirmationChx')
        self.__driver.find_element_by_xpath(
            "//*[@id='deleteClient']"
        ).click()

    @WebAction()
    def get_custodian_count(self, case_name):
        """
        Get the count of custodians
        Returns:
            int : the cont of custodians
        """
        if self.__table.is_entity_present_in_column('Name', case_name):
            try:
                column_data = self.__table.get_column_data('Custodians')
                return column_data[0]
            except ValueError:
                self.__table.display_hidden_column('Custodians')
                column_data = self.__table.get_column_data('Custodians')
                return column_data[0]

    @PageService()
    def add_custodian_to_definition(self, definition, custodians):
        """
        Adds additional custodian
        Args:
            definition (str):  Definition to which custodian should be added
            custodians (list): Email id of custodians

        """
        self.__adminconsole_base.select_hyperlink(definition)
        self.__adminconsole_base.expand_accordion('Custodians')
        for custodian in custodians:
            self.__enter_custodian_value(custodian)
            self.__add_custodian()
        self.__panel.submit()

    @PageService()
    def submit_collection_job(self):
        """
        Submits a collection job
        """
        self.__access_links('Submit collection job')
        self.__admin_console.wait_for_completion()

    @PageService()
    def enter_case_details(self, name, datatype, data_collection,
                           custodians, dcplan, server_plan, keyword):
        """
        Enters the basic details required to create the case
        Args:
            name (str):             Name of the Case
            datatype (str):         Type of data
            data_collection (str):  One time only, Continuous or Do not collect
            custodians (List):      Email id of custodian
            dcplan (str):           Data Classification Plan
            server_plan (str):      Server Plan
            keyword (str):          Keyword based upon which case is to be created

        """
        self.__adminconsole_base.fill_form_by_id('clientName', name)
        self.__dropdown.select_drop_down_values(0, [dcplan])
        self.__dropdown.select_drop_down_values(1, [server_plan])
        self.__dropdown.select_drop_down_values(2, [datatype])
        self.__dropdown.select_drop_down_values(3, [data_collection])

        self.__adminconsole_base.expand_accordion('Custodian')
        for custodian in custodians:
            self.__enter_custodian_value(custodian)
            self.__add_custodian()
            self.__admin_console.wait_for_completion()
        self.__adminconsole_base.expand_accordion('Additional criteria')
        self.__adminconsole_base.fill_form_by_id('keyword', keyword)

        self.__panel.submit()
        self.__admin_console.wait_for_completion()

    @PageService()
    def create_definition(self, name, datatype, data_collection,
                          custodians, keyword, filters=None):
        """
        Enters the details required to create a definition
        Args:
            name (str):             Name of the Definition
            datatype (str):         Type of data
            data_collection (str):  One time only, Continuous or Do not collect
            custodians (list):      Email id of custodian
            keyword (str):          Keyword based upon which definition is to be created
            filters (dictionary):   List of Additional filter criteria

        """
        self.__adminconsole_base.fill_form_by_id('definitionName', name)
        self.__dropdown.select_drop_down_values(0, [datatype])
        self.__dropdown.select_drop_down_values(1, [data_collection])

        self.__adminconsole_base.expand_accordion('Custodian')
        for custodian in custodians:
            self.__enter_custodian_value(custodian)
            self.__add_custodian()

        self.__access_links('Additional criteria')
        self.__adminconsole_base.fill_form_by_id('keyword', keyword)
        # self.__click_query_filter_buttons()
        try:  # In case filters is None
            for key in filters.keys():
                self.__add_filter(key, filters[key])
                self.__filter_count += 1
        except AttributeError:
            pass

        self.__panel.submit()
        self.__admin_console.wait_for_completion()

    @PageService()
    def get_keyword_email_count(self, value):
        """
        Gets the count of emails which matches the keyword
        Args:
            value (str): Keyword to be searched for

        Returns:
            int: No of emails
        """
        self.__adminconsole_base.fill_form_by_id('searchInput', value)
        email_num = int(self.__table.get_total_rows_count())
        return email_num

    @PageService()
    def select_add_case(self):
        """
        Clicks on Add Case link
        """
        self.__adminconsole_base.select_hyperlink('Add case')
        self.__click_legal_hold()
        self.__admin_console.wait_for_completion()

    @PageService()
    def select_case(self, name):
        """
        Clicks on the case with given name
        Args:
            name (str): Name of the case

        """
        self.__adminconsole_base.select_hyperlink(name)

    @PageService()
    def open_search_tab(self):
        """
        Opens the search tab in the Case Details Page
        """
        self.__adminconsole_base.access_tab('Search')

    @PageService()
    def open_overview_tab(self):
        """
        Opens the overview tab in the Case details Page
        """
        self.__adminconsole_base.access_tab('Overview')

    @PageService()
    def open_job_history(self):
        """
        Opens the Job History of that particular case
        """
        self.__access_links(self.view_jobs_label)
        self.__admin_console.wait_for_completion()

    @PageService()
    def select_add_definition(self):
        """
        Clicks on Add definition link
        """
        self.__adminconsole_base.select_hyperlink('Add definition')

    @PageService()
    def delete_case(self, case_name):
        """
        Delete case with given name
        Args:
            case_name: Case to be deleted
        """
        self.__table.access_action_item(case_name, 'Close case')
        self.__compl_search.click_confirm_delete_option()
        self.__admin_console.wait_for_completion()
        self.__table.access_action_item(case_name, 'Delete', partial_selection=True)
        self.__compl_search.click_confirm_delete_option()
        self.__admin_console.wait_for_completion()
        self.delete_case_confirmation()
        self.__admin_console.wait_for_completion()
