# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
subclient of the File System agent on the AdminConsole

Class:
    FsSubclientDetails

Functions:


    backup_history()               -- To view backup history of client

    restore_history()              -- To view restore history of client

    backup()                       -- To backup selected subclient

    delete_subclient()             --  To delete the subclient

    restore_recovery_points()      -- Performs a point in time restore

    assign_plan()                  -- To assign a plan to a subclient

    set_backup_activity()          -- Enable or disable the data backup option

    set_pre_backup_command()       -- Sets the pre backup process

    set_post_backup_command()      --  Sets the post backup process

    set_pre_post_advanced()        -- To set pre and post process scan/backup commands.

    clear_pre_post_commands()      --  Clears the pre post commands set already

    edit_content()                 -- edit the content of the subclient for backup

    enable_snapshot_engine()       -- Sets the snapshot engine for the subclient

    set_block_level()              -- To set block level option

"""
import os
from AutomationUtils.machine import Machine
from Web.AdminConsole.Components.browse import Browse, ContentBrowse
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.AdminConsole.Components.panel import Backup, PanelInfo, DropDown
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.FileServerPages.file_servers import RestorePanel
from Web.Common.page_object import PageService, WebAction


class FsSubclientDetails:

    """
    This class provides the function or operations that can be performed on the
    subclient of the File System iDA on the AdminConsole
    """
    def __init__(self, admin_console):
        """
        Args:
            admin_console (AdminConsole): adminconsole base object

        """
        self.__admin_console = admin_console
        self.__driver = admin_console.driver

        # Components required
        self.__table = Table(self.__admin_console)
        self.__modal_dialog = ModalDialog(self.__admin_console)
        self.__contentbrowse = ContentBrowse(self.__admin_console)
        self.__panel = PanelInfo(self.__admin_console)
        self.__snappanel = PanelInfo(self.__admin_console, 'Snapshot')
        self.__blockpanel = PanelInfo(self.__admin_console, 'Block level backup')
        self.__backuppanel = Backup(self.__admin_console)
        self.__restore_panel = RestorePanel(self.__admin_console)
        self.__browse = Browse(self.__admin_console)
        self.__dropdown = DropDown(self.__admin_console)
        self.__admin_console._load_properties(self)

    @PageService()
    def backup(self, backup_level, notify=False, drop_down=False):
        """"Back up option on the title bar
        Args:

            backup_level   (enum)   : type of backup

            notify         (bool)   : To notify via mail about the backup

            drop_down      (bool)   : To access the backup action from drop down

        Returns :
             job_id : Job ID of the backup job

        Raises:
            Exception :

             -- if fails to run the backup
        """
        if drop_down:
            self.__admin_console.access_menu_from_dropdown(self.__admin_console.props['label.globalActions.backup'])
        else:
            self.__admin_console.access_menu(self.__admin_console.props['label.globalActions.backup'])
        job_id = self.__backuppanel.submit_backup(backup_level, notify)

        return job_id

    @PageService()
    def backup_history(self):
        """"Backup history containing jobs of all subclients"""
        self.__admin_console.access_menu(self.__admin_console.props['label.BackupHistory'])

    @PageService()
    def restore_history(self):
        """"Restore history containing jobs of all subclients"""
        self.__admin_console.access_menu_from_dropdown(self.__admin_console.props['label.RestoreHistory'])

    @PageService()
    def delete_subclient(self):
        """
        Deletes the subclient

        Raises:
            Exception:
                if the subclient could not be deleted

        """
        self.__admin_console.access_menu_from_dropdown(self.__admin_console.props['action.delete'])
        self.__modal_dialog.type_text_and_delete("DELETE")
        self.__admin_console.check_error_message()

    @WebAction()
    def _change_plan(self, plan):
        """
        change the plan associated to subclient

            Args:
                     plan(Str)           : Plan to be selected

        """
        xp = f"//a[contains(text(),'Edit')]"
        self.__driver.find_elements_by_xpath(xp)[0].click()
        self.__admin_console.wait_for_completion()
        self.__dropdown.select_drop_down_values(0, [plan])
        self.__driver.find_element_by_xpath('//span[@class="k-icon k-i-check"]').click()

    @PageService()
    def assign_plan(self, plan):
        """
        To assign the specified plan to the subclient.

            Args:
                     plan(Str)           : Plan to be selected

        Raises:
            Exception:
                if the plan cannot be set
        """
        self._change_plan(plan)

    @PageService()
    def set_backup_activity(self, enable):
        """
        Enable or disable the Backup Enabled toggle.

            Args:
                     enable(bool)           : True: Enables the data backup field

                                              False : Disables the data backup field
        Returns:
            None

        Raises:
            Exception:
                Toggle button not found
        """
        if enable:
            self.__panel.enable_toggle(self.__admin_console.props['label.backupEnabled'])
        else:
            self.__panel.disable_toggle(self.__admin_console.props['label.backupEnabled'])

    @PageService()
    def restore_recovery_points(self, recovery_time, dest_client=None, restore_path=None, unconditional_overwrite=False,
                                notify=False, selected_files=None, hadoop_restore=False):
        """"Point in time restores by selecting the date for recovery points

                Args:

                     recovery_time(str)     : The date specified in DD--MM--YYYY format

                     dest_client (str)      : Name of the destination client

                     restore_path(str)      : The destination path to which content should be restored to

                     unconditional_overwrite(bool)  : To overwrite unconditionally on destination path

                     notify(bool)           : To notify via mail about the restore

                     selected_files (list)  : list of (str) paths of restore content

                     hadoop_restore(bool)   : To indicate restore is for hadoop server
                         default: False

                Returns :
                     job_id : job id of the restore

                Raises:
                    Exception :

                     -- if fails to run the restore operation

        """
        self.__admin_console.recovery_point_restore(recovery_time)
        if selected_files:
            delimiter = '\\'
            paths = os.path.dirname(selected_files[0])
            if '/' in selected_files[0]:
                delimiter = '/'
            if delimiter == '/':
                paths = paths.strip('/')
            paths = paths.split(delimiter)
            select_files = [os.path.basename(file) for file in selected_files]
            for folder in paths:
                self.__browse.access_folder(folder)
            self.__browse.select_for_restore(file_folders=select_files)

        else:
            self.__browse.select_for_restore(all_files=True)
        self.__browse.submit_for_restore()
        if hadoop_restore and restore_path is not None:
            self.__restore_panel.access_tab(self.__admin_console.props['label.OOPRestore'])
        if dest_client:
            self.__restore_panel.select_destination_client(dest_client)
        if restore_path:
            self.__restore_panel.select_browse_for_restore_path(toggle_inplace=not hadoop_restore)
            self.__contentbrowse.select_path(restore_path)
            self.__contentbrowse.save_path()
        if unconditional_overwrite:
            self.__restore_panel.select_overwrite()
        job_id = self.__restore_panel.submit_restore(notify)

        return job_id

    @WebAction()
    def _choose_pre_backup_process(self):
        """
        CLicks on the pre backup process folder icon

        """
        self.__driver.find_element_by_xpath(f'//a[@ng-click="prepostCommandsTileCtrl.setBackupCommands()"]/span').click()

        self.__admin_console.wait_for_completion()


    @WebAction()
    def _choose_post_backup_process(self):
        """
       click on the post backup process folder icon.

        """
        self.__driver.find_element_by_xpath(f'//a[@ng-click="prepostCommandsTileCtrl.setBackupCommands(true)"]/span').click()
        self.__admin_console.wait_for_completion()

    @PageService()
    def set_pre_backup_command(self, command_path):

        """
       Sets the pre backup processes.

        Returns:
            None

        Raises:
            Exception:
                command button not found

        """
        self._choose_pre_backup_process()
        self.__contentbrowse.select_path(command_path)
        self.__contentbrowse.save_path()

    @PageService()
    def set_post_backup_command(self, command_path):

        """
       Sets the post backup processes.

        Returns:
            None

        Raises:
            Exception:
                command button not present

        """
        self._choose_post_backup_process()
        self.__contentbrowse.select_path(command_path)
        self.__contentbrowse.save_path()

    @WebAction()
    def _add_command_process(self, index, path):
        """
        To click on folders to browse and select the pre/post commands
        """
        xp = f"//span[@class='k-icon k-i-folder-more font-size-page-title']"
        self.__driver.find_elements_by_xpath(xp)[index].click()
        self.__admin_console.wait_for_completion()
        self.__contentbrowse.select_path(path)
        self.__contentbrowse.save_path()

    @PageService()
    def set_pre_post_advanced(self, pre_scan_process=None, post_scan_process=None, pre_backup_process=None,
                              post_backup_process=None, run_scan_all_attempts=False, run_backup_all_attempts=False,
                              impersonate_user=False, username=None, password=None):
        """
        To set pre and post process commands.
        """
        self.__admin_console.select_hyperlink(self.__admin_console.props['label.advancedSetting'])

        if pre_scan_process:
            self._add_command_process(0, pre_scan_process)

        if post_scan_process:
            self._add_command_process(1, post_scan_process)

        if pre_backup_process:
            self._add_command_process(2, pre_backup_process)

        if post_backup_process:
            self._add_command_process(3, post_backup_process)

        if run_scan_all_attempts:
            self.__panel.enable_toggle(self.__admin_console.props['label.runPostScan'])

        if run_backup_all_attempts:
            self.__panel.enable_toggle(self.__admin_console.props['label.runPostBackup'])

        if impersonate_user:
            self.__panel.enable_toggle(self.__admin_console.props['label.impersonateUser'])
            if impersonate_user == "Enter credentials":
                self.__admin_console.check_radio_button(impersonate_user)
                self.__admin_console.fill_form_by_id("uname", username)
                self.__admin_console.fill_form_by_id("pass", password)
            else:
                self.__admin_console.check_radio_button(impersonate_user)

        self.__backuppanel.submit()

    @PageService()
    def clear_pre_post_commands(self, input_id):
        """

        Clears the pre post commands set already

        Args :

                input_id(str)   - The input field ID that needs to be cleared.

                    Examples:

                        preBackupCommand    -   For Pre backup command
                        postBackupCommand   -   For Post backup commands
                        preScanCommand      -   For Pre Scan command
                        postScanCommand     -   For Post Scan commands

        Returns:
            None

        Raises:
            Exception:
                The pre/post commands could not be reset.
        """
        self.__admin_console.select_hyperlink(self.__admin_console.props['label.advancedSetting'])
        self.__driver.find_element_by_id(input_id).clear()
        self.__backuppanel.submit()

    @PageService()
    def edit_content(self, browse_and_select_data, backup_data,
                     del_content=None, exceptions=None,
                     del_exceptions=None, exclusions=None,
                     del_exclusions=None, file_system='Windows'):
        """
        Edits the content of the subclient by adding or removing files and folders.

        Args:

            browse_and_select_data(bool)  : Pass True to browse and select data,
                                            False for custom path

            backup_data     (list(paths)) : Data to be backed up by new sub client created
                Eg. backup_data = ['C:\\TestBackupSet1', C:\\TestBackupSet2']

            del_content     (list(paths)) : Data to be removed in new sub client created
                Eg. backup_data = ['C:\\TestBackupSet1', C:\\TestBackupSet2']

            exclusions       (list(paths)) : Data to be backed up by new sub client created
                Eg. exclusions = ['C:\\TestBackupSet1', C:\\TestBackupSet2']

            del_exclusions     (list(paths)) : Removed exclsuions of new sub client created
                Eg. backup_data = ['C:\\TestBackupSet1', C:\\TestBackupSet2']

            exceptions       (list(paths)) : Data to be backed up by new sub client created
                Eg. exceptions = ['C:\\TestBackupSet1', C:\\TestBackupSet2']

            del_exceptions     (list(paths)) : Removed exclsuions of new sub client created
                Eg. backup_data = ['C:\\TestBackupSet1', C:\\TestBackupSet2']


            file_system       (string)     :  file system of the client
                Eg. - 'Windows' or 'Unix'

        Returns:
            None

        Raises:
            Exception:
                There is no option to edit the content of the collection
        """
        toggle = self.__panel.get_toggle_element(self.__admin_console.props['label.backupContentFromPlan'])
        status = self.__panel.is_toggle_enabled(toggle)
        self.__admin_console.tile_select_hyperlink(self.__admin_console.props['header.content'],
                                                   self.__admin_console.props['action.edit'])
        if status:
            self.__modal_dialog.click_submit()

        if del_content:
            for path in del_content:
                self.__remove_existing_path(path)

        if del_exclusions:
            self.__admin_console.select_hyperlink(self.__admin_console.props['label.Exclusions'])
            for path in del_exclusions:
                self.__remove_existing_path(path)

        if del_exceptions:
            self.__admin_console.select_hyperlink(self.__admin_console.props['label.Exceptions'])
            for path in del_exceptions:
                self.__remove_existing_path(path)

        self.__admin_console.select_hyperlink(self.__admin_console.props['header.content'])
        if browse_and_select_data:
            self.__admin_console.select_hyperlink(self.__admin_console.props['action.browse'])
            self.__browse_and_select_data(backup_data, file_system)

        else:
            for path in backup_data:
                self.__add_custom_path(path)
                self.__click_add_custom_path()

        if exclusions:
            self.__admin_console.select_hyperlink(self.__admin_console.props['label.Exclusions'])
            for path in exclusions:
                self.__add_custom_path(path)
                self.__click_add_custom_path()

        if exceptions:
            self.__admin_console.select_hyperlink(self.__admin_console.props['label.Exceptions'])
            for path in exceptions:
                self.__add_custom_path(path)
                self.__click_add_custom_path()

        self.__admin_console.click_button(self.__admin_console.props['OK'])
        self.__admin_console.check_error_message()
        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @WebAction()
    def __remove_existing_path(self, path):
        """ Deletes the existing files
                Args:
                     path(str)     :   file path to be removed
        """
        elem1 = f"//div[contains(@class,'ui-grid-cell ng-scope ui-grid-coluiGrid')]/span[(text()='{path}')]/parent::div"
        result = self.__driver.find_elements_by_xpath(elem1)
        get_id = ''
        for file in result:
            if file.is_displayed():
                get_id = self.__driver.find_element_by_xpath(elem1).get_attribute("id").split('-')
                get_id = get_id[0]+'-'+get_id[1]
        if get_id:
            elem1 = f"//div[contains(@class,'ui-grid-selection')]/ancestor::div[contains(@class,'ui-grid-cell ng-scope') and contains(@id,'{get_id}')]"
            self.__driver.find_element_by_xpath(elem1).click()
        self.__admin_console.select_hyperlink(self.__admin_console.props['label.globalActions.remove'])

    @WebAction()
    def __add_custom_path(self, path):
        """Add custom paths in the path input box
                Args:
                    path (str)      :   Data path to be added
        """
        custom_path_input_xpath = "//input[@placeholder='Enter custom path']"
        custom_path_input = self.__driver.find_elements_by_xpath(custom_path_input_xpath)
        for path_input in custom_path_input:
            if path_input.is_displayed():
                path_input.clear()
                path_input.send_keys(path)
        self.__admin_console.wait_for_completion()

    @WebAction()
    def __click_add_custom_path(self):
        """Clicks the add custom path icon"""
        add_path_icon_xpath = "//i[@title='Add']"
        custom_path_add = self.__driver.find_elements_by_xpath(add_path_icon_xpath)
        for path_add in custom_path_add:
            if path_add.is_displayed():
                path_add.click()
        self.__admin_console.wait_for_completion()

    @WebAction()
    def __browse_and_select_data(self, backup_data, file_system):
        """
        selects backup data through FS Browse

        Args:

            backup_data     (list(paths)) : Data to be backed up by new sub client created
                Eg. backup_data = ['C:\\TestBackupSet1', C:\\TestBackupSet2']

            file_system      (string)     :  file system of the client
                Eg. - 'Windows' or 'Unix'

        Returns:
            None

        Raises:
            Exception:
                if not able to select data
        """

        for data in backup_data:
            count = 0
            flag = True
            if file_system.lower() == 'windows':
                pattern = "\\"
            else:
                pattern = "/"
            directories = []
            start = 0
            while flag:
                tag = data.find(pattern, start)
                if tag == -1:
                    flag = False
                else:
                    count += 1
                    start = tag+1

            for i in range(0, count+1):
                directory, sep, folder = data.partition(pattern)
                data = folder
                if directory != '':
                    directories.append(directory)
            path = len(directories)

            for i in range(0, path-1):
                if self.__driver.find_element_by_xpath(
                        "//span[contains(text(),'"+str(directories[i])+"')]/../../button").\
                        get_attribute("class") == 'ng-scope collapsed':
                    self.__driver.find_element_by_xpath(
                        "//span[contains(text(),'"+str(directories[i])+"')]/../../button").click()
                dest = i+1
                self.__driver.find_element_by_xpath(
                    "//span[contains(text(),'" + str(directories[dest]) + "')]").click()
        self.__admin_console.click_button(self.__admin_console.props['label.save'])

    @PageService()
    def enable_snapshot_engine(self, enable_snapshot, engine_name=None):
        """
        Sets the snapshot engine for the subclient
        Args:
            enable_snapshot (bool):     to enable / disable snap backups on the subclient

            engine_name     (basestring):   name of the snapshot engine

        Returns:
            None

        Raises:
            Exception:
                if not able to change snapshot
        """
        toggle = self.__snappanel.get_toggle_element(self.__admin_console.props['label.enableHardwareSnapshot'])
        status = self.__snappanel.is_toggle_enabled(toggle)
        if enable_snapshot:
            if status:
                self.__admin_console.tile_select_hyperlink(self.__admin_console.props['label.snapshot'], self.__admin_console.props['action.edit'])
            else:
                self.__snappanel.enable_toggle(self.__admin_console.props['label.enableHardwareSnapshot'])
            if not engine_name:
                raise Exception("The engine name is not provided")
            self.__dropdown.select_drop_down_values(0, [engine_name])
        else:
            self.__snappanel.disable_toggle(self.__admin_console.props['label.enableHardwareSnapshot'])
        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @PageService()
    def set_block_level(self, blocklevel):
        """
        Sets the block level option for the subclient

        Args:
            blocklevel (bool):     To enable / disable block level on the subclient

        Returns:
            None

        Raises:
            Exception:
                if not able to change snapshot
        """
        if blocklevel:
            self.__blockpanel.enable_toggle(self.__admin_console.props['label.enabled'])

        else:
            self.__blockpanel.disable_toggle(self.__admin_console.props['label.enabled'])
