# -*- coding: utf-8 -*-

# ---------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# ----------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
File servers page in admin console

FileServers:

    access_server()       -> Method to access the required file server

    backup_subclient()    -> Method to run backup operation from file servers page

    run_check_readiness() -> Method to run check readiness operation from file servers page

    restore_client()      -> Method to run restore operation from file servers page

    view_jobs()           -> Method to view the jobs of the sever

    install_windows_unix_client()	-> Method to add Windows/Unix Server

    add_distributed_app()           -> Create distributed file server with given parameters.

    add_ibmi_client()               -> Method to add a new IBMi client.

    is_client_exists()              -> check client entry from file servers page

    retire_server()                 -> retire client

    action_uninstall_software()     -> uninstall all/selected packages

    action_update_software()        -> Update client

    action_add_software()           -> Add packages to existing client

    delete_server()                 -> delete client

    reconfigure_server()             -> reconfigure client



RestorePanel:

    select_destination_client()     :       Select destination client for restore

    select_nas_destination_client()     :   Select nas destination for nas


    select_backupset_and_subclient():       Selects the backup set and subclient to be restored

    search_files_for_restore()      :       Search for the files to restore

    select_overwrite()              :       Selects the unconditional overwrite option

    select_browse_for_restore_path():       click on browse during selecting destination path

    submit_restore()                :       restores the data as per options selected


AddPanel:

    add_custom_path()               :       Adds data paths to content,exclusions,exceptions tab

    click_add_custom_path()         :       clicks the add custom path plus icon

    toggle_own_content()            :       Toggle to override backup content

    browse_and_select_data()        :       To browse and choose content data

    remove_plan_content()           :       To remove inherited plan content

    add()                           :       Method to add content,exclusions,exceptions

"""
import os
from selenium.webdriver.common.action_chains import ActionChains
from AutomationUtils.constants import DistributedClusterPkgName
from Web.Common.exceptions import CVWebAutomationException
from Web.Common.page_object import (WebAction, PageService)
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import Backup, ModalPanel, DropDown
from Web.AdminConsole.Components.browse import Browse, ContentBrowse
from Web.AdminConsole.NAS.nas_file_servers import NASFileServers


class FileServers:
    """class for file servers page"""

    def __init__(self, admin_console):
        """
        Args:
        admin_console(AdminConsole): adminconsole
        object
        """
        self._admin_console = admin_console
        self._driver = admin_console.driver
        self._admin_console._load_properties(self)
        self.nas_file_server = NASFileServers(self._admin_console)

        # Components required
        self.__table = Table(self._admin_console)
        self.__panel = Backup(self._admin_console)
        self.__restore_panel = RestorePanel(self._admin_console)
        self.__browse = Browse(self._admin_console)
        self.__contentbrowse = ContentBrowse(self._admin_console)
        self.__drop_down = DropDown(self._admin_console)
        self.__modal_panel = ModalPanel(self._admin_console)
        self.__addpanel = AddPanel(self._admin_console)

    @WebAction()
    def __get_affected_client(self):
        """ Method to get affected client list"""
        return self._driver.find_element_by_xpath(
            "//p[@ng-bind-html='previewCompanyChangeCtrl.eligibleServersString']").text

    @WebAction()
    def __click_add_client_button(self):
        """Method to click plus icon button on Add file server pane"""
        xpath = "//span[@id='add-host-name']"
        self._driver.find_element_by_xpath(xpath).click()

    @WebAction()
    def __click_type_filter_drop_down(self):
        """Method to click on the type filter drop down"""
        type_filter_xpath = \
            "//div[@class='uib-dropdown cv-server-filter dropdown cv-k-header-dropdown']"
        self._driver.find_element_by_xpath(type_filter_xpath).click()

    @WebAction()
    def __select_file_server_type(self, fs_type):
        """Method to select an option from type filter drop down"""
        fs_type_option_xpath = f"//a[contains(text(), '{fs_type}')]"
        self._driver.find_element_by_xpath(fs_type_option_xpath).click()

    @WebAction()
    def __expand_backup_configuration_accordion(self):
        """Method to expand Backup Configuration accordion which contains plan input"""
        acc_xp = \
            "//span[contains(text(), 'Backup configuration')]/ancestor::div[contains(@class, 'cv-accordion-header')]"
        accordion_elt = self._driver.find_element_by_xpath(acc_xp)
        if "expanded" not in accordion_elt.get_attribute("class"):
            ActionChains(self._driver).move_to_element(
                accordion_elt).click().perform()

    @WebAction()
    def __get_job_id_page(self):
        """
        Returns job_id from the page

        """
        return self._driver.find_element_by_xpath(
            '//div[@data-ng-controller="pushInstallController"]//a[contains(@href,"jobs")]').text

    @WebAction()
    def __get_job_id_dialog(self):
        """
        Returns job_id from the dialog that appears

        """
        return self._driver.find_element_by_xpath('//div[contains(text(), "Job ID:")]').text

    @PageService()
    def access_server(self, client_name):
        """
        Method to access the required server

        Args:
            client_name (str)      : Name of the client to be accessed

        Returns :
            None

        Raises:
            Exception :

            -- if the client doesn't exist

        """

        self.__table.access_link(client_name)
        self._admin_console.wait_for_completion()

    @PageService()
    def backup_subclient(
            self,
            client_name,
            backup_level,
            backupset_name=None,
            subclient_name=None,
            notify=False,
            **kwargs):
        """
        Method to backup the given subclient

        Args:
             client_name (str)      : Name of the client to be backed up

             backupset_name (str)   : backup set name of the client

             subclient_name (str)   : subclient to be backed up

             backup_level   (enum)   : type of backup

            notify(bool)           : To notify via mail about the backup

            \*\*kwargs  (dict)              --  Optional arguments.

            Available kwargs Options:

                    agent   (str)   :   The agent, if applicable.
                    Accepted values are NDMP, CIFS and NFS.

        Returns :
             job_id : Job ID of the backup job

        Raises:
            Exception :

             -- if fails to run the backup
        """

        if str(backup_level).lower() == "full":
            backup_level = self.__panel.BackupType.FULL
        elif str(backup_level).lower() == "incremental":
            backup_level = self.__panel.BackupType.INCR
        elif str(backup_level).lower() == "synthfull":
            backup_level = self.__panel.BackupType.SYNTH

        self.__table.access_action_item(client_name, "Back up")

        if kwargs.get('agent', False):
            self._admin_console.click_by_id('agent-filter')
            self._admin_console.select_hyperlink(kwargs.get('agent').upper())

        if backupset_name:
            job_id = self.__panel.submit_backup(
                backup_level, backupset_name, subclient_name, notify)
        else:
            job_id = self.__panel.submit_backup(backup_type=backup_level, notify=notify)

        return job_id

    @PageService()
    def run_check_readiness(self, client_name):
        """ Method to run check readiness on the given client

              Args:
                  client_name (str) --- Name of client on which check readiness operation to run

              Returns:
                      None

              Raises:
                      Exception:

                         -- if fails to run check readiness operation

              """
        self.__table.access_action_item(
            client_name, self._admin_console.props['label.readinessCheck'])
        self._admin_console.check_error_message()

    @PageService()
    def view_live_logs(self, client_name):
        """ Method to run check readiness on the given client

              Args:
                  client_name (str) --- Name of client on which check readiness operation to run

              Returns:
                      None

              Raises:
                      Exception:

                         -- if fails to run check readiness operation

              """
        self.__table.access_action_item(client_name, 'View logs')
        self._admin_console.check_error_message()

    @PageService()
    def restore_subclient(
            self,
            client_name,
            dest_client=None,
            restore_path=None,
            backupset_name=None,
            subclient_name=None,
            unconditional_overwrite=False,
            notify=False,
            selected_files=None,
            **kwargs):
        """
                Method to Restore the given client data

                Args:
                     client_name (str)      : Name of the client

                     dest_client (str)      : Name of the destination client

                     restore_path(str)      : The destination path to which content
                                              should be restored to

                     backupset_name (str)   : backup set name of the client

                     subclient_name (str)   : subclient name

                     unconditional_overwrite(bool)  : To overwrite unconditionally
                                                      on destination path

                     notify(bool)           : To notify via mail about the restore
                     selected_files (list)  : list of (str) paths pf restore content

                    kwargs  (dict)          :   Optional keyword arguments.

                        impersonate_user    (dict)  :  username and password are keys.

                Returns :
                     job_id : job id of the restore

                Raises:
                    Exception :

                     -- if fails to run the restore operation
                """

        self.__table.access_action_item(
            client_name, self._admin_console.props['label.restore'])
        if backupset_name:
            self.__restore_panel.select_backupset_and_subclient(backupset_name, subclient_name)
            self.__restore_panel.submit(wait_for_load=True)
        if selected_files:
            delimiter = '\\'
            paths = os.path.dirname(selected_files[0])
            if '/' in selected_files[0]:
                delimiter = '/'
            if delimiter == '/':
                paths = paths.strip('/')
            paths = paths[:-1:].rsplit(delimiter, paths[:-1:].count("\\") - 2) if paths.startswith(
                r'\\') else paths.split(delimiter)
            select_files = [os.path.basename(file) for file in selected_files]
            for folder in paths:
                self.__browse.access_folder(folder)
            self.__browse.select_for_restore(file_folders=select_files)

        else:
            self.__browse.select_for_restore(all_files=True)
        self.__browse.submit_for_restore()
        self._admin_console.wait_for_completion()

        if dest_client:
            self.__restore_panel.select_destination_client(dest_client)
        if restore_path:
            self.__restore_panel.select_browse_for_restore_path()
            self.__contentbrowse.select_path(restore_path)
            self.__contentbrowse.save_path()
        if unconditional_overwrite:
            self.__restore_panel.select_overwrite()
        if kwargs.get('impersonate_user', False):
            self._admin_console.toggle_enable("Impersonate user")
            self._admin_console.fill_form_by_name("impersonateUserName", kwargs['impersonate_user']['username'])
            self._admin_console.fill_form_by_name("impersonatePassword", kwargs['impersonate_user']['password'])
        job_id = self.__restore_panel.submit_restore(notify)

        return job_id

    @PageService()
    def view_jobs(self, client_name):
        """
                Method to view job history for the client

                Args:
                     client_name (str)      : Name of the client

                Returns :
                     None

                Raises:
                    Exception :

                     -- if fails to run the view jobs operation
                """
        self.__table.access_action_item(
            client_name, self._admin_console.props['label.globalActions.viewJobs'])

    @PageService()
    def migrate_client_to_company(self, client_name, company_name):
        """
        Method to migrate client from one company to another

        Args:
            client_name (str) ; Name of the client to be migrated

            company_name (str) : Name of the company to be migrated to

        """
        self.__table.access_action_item(client_name, 'Change company')
        self.__drop_down.select_drop_down_values(0, list(company_name))
        self.__modal_panel.submit()
        client = self.__get_affected_client()
        if not client_name == client:
            raise CVWebAutomationException('Affected client list displays %s instead of %s' % (client, client_name))
        self.__modal_panel.submit()
        self._admin_console.check_error_message()

    @WebAction()
    def _enable_toggle(self, text):
        """ Method to select toggle
        Args:

            text(str): The text beside the toggle button

        """
        self._driver.find_element_by_xpath(f"//span[contains(text(),'{text}')]").click()

    @WebAction()
    def _set_port_number(self, port):
        """ Method to select toggle
        Args:

            port(int): The text beside the toggle button

        """
        self._driver.find_element_by_id("sshPortNumber").send_keys(port)

    @WebAction()
    def _clear_port_number(self):
        """ Method to select toggle"""
        self._driver.find_element_by_id("sshPortNumber").clear()

    @WebAction()
    def _select_server_type(self, server_type):
        """ Selects Server Type"""
        self._driver.find_element_by_xpath(f"//div[contains(text(),'{server_type}')]").click()

    @WebAction()
    def validate_type_filter(self):
        """ Validates if Type filter is present in File Servers page """
        return self._admin_console.check_if_entity_exists(
            "xpath", "//span[text()='Type']/../a[contains(@class,'uib-dropdown-toggle')]")

    @PageService()
    def install_windows_unix_client(self,
                                    file_server_host_name,
                                    username,
                                    password,
                                    os_type="Windows",
                                    reboot_required=False,
                                    plan=None,
                                    port=None,
                                    define_own_content=False,
                                    exclusions=None,
                                    exceptions=None,
                                    remove_plan_content=False,
                                    backup_system_state=False,
                                    browse_and_select_data=False,
                                    impersonate_user=None,
                                    backup_data=None):
        """Method to install a new file server

        Args:

            file_server_host_name(str)   :   hostname of the server to be added

            username(str)   :   username of the server being added

            password(str)   :   password of the server being added

            os_type(Str)         :   The Operating system of the server(windows/Unix)

            reboot_required(bool)     :  Reboot server

            port(int)        :  The non-standard SSH port number. Applicable only to Unix/Linux

            plan(str)        :  The plan to be associated with the server

            define_own_content(bool)    :   True if we want to override plan content
                                            False if we don't want to override plan content

            browse_and_select_data(bool)  : Pass True to browse and select data,
                                            False for custom path

            backup_data     (list(paths)) : Data to be backed up by new sub client created
                Eg. backup_data = ['C:\\TestBackupSet1', C:\\TestBackupSet2']


            exclusions       (list(paths)) : Data to be backed up by new sub client created
                Eg. exclusions = ['C:\\TestBackupSet1', C:\\TestBackupSet2']

            exceptions       (list(paths)) : Data to be backed up by new sub client created
                Eg. exceptions = ['C:\\TestBackupSet1', C:\\TestBackupSet2']

            backup_system_state (boolean)  :  boolean values to determine if system state to
                                            be backed up or not

            impersonate_user    (string): Specify username (eg: for UNC paths).

            remove_plan_content (bool)      : True for removing content inherited from plan
                                              False for keeping content inherited from plan

        """
        self._admin_console.select_hyperlink("Add server")
        self._select_server_type(self._admin_console.props['label.dataSource'])
        self._admin_console.wait_for_completion()
        self._admin_console.fill_form_by_id("hostName", file_server_host_name)
        self._admin_console.fill_form_by_id("fakeusernameremembered", username)
        self._admin_console.fill_form_by_id("fakepasswordremembered", password)
        self._admin_console.fill_form_by_id(
            "confirmPasswordPushInstall", password)
        if os_type == "Windows":
            self._admin_console.select_radio("osTypeWINDOWS")
        elif os_type == "Unix" or os_type == 'Linux':
            self._admin_console.select_radio("osTypeUNIX")
            if port:
                self._enable_toggle(self._admin_console.props['label.userSSHPortNumber'])
                self._admin_console.wait_for_completion()
                self._clear_port_number()
                self._admin_console.wait_for_completion()
                self._set_port_number(port)
                self._admin_console.wait_for_completion()
        else:
            raise Exception("File server OS type not valid")
        if reboot_required:
            self._enable_toggle(self._admin_console.props['label.foreceReboot'])

        if plan:
            self.__expand_backup_configuration_accordion()
            self.__addpanel.add(
                plan=plan,
                define_own_content=define_own_content,
                browse_and_select_data=browse_and_select_data,
                backup_data=backup_data,
                impersonate_user=impersonate_user,
                exclusions=exclusions,
                exceptions=exceptions,
                file_system=os_type,
                remove_plan_content=remove_plan_content,
                backup_system_state=backup_system_state,
                submit=False)

        self._admin_console.click_button_using_text("Install")
        job_id = self._admin_console.get_jobid_from_popup()
        return job_id

    @PageService()
    def add_distributed_app(self, pkg, server_name, access_nodes, plan_name):
        """Create distributed file server with given parameters.

            Args:
                pkg              (enum)  -- Instance of constants.DistributedClusterPkgName

                server_name      (str)   -- Name of the server to be created.

                access_nodes     (list)  -- list of access nodes to select

                plan_name        (str)   -- plan name to select

        """
        add_menu_id = None
        if pkg == DistributedClusterPkgName.GLUSTERFS:
            add_menu_id = 'addGlusterServer'
        elif pkg == DistributedClusterPkgName.LUSTREFS:
            add_menu_id = 'addLustreServer'
        elif pkg == DistributedClusterPkgName.GPFS:
            add_menu_id = 'addGPFSServer'
        if add_menu_id is None:
            raise CVWebAutomationException("Invalid Distributed Cluster Package name provided")
        self._admin_console.click_by_id("addFileServer")
        self._admin_console.click_by_id(add_menu_id)
        self._admin_console.fill_form_by_id('serverName', server_name)
        self.__drop_down.select_drop_down_values(drop_down_id='accessnode-selection-dropdown', values=access_nodes)
        self.__drop_down.select_drop_down_values(drop_down_id='planSummaryDropdown', values=[plan_name])
        self._admin_console.submit_form()

    @PageService()
    def add_ibmi_client(self,
                        server_name,
                        file_server_host_name,
                        username,
                        password,
                        access_node,
                        data_path=None,
                        port=None,
                        plan=None,
                        subsystem_description=None,
                        job_queue=None,
                        create_jobq=None,
                        job_priority=None,
                        run_priority=None
                        ):
        """Method to add a new IBMi client.

        Args:

            server_name(str)                :   Name of the client

            file_server_host_name(str)      :   hostname of the server to be added

            username(str)                   :   username of the server being added

            password(str)                   :   password of the server being added

            access_node(list)                :   Name of the access node client

            data_path(str)                  :   Commvault data path on client machine.
                default: None (to take default value populated in CC)

            port(int)                       :  The port number on client machine
                default: None

            plan(str)                       :  The plan to be associated with the server
                default: None

            subsystem_description(str)      :   Subsystem description in <LIB>/<SBSD> format.
                default: None

            job_queue(str)                  :   job queue in <LIB>/<JOBQ> format.
                default: None

            create_jobq(bool)               :   Create the jobq under CVLIBOBJ library
                default: None

            job_priority(int)               :   job priority of Commvault jobs on client machine
                defalt: None

            run_priority(int)               : run priority of Commvault jobs on client machine
                default: None
        Return: notification

        """
        self._admin_console.select_hyperlink(self._admin_console.props["label.installSoftware"])
        self._admin_console.access_sub_menu(self._admin_console.props["label.iBMiServer"])

        self._admin_console.fill_form_by_id("clientName", server_name)
        self._admin_console.fill_form_by_id("hostName", file_server_host_name)
        if data_path is not None:
            self._admin_console.fill_form_by_id("dataFolder", data_path)

        if port is not None:
            self._admin_console.fill_form_by_id("cvdPort", port)

        if plan is not None:
            self.__select_ibmi_plan(plan=plan)

        self._admin_console.fill_form_by_id("addIBMiServer_userName_input1", username)
        self._admin_console.fill_form_by_id("password", password)
        self._admin_console.fill_form_by_id("confirmPassword", password)

        self.__drop_down.select_drop_down_values(drop_down_id='addIBMiServer_isteven-multi-select_#3689',
                                                 values=access_node)

        if subsystem_description is not None:
            self._admin_console.fill_form_by_id("subsystemDescription", subsystem_description)
        if job_queue is not None:
            self._admin_console.fill_form_by_id("jobQueue", job_queue)
        if create_jobq is not None:
            self._admin_console.checkbox_select("createJobQueue")
        if job_priority is not None:
            self._admin_console.fill_form_by_id("jobPriority", job_priority)
        if run_priority is not None:
            self._admin_console.fill_form_by_id("runPriority", run_priority)

        self._admin_console.click_button_using_text(self._admin_console.props["label.save"])
        notification = self._admin_console.get_notification(wait_time=300)
        return notification

    @WebAction()
    def __select_ibmi_plan(self, plan):
        """Select the time tab
                Args:
                    plan   (String): select existing plan
        """
        self._driver.find_element_by_xpath(
            "//span[@class='multiSelect inlineBlock']/button[@class='ng-binding' and 1]").click()
        self._driver.find_element_by_xpath(f"//span[contains(text(),'{plan}')]").click()

    @PageService()
    def is_client_exists(self, server_name):
        """ check client entry existence from file server page
        Args:
                server_name     (str) -- server name to retire

        returns: boolean
            True: if server exists
            false: if server does not exists
        """
        status = self.__table.is_entity_present_in_column(column_name='Name',
                                                          entity_name=server_name)
        return status

    @PageService()
    def retire_server(self, server_name):
        """Performs retire action for the given server

        Args:
                server_name     (str) -- server name to retire
        """
        self.__table.access_action_item(server_name, self._admin_console.props['action.commonAction.retire'])

        self._admin_console.fill_form_by_id("deleteTypedConfirmation",
                                            self._admin_console.props['action.commonAction.retire'])
        self._admin_console.click_button_using_text(self._admin_console.props['action.commonAction.retire'])
        return self._admin_console.get_jobid_from_popup()

    @PageService()
    def action_add_software(
            self,
            client_name=None,
            select_all_packages=False,
            packages=None,
            reboot=False):
        """selects the Add software option for the given client

        Args:
            client_name     (str)       -- client to add software on

            select_all_packages  (bool)  -- selects all the packages if set True
                                            (default: False)

            packages        (list)      -- list of packages to be installed

            reboot          (bool)      -- set to True if reboot required
                                            (default: False)
        """
        self._admin_console.refresh_page()
        self.__table.access_action_item(
            client_name, self._admin_console.props['action.addSoftware'])
        if select_all_packages:
            self.__drop_down.select_drop_down_values(0, select_all=True)
        elif packages:
            self.__drop_down.select_drop_down_values(0, packages)
        else:
            raise CVWebAutomationException('Packages list is not provided')
        if reboot:
            self._enable_toggle(self._admin_console.props['label.foreceReboot'])
        self._admin_console.submit_form()
        job_id = self.__get_job_id_page()
        self._admin_console.click_button(self._admin_console.props['OK'])
        return job_id

    @PageService()
    def action_update_software(self, client_name=None, reboot=False):
        """selects the update software option for the given client

        Args:
            client_name     (str) -- client to update software on

            reboot      (bool)    -- set to True if reboot required
         """

        self.__table.access_action_item(
            client_name, self._admin_console.props['label.updateSoftware'])
        if reboot:
            self._enable_toggle(self._admin_console.props['label.foreceReboot'])

        self._admin_console.click_button_using_text(self._admin_console.props['button.yes'])
        self._admin_console.click_button_using_text(self._admin_console.props['button.yes'])
        job_id = self.__get_job_id_dialog().split(': ')[1]
        self._admin_console.click_button(self._admin_console.props['OK'])
        return job_id

    @PageService()
    def action_uninstall_software(self, client_name=None, packages=None):
        """
        uninstalls selected packages for a client
        Args:
            client_name     (str)       -- client to uninstall software on
            packages        (list)      -- list of packages to be uninstalled

        """
        self.__table.access_action_item(
            client_name, self._admin_console.props['label.uninstallSoftware'])
        if packages:
            self.__drop_down.select_drop_down_values(0, packages)
            self._admin_console.click_button(self._admin_console.props['action.uninstall'])
        else:
            raise CVWebAutomationException('Packages list is not provided')
        self._admin_console.click_button_using_text(self._admin_console.props['button.yes'])
        return self._admin_console.get_jobid_from_popup()

    @PageService()
    def action_sendlogs(self, client_name=None):
        """selects sendlogs option for the given client"""
        self.__table.access_action_item(
            client_name, self._admin_console.props['action.commonAction.sendLogs'])

    @PageService()
    def filter_file_server_by_type(self, fs_type='Windows'):
        """Method to filter file server list from the Type filter drop down"""
        self.__click_type_filter_drop_down()
        self.__select_file_server_type(fs_type)
        self._admin_console.wait_for_completion()

    @PageService()
    def add_nas_client(self, name, host_name, plan, vendor=None, **kwargs):
        """
        Adds a new NAS File Server with the Chosen iDAs and Access Node.

        Args:
            name        (basestring)    :   The  name of the NAS/Network Share client to be created.

            host_name   (basestring)    :   The host name of the NAS/Network Share client to be created.

            plan        (basestring)    :   The name of the plan that needs to be associated to the client.

            vendor      (Vendor(Enum))  :   The name of the vendor, supports following values.
                -   DELL_EMC_ISILON

            \*\*kwargs  (dict)              --  Optional arguments.

            Available kwargs Options:

                    array_details   (dict)  :   The dictionary of array details.
                     Dictionary contains the keys array_name, control_host, username and password.

                        array_name      (basestring)    :   Name of the array.

                        control_host    (basestring)    :   Control host name if applicable to the array.

                        username        (basestring)    :   Username for the array, defined in CoreUtils\config.json.

                        password        (basestring)    :   Password for the array, defined in CoreUtils\config.json.

                    cifs            (dict)  :   The dictionary of CIFS Agent details.
                    Dictionary contains the keys access_nodes, impersonate_user and content.

                        access_nodes        (list)  :   List of access node names, access node names are strings.

                        impersonate_user    (dict)  :   The dictionary of impersonation account details.
                        Dictionary contains the keys username and password.

                            username    (basestring)    :   Username of the account, defined in CoreUtils\config.json.

                            password    (basestring)    :   Password of the account, defined in CoreUtils\config.json.

                        content             (list)  :   List of content paths, content paths are strings.

        """
        self.nas_file_server.add_nas_client(name, host_name, plan, vendor, **kwargs)

    @PageService()
    def delete_client(self, server_name):
        """
        Performs delete action for given server

        Args:
            server_name     (basestring):   the name of the server

        """
        self.__table.access_action_item(server_name, "Delete")
        self._admin_console.fill_form_by_id("deleteTypedConfirmation", "Delete")
        self._admin_console.click_button("Delete")
        self._admin_console.get_notification(wait_time=60)

    @PageService()
    def reconfigure_server(self, server_name):
        """
        Performs reconfigure action for the given server

        Args:
            server_name     (basestring):   the name of the server
        """

        self.__table.access_action_item(server_name, self._admin_console.props['label.reconfigure'])
        self._admin_console.click_button_using_text(self._admin_console.props['action.commonAction.reconfigure'])
        self._admin_console.click_button_using_text('Yes')
        self._admin_console.get_notification(wait_time=60)


class RestorePanel(ModalPanel):
    """ Class to handle restore panel related operations """

    @WebAction()
    def _select_checkbox(self, element):
        """ Method to toggle inplace option"""
        self._driver.find_element_by_xpath(
            f'//label[@for="{element}"]'
        ).click()

    @WebAction()
    def _click_browse(self):
        """ Method to click on browse button during destination path selection screen """
        if self._admin_console.check_if_entity_exists('id', 'BrowsePath'):
            self._driver.find_element_by_id('BrowsePath').click()
        else:
            self._driver.find_element_by_xpath(
                '//button[@id="fsRestoreOptions_button_#6719"]'
            ).click()

    @WebAction()
    def __enable_notify_via_email(self):
        """ Enables notify via email checkbox """
        self._driver.find_element_by_xpath("*//span[contains(text(),'Notify via email')]").click()

    @WebAction()
    def _click_backupset_and_subclient(self, backupset_name, subclient_name):
        """Selects the required backup set and subclient
        Args:
            backupset_name (String) : Name of backup set to be selected

            subclient_name (String) : Name of the subclient to be selected

        """
        self._driver.find_element_by_xpath(
            f'//label[text()="{backupset_name}"]/../..//label[text()="{subclient_name}"]'
            "/../preceding-sibling::span[1]//input"
        ).click()

    @PageService()
    def select_backupset_and_subclient(self, backupset_name, subclient_name):
        """Selects the required backup set and subclient
        Args:
            backupset_name (String): Name of backup set to be selected

            subclient_name (String): Name of the subclient to be selected

        """
        self._click_backupset_and_subclient(backupset_name, subclient_name)

    @WebAction()
    def __click_search(self):
        """Clicks the search option in restore page"""
        self.__clear_search()
        self._driver.find_element_by_xpath(
            "//input[contains(@placeholder,'Search for files')]").click()
        self._admin_console.wait_for_completion()

    @WebAction()
    def __clear_search(self):
        """Clears the search"""
        clear = self._driver.find_elements_by_xpath(
            "//span[contains(@class,'form-control-clear glyphicon')]")
        if clear:
            if clear[0].is_displayed():
                clear[0].click()

    @WebAction()
    def _enter_search_value(self, search_type, value):
        """Enters the value in search
                Args:
                    search_type (String): Category type (eg:filename, contains)
                    value   (String): Value to be entered
        """
        search = self._driver.find_element_by_xpath(
            f"//span[@class='ng-binding ng-scope']//label[text()='{search_type}']/following-sibling::input")
        search.click()
        search.clear()
        search.send_keys(value)

    @WebAction()
    def _select_value(self, label, value):
        """Selects file type
                Args:
                    label (String): Category type (eg:Modified, fileType)
                    value (String): Value to be entered
                                        (eg:'file.txt', 'test.html')
        """
        self._driver.find_element_by_xpath(
            f"//label[text()='{label}']/following-sibling::select/option[@label='{value}']").click()

    @WebAction()
    def __click_time_tab(self, label):
        """Select the time tab
                Args:
                    label   (String): Category type (eg:From Time, To Time)
        """
        self._driver.find_element_by_xpath(
            f"//label[text()='{label}']/following-sibling::span/button[@class='btn btn-default']").click()

    @WebAction()
    def _select_time_range(self, label, time):
        """Selects the time range
                Args:
                    label   (String): Category type (eg:From Time, To Time)
                    time (basestring): the backup date in 01-March-2019 format
        """
        self.__click_time_tab(label)
        if time:
            calender = {'date': time.split("-")[0], 'month': time.split("-")[1], 'year': time.split("-")[2]}
            self._admin_console.date_picker(calender)

    @WebAction()
    def __check_check_box(self, label, value):
        """ Checks if the checkbox is selected or not
                Args:
                    label (String): Category type (eg:IncludeFolders:, showDeletedFiles:)
                    value (bool): Value to be entered
                                        (eg:'file.txt', 'test.html')
        """
        elem = self._driver.find_elements_by_xpath(f"//*[@id='{label}' and contains(@class,'ng-not-empty')]")
        elem1 = self._driver.find_elements_by_xpath(f"//*[@id='{label}' and contains(@class,'ng-empty')]")
        if elem1 and value:
            self._select_checkbox(label)
        if elem and not value:
            self._select_checkbox(label)
        self._admin_console.wait_for_completion()

    @WebAction()
    def __submit_search(self):
        """Submit the search option"""
        self._driver.find_element_by_xpath(f"//*[@id='cv-search-box_button_#9364']").click()
        self._admin_console.wait_for_completion()

    @PageService()
    def search_files_for_restore(self, file_name=None, contains=None, file_type=None, modified=None,
                                 from_time=None, to_time=None, include_folders=True,
                                 show_deleted_files=True):
        """Search the files based on the parametes provided
                Args:
                    file_name (String): Name of file to be searched
                                        (eg:'file.txt', 'test.html')
                    contains  (String): pattern string that the files contain
                                        (eg:'html', 'automation')
                    file_type  (String): The type of the file to be searched
                                         (eg:'Audio', 'Image', 'Office', 'Video',
                                             'System', 'Executable')
                    modified   (String): Modified time of file to be searched
                                         (eg:'Today', 'Yesterday', 'This week')
                    from_time   (String): The files backed up from date
                    to_time     (String): The files backed up to this date
                    include_folders (bool): True is to include folder while search is applied
                    show_deleted_files (bool): True is to apply search for deleted items
        """
        self.__click_search()
        if file_name:
            self._enter_search_value('Filename:', file_name)

        if contains:
            self._enter_search_value('Contains:', contains)

        if file_type:
            self._select_value("File Type:", file_type)

        if modified:
            if modified != 'Time Range':
                self._select_value("Modified:", modified)
            else:
                self._select_value("Modified:", modified)
                if from_time:
                    self._select_time_range("From time", from_time)
                if to_time:
                    self._select_time_range("To time", to_time)

        self.__check_check_box("IncludeFolders:", include_folders)
        self.__check_check_box("showDeletedFiles:", show_deleted_files)
        self.__submit_search()

    @PageService()
    def select_destination_client(self, client_name):
        """Method to select client from drop down in restore panel"""

        self._admin_console.wait_for_completion()
        self._dropdown.select_drop_down_values(0, values=[client_name])
        self._admin_console.wait_for_completion()

    @PageService()
    def select_overwrite(self):
        """Method to select unconditional overwrite in restore panel"""
        self._select_checkbox("overwrite")
        self._admin_console.wait_for_completion()

    @PageService()
    def deselect_acl_for_restore(self):
        """Method to deselect acls option in restore panel"""
        self._select_checkbox("acls")
        self._admin_console.wait_for_completion()

    @PageService()
    def deselect_data_for_restore(self):
        """Method to deselect data option in restore panel"""
        self._select_checkbox("data")
        self._admin_console.wait_for_completion()

    @PageService()
    def select_browse_for_restore_path(self, toggle_inplace=True):
        """Method to click on browse in restore panel
        Args:

            toggle_inplace (bool): to toggle inplace option
                default: True

        """
        if toggle_inplace:
            self._select_checkbox("inplace")
        self.select_browse_in_restore()

    @PageService()
    def select_browse_in_restore(self):
        """Method to click on browse in restore panel"""
        self._click_browse()
        self._admin_console.wait_for_completion()

    @PageService()
    def submit_restore(self, notify=False):
        """ Method to submit restore job
        Args:

            notify (bool): to enable by email

        Returns:
            job_id: job id from notification
        """

        if notify:
            self.__enable_notify_via_email()
        self.submit(wait_for_load=False)
        return self._admin_console.get_jobid_from_popup()


class AddPanel:

    def __init__(self, admin_console):
        """
        Args:
        admin_console(AdminConsole): adminconsole
        object
        """
        self.__admin_console = admin_console
        self.__driver = admin_console.driver

        # Components required
        self.__table = Table(self.__admin_console)
        self.__panel = Backup(self.__admin_console)
        self.__browse = Browse(self.__admin_console)
        self.__contentbrowse = ContentBrowse(self.__admin_console)
        self.__dropdown = DropDown(self.__admin_console)

    @WebAction()
    def add_custom_path(self, path):
        """Add custom paths in the path input box
                Args:
                    path (str)      :   Data path to be added
        """
        custom_path_input_xpath = "//input[@placeholder='Enter custom path']"
        custom_path_input = self.__driver.find_elements_by_xpath(custom_path_input_xpath)
        for path_input in custom_path_input:
            if path_input.is_displayed():
                path_input.clear()
                path_input.send_keys(path)

    @WebAction()
    def click_add_custom_path(self):
        """Clicks the add custom path icon"""
        add_path_icon_xpath = "//i[@title='Add']"
        custom_path_add = self.__driver.find_elements_by_xpath(add_path_icon_xpath)
        for path_add in custom_path_add:
            if path_add.is_displayed():
                path_add.click()

    @WebAction()
    def toggle_own_content(self):
        """toggles the override backup content option"""
        self.__driver.find_element_by_xpath(
            f'//span[@class="help-label ng-binding" and text()="Define your own backup content"]').click()

    @WebAction()
    def browse_and_select_data(self, backup_data, file_system):
        """
        selects backup data through FS Browse

        Args:

            backup_data     (list(paths)) : Data to be backed up by new sub client created
                Eg. backup_data = ['C:\\TestBackupSet1', C:\\TestBackupSet2']

            file_system      (string)     :  file system of the client
                Eg. - 'Windows' or 'Unix'

        Returns:
            None

        Raises:
            Exception:
                if not able to select data
        """

        for data in backup_data:
            count = 0
            flag = True
            if file_system.lower() == 'windows':
                pattern = "\\"
            else:
                pattern = "/"
            directories = []
            start = 0
            while flag:
                tag = data.find(pattern, start)
                if tag == -1:
                    flag = False
                else:
                    count += 1
                    start = tag + 1

            for i in range(0, count + 1):
                directory, sep, folder = data.partition(pattern)
                data = folder
                if directory != '':
                    directories.append(directory)
            path = len(directories)

            for i in range(0, path - 1):
                if self.__driver.find_element_by_xpath(
                        "//span[contains(text(),'" + str(directories[i]) + "')]/../../button"). \
                        get_attribute("class") == 'ng-scope collapsed':
                    self.__admin_console.click_by_xpath(
                        "//span[contains(text(),'" + str(directories[i]) + "')]/../../button")
                dest = i + 1
            self.__admin_console.click_by_xpath(
                "//span[contains(text(),'" + str(directories[dest]) + "')]")
        self.__admin_console.click_button(self.__admin_console.props['label.save'])

    @WebAction(delay=5)
    def remove_plan_content(self):
        """
        removes the content inherited from plan
        """
        elem = "//div[contains(@class,'ui-grid-selection-row-header-buttons ui-grid-icon-ok ng-pristine ng-untouched ng-valid ng-scope')]"
        checkbox = self.__driver.find_elements_by_xpath(elem)
        for files in checkbox:
            if files.is_displayed():
                files.click()

    @WebAction()
    def __select_credentials(self, value):
        """ Selects radio button for Impersonate User"""
        self.__driver.find_element_by_xpath(f"//span[contains(text(),'{value}')]").click()

    @PageService()
    def add(self,
            plan,
            define_own_content=False,
            browse_and_select_data=False,
            backup_data=None,
            impersonate_user=None,
            exclusions=None,
            exceptions=None,
            backup_system_state=None,
            file_system='Windows',
            remove_plan_content=False,
            submit=True,
            toggle_own_content=True):
        """
        Method to Add content,exclusions,exceptions

        Args:
            plan (string): plan name to be used as policy for new sub client backup.

            define_own_content(bool): Pass True to define own content
                                        False for associated plan content

            browse_and_select_data(bool): Pass True to browse and select data,
                                            False for custom path

            backup_data     (list(paths)): Data to be backed up by new sub client created
                Eg. backup_data = ['C:\\TestBackupSet1', C:\\TestBackupSet2']

            impersonate_user    (string): Specify username (eg: for UNC paths).

            exclusions       (list(paths)): Data to be backed up by new sub client created
                Eg. exclusions = ['C:\\TestBackupSet1', C:\\TestBackupSet2']

            exceptions       (list(paths)): Data to be backed up by new sub client created
                Eg. exceptions = ['C:\\TestBackupSet1', C:\\TestBackupSet2']

            backup_system_state (String): System state to be enabled or not.

            file_system       (string):  file system of the client
                Eg. - 'Windows' or 'Unix'

            remove_plan_content (bool): True for removing content inherited from plan
                                              False for keeping content inherited from plan

            submit              (bool): True if the user wants to submit the form
                                      : False if the user doesn't want to submit the form

            toggle_own_content  (bool): Pass True to toggle override backup content
                                        False for not toggling
        Raises:
            Exception:
                -- if fails to add entity

        """

        self.__dropdown.select_drop_down_values(drop_down_id='planSummaryDropdown', values=[plan])

        if define_own_content:
            if toggle_own_content:
                self.toggle_own_content()
                self.__admin_console.wait_for_completion()

            if remove_plan_content:
                self.remove_plan_content()
                self.__admin_console.select_hyperlink(
                    self.__admin_console.props['label.globalActions.remove'])

            if browse_and_select_data:
                self.__admin_console.select_hyperlink(self.__admin_console.props['header.content'])
                self.__admin_console.select_hyperlink(self.__admin_console.props['action.browse'])
                self.browse_and_select_data(backup_data, file_system)

            else:
                for path in backup_data:
                    self.add_custom_path(path)
                    self.__admin_console.wait_for_completion()
                    self.click_add_custom_path()
                    self.__admin_console.wait_for_completion()
                    if self.__admin_console.check_if_entity_exists("xpath", "//h1[text()='Impersonate user']"):
                        self.__admin_console.cancel_form()

            if impersonate_user:

                self.__admin_console.select_hyperlink(
                    self.__admin_console.props['label.impersonateUser'])
                self.__admin_console.wait_for_completion()

                if isinstance(impersonate_user, str):
                    self.__select_credentials("Use saved credentials")
                    self.__admin_console.cv_single_select("Saved credentials", impersonate_user)
                    self.__admin_console.submit_form()

                elif isinstance(impersonate_user, dict):
                    self.__admin_console.fill_form_by_id('loginName', impersonate_user['username'])
                    self.__admin_console.fill_form_by_id('password', impersonate_user['password'])
                    self.__admin_console.submit_form()

            if exclusions:
                self.__admin_console.select_hyperlink(
                    self.__admin_console.props['label.Exclusions'])
                for path in exclusions:
                    self.add_custom_path(path)
                    self.__admin_console.wait_for_completion()
                    self.click_add_custom_path()
                    self.__admin_console.wait_for_completion()

            if exceptions:
                self.__admin_console.select_hyperlink(
                    self.__admin_console.props['label.Exceptions'])
                for path in exceptions:
                    self.add_custom_path(path)
                    self.__admin_console.wait_for_completion()
                    self.click_add_custom_path()
                    self.__admin_console.wait_for_completion()

            if backup_system_state is True:
                self.__admin_console.enable_toggle(0)
            elif backup_system_state is False:
                self.__admin_console.disable_toggle(0)

        if submit:
            self.__admin_console.submit_form()
            self.__admin_console.check_error_message()