# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on
subclient/database group/table groups details page

SubClient:
----------
    get_subclient_general_properties()      --      Returns the subclient properties under
    general panel

    get_subclient_snapshot_properties()     --      Returns the Intellisnap details of the
    subclient
    displayed under Snapshot panel

    enable_backup()                         --      Enables the 'Data backup' toggle if it
    is disabled

    disable_backup()                        --      Disables the 'Data backup' toggle if it
    is enabled

    backup()                                --      Submits backup job from the subclient page

    access_restore()                        --      Clicks on the restore button below the
    recovery points in subclient page

    clear_all_selection()                   --      Clicks on clear all checkbox in browse
    page if present

    restore_folders()                       --      Clicks on the given items, analogous to
    folders in the browse page and submits restore. Also returns the object of restore panel
    class class corresponding to database_type argument

    restore_files_from_multiple_pages()     --      Clicks on items from multiple pages
    recursively in browse page and submits restore. Also returns the object of restore
    panel class corresponding to database_type argument

    is_snapshot_enabled()                   --      method to check if snapshot option is enabled
    for subclient

    get_snap_engine()                       --      method to get snap engine type

    enable_snapshot()                       --      method to enable snapshot option for subclient

    disable_snapshot()                      --      method to disable snapshot option for subclient

    delete_subclient()                      --      method to delete subclient

    is_blocklevel_backup_enabled()          --      method to check if block level backup for
    subclient is enabled

    enable_blocklevel_backup()              --      method to enable block level backup for
    subclient

    disable_blocklevel_backup()             --      method to disable cblock level backup for
    subclient

    get_items_in_browse_page()              --      Returns all the items shown in current browse

                                                    page for given column

PostgreSQLSubclient:
--------------------
    enable_collect_objectlist()             --      method to enable collect object list option in
    postgreSQL subclient

    disable_collect_objectlist()            --      method to disable collect object list option in
    postgreSQL subclient

DB2Subclient:
------------
    backup()        --      method to trigger backup job from subclient level page

MySQLSubclient:
----------------
    edit_content()              --      method to edit content of subclient

    validate_autodiscover()     --      Method to verify database in client
                                        is auto discovered by subclient

    subclient_autodiscovered_content()--Method to get all the autodiscovered
                                        databases

    enable_standby_backup()     --      Method to enable standby backup option

    is_all_databases_in_content()--     Method to check if [All databases] in
                                        subclient content

InformixSubclient:
----------------
    edit_content()              --      Method to edit content for subclient

RDSSubclient:
----------------
    enable_replication()        --      Method to enable replication in Instance group page

    edit_region_mapping()       --      Method to add region mapping or
                                        to edit existing region mapping

    enable_cross_account_operations()-- Method to enable cross account
                                        operations in Instance group page
"""
from enum import Enum
from Web.Common.page_object import (
    PageService
)
from Web.AdminConsole.Components.table import Table, CVTable
from Web.AdminConsole.Components.panel import PanelInfo, Backup, ModalPanel, DropDown
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.AdminConsole.Components.browse import Browse
from Web.AdminConsole.Databases.db_instances import DBInstances
from Web.AdminConsole.Databases.Instances.restore_panels import DynamoDBRestorePanel
from Web.AdminConsole.Databases.Instances.restore_panels import MySQLRestorePanel
from Web.AdminConsole.Databases.Instances.restore_panels import RedshiftRestorePanel
from Web.AdminConsole.Databases.Instances.restore_panels import RDSRestorePanel
from Web.AdminConsole.Databases.Instances.restore_panels import SybaseRestorePanel
from Web.AdminConsole.Databases.Instances.restore_panels import InformixRestorePanel


class SubClient:
    """Class for database subclients page"""

    class DBTypes(Enum):
        """Enum to represent supported database types"""
        DYNAMODB = "DynamoDBRestorePanel"
        MYSQL = "MySQLRestorePanel"
        RDS = "RDSRestorePanel"
        SYBASE = "SybaseRestorePanel"
        REDSHIFT = "RedshiftRestorePanel"
        INFORMIX = "InformixRestorePanel"

    def __init__(self, admin_console):
        """
            Args:
                admin_console (AdminConsole): Object of AdminConsole class
        """
        self._admin_console = admin_console
        self._browse = Browse(self._admin_console)
        self._table = Table(self._admin_console)
        # self._admin_console._load_properties(self)
        self.props = self._admin_console.props
        self._subclient_general_panel = PanelInfo(self._admin_console,
                                                  title=self.props['header.general'])
        self.__subclient_snapshot_panel = PanelInfo(
            self._admin_console, title=self.props['header.snapshotManagement'])
        self._panel = ModalPanel(self._admin_console)
        self._panel_info = PanelInfo(self._admin_console)
        self._panel_dropdown = DropDown(self._admin_console)
        self.__restore_panel_map = {
            DBInstances.Types.DYNAMODB: SubClient.DBTypes.DYNAMODB,
            DBInstances.Types.MYSQL: SubClient.DBTypes.MYSQL,
            DBInstances.Types.SYBASE: SubClient.DBTypes.SYBASE,
            DBInstances.Types.REDSHIFT: SubClient.DBTypes.REDSHIFT,
            DBInstances.Types.INFORMIX: SubClient.DBTypes.INFORMIX,
            DBInstances.Types.RDS: SubClient.DBTypes.RDS
            }

    @PageService()
    def get_subclient_general_properties(self):
        """Returns the subclient properties under general panel"""
        return self._subclient_general_panel.get_details()

    @PageService()
    def get_subclient_snapshot_properties(self):
        """Returns the Intellisnap details of the subclient
            displayed under Snapshot panel"""
        return self.__subclient_snapshot_panel.get_details()

    @PageService()
    def enable_backup(self):
        """Enables the 'Data backup' toggle if it is disabled"""
        self._subclient_general_panel.enable_toggle(
            self.props['label.enableBackup'])

    @PageService()
    def disable_backup(self):
        """Disables the 'Data backup' toggle if it is enabled"""
        self._subclient_general_panel.disable_toggle(
            self.props['label.enableBackup'])

    @PageService()
    def backup(self, backup_type=Backup.BackupType.INCR, enable_data_for_incremental=False):
        """
        Submits backup job from the subclient page
        Args:
            backup_type                 (Backup.BackupType) -- backup type

            enable_data_for_incremental (bool)              -- flag to check if
            data needs to be backed up during incremental

                default: False


        Returns
            (str) -- Backup job id

        """
        self._admin_console.access_menu(self.props['label.globalActions.backup'])
        return Backup(self._admin_console).submit_backup(backup_type=backup_type,
                                                         incremental_with_data=
                                                         enable_data_for_incremental)

    @PageService()
    def access_restore(self):
        """Clicks on the restore button below the recovery points in subclient page"""
        self._admin_console.access_action(self.props['label.globalActions.restore'])

    @PageService()
    def clear_all_selection(self):
        """Clicks on clear all checkbox in browse page if present"""
        self._browse.clear_all_selection()

    @PageService()
    def restore_folders(self, database_type, items_to_restore=None, all_files=False,
                        copy=None):
        """Clicks on the given items, analogous to folders in the browse page and
        submits restore. Also returns the object of restore panel class
        corresponding to database_type argument

        Args:
            database_type (Types):   Type of database should be one among the types defined
                                      in 'Types' enum in DBInstances.py file

            items_to_restore (list) : list of files and folders to select for restore
                                    list of items analogous to folders in browse page

            all_files        (bool):  select all the files shown for restore / download

                default: False

            copy            (str):  The name of the copy to browse from
                                    Example- "Secondary" or "Copy-2"

        Returns:
            Object (RestorePanelTypes): Object of class in RestorePanelTypes corresponding
                                        to database_type
        """
        if copy:
            self._browse.select_adv_options_submit_restore(copy, database=True)
        if all_files:
            self._browse.select_for_restore(all_files=all_files)
        else:
            self._browse.select_for_restore(file_folders=items_to_restore)
        self._browse.submit_for_restore()
        return globals()[self.__restore_panel_map[database_type].value](self._admin_console)

    @PageService()
    def restore_files_from_multiple_pages(self, database_type, mapping_dict,
                                          root_node, rds_agent=False, copy=None):
        """Clicks on items from multiple pages recursively in browse page
        and submits restore. Also returns the object of restore panel class
        corresponding to database_type argument

        Args:
            database_type (Types):   Type of database should be one among the types defined
                                      in 'Types' enum in DBInstances.py file

            mapping_dict (dict) : The dictionary containing the folder names as keys
                                and list of files to be selected under them as value

                                    Example:
                                        mapping_dict={
                                        'FOLDER1':['file1','file2','file3']
                                        'FOLDER2':['fileA','fileB','fileC']
                                        }

            root_node   (str):  The name of the subclient/instance on which browse operation
                                    was performed or the name of the root folder that
                                    appears on the browse page

            rds_agent  (Boolean):  Flag to indicate if browse is performed for RDS agent

                                   True if browsing from amazon RDS instance or subclient
                                   False for any other agent

            copy        (str):  The name of the copy to browse from
                                Example- "Secondary" or "Copy-2"
        Returns:
            Object (RestorePanelTypes): Object of class in RestorePanelTypes corresponding
                                        to database_type

        """
        if copy:
            self._browse.select_adv_options_submit_restore(copy, database=True)
        if not rds_agent:
            self._browse.select_from_multiple_pages(mapping_dict, root_node)
        else:
            self._browse.select_from_multiple_pages(mapping_dict, root_node, rds_agent=True)
        self._browse.submit_for_restore()
        return globals()[self.__restore_panel_map[database_type].value](self._admin_console)

    @PageService()
    def is_snapshot_enabled(self):
        """method to check if snapshot option is enabled for subclient
        """
        toggle_element = self.__subclient_snapshot_panel.get_toggle_element(
            self.props['label.enableHardwareSnapshot'])
        return self.__subclient_snapshot_panel.is_toggle_enabled(toggle_element)

    @PageService()
    def get_snap_engine(self):
        """method to get snap engine type
        """
        if self.is_snapshot_enabled:
            panel_info = self.__subclient_snapshot_panel.get_details()
            return panel_info[self.props['label.engineName']]
        return None

    @PageService()
    def enable_snapshot(self, snap_engine, proxy_node=None):
        """method to enable snapshot option for subclient

        Args:
            snap_engine (str)  -- Snap engine name to be selected

            proxy_node  (str)  -- proxy/access node to be used for backup copy

                default: None

        """
        self.__subclient_snapshot_panel.enable_toggle(self.props['label.enableHardwareSnapshot'])
        self._panel_dropdown.select_drop_down_values(values=[snap_engine], drop_down_id='engine')
        if proxy_node:
            self._panel_dropdown.select_drop_down_values(values=[proxy_node], drop_down_id='useProxy')
        self._admin_console.submit_form()

    @PageService()
    def disable_snapshot(self):
        """method to disable snapshot option for subclient

        """
        self.__subclient_snapshot_panel.disable_toggle(self.props['label.enableHardwareSnapshot'])

    @PageService()
    def delete_subclient(self):
        """Deletes the subclient"""
        self._admin_console.access_menu_from_dropdown(
            self._admin_console.props['action.delete']
        )
        _dialog = ModalDialog(self._admin_console)
        _dialog.type_text_and_delete('DELETE')
        self._admin_console.wait_for_completion()

    @PageService()
    def is_blocklevel_backup_enabled(self):
        """
        method to check if block level backup for subclient is enabled

        """
        toggle_element = self._panel_info.get_toggle_element(
            self.props['label.blockLevelOption'])
        return self._panel_info.is_toggle_enabled(toggle_element)

    @PageService()
    def enable_blocklevel_backup(self):
        """
        method to enable block level backup for subclient

        """
        self._panel_info.enable_toggle(self.props['label.blockLevelOption'])

    @PageService()
    def disable_blocklevel_backup(self):
        """
        method to disable cblock level backup for subclient

        """
        self._panel_info.disable_toggle(self.props['label.blockLevelOption'])

    @PageService()
    def get_items_in_browse_page(self, column_name):
        """Returns all the items shown in current browse page for given column
        Args:
            column_name (str):  The name of the column to get data

        Returns:
            list    : List of items under given column
        """
        return self._browse.get_column_data(column_name)


class PostgreSQLSubclient(SubClient):
    """

    This class provides the function or operations to perform on
    postgreSQL subclient page

    """

    @PageService()
    def enable_collect_objectlist(self):
        """
        method to enable collect object list option in postgreSQL subclient

        """
        self._panel_info.enable_toggle(self.props['label.CollectObjectList'])

    @PageService()
    def disable_collect_objectlist(self):
        """
        method to disable collect object list option in postgreSQL subclient

        """
        self._panel_info.disable_toggle(self.props['label.CollectObjectList'])


class DB2Subclient(SubClient):
    """

    This class provides the function or operations to perform on
    DB2 subclient page

    """

    @PageService()
    def backup(self, backup_type=Backup.BackupType.INCR, enable_data_for_incremental=False):
        """
        Submits backup job from the subclient page
        Args:
            backup_type                 (Backup.BackupType) -- backup type

            enable_data_for_incremental (bool)              -- flag to check if
            data needs to be backed up during incremental

                default: False


        Returns
            (str) -- Backup job id

        """
        self._admin_console.access_menu(self.props['label.globalActions.backup'])
        if enable_data_for_incremental and backup_type == Backup.BackupType.INCR:
            self._admin_console.checkbox_select('dataType')
        return Backup(self._admin_console).submit_backup(backup_type)


class MySQLSubclient(SubClient):
    """Class for MySQL subclient"""

    def __init__(self, admin_console):
        super().__init__(admin_console)
        self.__table = CVTable(self._admin_console)
        self.__subclient_databases_panel = PanelInfo(self._admin_console,
                                                     title=self.props['pageHeader.databases'])

    @PageService()
    def edit_content(self, database_list):
        """
        Method to edit content of subclient to databases
        in list argument passed as input
        Args:
            database_list: list of databases to be in database group
        """
        self._admin_console.select_hyperlink(self.props['label.viewOrEdit'])
        self._admin_console.wait_for_completion()
        self._browse.clear_all_selection()
        self._browse.select_for_restore(database_list)
        self._panel.submit()

    @PageService()
    def validate_autodiscover(self, database_list):
        """
        Method to verify database in client is auto discovered by subclient
        Args
            database_list: list of databases to verify are in database group
        """
        self._admin_console.select_hyperlink(self.props['label.viewOrEdit'])
        self._admin_console.wait_for_completion()
        content = self.__table.get_column_data(column_name="Name", data_from_all_pages=True)
        self._panel.cancel()
        if not all(database in content for database in database_list):
            raise Exception("Auto discover validation failed..!!")

    @PageService()
    def subclient_autodiscovered_content(self):
        """
        Method to get the auto discovered databases in subclient
        Returns:
            List of databases
        """
        self._admin_console.select_hyperlink(self.props['label.viewOrEdit'])
        self._admin_console.wait_for_completion()
        content = self.__table.get_column_data(column_name="Name", data_from_all_pages=True)
        self._panel.cancel()
        return content

    @PageService()
    def enable_standby_backup(self):
        """Method to enable standby backup in subclient details page"""
        self._subclient_general_panel.enable_toggle(
            self.props['label.EnableStandbyBackup'])
        self._admin_console.wait_for_completion()

    @PageService()
    def is_all_databases_in_content(self):
        """ Checks if subclient content is set to all databases"""
        return self.props['label.AllDatabases'] in self.__subclient_databases_panel.get_details()


class InformixSubclient(SubClient):
    """Class for Informix subclient"""

    @PageService()
    def edit_content(self, bkp_mode):
        """
        Method to edit informix subclient content
        Args:
            bkp_mode(str)       : Backup mode as in command center with no space between words
                Accepted values = 'Entireinstance', 'Wholesystem', 'Selective',
                                  'Fulllogicallogs' and 'Fullandcurrentlogicallogs'
        """
        self._admin_console.select_hyperlink(self.props['label.viewOrEdit'])
        self._admin_console.wait_for_completion()
        backuptypes = {
            'entireinstance': 'label.entireInstance',
            'wholesystem': 'label.wholeSystem',
            'selective': 'label.selective',
            'fulllogicallogs': 'label.fullLogicalLogs',
            'fullandcurrentlogicallogs': 'label.fullCurrentLogicalLogs'
        }
        self._admin_console.select_value_from_dropdown(
            select_id='restoreMode', value=self.props[backuptypes[bkp_mode.lower()]])
        self._panel.submit()


class RDSSubclient(SubClient):
    """Class for RDS subclient"""

    def __init__(self, admin_console):
        super().__init__(admin_console)
        self.__subclient_snapshot_panel = PanelInfo(self._admin_console,
                                                    title=self.props['label.snapshot'])

    @PageService()
    def enable_replication(self, source_region, destination_region):
        """Method to enable replication in Instance group page and add region mapping
            Args:
                    source_region: Source region for replication
                    destination_region: Destination region for replication
            Returns: Details in the snapshot panel of the subclient page as dict
        """
        if self.__subclient_snapshot_panel.is_toggle_enabled(
                self.__subclient_snapshot_panel.get_toggle_element(
                self.props['header.label.replication'])):
            self.edit_region_mapping(source_region, destination_region)
        else:
            self.__subclient_snapshot_panel.enable_toggle(self.props['header.label.replication'])
            self._panel_dropdown.select_drop_down_values(values=[source_region],
                                                         drop_down_id='sourceRegionId')
            self._panel_dropdown.select_drop_down_values(values=[destination_region],
                                                         drop_down_id='destinationRegionId')
            self._admin_console.submit_form()
        return self.__subclient_snapshot_panel.get_details()

    @PageService()
    def edit_region_mapping(self, source_region, destination_region):
        """Method to add region mapping or to edit existing region mapping
            Args:
                source_region: Source region for replication
                destination_region: Destination region for replication
        """
        if len(self._table.get_table_data()) == 0 or\
                source_region not in self._table.get_column_data('Source region'):
            self._admin_console.select_hyperlink(self.props['action.add'])
            self._panel_dropdown.select_drop_down_values(values=[source_region],
                                                         drop_down_id='sourceRegionId')
            self._panel_dropdown.select_drop_down_values(values=[destination_region],
                                                         drop_down_id='destinationRegionId')
            self._admin_console.submit_form()
        elif dict(zip(self._table.get_column_data('Source region'),
                      self._table.get_column_data(
                          'Destination region')))[source_region] != destination_region:
            self._table.access_action_item(source_region, self.props['action.edit'])
            self._panel_dropdown.select_drop_down_values(values=[destination_region],
                                                         drop_down_id='destinationRegionId')
            self._admin_console.submit_form()

    @PageService()
    def enable_cross_account_operations(self, destination_account):
        """Method to enable cross account operations in Instance group page
            Args:
                destination_account(str)       : Name of the destination account  to be used for
                                                cross account operations
        """
        self.__subclient_snapshot_panel.enable_toggle(
            self.props['label.crossAccountOperations'])
        self._panel_dropdown.select_drop_down_values(values=[destination_account],
                                                     drop_down_id='accountList')
        self._admin_console.submit_form()
