# -*- coding: utf-8 -*-s

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the functions or operations that can be performed
on the Database instance

DBInstances:

    select_instance()           --  Select the database instance in DB instances page

    is_instance_exists()        --  Checks if instance with given name exists

    add_postgresql_instance()   --  Adds new postgresql instance

    __click_add_cloud_db()        --  Clicks on 'CLoud DB' under Add Instance

    add_dynamodb_instance()     --  Adds new DynamoDB instance

    add_redshift_instance()     --  Adds new Redshift instance

    add_rds_instance()          -- Adds new Amazon RDS instance

    add_db2_instance()          --  Adds new db2 instance

    add_mysql_instance()        --  Adds new MySQL instance

    add_sybase_instance()       --  Adds new Sybase instance

    access_databases_tab()      --  Clicks on 'Databases' tab in Databases page

    access_instant_clones_tab()  -- Clicks on 'Instant clones' tab in Databases page

    add_informix_instance()     --  Adds new Informix instance

    add_cosmosdb_sql_instance() --  Adds new Azure CosmosDB SQL API instance

    add_oracle_instance()       --  Adds new Oracle instance

    discover_instances()        --  Discovers instances of specified client and agent

    backup()                    --  triggers backup from instances page

    add_server()                --  adds database server
"""

from abc import ABC, abstractmethod
from enum import Enum

from Web.Common.page_object import (
    PageService, WebAction
)
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import DropDown, Backup
from Web.AdminConsole.Databases._migrate_to_cloud import (
    MigrateToCloud,
    MigrateSQLToAzure,
    MigrateOracleToAzure
)
from Web.AdminConsole.Databases.Instances import add_instance
from Web.AdminConsole.Databases.Instances.add_cloud_db_instance import DynamoDBInstance
from Web.AdminConsole.Databases.Instances.add_cloud_db_instance import AmazonRDSInstance
from Web.AdminConsole.Databases.Instances.add_cloud_db_instance import RedshiftInstance
from Web.AdminConsole.Databases.Instances.add_cloud_db_instance import CosmosDBSQLInstance

class DBInstances:
    """This class provides the function or operations that can be performed on the DB Instances
    page under solutions in AdminConsole
    """
    class Types(Enum):
        """Available database types supported"""
        MSSQL = "Microsoft SQL Server"
        ORACLE = "Oracle"
        POSTGRES = "PostgreSQL"
        CLOUD_DB = "Cloud DB"
        DYNAMODB = "DynamoDB"
        DB2 = "DB2"
        MYSQL = "MySQL"
        RDS = "RDS"
        SYBASE = "Sybase"
        REDSHIFT = "Redshift"
        INFORMIX = "Informix"
        SAP_HANA = "SAP HANA"
        ORACLE_RAC = "Oracle RAC"
        COSMOSDB_SQL = "Cosmos DB"

    def __init__(self, admin_console: AdminConsole):
        """Class constructor

            Args:
                admin_console   (obj)   --  The admin console class object

        """
        self.admin_console = admin_console
        self.instances_table = Table(self.admin_console)
        self.__panel_dropdown = DropDown(self.admin_console)
        self.add_instance = add_instance.AddDBInstance(self.admin_console)
        self.admin_console._load_properties(self)
        self.props = self.admin_console.props

    @PageService()
    def select_instance(self, database_type, instance_name, client_name=None):
        """Select the database instance in DB instances page

            Args:
                database_type   (DBInstances.Types) --  Use the enum to select any
                specific database

                instance_name   (str)               --  The name of the SQL instance to select

                client_name     (str)               --  Name of the client underwhich the instance
                is present

                    default: None

            Returns:
                (obj)   --      The SQL instance page object

        """
        if not isinstance(database_type, self.Types):
            raise Exception('Invalid database type')
        self.instances_table.view_by_title(database_type.value)
        self.admin_console.refresh_page()
        if client_name:
            self.instances_table.access_link_by_column(client_name, instance_name)
        else:
            self.instances_table.access_link_by_column(instance_name, instance_name)

    @PageService()
    def is_instance_exists(self, database_type, instance_name, client_name):
        """Check if instance exists

            Args:

                database_type            (str)  --  Database type

                instance_name            (str)  --  Instance Name

                client_name              (str)  --  Client Name
        """
        if not isinstance(database_type, self.Types):
            raise Exception('Invalid database type')
        self.instances_table.view_by_title(database_type.value)
        self.admin_console.refresh_page()
        self.instances_table.is_entity_present_in_column('Server', client_name)
        instances_data = self.instances_table.get_column_data('Name')
        return instance_name in instances_data

    @PageService()
    def add_postgresql_instance(self, server_name, instance_name, plan, database_user, password,
                                port, binary_directory, lib_directory, archive_log_directory,
                                maintenance_db="postgres"):
        """Adds new postgresql instance

            Args:
                server_name             (str)   --  Server name

                instance_name           (str)   --  postgresql instance name

                plan                    (str)   --  Plan to be associated with the instance

                database_user           (str)   --  PostgreSQL user

                password                (str)   --  PostgreSQL user password

                port                    (str)   --  PostgreSQL port

                binary_directory        (str)   --  Binary directory path

                lib_directory           (str)   --  Library directory path

                archive_log_directory   (str)   --  archive log directory path

                maintenance_db          (str)   --  postgreSQL maintenance database name

                    default: Postgres



        """
        self.add_instance.add_postgresql_instance(
            server_name, instance_name, plan, database_user, password, port,
            binary_directory, lib_directory, archive_log_directory, maintenance_db=maintenance_db)

    @PageService()
    def add_db2_instance(self, server_name, instance_name, plan, db2_home, db2_username, db2_user_password):
        """Adds new db2 instance

            Args:
                server_name             (str)   --  Server name

                instance_name           (str)   --  postgresql instance name

                plan                    (str)   --  Plan to be associated with the instance

                db2_home                (str)   --  db2 home path

                db2_username           (str)   --  db2 user name

                db2_user_password       (str)   --  db2 user password


        """
        self.add_instance.add_db2_instance(
            server_name, instance_name, plan, db2_home, db2_username, db2_user_password)

    @PageService()
    def add_mysql_instance(self, server_name, instance_name, plan,
                           database_user, password, binary_directory, log_directory,
                           config_directory, unix, socketfile_directory=None, unix_username=None,
                           nt_username=None, nt_password=None, port=None,
                           ssl_ca_file=None, xtra_backup_bin_path=None):
        """ Adds new MySQL database instance

            Args:
                server_name             (str)   --  Server name

                instance_name           (str)   --  MySQL instance name

                plan                    (str)   --  Plan to be associated with the instance

                database_user           (str)   --  MySQL user

                password                (str)   --  MySQL user password

                unix                    (bool)  --  True if server os is UNIX. Else false

                unix_username           (str)   --  UNIX user name (unix server specific)

                nt_username             (str)   --  NT username  (windows server specific)

                nt_password             (str)   --  NT password  (windows server specific)

                port                    (int)   --  Port  (windows server specific)

                socketfile_directory    (str)   --  Socket file directory  (unix server specific)

                    default: None

                binary_directory        (str)   --  Binary directory path

                log_directory           (str)   --  Log directory path

                config_directory        (str)   --  configuration file directory path

                ssl_ca_file             (str)   --  SSL CA file directory path

                    default: None

                xtra_backup_bin_path    (str)   --  XtraBackup bin path. If None, XtraBackup for
                                                    hot backup will not be enabled.

                    default: None

        """

        self.add_instance.add_mysql_instance(server_name=server_name,
                                             instance_name=instance_name,
                                             plan=plan, database_user=database_user,
                                             password=password,
                                             unix=unix, unix_username=unix_username,
                                             nt_username=nt_username, port=port,
                                             nt_password=nt_password,
                                             socketfile_directory=socketfile_directory,
                                             binary_directory=binary_directory,
                                             log_directory=log_directory,
                                             config_directory=config_directory,
                                             ssl_ca_file=ssl_ca_file,
                                             xtra_backup_bin_path=xtra_backup_bin_path)

    @WebAction()
    def __click_add_cloud_db(self):
        """Click on Cloud DB under add instance in Instances Page"""
        self.admin_console.access_menu(self.props['pageHeader.addInstance'])
        self.admin_console.click_by_id(134)
        self.admin_console.wait_for_completion()

    @PageService()
    def add_dynamodb_instance(self, cloud_account, plan, adjust_read_capacity=0, content='default'):
        """
        Creates DynamoDB instance
        Args:
            cloud_account   (str) : The cloud account that needs to be used for
                                    configuring instance

            plan            (str):  The name of the plan

            adjust_read_capacity(int)   :   The value that needs to be set for
                                        adjust read capacity parameter

            content         (dict or list):  The content to be selected

                            1. To set complete regions as content:
                            Provide list of strings of region names
                            Example: ['Asia Pacific (Mumbai) (ap-south-1)',
                                   'Asia Pacific (Singapore) (ap-southeast-1)']


                            2. To set one more tables under regions as content:
                            Provide a dictionary containing the full region names as keys
                            and LIST of strings with items to be selected under them as value
                                Example:
                                {
                                'US East (Ohio) (us-east-2)':['table1','table2','table3']
                                'US East (Virginia) (us-east-1)':['tableA','tableB'']
                                }

                            Default value is 'default', default content set in UI will be used
        """
        self.__click_add_cloud_db()
        dynamodb_instance = DynamoDBInstance(self.admin_console)
        dynamodb_instance.create_instance(cloud_account, plan, adjust_read_capacity, content)

    @PageService()
    def add_redshift_instance(self, cloud_account, plan, content='default'):
        """
        Creates Redshift instance
        Args:
            cloud_account   (str):  The cloud account that needs to be used for
                                    configuring instance

            plan            (str):  The name of the plan

            content         (dict or list):  The content to be selected

                            1. To set complete regions as content:
                            Provide list of strings of region names
                            Example: ['Asia Pacific (Mumbai) (ap-south-1)',
                                   'Asia Pacific (Singapore) (ap-southeast-1)']


                            2. To set one or more tables under regions as content:
                            Provide a dictionary containing the full region names as keys
                            and LIST of strings with items to be selected under them as value
                                Example:
                                {
                                'US East (Ohio) (us-east-2)':['table1','table2','table3']
                                'US East (Virginia) (us-east-1)':['tableA','tableB'']
                                }

                            Default value is 'default', default content set in UI will be used
        """
        self.__click_add_cloud_db()
        RedshiftInstance(self.admin_console).create_instance(cloud_account, plan, content)

    @PageService()
    def add_rds_instance(self, cloud_account, plan, content='default'):
        """
        Creates Amazon RDS instance
        Args:
            cloud_account   (str) : The cloud account that needs to be used for
                                            configuring instance

            plan            (str):  The name of the plan

            content         (dict or list):  The content to be selected

                            1. To set complete regions as content:
                            Provide a list of strings of region names
                            Example: ['Asia Pacific (Mumbai) (ap-south-1)',
                                   'Asia Pacific (Singapore) (ap-southeast-1)']


                            2. To set one more tables under regions as content:
                            Provide a dictionary containing the full region names as keys
                            and LIST of strings with items to be selected under them as value
                                Example:
                                {
                                'US East (Ohio) (us-east-2)':['table1','table2','table3']
                                'US East (Virginia) (us-east-1)':['tableA','tableB'']
                                }

                            Default value is 'default', default content set in UI will be used
        """
        self.__click_add_cloud_db()
        rds_instance = AmazonRDSInstance(self.admin_console)
        rds_instance.create_rds_instance(cloud_account, plan, content)

    @PageService()
    def add_sybase_instance(self, server_name, instance_name, plan, sa_user_name,
                            password, unix, os_user_name=None, os_password=None):
        """
        Creates Sybase instance
        Args:
            server_name     (str):  Server name
            instance_name:  (str):  Name of instance to be created
            plan:           (str):  Plan name
            sa_user_name:   (str)   SA user name
            password:       (str): Password
            unix            (bool)  --  True if server os is UNIX. Else false
            os_user_name    (str): OS username
            os_password     (str): OS password
        """
        self.add_instance.add_sybase_instance(server_name=server_name,
                                              instance_name=instance_name, plan=plan,
                                              sa_user_name=sa_user_name, password=password,
                                              unix=unix, os_user_name=os_user_name,
                                              os_password=os_password)

    @PageService()
    def access_databases_tab(self):
        """Clicks on 'Databases' tab in Databases page"""
        self.admin_console.access_tab(self.admin_console.props['label.databases'])

    @PageService()
    def access_instant_clones_tab(self):
        """Clicks on 'Instant clones' tab in Databases page"""
        self.admin_console.access_tab(self.admin_console.props['pageHeader.clones'])

    @PageService()
    def add_informix_instance(self, server_name, instance_name, plan,
                              informix_username, informix_home, onconfig,
                              sqlhosts, is_windows_os, informix_password=None):
        """Adds new informix instance

            Args:
                server_name             (str)   --  Server name
                instance_name           (str)   --  informix instance name, INFORMIXSERVER
                plan                    (str)   --  Plan to be associated with the instance
                informix_username       (str)   --  informix user name
                informix_home           (str)   --  informix home directory, INFORMIXDIR
                onconfig                (str)   --  onconfig filename, ONCONFIG
                sqlhosts                (str)   --  sqlhosts file path, INFORMIXSQLHOSTS
                is_windows_os           (bool)  --  True if server OS is windows
                informix_password       (str)   --  informix user password

        """
        self.add_instance.add_informix_instance(
            server_name=server_name, instance_name=instance_name, plan=plan,
            informix_username=informix_username, informix_home=informix_home,
            onconfig=onconfig, sqlhosts=sqlhosts, is_windows_os=is_windows_os,
            informix_password=informix_password)

    @PageService()
    def add_cosmosdb_sql_instance(self, cloud_account, plan, content='default'):
        """Creates new Azure CosmosDB SQL API instance
        Args:
            cloud_account   (str) : The cloud account that needs to be used for
                                    configuring instance

            plan            (str):  The name of the plan

            content         (list or nested dict):  The content to be selected
                Default value is 'default', default content set in UI will be used

                            1. To set complete CosmosDB account as content:

                            Provide a list of strings of account names
                                Example: ['cosmos-account-1', cosmos-account-2]

                            2. To set one more databases and containers as content:

                            Provide a nested dictionary containing the account names as keys
                            and value as another dictionary whose keys are the database names
                            and values is a LIST of containers under the database
                                Example:
                                {
                                'cosmos-account-1': {
                                'database1':['container1', 'container2'],
                                'database2':['container3', 'container4']
                                        }
                                'cosmos-account-2':{
                                'database3': ['container5', 'container6'],
                                'database4': ['container7', 'container8']
                                        }
                                }

                            3. To set complete database as content:

                            Provide the nested dictionary same as above #2 but
                            in the nested dictionary, provide an empty LIST as value
                                Example:
                                    {
                                'cosmos-account-1': {
                                'database1':[],
                                        }
                                'cosmos-account-2':{
                                'database2': [],
                                'database3': []
                                        }
                                }
        """
        self.__click_add_cloud_db()
        CosmosDBSQLInstance(self.admin_console).create_instance(cloud_account, plan, content)

    @PageService()
    def add_oracle_instance(self, server_name, oracle_sid, plan, oracle_home,
                            connect_string, catalog_connect_string=None):
        """Adds new oracle instance

            Args:
                server_name             (str)   --  Server name
                oracle_sid              (str)   --  Oracle server SID
                plan                    (str)   --  Plan to be associated with the instance
                oracle_home             (str)   --  oracle home directory
                connect_string          (str)   --  Connect string for the server in the format:
                                                    <Username>/<Password>@<Service Name>
                catalog_connect_string  (str)   --  Connect string for catalog connect in format:
                                                    <Username>/<Password>@<Service Name>
                    default:    None

        """
        credentials, service_name = connect_string.split("@")
        username, password = credentials.split("/")
        catalog_username = None
        catalog_password = None
        catalog_service_name = None
        use_catalog_connect = False
        if catalog_connect_string:
            use_catalog_connect = True
            catalog_credentials, catalog_service_name = catalog_connect_string.split("@")
            catalog_username, catalog_password = catalog_credentials.split("/")
        self.add_instance.add_oracle_instance(
            server_name=server_name, oracle_sid=oracle_sid, plan=plan, oracle_home=oracle_home,
            username=username, password=password, service_name=service_name,
            use_catalog_connect=use_catalog_connect, catalog_username=catalog_username,
            catalog_password=catalog_password, catalog_service_name=catalog_service_name)

    @PageService()
    def discover_instances(self, database_engine, server_name):
        """Discovers instances
            Args:
                database_engine (DBInstances.Types) --  Use the enum to select any
                                                        specific database
                server_name     (str):  Server name
        """
        self.admin_console.access_menu_from_dropdown('Discover instances')
        self.admin_console.select_value_from_dropdown(select_id='agentTypeSelection',
                                                      value=database_engine.value)
        self.__panel_dropdown.select_drop_down_values(values=[server_name],
                                                      drop_down_id="cvClients",
                                                      partial_selection=True)
        self.admin_console.submit_form()

    @PageService()
    def backup(self, instance, backupset=None, subclient=None,
               backup_type=Backup.BackupType.INCR,
               enable_data_for_incremental=False, cumulative=False):
        """
        Submits backup job from the subclient page
        Args:
            backup_type                 (Backup.BackupType) -- backup type

            instance                    (str)               -- Name of instance

            backupset                   (str)               -- Name of backupset
                None

            subclient                   (str)               -- Name of subclient
                None

            enable_data_for_incremental (bool)              -- flag to check if
            data needs to be backed up during incremental
                default: False

            cumulative                  (bool)              -- flag to check for
            cumulative backup


        Returns
            (str) -- Backup job id

        """
        self.instances_table.access_action_item(
            instance, self.admin_console.props['label.globalActions.backup'])
        if backupset is None and subclient is not None:
            backupset = instance
        return Backup(self.admin_console).submit_backup(
            backup_type=backup_type, backupset_name=backupset, subclient_name=subclient,
            incremental_with_data=enable_data_for_incremental, cumulative=cumulative)

    @PageService()
    def add_server(self, database_type, server_name, username, password, plan,
                   unix_group=None, install_location=None, os_type="windows"):
        """
        Method to add server of database type
            Args:
                database_type   (DBInstances.Types) --  Use the enum to select any
                                                        specific database

                server_name     (str)               --  Name of server

                username        (str)               --  Server username

                password        (str)               --  Server password

                plan            (str)               --  Plan to associate to the server

                unix_group      (str)               --  Unix group to associate the server to

                install_location(str)               --  Install location on server
                    default:    None

                os_type         (str)               --  "windows"/"unix"
                    default:    "windows"

            Returns:    job id of add server job launched
        """
        self.admin_console.access_menu(self.props['label.installSoftware'])
        self.admin_console.access_sub_menu(database_type.value)

        if 'windows' in os_type.lower():
            self.admin_console.select_radio('osTypeWINDOWS')
            if "\\" not in username:
                raise Exception("Include domain in username")
        else:
            self.admin_console.select_radio('osTypeUNIX')
            if unix_group:
                self.admin_console.fill_form_by_id("uGroup", unix_group)
        self.admin_console.fill_form_by_id("serverName", server_name)
        self.admin_console.fill_form_by_id("userName", username)
        self.admin_console.fill_form_by_id("password", password)
        self.__panel_dropdown.select_drop_down_values(
            drop_down_id='planSummaryDropdown', values=[plan])
        if install_location:
            self.admin_console.fill_form_by_id("installLocation", install_location)
        self.admin_console.submit_form()
        _jobid = self.admin_console.get_jobid_from_popup()
        self.admin_console.wait_for_completion()
        return _jobid


class DBInstance(ABC):
    """This class provides the functions or operations that can be performed in common
    on any DB Instance"""

    def __init__(self, admin_console: AdminConsole):
        """Class constructor

            Args:
                admin_console   (obj)   --  The admin console class object

        """
        self.admin_console = admin_console

    @property
    @abstractmethod
    def db_type(self):
        raise NotImplementedError


class _DBInstanceFeatures:

    class MigrationVendor(Enum):
        """Available Vendor types supported"""
        AZURE = "AZURE"

    def __init__(self, admin_console: AdminConsole, db_type):
        """Class constructor

            Args:
                admin_console   (obj)   --  The admin console class object

                db_type (str) -- Type of the DB instance

        """
        self.admin_console = admin_console
        self._db_type = db_type

    @PageService()
    def configure_migrate_to_cloud(self, vendor: MigrationVendor):
        """Starts the "Migrate to cloud" option of the instance and submits the form

            Args:
                vendor      (MigrationVendor)   --  The cloud vendor to migrate the SQL DB to.



             Returns:
                 MigrateToCloud   --      an implemented instance of MigrateToCloud

        """

        factory_dict = {
            "AZURE": {
                "SQL Server": MigrateSQLToAzure,
                "Oracle":  MigrateOracleToAzure
            }
        }
        self.admin_console.access_menu_from_dropdown('Migrate to cloud')
        return factory_dict[vendor.value][self._db_type](self.admin_console)


class SQLInstance(DBInstance, _DBInstanceFeatures):

    def __init__(self, admin_console: AdminConsole):
        DBInstance.__init__(self, admin_console)
        _DBInstanceFeatures.__init__(self, admin_console, self.db_type)

    @property
    def db_type(self):
        return "SQL Server"


class OracleInstance(DBInstance, _DBInstanceFeatures):

    def __init__(self, admin_console: AdminConsole):
        DBInstance.__init__(self, admin_console)
        _DBInstanceFeatures.__init__(self, admin_console, self.db_type)

    @property
    def db_type(self):
        return "Oracle"
