# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the common functions/operations that can be performed on the Database
instance details page, the page that opens after selecting an instance from Databases page.

Since some agents have restore option only from instance and few have from
both instance and subclient level, We have two ENUMs and maps.
One is to associate Database types with add subclient class
Another one is to associate Database types with restore panel class

DBInstanceDetails:
------------------
    get_instance_details()              --  Returns details of Instance

    get_instance_entities()             --  Returns the list of items/entities in the Instance
    details page which could be one of the following:subclients or Table groups or Backupsets
    or Database groups

    delete_instance()                   --  Deletes the instance

    list_backup_history_of_entity()     --  Clicks on 'Backup history' from given entity's action items

    click_on_entity()                   --  Clicks the given entity which can be subclient or
    backupset or table group

    click_add_subclient()               --  Clicks on add subclient/table group/ database group
    and returns the object of add subclient class of database corresponding to database_type
    argument

    access_restore()                    --  Clicks on the restore button below the recovery
    points in instance page

    clear_all_selection()               --  Clicks on clear all checkbox in browse page if present

    restore_folders()                   --  Clicks on the given items, analogous to folders in the
    browse page and submits restore. Also returns the object of restore panel class corresponding
    to database_type argument

    restore_files_from_multiple_pages() --  Clicks on items from multiple pages recursively in
    browse page and submits restore. Also returns the object of restore panel class corresponding
    to database_type argument

    access_configuration_tab()          --  Clicks on 'Configuration' tab on instance details page

    access_overview_tab()               --  Clicks on 'Overview' tab on instance details page
    
    edit_instance_change_plan()         --  Edits instance to change plan associated

MySQLInstanceDetails:
---------------------
    enable_xtrabackup()                 --  Edits instance to enable xtrabackup option and
                                            specify xtrabackup log directory

    enable_standby_instance()           --  Edits instance to enable standby instance option
                                            and specify standby instance

    


PostgreSQLInstanceDetails instance Attributes:
----------------------------------------------

    **binary_directory**                --  returns postgres binary directory

    **library_directory**               --  returns postgres Library directory

    **archive_log_directory**           --  returns postgres archive log directory

    **version**                         --  returns postgres server version

    **user**                            --  returns postgres user

    **port**                            --  returns postgres server port

Db2InstanceDetails instance Attributes:
---------------------------------------

    **db2_home_directory**              --  returns db2 home directory

    **db2_version**                     --  returns db2 application version

    **db2_user_name**                   --  returns db2 instance user name

CloudDBInstanceDetails:
-----------------------
    modify_region_for_cloud_account()   --  Method to set region restriction for cloud account

InformixInstanceDetails instance Attributes:
---------------------------------------

    **informix_home**                   --  returns informix home directory

    **onconfig**                        --  returns informix onconfig filename

    **sqlhosts**                        --  returns informix sqlhosts file path

    **informix_username**               --  returns informix user name


OracleInstanceDetails:
----------------------
    edit_instance_update_credentials()  --  Edits instance to change connect string and
                                            plan associated

"""
from enum import Enum
from selenium.common.exceptions import NoSuchElementException, ElementNotInteractableException
from Web.Common.page_object import (
    PageService,
)
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import PanelInfo
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.Components.browse import Browse
from Web.AdminConsole.Components.panel import DropDown
from Web.AdminConsole.Databases.db_instances import DBInstances
from Web.AdminConsole.Databases.Instances.add_subclient import AddDynamoDBSubClient
from Web.AdminConsole.Databases.Instances.add_subclient import AddMySQLSubClient
from Web.AdminConsole.Databases.Instances.add_subclient import AddRedshiftSubClient
from Web.AdminConsole.Databases.Instances.add_subclient import AddInformixSubClient
from Web.AdminConsole.Databases.Instances.add_subclient import AddOracleSubClient
from Web.AdminConsole.Databases.Instances.restore_panels import DynamoDBRestorePanel
from Web.AdminConsole.Databases.Instances.restore_panels import OracleRestorePanel
from Web.AdminConsole.Databases.Instances.restore_panels import DB2RestorePanel
from Web.AdminConsole.Databases.Instances.restore_panels import RedshiftRestorePanel
from Web.AdminConsole.Databases.Instances.restore_panels import InformixRestorePanel
from Web.AdminConsole.Databases.Instances.restore_panels import CosmosDBSQLRestorePanel


class DBInstanceDetails:
    """Class for Database Instances page"""
    class SubclientTypes(Enum):
        """Enum to represent classes for adding subclient"""
        DYNAMODB = "AddDynamoDBSubClient"
        MYSQL = "AddMySQLSubClient"
        REDSHIFT = "AddRedshiftSubClient"
        INFORMIX = "AddInformixSubClient"
        ORACLE = "AddOracleSubClient"

    class RestorePanelTypes(Enum):
        """Enum to represent classes for implementing restore panel"""
        DYNAMODB = "DynamoDBRestorePanel"
        ORACLE = "OracleRestorePanel"
        DB2 = "DB2RestorePanel"
        MYSQL = "MySQLRestorePanel"
        REDSHIFT = "RedshiftRestorePanel"
        INFORMIX = "InformixRestorePanel"
        COSMOSDB_SQL = "CosmosDBSQLRestorePanel"

    def __init__(self, admin_console):
        """
        Args:
            admin_console (AdminConsole): Object of AdminConsole class
        """
        self.__admin_console = admin_console
        self.__table = Table(self.__admin_console)
        self.__browse = Browse(self.__admin_console)
        self._panel_dropdown = DropDown(self.__admin_console)
        self.__instance_panel = None
        self.__add_subclient_map = {
            DBInstances.Types.DYNAMODB: DBInstanceDetails.SubclientTypes.DYNAMODB,
            DBInstances.Types.MYSQL: DBInstanceDetails.SubclientTypes.MYSQL,
            DBInstances.Types.REDSHIFT: DBInstanceDetails.SubclientTypes.REDSHIFT,
            DBInstances.Types.INFORMIX: DBInstanceDetails.SubclientTypes.INFORMIX,
            DBInstances.Types.ORACLE: DBInstanceDetails.SubclientTypes.ORACLE
            }
        self.__restore_panel_map = {
            DBInstances.Types.DYNAMODB: DBInstanceDetails.RestorePanelTypes.DYNAMODB,
            DBInstances.Types.ORACLE: DBInstanceDetails.RestorePanelTypes.ORACLE,
            DBInstances.Types.MYSQL: DBInstanceDetails.RestorePanelTypes.MYSQL,
            DBInstances.Types.REDSHIFT: DBInstanceDetails.RestorePanelTypes.REDSHIFT,
            DBInstances.Types.INFORMIX: DBInstanceDetails.RestorePanelTypes.INFORMIX,
            DBInstances.Types.COSMOSDB_SQL: DBInstanceDetails.RestorePanelTypes.COSMOSDB_SQL
            }

    @PageService()
    def get_instance_details(self):
        """Returns details of Instance"""
        self.__instance_panel = PanelInfo(self.__admin_console, title='General')
        return self.__instance_panel.get_details()

    @PageService()
    def get_instance_entities(self):
        """
        Returns the list of items/entities in the Instance details page which could
        be one of the following:
        subclients or Table groups or Backupsets or Database groups
        """
        return self.__table.get_column_data(self.__admin_console.props['label.name'])

    @PageService()
    def delete_instance(self):
        """Deletes the instance"""
        self.__admin_console.access_menu_from_dropdown(
            self.__admin_console.props['action.delete']
        )
        _dialog = ModalDialog(self.__admin_console)
        _dialog.type_text_and_delete('DELETE')
        self.__admin_console.wait_for_completion()

    @PageService()
    def click_on_entity(self, entity_name):
        """Clicks the given entity which can be subclient or backupset or table group
        Args:
            entity_name (str)  :    Name of the entity which needs to be clicked

        """
        self.__table.access_link(entity_name)

    @PageService()
    def click_add_subclient(self, database_type):
        """Clicks on add subclient/table group/ database group and returns the object
        of add subclient class of database corresponding to database_type argument

        Args:
            database_type (Types):   Type of database should be one among the types
                                     defined in 'Types' enum in DBInstances.py file

        Returns:
            Object (SubclientTypes):  Object of class in SubclientTypes corresponding to
                                        database_type
        """
        self.__table.access_toolbar_menu('addSubclient')
        return globals()[self.__add_subclient_map[database_type].value](self.__admin_console)

    @PageService()
    def list_backup_history_of_entity(self, entity_name):
        """Clicks on 'Backup history' from any entity's action items(like subclient)
        Args:
            entity_name (str)   :   Name of entity to view backup history
        """
        self.__table.access_action_item(entity_name,
                                        self.__admin_console.props['label.BackupHistory'])

    @PageService()
    def access_restore(self):
        """Clicks on the restore button below the recovery points in instance page"""
        self.__admin_console.access_action(self.__admin_console.props['action.restore'])

    @PageService()
    def clear_all_selection(self):
        """Clicks on clear all checkbox in browse page if present"""
        self.__browse.clear_all_selection()

    @PageService()
    def restore_folders(self, database_type, items_to_restore=None, all_files=False,
                        copy=None):
        """ Selects files and folders to restore

        Args:
            database_type (Types):   Type of database should be one among the types defined
                                      in 'Types' enum in DBInstances.py file

            items_to_restore (list):  the list of files and folders to select for restore

                default: None

            all_files        (bool):  select all the files shown for restore / download

                default: False

            copy            (str):  The name of the copy to browse from
                                    Example- "Secondary" or "Copy-2"
                default: None

        Returns:
            Object (RestorePanelTypes): Object of class in RestorePanelTypes corresponding
                                        to database_type
        """
        if copy:
            self.__browse.select_adv_options_submit_restore(copy, database=True)
        self.__browse.select_for_restore(items_to_restore, all_files)
        self.__browse.submit_for_restore()
        return globals()[self.__restore_panel_map[database_type].value](self.__admin_console)

    @PageService()
    def restore_files_from_multiple_pages(self, database_type, mapping_dict, root_node,
                                          copy=None):
        """Clicks on items from multiple pages recursively in browse page
        and submits restore. Also returns the object of restore panel class
        corresponding to database_type argument

        Args:
            database_type (Types):   Type of database should be one among the types defined
                                      in 'Types' enum in DBInstances.py file

            mapping_dict (dict) : The dictionary containing the folder names as keys
                                and list of files to be selected under them as value

                Example:
                    mapping_dict={
                    'FOLDER1':['file1','file2','file3']
                    'FOLDER2':['fileA','fileB','fileC']
                    }

            root_node   (str):  The name of the subclient/instance on which browse operation
                                    was performed or the name of the root folder that
                                    appears on the browse page

            copy        (str):  The name of the copy to browse from
                                Example- "Secondary" or "Copy-2"

        Returns:
            Object (RestorePanelTypes): Object of class in RestorePanelTypes corresponding
                                        to database_type
        """
        if copy:
            self.__browse.select_adv_options_submit_restore(copy, database=True)
        self.__browse.select_from_multiple_pages(mapping_dict, root_node)
        self.__browse.submit_for_restore()
        return globals()[self.__restore_panel_map[database_type].value](self.__admin_console)

    @PageService()
    def access_configuration_tab(self):
        """Clicks on 'Confiugration' tab on the instance details page"""
        self.__admin_console.access_tab(
            self.__admin_console.props['heading.settings'])

    @PageService()
    def access_overview_tab(self):
        """Clicks on 'Overview' tab on the instance details page"""
        self.__admin_console.access_tab(
            self.__admin_console.props['heading.overview'])

    @PageService()
    def edit_instance_change_plan(self, new_plan):
        """ Edits instance to change plan
            Args:
                new_plan    (str):   New plan for the instance
        """
        try:
            self.__instance_panel = PanelInfo(self.__admin_console, "General")
            self.__instance_panel.edit_tile_entity('Plan')
        except (NoSuchElementException, ElementNotInteractableException):
            self.__admin_console.select_hyperlink(
                self.__admin_console.props['action.edit'])
        self._panel_dropdown.select_drop_down_values(
            values=[new_plan],
            drop_down_id='planSummaryDropdown')
        try:
            self.__instance_panel.save_dropdown_selection('Plan')
        except (NoSuchElementException, ElementNotInteractableException):
            self.__admin_console.submit_form()
        self.__admin_console.wait_for_completion()


class PostgreSQLInstanceDetails(DBInstanceDetails):
    """This class provides the function or operations to perform on postgres instance page
    """

    def __init__(self, admin_console):
        """Class constructor

            Args:
                admin_console   (obj)                 --  The admin console class object

        """
        super(PostgreSQLInstanceDetails, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.postgres_instance_properties = None

    @property
    def binary_directory(self):
        """returns postgres binary directory"""
        if not self.postgres_instance_properties:
            self.postgres_instance_properties = self.get_instance_details()
        return self.postgres_instance_properties['Binary directory']

    @property
    def library_directory(self):
        """returns postgres Library directory"""
        if not self.postgres_instance_properties:
            self.postgres_instance_properties = self.get_instance_details()
        return self.postgres_instance_properties['Lib directory']

    @property
    def archive_log_directory(self):
        """returns postgres archive log directory"""
        if not self.postgres_instance_properties:
            self.postgres_instance_properties = self.get_instance_details()
        return self.postgres_instance_properties['Archive log directory']

    @property
    def version(self):
        """returns postgres server version"""
        if not self.postgres_instance_properties:
            self.postgres_instance_properties = self.get_instance_details()
        return self.postgres_instance_properties['Version']

    @property
    def user(self):
        """returns postgres user"""
        if not self.postgres_instance_properties:
            self.postgres_instance_properties = self.get_instance_details()
        return self.postgres_instance_properties['PostgreSQL user name']

    @property
    def port(self):
        """returns postgres server port"""
        if not self.postgres_instance_properties:
            self.postgres_instance_properties = self.get_instance_details()
        return self.postgres_instance_properties['Port']


class Db2InstanceDetails(DBInstanceDetails):
    """This class provides the function or operations to perform on postgres instance page
    """

    def __init__(self, admin_console):
        """Class constructor

            Args:
                admin_console   (obj)                 --  The admin console class object

        """
        super(Db2InstanceDetails, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.db2_instance_properties = None

    @property
    def db2_home_directory(self):
        """returns db2 home directory"""
        if not self.db2_instance_properties:
            self.db2_instance_properties = self.get_instance_details()
        return self.db2_instance_properties['Home']

    @property
    def db2_version(self):
        """returns db2 version"""
        if not self.db2_instance_properties:
            self.db2_instance_properties = self.get_instance_details()
        return self.db2_instance_properties['Version']

    @property
    def db2_user_name(self):
        """returns db2 user name"""
        if not self.db2_instance_properties:
            self.db2_instance_properties = self.get_instance_details()
        return self.db2_instance_properties['User name']


class MySQLInstanceDetails(DBInstanceDetails):
    """This class provides the function or operations to perform on MySQL instance page
        """

    def __init__(self, admin_console):
        """Class constructor

            Args:
                admin_console   (obj)                 --  The admin console class object

        """
        super(MySQLInstanceDetails, self).__init__(admin_console)
        self.__admin_console = admin_console

    @PageService()
    def enable_xtrabackup(self, xtra_backup_bin_path):
        """Edits the instance to enable Xtrabackup
            Args:
                xtra_backup_bin_path    (str):  XtraBackup Bin path
        """
        self.__admin_console.select_hyperlink(
            self.__admin_console.props['action.edit']
        )
        self.__admin_console.expand_accordion(
            self.__admin_console.props['label.assets.database.advancedOptions'])
        self.__admin_console.enable_toggle(index=-2)
        self.__admin_console.fill_form_by_name('XtraBackupPath', xtra_backup_bin_path)
        self.__admin_console.submit_form()

    @PageService()
    def enable_standby_instance_backup(self, standby_instance, run_logs_on_source=False):
        """Edits the instance to enable standby instance for backup
            Args:
                standby_instance    (str):  Name of the standby instance
                run_logs_on_source  (Boolean):  True if transaction logs are to be
                                                run on source
                    Default:    False
        """
        self.__admin_console.select_hyperlink(
            self.__admin_console.props['action.edit']
        )
        self.__admin_console.expand_accordion(
            self.__admin_console.props['label.assets.database.advancedOptions'])
        self.__admin_console.enable_toggle(index=-1)
        self._panel_dropdown.select_drop_down_values(values=[standby_instance], index=3)
        if run_logs_on_source:
            self.__admin_console.checkbox_select(checkbox_id="runLogBackupsOnSource")
        self.__admin_console.submit_form()
        self.__admin_console.wait_for_completion()


class CloudDBInstanceDetails(DBInstanceDetails):
    """This class provides the common functions or operations that can be performed on
        instances of cloud databases like RDS, dynamodb, redshift, documentdb"""

    def __init__(self, admin_console):
        """Method to initialize CloudDBIstanceDetails class
        Args:
            admin_console (AdminConsole):   Object of AdminConsole class
        """
        super().__init__(admin_console)
        self.__admin_console = admin_console

    @PageService()
    def modify_region_for_cloud_account(self, region_name):
        """Method to set a region restriction for cloud account
        Args:
            region_name (list): The list of cloud regions that needs to be set/enabled
                                for the cloud account
        """
        self.access_configuration_tab()
        settings_panel = PanelInfo(self.__admin_console,
                                   title=self.__admin_console.props['label.nav.settings'])
        settings_panel.edit_tile_entity(
            self.__admin_console.props['dbObjType.region'])
        _panel_dropdown = DropDown(self.__admin_console)
        _panel_dropdown.select_drop_down_values(values=region_name,
                                                drop_down_id='regionSelect')
        settings_panel.save_dropdown_selection(
            self.__admin_console.props['dbObjType.region'])


class InformixInstanceDetails(DBInstanceDetails):
    """This class provides the function or operations to perform on Informix instance page
    """

    def __init__(self, admin_console):
        """Class constructor

            Args:
                admin_console   (obj)                 --  The admin console class object

        """
        super(InformixInstanceDetails, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.informix_instance_properties = None

    @property
    def informix_home(self):
        """returns informix home directory"""
        if not self.informix_instance_properties:
            self.informix_instance_properties = self.get_instance_details()
        return self.informix_instance_properties['Informix home']

    @property
    def onconfig(self):
        """returns informix onconfig file name"""
        if not self.informix_instance_properties:
            self.informix_instance_properties = self.get_instance_details()
        return self.informix_instance_properties['Onconfig file']

    @property
    def sqlhosts(self):
        """returns informix sqlhosts file path"""
        if not self.informix_instance_properties:
            self.informix_instance_properties = self.get_instance_details()
        return self.informix_instance_properties['SQLHOSTS file']

    @property
    def informix_username(self):
        """returns informix user name"""
        if not self.informix_instance_properties:
            self.informix_instance_properties = self.get_instance_details()
        return self.informix_instance_properties['User name']


class OracleInstanceDetails(DBInstanceDetails):
    """This class provides the function or operations to perform on Oracle instance page
       """

    def __init__(self, admin_console):
        """Class constructor

            Args:
                admin_console   (obj)                 --  The admin console class object

        """
        super(OracleInstanceDetails, self).__init__(admin_console)
        self.__admin_console = admin_console

    @PageService()
    def edit_instance_update_credentials(self, connect_string, plan=None):
        """Method to Update connect string for oracle instance
            Args:
                connect_string  (str):  Connect string for the server in the format:
                                        <Username>/<Password>@<Service Name>
                plan            (str):  New plan for the instance
                    default: None

        """
        credentials, service_name = connect_string.split("@")
        username, password = credentials.split("/")
        self.__admin_console.select_hyperlink(
            self.__admin_console.props['action.edit'])
        if plan:
            self._panel_dropdown.select_drop_down_values(
                values=[plan], drop_down_id='oracleCreateInstance_isteven-multi-select_#9446')
        self.__admin_console.fill_form_by_name('dbUserName', username)
        self.__admin_console.fill_form_by_name('dbPassword', password)
        self.__admin_console.fill_form_by_name('dbInstanceName', service_name)
        self.__admin_console.submit_form()
