# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on the Database
backupset page, the page that opens after selecting a backupset from instance page.

Backupset:
----------
    get_backupset_general_properties()  --  method to get properties of backupset

    list_subclients()                   --  Method returns the list of subclients/database
    groups under the backupset

    access_subclient()                  --  Clicks the given subclient

    access_restore()                    --  Clicks on the restore button below the recovery points

    restore_folders()                   --  Selects files and folders to restore

    delete_subclient()                  --  method to delete the subclient

PostgreSQLBackupset:
--------------------
    add_subclient()                     --  method to add postgreSQL dumpbased subclient

DB2Subclient:
-------------

"""
from enum import Enum
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.Components.panel import PanelInfo
from Web.AdminConsole.Components.panel import DropDown
from Web.AdminConsole.Components.browse import Browse
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.Common.page_object import PageService
from Web.AdminConsole.Databases.db_instances import DBInstances
from Web.AdminConsole.Databases.Instances.restore_panels import PostgreSQLRestorePanel
from Web.AdminConsole.Databases.Instances.restore_panels import DB2RestorePanel
from Web.AdminConsole.Databases.Instances.add_subclient import AddPostgreSQLSubClient


class Backupset:
    """This class provides the function or operations to perform on backupset page
    """

    class SubclientTypes(Enum):
        """Enum to represent classes for adding subclient"""
        POSTGRES = "AddPostgreSQLSubClient"
        DB2 = "AddDB2Subclient"

    class RestorePanelTypes(Enum):
        """Enum to represent classes for implementing restore panel"""
        POSTGRES = "PostgreSQLRestorePanel"
        DB2 = "DB2RestorePanel"

    def __init__(self, admin_console: AdminConsole):
        """Class constructor

            Args:
                admin_console   (obj)                 --  The admin console class object

        """
        self.__admin_console = admin_console
        self.__browse = Browse(self.__admin_console)
        self.__table = Table(self.__admin_console)
        self._panel_dropdown = DropDown(self.__admin_console)
        self.__admin_console._load_properties(self)
        self.props = self.__admin_console.props
        self.__backupset_general_panel = None
        self._restore_panel_map = {
            DBInstances.Types.POSTGRES: Backupset.RestorePanelTypes.POSTGRES,
            DBInstances.Types.DB2: Backupset.RestorePanelTypes.DB2,
        }
        self._add_subclient_map = {
            DBInstances.Types.POSTGRES: Backupset.SubclientTypes.POSTGRES
        }

    @PageService()
    def get_backupset_general_properties(self):
        """method to get properties of backupset"""
        self.__backupset_general_panel = PanelInfo(
            self.__admin_console, title=self.props['label.nav.general'])
        return self.__backupset_general_panel.get_details()

    @PageService()
    def list_subclients(self):
        """Method returns the list of subclients/database groups under the backupset

        Returns:
            list of subclients under the backupset

        """
        return self.__table.get_column_data(self.props['label.name'])

    @PageService()
    def access_subclient(self, subclient_name):
        """Clicks the given subclient"""
        self.__table.access_link(subclient_name)

    @PageService()
    def access_restore(self):
        """Clicks on the restore button below the recovery points"""
        self.__admin_console.access_action(self.props['action.restore'])

    @PageService()
    def restore_folders(
            self, database_type, items_to_restore=None,
            all_files=False, from_time=None, to_time=None):
        """ Selects files and folders to restore

        Args:
            database_type (Types):   Type of database should be one among the types defined
            in 'Types' enum in DBInstances.py file

            items_to_restore (list):  the list of files and folders to select for restore

                default: None

            all_files        (bool):  select all the files shown for restore / download

                default: False

            from_time   :   Time from when to backup
                format: %d-%B-%Y-%I-%M-%p
                        (dd-Month-yyyy-hour(12 hour)-minutes-session(AM/PM))
                        (01-January-2000-11-59-PM)

            to_time   :   Time till when to backup
                format: %d-%B-%Y-%I-%M-%p
                        (dd-Month-yyyy-hour(12 hour)-minutes-session(AM/PM))
                        (01-January-2000-11-59-PM)

        Returns:
            object of relevant class in restore_panels file

        """
        if to_time:
            self.__browse.show_backups_by_date_range(from_time=from_time, to_time=to_time)
        self.__browse.clear_all_selection()
        self.__browse.select_for_restore(items_to_restore, all_files)
        self.__browse.submit_for_restore()
        return globals()[self._restore_panel_map[database_type].value](self.__admin_console)

    @PageService()
    def delete_subclient(self, subclient_name):
        """
        method to delete the subclient

        Args:
            subclient_name (str):   name of the subclient to be deleted

        """
        self.__table.access_action_item(subclient_name, 'Delete')
        _dialog = ModalDialog(self.__admin_console)
        _dialog.type_text_and_delete('DELETE')
        self.__admin_console.wait_for_completion()


class PostgreSQLBackupset(Backupset):
    """This class provides the function or operations to perform on backupset page
    """

    def __init__(self, admin_console):
        """Class constructor

            Args:
                admin_console   (obj)                 --  The admin console class object

        """
        super(PostgreSQLBackupset, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.__table = Table(self.__admin_console)
        self.__admin_console._load_properties(self)
        self.props = self.__admin_console.props

    @PageService()
    def add_subclient(self, subclient_name, number_backup_streams,
                      collect_object_list, plan, database_list):
        """
        method to add postgreSQL dumpbased subclient

        Args:
            subclient_name          (str):  Name of the subclient

            number_backup_streams   (int): number of streams used for backup

            collect_object_list     (bool): boolean value to specify if collect object
            list needs to be enabled

            plan                    (str):  plan name to be assigned to subclient

            database_list           (list): list of databases which needs to be part
            of subclient content

        """
        self.__table.access_toolbar_menu('addSubclient')
        add_subclient_object = globals()[
            self._add_subclient_map[DBInstances.Types.POSTGRES].value](self.__admin_console)
        add_subclient_object.add_subclient(
            subclient_name, number_backup_streams, collect_object_list, plan, database_list)


class DB2Backupset(Backupset):
    """This class provides the function or operations to perform on backupset page
    """

    def __init__(self, admin_console):
        """Class constructor

            Args:
                admin_console   (obj)                 --  The admin console class object

        """
        super(DB2Backupset, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.__table = Table(self.__admin_console)
        self.__admin_console._load_properties(self)
        self.props = self.__admin_console.props
        self.browse = Browse(admin_console)
