# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module implements the methods that fill in the various restore
options for databases. Each class represents the options available per database

DynamoDBRestorePanel:

    adjust_write_capacity()         --      Sets temporary write capacity for restore

    enable_overwrite()              --      Enables the overwrite destination tables toggle

    set_streams()                   --      Sets the number of streams for restore

    same_account_same_region()      --      Submits a restore with same cloud account as source
    and restores tables to same regions as source

    cross_account_same_region()     --      Submits a cross account restore to the given destination
    account and tables are restored to same region as source

RedshiftRestorePanel:

    same_account_same_region        --      Submits a restore with same cloud account as source
    and restores snapshots to same regions as source

PostgreSQLRestorePanel
    in_place_restore()              --      Submits a in place restore of postgreSQL database

    out_of_place_restore()          --      Submits a out of place restore of postgreSQL database

OracleRestorePanel
    _redirect_tablespace            --      Enters redirect path for datafile restore for
     corresponding tablespace

    _redirect                       --      Enters redirect restore options

    _recover                        --      Enter recover to options

    in_place_restore()              --      Submits a in place restore of Oracle database

    out_of_place_restore()          --      Submits an out of place restore of Oracle database

DB2RestorePanel
    in_place_restore()              --      Submits a in place restore of db2 database


MySQLRestorePanel
    mark_email()                    --      Marks notify by email option

    in_place_restore()              --      Submits a in place restore of MySQL database

RDSRestorePanel:
    restore()                       --      Submits RDS restore without setting any advanced options

SybaseResorePanel
    mark_email()                    --      Marks notify by email option

    mark_recover_databases()        --      Marks recover databases option

    _click_device()                 --      Clicks on device in redirect options

    _construct_redirect_dict()      --      Constructs redirect options dict

    in_place_restore()              --      Submits an in place restore of Sybase database

    out_of_place_restore()          --      Submits an out of place restore of Sybase database

InformixRestorePanel
    informix_restore()              --      To perform inplace and out of place restore for Informix
    __get_value()                   --      Gets element value from restore options using element id

CosmosDBSQLRestorePanel:

    disable_write_throughput()      --      Disables the adjust throughput toggle

    select_overwrite()              --      Enables overwrites destination containers option

    in_place_restore()              --      Submits restore to same cloud account

"""
from datetime import datetime
from Web.Common.page_object import (
    PageService,
    WebAction
)
from Web.Common.exceptions import CVWebAutomationException
from Web.AdminConsole.Components.panel import ModalPanel, DropDown
from Web.AdminConsole.Components.dialog import ModalDialog

class DynamoDBRestorePanel(ModalPanel):
    """Class to represent the restore panel for DynamoDB"""
    def __init__(self, admin_console):
        super(DynamoDBRestorePanel, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.__dialog = ModalDialog(self.__admin_console)

    @PageService()
    def adjust_write_capacity(self, write_capacity):
        """Sets temporary write capacity for restore
        Args:
            write_capacity   (int):  The value to be set as write capacity

        """
        self.__admin_console.enable_toggle(0)
        self.__admin_console.fill_form_by_id('writeCap', write_capacity)

    @PageService()
    def enable_overwrite(self):
        """Enables the overwrite destination tables toggle"""
        self.__admin_console.enable_toggle(1)

    @PageService()
    def set_streams(self, number_of_streams):
        """Sets the number of streams for restore

        Args:
            number_of_streams   (int):  Number of streams to be set

        """
        self.__admin_console.fill_form_by_id('noOfStreams',
                                             number_of_streams)

    @PageService()
    def same_account_same_region(self, overwrite=True, adjust_write_capacity=0):
        """Submits a restore with same cloud account as source
        and restores tables to same regions as source

        Args:
            overwrite (Boolean): True to overwrite tables
                                False to not overwrite

            adjust_write_capacity (int):  The value to be set as write capacity
        """
        if overwrite:
            self.enable_overwrite()
        if adjust_write_capacity != 0:
            self.adjust_write_capacity(adjust_write_capacity)
        self.submit()
        self.__dialog.click_submit()
        _jobid = self.__admin_console.get_jobid_from_popup()
        return _jobid

    @PageService()
    def cross_account_same_region(self, dest_account, overwrite=True):
        """Submits a cross account restore to the given destination
        account and tables are restored to same region as source

        Args:
            dest_account (str):  The name of destination cloud account
            overwrite (Boolean): True to overwrite tables
                                False to not overwrite

        """
        self._dropdown.select_drop_down_values(
            values=[dest_account], drop_down_id='destinationServer')
        if overwrite:
            self.enable_overwrite()
        self.submit()
        self.__dialog.click_submit()
        _jobid = self.__admin_console.get_jobid_from_popup()
        return _jobid

class RedshiftRestorePanel(ModalPanel):
    """Class to represent the restore panel for Redshift"""
    def __init__(self, admin_console):
        super(RedshiftRestorePanel, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.__dialog = ModalDialog(self.__admin_console)

    @PageService()
    def same_account_same_region(self, cluster_identifier):
        """Submits a restore with same cloud account as source
        and restores snapshots to same regions as source

        Args:
            cluster_identifier(str) --  Name of the cluster to be created

        """
        self.__admin_console.fill_form_by_id('dbIdentifier', cluster_identifier)
        self.submit()
        self.__dialog.click_submit()
        _jobid = self.__admin_console.get_jobid_from_popup()
        return _jobid

class PostgreSQLRestorePanel(ModalPanel):
    """Class to represent the restore panel for PostgreSQl DB"""
    def __init__(self, admin_console):
        super(PostgreSQLRestorePanel, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.__dialog = ModalDialog(self.__admin_console)

    @PageService()
    def in_place_restore(self, fsbased_restore=False):
        """Submits a in place restore of postgreSQL database

        Args:

            fsbased_restore      (bool): Boolean value to specify if
            the restore is being performed on fsbasedbackupset

        Returns:
            restore job id on succesful restore submission

        """
        self.access_tab('In place')
        self.submit()
        if fsbased_restore:
            self.__dialog.click_submit()
        return self.__admin_console.get_jobid_from_popup()

    @PageService()
    def out_of_place_restore(self, destination_client, destination_instance, fsbased_restore=False):
        """Submits a out of place restore of postgreSQL database

        Args:

            destination_client      (str): Destination client name

            destination_instance    (str): Destination instance name

            fsbased_restore      (bool): Boolean value to specify if
            the restore is being performed on fsbasedbackupset

        Returns:
            restore job id on succesful restore submission

        """
        self.access_tab('Out of place')
        self._dropdown.select_drop_down_values(
            values=[destination_client], drop_down_id='destinationclient')
        self._dropdown.select_drop_down_values(
            values=[destination_instance], drop_down_id='destinationDatabase')
        self.submit()
        if fsbased_restore:
            self.__dialog.click_submit()
        return self.__admin_console.get_jobid_from_popup()


class OracleRestorePanel(ModalPanel):
    """Class to represent the restore panel for Oracle database"""

    def __init__(self, admin_console):
        super(OracleRestorePanel, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.__dialog = ModalDialog(self.__admin_console)
        self._panel_dropdown = DropDown(self.__admin_console)

    @WebAction()
    def _redirect_tablespace(self, tablespace, redirect_paths):
        """Method to fill target path for specific tablespace for redirect restore
            Args:
                tablespace      (str):  Name of tablespace to redirect
                redirect_paths  (list): Redirect paths for datafiles of tablespace
        """
        self.__admin_console.expand_accordion(tablespace)
        xpath = f"//span[contains(text(), '{tablespace}')]/ancestor::div[@role='tab']" \
            f"/following-sibling::div/descendant-or-self::input"
        elems = self.__admin_console.driver.find_elements_by_xpath(xpath)
        for elem, path in zip(elems, redirect_paths):
            elem.clear()
            elem.send_keys(path)

    @WebAction()
    def _redirect(self, redirect_all_path=None, redirect_datafiles=None,
                  redirect_redo=None):
        """Method to enter redirect options

        Args:
            redirect_all_path   (str)   : Redirect all Path
                default: None
            redirect_datafiles  (dict)  : Dict mapping tablespaces with redirect path of datafiles
                default: None
            redirect_redo       (str)   : Redo logs redirect path
                default: None
        """
        self.__admin_console.select_hyperlink('Redirect')
        if redirect_all_path:
            self.__admin_console.select_radio("redirectAllCB")
            self.__admin_console.fill_form_by_name('redirectAll', redirect_all_path)
        if redirect_datafiles:
            self.__admin_console.select_radio("individualTSCB")
            for tablespace in redirect_datafiles:
                self._redirect_tablespace(tablespace.upper(), redirect_datafiles[tablespace])
        if redirect_redo:
            self.__admin_console.enable_toggle(0)
            self.__admin_console.fill_form_by_name('redoLogsPath', redirect_redo)
        self.submit()

    @WebAction()
    def _recover(self, recover_to=None):
        """Method to enter recover options

        Args:
            recover_to          (str/int):"most recent backup"/"current time"/SCN number/
            Point in time in format "%m/%d/%Y %H:%M:%S" (eg. 12/31/2020 23:59:59)
        """
        if recover_to:
            recover_to_map = {"most recent backup": "mostRecent", "current time": "currentTime"}
            if isinstance(recover_to, int):
                self.__admin_console.select_radio("scn")
                self.__admin_console.fill_form_by_name('pitScn', recover_to)
            elif recover_to.lower() in recover_to_map:
                self.__admin_console.select_radio(recover_to_map[recover_to.lower()])
            else:
                self.__admin_console.select_radio("pitDate1")
                self.__admin_console.fill_form_by_name('dateTimeValue', recover_to)

    @PageService()
    def in_place_restore(self, redirect_all_path=None, redirect_datafiles=None,
                         redirect_redo=None, recover_to=None):
        """
        submits inplace restore for Oracle database

        Args:
            redirect_all_path   (str)   : Redirect all Path
                default: None
            redirect_datafiles  (dict)  : Dict mapping tablespaces with redirect path of datafiles
                default: None
            redirect_redo       (str)   : Redo logs redirect path
                default: None
            recover_to          (str/int):"most recent backup"/"current time"/SCN number/
            Point in time in format "%m/%d/%Y %H:%M:%S" (eg. 12/31/2020 23:59:59)
                default: None

        Returns:
             restore job id on succesful restore submission
        """
        self.access_tab('In place')
        if redirect_all_path or redirect_datafiles or redirect_redo:
            self._redirect(redirect_all_path, redirect_datafiles, redirect_redo)
        self._recover(recover_to)
        self.submit()
        self.__dialog.click_submit()
        _jobid = self.__admin_console.get_jobid_from_popup()
        return _jobid

    @PageService()
    def out_of_place_restore(self, destination_client, destination_instance,
                             redirect_all_path=None, redirect_datafiles=None,
                             redirect_redo=None, rman_duplicate=False,
                             duplicate_standby=False, recover_to=None):
        """
        submits out of place restore for Oracle database

        Args:
            destination_client  (str)   : Destination client to restore data to
            destination_instance(str)   : Destination instance to restore data to
            redirect_all_path   (str)   : Redirect all Path
                default: None
            redirect_datafiles  (dict)  : Dict mapping tablespaces with redirect path of datafiles
                default: None
            redirect_redo       (str)   : Redo logs redirect path
                default: None
            rman_duplicate      (bool)  : True if RMAN duplicate option must be enabled
                default: False
            duplicate_standby   (bool)  : True if duplicate for standby option is to be enabled
                default: False
            recover_to          (str/int):"most recent backup"/"current time"/SCN number/
            Point in time in format "%m/%d/%Y %H:%M:%S" (eg. 12/31/2020 23:59:59)
                default: None

        Returns:
             restore job id on succesful restore submission
        """
        self.access_tab("Out of place")
        self._panel_dropdown.select_drop_down_values(values=[destination_client],
                                                     drop_down_id='destinationServer',
                                                     partial_selection=True)
        self._panel_dropdown.select_drop_down_values(values=[destination_instance],
                                                     drop_down_id='destinationInstance')
        self.__admin_console.disable_toggle(0)
        if rman_duplicate:
            self.__admin_console.enable_toggle(0)
            if duplicate_standby:
                self.__admin_console.checkbox_select("database")
        if redirect_all_path or redirect_datafiles or redirect_redo:
            self._redirect(redirect_all_path, redirect_datafiles, redirect_redo)
        self._recover(recover_to)
        self.submit()
        self.__dialog.click_submit()
        _jobid = self.__admin_console.get_jobid_from_popup()
        return _jobid


class DB2RestorePanel(ModalPanel):
    """Class to represent the restore panel for DB2 database"""

    def __init__(self, admin_console):
        super(DB2RestorePanel, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.__dialog = ModalDialog(self.__admin_console)

    @PageService()
    def in_place_restore(self):
        """
        submits inplace restore for DB2 database

        Returns:
             restore job id on succesful restore submission
        """
        self.access_tab('In place')
        self.submit()
        return self.__admin_console.get_jobid_from_popup()


class MySQLRestorePanel(ModalPanel):
    """Class to represent the restore panel for MySQL"""
    def __init__(self, admin_console):
        super().__init__(admin_console)
        self.__admin_console = admin_console
        self.__dialog = ModalDialog(self.__admin_console)

    @WebAction()
    def mark_email(self, select=False):
        """
        Marks notify by email option if enabled
        Args:
            select  (bool)  :   True if notify by email should be enabled
                default:    False
        """
        elem = self.__admin_console.driver.find_element_by_xpath(
            "*//span[contains(text(),'"
            +self.__admin_console.props["label.notifyUserOnJobCompletion"]+"')]")
        checkbox_already_selected = elem.is_selected()
        if select and not checkbox_already_selected:
            elem.click()
        if not select and checkbox_already_selected:
            elem.click()

    @PageService()
    def in_place_restore(self, data_restore=True, log_restore=True,
                         staging_location=None, notify_job_completion=None):
        """Submits Restore in place job

        Args:
            data_restore (Boolean):  Checks data restore option
                default: True
            log_restore (Boolean):  Checks log restore option
                default: True
            staging_location  (str): Location of data agent job details.
                                    Default value is already filled
                default: None
            notify_job_completion (Boolean): Notify on job completion by email
                default: None

        Returns:
            job id  (int): job id of restore job

        """
        self.access_tab("In place")
        if not data_restore:
            self.__admin_console.checkbox_deselect("data")
        else:
            self.__admin_console.checkbox_select("data")
        if not log_restore:
            self.__admin_console.checkbox_deselect("log")
        else:
            self.__admin_console.checkbox_select("log")
        if staging_location:
            self.__admin_console.fill_form_by_id("StagingLocation", staging_location)
        self.mark_email(select=notify_job_completion)
        self.submit()
        self.__dialog.click_submit()
        _jobid = self.__admin_console.get_jobid_from_popup()
        return _jobid


class RDSRestorePanel(ModalPanel):
    """Class to represent the restore panel for Amazon RDS database"""

    def __init__(self, admin_console):
        super(RDSRestorePanel, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.__dialog = ModalDialog(self.__admin_console)

    @PageService()
    def restore(self, database_identifier):
        """Submits basic restore of RDS without any advanced options
        Args:
            database_identifier (str)   -- The identifier to be set during restore

        """
        self.__admin_console.fill_form_by_id('dbIdentifier', database_identifier)
        self.submit()
        self.__dialog.click_submit()
        _jobid = self.__admin_console.get_jobid_from_popup()
        return _jobid


class SybaseRestorePanel(ModalPanel):
    """Class to represent the restore panel for Sybase"""

    def __init__(self, admin_console):
        super().__init__(admin_console)
        self.__admin_console = admin_console
        self.__dialog = ModalDialog(self.__admin_console)
        self._panel_dropdown = DropDown(self.__admin_console)
        self.redirect_options_dict = None

    @WebAction()
    def mark_email(self, select=False):
        """
        Marks notify by email option if enabled
        Args:
            select  (bool)  :   True if notify by email should be enabled
                default:    False
        """
        elem = self.__admin_console.driver.find_element_by_xpath(
            "*//span[contains(text(),'"
            + self.__admin_console.props["label.notifyUserOnJobCompletion"] + "')]")
        checkbox_already_selected = elem.is_selected()
        if select and not checkbox_already_selected:
            elem.click()
        if not select and checkbox_already_selected:
            elem.click()

    @WebAction()
    def mark_recover_databases(self, select=True):
        """
        Marks recover databases option if enabled
        Args:
            select  (bool)  :   True if recover databases should be enabled
                default:    True
        """
        if select:
            self._admin_console.checkbox_select("recoverDatabases")
        else:
            self.__admin_console.checkbox_deselect("recoverDatabases")

    @WebAction()
    def _click_device(self, database_name, device_name):
        """
        Clicks on device of a particular database in redirect restore
        Args:
            database_name   (str)   :   Name of the database of the device
            device_name     (str)   :   Name of the device to be selected
                    default: True
        """
        xpath = f"//b[contains(text(),'{database_name}')]/../following-sibling::" \
            f"ul/li/a[normalize-space()='{device_name}']"
        self.__admin_console.driver.find_element_by_xpath(xpath).click()

    def _construct_redirect_dict(self, database_names, prefix, path):
        """
        Generates redirect options dictionary
        Args:
            database_names  (list)  :   Names of the source databases
            prefix          (str)   :   Prefix for dict values
            path            (str)   :   Physical path to redirect to
        """
        self.redirect_options_dict = {}
        for database_name in database_names:
            self.redirect_options_dict[database_name] = {
                f"data_{database_name}": {
                    "target_db": f"{prefix}{database_name}",
                    "device_name": f"{prefix}data_{database_name}",
                    "physical_path": f"{path}{prefix}data_{database_name}.dat"
                },
                f"log_{database_name}": {
                    "target_db": f"{prefix}{database_name}",
                    "device_name": f"{prefix}log_{database_name}",
                    "physical_path": f"{path}{prefix}log_{database_name}.dat"
                }
            }

    @PageService()
    def in_place_restore(self, notify_job_completion=False, recover_databases=True,
                         redirect_options=None, database_names=None, path=None):
        """
        Submits Restore in place job for Sybase
            Args:
                notify_job_completion: True if user to be notified by mail on job completion
                    default: False
                recover_databases: True if recover databases is to be enabled
                    default: True
                redirect_options: Dictionary containing redirect options for each device
                                    in the format
                                    {database1:
                                        device1:{
                                            target_db:"target db",
                                            device_name:"device_name",
                                            physical_path:"physical/path/",
                                            device_size:123456
                                        }
                                    }
                database_names(list):  Names of the databases in case databases need to be
                                    redirected during restore for redirect_options dict
                                     construction in house
                path                :   Path to redirect database to including separator
            Returns:
                job id  (int): job id of restore job
        """
        self.access_tab("In place")
        self.mark_recover_databases(select=recover_databases)
        if redirect_options or database_names:
            self.__admin_console.expand_accordion(
                self.__admin_console.props['header.redirectOptions'])
            if database_names:
                self._construct_redirect_dict(database_names, "inplace_", path)
                redirect_options = self.redirect_options_dict
            else:
                self.redirect_options_dict = redirect_options
            for database, devices in redirect_options.items():
                for device, properties in devices.items():
                    self._click_device(database, device)
                    if properties.get("target_db"):
                        self.__admin_console.fill_form_by_name('databaseName',
                                                               properties['target_db'])
                    if properties.get("device_name"):
                        self.__admin_console.fill_form_by_name('logName',
                                                               properties['device_name'])
                    if properties.get('physical_path'):
                        self.__admin_console.fill_form_by_name('physicalPath',
                                                               properties['physical_path'])
                    if properties.get("device_size"):
                        self.__admin_console.fill_form_by_name('deviceSize',
                                                               properties['device_size'])
        self.mark_email(select=notify_job_completion)
        self.submit()
        _jobid = self.__admin_console.get_jobid_from_popup()
        return _jobid

    @PageService()
    def out_of_place_restore(self, destination_client, destination_instance,
                             notify_job_completion=False, recover_databases=True,
                             redirect_options=None, database_names=None, path=None):
        """
        Submits Restore in place job for Sybase
            Args:
                destination_client: Destination client to restore data to
                destination_instance: Destination instance to restore data to
                notify_job_completion: True if user to be notified by mail on job completion
                    default: False
                recover_databases: True if recover databases is to be enabled
                    default: True
                redirect_options: Dictionary containing redirect options for each device
                                    in the format
                                    {database1:
                                        device1:{
                                            target_db:"target db",
                                            device_name:"device_name",
                                            physical_path:"physical/path/",
                                            device_size:123456
                                        }
                                    }
                database_names(list):  Names of the databases in case databases need to be
                                    redirected during restore for redirect_options dict
                                     construction in house
                path                :   Path to redirect database to including separator
            Returns:
                job id  (int): job id of restore job
        """
        self.access_tab("Out of place")
        self._panel_dropdown.select_drop_down_values(values=[destination_client],
                                                     drop_down_id='destinationclient',
                                                     partial_selection=True)
        self._panel_dropdown.select_drop_down_values(values=[destination_instance],
                                                     drop_down_id='destinationDatabase')
        self.mark_recover_databases(select=recover_databases)
        if redirect_options or database_names:
            self.__admin_console.expand_accordion(
                self.__admin_console.props['header.redirectOptions'])
            if database_names:
                self._construct_redirect_dict(database_names, "outofplace_", path)
                redirect_options = self.redirect_options_dict
            else:
                self.redirect_options_dict = redirect_options
            for database, devices in redirect_options.items():
                for device, properties in devices.items():
                    self._click_device(database, device)
                    if properties.get("target_db"):
                        self.__admin_console.fill_form_by_name('databaseName',
                                                               properties['target_db'])
                    if properties.get("device_name"):
                        self.__admin_console.fill_form_by_name('logName',
                                                               properties['device_name'])
                    if properties.get('physical_path'):
                        self.__admin_console.fill_form_by_name('physicalPath',
                                                               properties['physical_path'])
                    if properties.get("device_size"):
                        self.__admin_console.fill_form_by_name('deviceSize',
                                                               properties['device_size'])
        self.mark_email(select=notify_job_completion)
        self.submit()
        _jobid = self.__admin_console.get_jobid_from_popup()
        return _jobid


class InformixRestorePanel(ModalPanel):
    """Class to represent the restore panel for Informix database"""

    def __init__(self, admin_console):
        super(InformixRestorePanel, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.props = self.__admin_console.props

    @WebAction()
    def __get_value(self, elem_id):
        """
        Get element value by id for any element from restore options
        Returns:
            Value of the element from command center restore options
        """
        return self.__admin_console.driver.find_element_by_id(elem_id).get_attribute("value")

    @PageService()
    def informix_restore(self, restore_type=None, destination_client=None,
                         destination_instance=None, physical=True, logical=True,
                         config_files=False, restore_time=None, tolog=0):
        """
        Submits inplace or out of place restore job for Informix
            Args:
                restore_type        (str) -- Restore mode with no space between words.Default = None
                    Accepted values       = 'Entireinstance' or 'Wholesystem'
                destination_client  (str) -- Name of destination client,
                                             required for out of place restore only
                destination_instance(str) -- Name of destination instance,
                                             required for out of place restore only
                physical            (bool)-- True to perform physical restore. Default = True
                logical             (bool)-- True to perform logical restore. Default = True
                config_files        (bool)-- True to perform config files restore. Default = False
                restore_time        (str) -- Actual to-time to perform point-in-time restore
                tolog               (int) -- Last log number to be restored for point-in-log restore
            Returns:
                job id  (int): job id of restore job
        """
        if destination_client and destination_instance is not None:
            self.access_tab("Out of place")
            self._dropdown.select_drop_down_values(
                values=[destination_client], drop_down_id='destinationServer')
            self._dropdown.select_drop_down_values(
                values=[destination_instance], drop_down_id='destinationDatabase')
        else:
            self.access_tab("In place")
        restoretypes = {
            'entireinstance': 'label.entireInstance',
            'wholesystem': 'label.wholeSystem'
        }
        if restore_type is not None:
            self.__admin_console.select_value_from_dropdown(
                select_id='restoreMode', value=self.props[restoretypes[restore_type.lower()]])
        if not physical:
            self.__admin_console.checkbox_deselect("physical")
            if not logical:
                if not config_files:
                    raise Exception("Select the restore content correctly")
        if not logical:
            self._admin_console.checkbox_deselect("logical")
        if config_files:
            self._admin_console.checkbox_select("restoreConfigFile")
        if tolog > 0:
            self.__admin_console.select_radio("toLogicalLog")
            self.__admin_console.fill_form_by_id("logicalLogValue", tolog)
        elif restore_time is not None:
            self.__admin_console.select_radio("restorePIT")
            pit_now = self.__get_value("dateTimeValue")
            pit_now = datetime.strptime(pit_now, '%m/%d/%Y %H:%M:%S').strftime("%m/%d/%Y %H:%M")
            if pit_now != restore_time:
                if int(pit_now[-2:])-int(restore_time[-2:])>1:
                    raise CVWebAutomationException("PIT %(p)s and browse-restore time %(r)s "
                                               "differ" %{'p': pit_now, 'r': restore_time})
        else:
            self.__admin_console.select_radio("mostRecent")
        self.submit()
        return self.__admin_console.get_jobid_from_popup()


class CosmosDBSQLRestorePanel(ModalPanel):
    """Class to represent restore panel for CosmosDB SQL API"""
    def __init__(self, admin_console):
        super(CosmosDBSQLRestorePanel, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.__dialog = ModalDialog(self.__admin_console)

    @PageService()
    def disable_write_throughput(self):
        """Disables the adjust write throughput toggle"""
        self.__admin_console.disable_toggle(1)

    @PageService()
    def select_overwrite(self):
        """Enables overwrites destination containers option"""
        self.__admin_console.enable_toggle(2)

    @PageService()
    def in_place_restore(self, adjust_throughput=0, streams=2, overwrite=True):
        """Submits restore to same cloud account and restores containers with same name
        Args:
            adjust_throughput  (int)  --  The value to be set as adjust throughput

            streams     (str)   --  The number of streams for restore

            overwrite   (bool)  --  True to enable overwrite containers
                                    False to not overwrite containers
        Returns:
            int     --  The jobid of the restore
        """
        if adjust_throughput != 0:
            self.__admin_console.fill_form_by_id('throughput', adjust_throughput)
        else:
            self.disable_write_throughput()
        if streams != 2:
            self.__admin_console.fill_form_by_id('noOfStreams', streams)
        if overwrite:
            self.select_overwrite()
        self.submit()
        self.__dialog.click_submit()
        _jobid = self.__admin_console.get_jobid_from_popup()
        return _jobid
