# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module implements the methods for creating a new subclient
or table group or database group or similar entities
Each database has a class that contains methods to add subclient for that database

AddCloudDBSubClient:
Functions:

select_cloud_regions()      :   Clicks on the region to add cloud content

AddDynamoDBSubClient:
Functions:

add_dynamodb_subclient()    :   Method to add subclient for dynamodb

AddRedshiftSubClient:
Functions:

add_redshift_subclient()    :   Method to add subclient for redshift

AddPostgreSQLSubClient:
Functions:

add_subclient()             :   method to add postgreSQL dumpbased subclient


AddMySQLSubClient:
Functions:

add_subclient()             :   method to add MySQL subclient

AddInformixSubClient:
Functions:

add_subclient()             :   Method to add Informix subclient

AddOracleSubClient:
Functions:

add_subclient()             :   Method to add Oracle subclient
"""

from Web.Common.page_object import (
    PageService, WebAction
)
from Web.AdminConsole.Components.panel import ModalPanel
from Web.AdminConsole.Components.panel import PanelInfo, DropDown
from Web.AdminConsole.Components.table import CVTable
from Web.AdminConsole.Components.browse import Browse


class AddCloudDBSubClient(ModalPanel):
    """Class to implement common methods for adding cloud database subclient"""
    def __init__(self, admin_console):
        super(AddCloudDBSubClient, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.__panel = PanelInfo(self.__admin_console)

    @WebAction()
    def __click_on_cloud_region(self, region_name):
        """Clicks on the region to add cloud content
        Args:
            region_name (str):  Name of the region to be selected
        """
        self._driver.find_element_by_xpath(
            f"//span[contains(text(),'{region_name}')]").click()

    @PageService()
    def select_cloud_regions(self, region_list):
        """Clicks on the region to add cloud content
        Args:
            region_list (str):  Name of the region to be selected
        """
        for region in region_list:
            self.__click_on_cloud_region(region)


class AddDynamoDBSubClient(AddCloudDBSubClient):
    """Class to represent the add subclient panel for DynamoDB"""
    def __init__(self, admin_console):
        super(AddDynamoDBSubClient, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.__panel = PanelInfo(self.__admin_console)

    def add_dynamodb_subclient(self, subclient_name, plan, content,
                               streams=2, adjust_read_capacity=0):
        """Method to add subclient for dynamodb

        Args:
        subclient_name  (str):  Name of subclient
        plan            (str):  Name of plan
        content (list): List of names of regions to be added as content
        streams (int):  Number of streams to be set
        adjust_read_capacity (int): value that needs to be set for
                                    adjust read capacity
        """

        self.__admin_console.fill_form_by_id('subclientName', subclient_name)
        self._dropdown.select_drop_down_values(
            values=[plan],
            drop_down_id='planSummaryDropdown')
        if streams != 2:
            self.__admin_console.fill_form_by_id('numberBackupStreams', streams)
        if adjust_read_capacity != 0:
            self.__panel.enable_toggle('Adjust read capacity')
            self.__admin_console.fill_form_by_id('capacity', adjust_read_capacity)
        self.select_cloud_regions(content)
        self.submit()
        self.__admin_console.check_error_message()

class AddRedshiftSubClient(AddCloudDBSubClient):
    """Class to represent the restore panel for Redshift"""
    def __init__(self, admin_console):
        super(AddRedshiftSubClient, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.__panel = PanelInfo(self.__admin_console)

    @PageService()
    def add_redshift_subclient(self, subclient_name, plan, content):
        """Method to add subclient for redshift

        Args:
        subclient_name  (str):  Name of subclient
        plan            (str):  Name of plan
        content         (list): List of names of regions to be added as content
        """

        self.__admin_console.fill_form_by_id('subclientName', subclient_name)
        self._dropdown.select_drop_down_values(
            values=[plan],
            drop_down_id='planSummaryDropdown')
        self.select_cloud_regions(content)
        self.submit()
        self.__admin_console.check_error_message()

class AddPostgreSQLSubClient(ModalPanel):
    """Class to represent the restore panel for PostgreSQL"""
    def __init__(self, admin_console):
        super(AddPostgreSQLSubClient, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.__panel_dropdown = DropDown(self.__admin_console)

    def add_subclient(self, subclient_name, number_backup_streams,
                      collect_object_list, plan, database_list):
        """
        method to add postgreSQL dumpbased subclient

        Args:
            subclient_name          (str):  Name of the subclient

            number_backup_streams   (int): number of streams used for backup

            collect_object_list     (bool): boolean value to specify if collect object
            list needs to be enabled

            plan                    (str):  plan name to be assigned to subclient

            database_list           (list): list of databases which needs to be part
            of subclient content

        """
        self.__admin_console.fill_form_by_name('subclientName', subclient_name)
        self.__admin_console.fill_form_by_name('numberBackupStreams', number_backup_streams)
        if collect_object_list:
            self.__admin_console.checkbox_select('collectObjectList')
        self.__panel_dropdown.select_drop_down_values(
            values=[plan], drop_down_id='planSummaryDropdown')
        _cvtable = CVTable(self.__admin_console)
        _cvtable.select_values_from_table(database_list)
        self.__admin_console.submit_form()


class AddMySQLSubClient(ModalPanel):
    """Class to represent the add subclient panel of MySQL"""

    def __init__(self, admin_console):
        super(AddMySQLSubClient, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.__panel_dropdown = DropDown(self.__admin_console)
        self.__browse = Browse(self.__admin_console)

    @PageService()
    def add_subclient(self, subclient_name, number_backup_streams, database_list, plan):
        """
        method to add MySQL subclient

        Args:
            subclient_name          (str):  Name of the subclient

            number_backup_streams   (int): number of streams used for backup

            database_list           (list): list of databases which needs to be part
                                            of subclient content

            plan                    (str):  plan name to be assigned to subclient

        """
        self.__admin_console.fill_form_by_name('subclientName', subclient_name)
        self.__admin_console.fill_form_by_name('numberBackupStreams', number_backup_streams)
        self.__panel_dropdown.select_drop_down_values(values=[plan],
                                                      drop_down_id='planSummaryDropdown')
        self.__browse.select_for_restore(database_list)
        self.__admin_console.submit_form()

class AddInformixSubClient(ModalPanel):
    """Class to represent the add subclient panel of Informix"""
    def __init__(self, admin_console):
        super(AddInformixSubClient, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.props = self.__admin_console.props
        self.__panel_dropdown = DropDown(self.__admin_console)

    @WebAction()
    def __select_dbspace(self):
        """
        Method to select automation created dbspace cvauto1 as content for selective subclient
        """
        self._driver.find_element_by_xpath("//span[contains(text(),'cvauto1')]").click()

    @PageService()
    def add_subclient(self, subclient_name, plan, bkp_mode, incr_level=1):
        """
        Method to add Informix subclient

        Args:
            subclient_name  (str):  Name of the subclient
            plan            (str):  Plan to be assigned to subclient
            bkp_mode        (str):  Backup mode as in command center with no space between words
                Accepted Values  :  'Entireinstance', 'Wholesystem', 'Selective',
                                    'Fulllogicallogs' and 'Fullandcurrentlogicallogs'
            incr_level      (int):  backup level for incremental backups
                Accepted Values  :  1 or 2

        """
        backuptypes = {
            'entireinstance': 'label.entireInstance',
            'wholesystem': 'label.wholeSystem',
            'selective': 'label.selective',
            'fulllogicallogs': 'label.fullLogicalLogs',
            'fullandcurrentlogicallogs': 'label.fullCurrentLogicalLogs'
        }
        self.__admin_console.fill_form_by_name('subclientName', subclient_name)
        self.__panel_dropdown.select_drop_down_values(
            values=[plan], drop_down_id='planSummaryDropdown')
        self.__admin_console.select_value_from_dropdown(
            select_id='restoreMode', value=self.props[backuptypes[bkp_mode.lower()]])
        if bkp_mode.lower() == "selective":
            self.__select_dbspace()
        self.__admin_console.fill_form_by_id('backupLevel', incr_level)
        self.__admin_console.submit_form()


class AddOracleSubClient(ModalPanel):
    """Class to represent the add subclient panel of Oracle"""

    def __init__(self, admin_console):
        super(AddOracleSubClient, self).__init__(admin_console)
        self.__admin_console = admin_console
        self.props = self.__admin_console.props
        self.__panel_dropdown = DropDown(self.__admin_console)

    @PageService()
    def add_subclient(self, subclient_name, plan, number_backup_streams=2,
                      selective_online_full=True, data_backup="Online",
                      archive_log_backup=True, delete_archive_logs=True):
        """
        method to add Oracle subclient

        Args:
            subclient_name          (str):  Name of the subclient

            number_backup_streams   (int): number of streams used for backup
                default: 2

            plan                    (str):  plan name to be assigned to subclient

            selective_online_full   (bool): True if selective online full is to be enabled
                default: True

            data_backup             (str):  "Online"/"Online subset"/"Offline"
                default: Online

            archive_log_backup      (bool): True if Archive log backups
                default: True

            delete_archive_logs     (bool): True if delete archive logs is to be enabled
                default: True

        """
        data_backup_map = {"online": "onlineData", "online subset": "onlineSubset",
                           "offline": "offlineData"}
        self.__admin_console.fill_form_by_name('subclientName', subclient_name)
        self.__admin_console.fill_form_by_name('numberBackupStreams', number_backup_streams)
        self.__panel_dropdown.select_drop_down_values(values=[plan],
                                                      drop_down_id='planSummaryDropdown')
        if selective_online_full:
            self.__admin_console.checkbox_select('selectiveOnlineFull')
            if delete_archive_logs:
                self.__admin_console.checkbox_select('deleteArchiveLogs')
        else:
            self.__admin_console.checkbox_deselect('selectiveOnlineFull')
            if data_backup:
                self.__admin_console.checkbox_select('dataBackup')
                self.__admin_console.select_radio(data_backup_map[data_backup.lower()])
            else:
                self.__admin_console.checkbox_deselect('dataBackup')
            if data_backup and data_backup.lower() == "Online":
                if archive_log_backup:
                    self.__admin_console.checkbox_select('logBackup')
                    if delete_archive_logs:
                        self.__admin_console.checkbox_select('deleteArchiveLogs')
                    else:
                        self.__admin_console.checkbox_deselect('deleteArchiveLogs')
                else:
                    self.__admin_console.checkbox_deselect('logBackup')
        self.__admin_console.submit_form()
