# -*- coding: utf-8 -*-s

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the functions or operations to create non cloud database instances

AddInstance:

    add_postgresql_instance()   --  Adds new postgreSQL database instance
    add_mysql_instance()        --  Adds new MySQL database instance
    add_db2_instance()          --  Adds new db2 instance
	add_sybase_instance()		--	Adds new Sybase instance
	add_informix_instance()     --  Adds new Informix instance
	add_oracle_instance()       --  Adds new Oracle instance

"""
from Web.Common.page_object import PageService
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.Components.panel import DropDown


class AddDBInstance:
    """This class provides the function or operations to create non cloud database instances
    """

    def __init__(self, admin_console: AdminConsole):
        """Class constructor

            Args:
                admin_console   (obj)   --  The admin console class object

        """
        self.__admin_console = admin_console
        self._panel_dropdown = DropDown(self.__admin_console)
        self.__admin_console._load_properties(self)
        self.props = self.__admin_console.props

    @PageService(hide_args=True)
    def add_postgresql_instance(self, server_name, instance_name, plan,
                                database_user, password, port, binary_directory,
                                lib_directory, archive_log_directory, maintenance_db="postgres"):
        """ Adds new postgreSQL database instance

            Args:
                server_name             (str)   --  Server name

                instance_name           (str)   --  postgresql instance name

                plan                    (str)   --  Plan to be associated with the instance

                database_user           (str)   --  PostgreSQL user

                password                (str)   --  PostgreSQL user password

                port                    (str)   --  PostgreSQL port

                binary_directory        (str)   --  Binary directory path

                lib_directory           (str)   --  Library directory path

                archive_log_directory   (str)   --  archive log directory path

                maintenance_db          (str)   --  postgreSQL maintenance database name

                    default: Postgres

        """
        self.__admin_console.access_menu(self.props['pageHeader.addInstance'])
        self.__admin_console.access_sub_menu(self.props['agentType.postgreSQL'])
        self._panel_dropdown.select_drop_down_values(values=[server_name], drop_down_id='cvClients')
        self.__admin_console.fill_form_by_name('databaseName', instance_name)
        self._panel_dropdown.select_drop_down_values(
            values=[plan],
            drop_down_id='planSummaryDropdown')
        self.__admin_console.expand_accordion(self.props['label.assets.database.connectionDetails'])
        self.__admin_console.fill_form_by_name('dbUsername', database_user)
        self.__admin_console.fill_form_by_name('dbPassword', password)
        self.__admin_console.fill_form_by_name('maintenanceDB', maintenance_db)
        self.__admin_console.fill_form_by_name('port', port)
        self.__admin_console.expand_accordion(self.props['label.paths'])
        self.__admin_console.fill_form_by_name('BinaryDirectory', binary_directory)
        self.__admin_console.fill_form_by_name('LibDirectory', lib_directory)
        self.__admin_console.fill_form_by_name('ArchiveLogDirectory', archive_log_directory)
        self.__admin_console.submit_form()

    @PageService(hide_args=True)
    def add_db2_instance(self, server_name, instance_name, plan, db2_home, db2_username, db2_user_password):
        """ Adds new db2 database instance

                    Args:
                        server_name             (str)   --  Server name

                        instance_name           (str)   --  db2 instance name

                        plan                    (str)   --  Plan to be associated with the instance

                        db2_home               (str)   --  db2 home path

                        db2_username           (str)   --  db2 user name

                        db2_user_password      (str)   --  db2 user password

                """
        self.__admin_console.access_menu(self.props['pageHeader.addInstance'])
        self.__admin_console.access_sub_menu(self.props['agentType.DB2'])
        self._panel_dropdown.select_drop_down_values(values=[server_name], drop_down_id='cvClients')
        self.__admin_console.fill_form_by_name('instanceName', instance_name)
        self._panel_dropdown.select_drop_down_values(
            values=[plan],
            drop_down_id='planSummaryDropdown')
        self.__admin_console.fill_form_by_name('dbHome', db2_home)
        self.__admin_console.fill_form_by_name('dbUsername', db2_username)
        self.__admin_console.fill_form_by_name('dbUserPassword', db2_user_password)
        self.__admin_console.submit_form()

    @PageService(hide_args=True)
    def add_mysql_instance(self, server_name, instance_name, plan, database_user, password,
                           binary_directory, log_directory, config_directory, unix,
                           socketfile_directory=None, unix_username=None, nt_username=None,
                           nt_password=None, port=None, ssl_ca_file=None,
                           xtra_backup_bin_path=None):
        """ Adds new MySQL database instance

            Args:
                server_name             (str)   --  Server name

                instance_name           (str)   --  MySQL instance name

                plan                    (str)   --  Plan to be associated with the instance

                database_user           (str)   --  MySQL user

                password                (str)   --  MySQL user password

                socketfile_directory    (str)   --  Socket file directory psth

                binary_directory        (str)   --  Binary directory path

                log_directory           (str)   --  Log directory path

                config_directory        (str)   --  configuration file directory path

                unix                    (bool)  --  True if server os is UNIX. Else false

                unix_username           (str)   --  UNIX user name  (unix server specific)

                    default: None

                nt_username             (str)   --  NT username  (windows server specific)

                    default: None

                nt_password             (str)   --  NT password  (windows server specific)

                    default: None

                port                    (int)   --  Port  (windows server specific)

                    default: None

                ssl_ca_file             (str)   --  SSL CA file directory path

                    default: None

                xtra_backup_bin_path    (str)   --  XtraBackup bin path. If None, XtraBackup
                                                    for hot backup will not be enabled.

                    default: None

        """
        self.__admin_console.access_menu(self.props['pageHeader.addInstance'])
        self.__admin_console.access_sub_menu(self.props['agentType.MySQL'])
        self._panel_dropdown.select_drop_down_values(values=[server_name], drop_down_id='cvClients')
        self.__admin_console.fill_form_by_name('databaseName', instance_name)
        self._panel_dropdown.select_drop_down_values(values=[plan],
                                                     drop_down_id='planSummaryDropdown')
        self.__admin_console.expand_accordion(self.props['label.assets.database.connectionDetails'])
        self.__admin_console.fill_form_by_name('dbUsername', database_user)
        self.__admin_console.fill_form_by_name('dbPassword', password)
        if unix:
            if unix_username:
                self.__admin_console.fill_form_by_name('UnixUsername', unix_username)
            if socketfile_directory:
                self.__admin_console.fill_form_by_name('SocketFile', socketfile_directory)
        else:
            if nt_username and nt_password:
                self.__admin_console.fill_form_by_name('NTUsername', nt_username)
                self.__admin_console.fill_form_by_name('NTPassword', nt_password)
            if port:
                self.__admin_console.fill_form_by_name('port', port)
        self.__admin_console.expand_accordion(self.props['label.configuration'])
        self.__admin_console.fill_form_by_name('BinaryDirectory', binary_directory)
        self.__admin_console.fill_form_by_name('LibDirectory', log_directory)
        self.__admin_console.fill_form_by_name('ConfigFile', config_directory)
        self.__admin_console.expand_accordion(self.props['label.assets.database.advancedOptions'])
        if ssl_ca_file:
            self.__admin_console.fill_form_by_name('SSLCAFile', ssl_ca_file)
        if xtra_backup_bin_path:
            self.__admin_console.enable_toggle(index=0)
            self.__admin_console.fill_form_by_name('XtraBackupPath', xtra_backup_bin_path)
        self.__admin_console.submit_form()

    @PageService(hide_args=True)
    def add_sybase_instance(self, server_name, instance_name, plan, sa_user_name,
                            password, unix, os_user_name=None, os_password=None):
        """ Adds new Sybase instance
        Args:
            server_name     (str):  Server name
            instance_name:  (str):  Name of instance to be created
            plan:           (str):  Plan name
            sa_user_name:   (str):  SA user name
            password:       (str):  Password
            unix            (bool): True if server os is UNIX. Else false
            os_user_name    (str):  OS username
            os_password     (str):  OS password
        """
        self.__admin_console.access_menu(self.props['pageHeader.addInstance'])
        self.__admin_console.access_sub_menu(self.props['agentType.Sybase'])
        self._panel_dropdown.select_drop_down_values(values=[server_name],
                                                     drop_down_id='cvClients')
        self.__admin_console.fill_form_by_name('databaseName', instance_name)
        self._panel_dropdown.select_drop_down_values(values=[plan],
                                                     drop_down_id='planSummaryDropdown')
        self.__admin_console.fill_form_by_name('saUsername', sa_user_name)
        self.__admin_console.fill_form_by_name('saPassword', password)
        if not unix:
            self.__admin_console.fill_form_by_name('username', os_user_name)
            self.__admin_console.fill_form_by_name('password', os_password)
        self.__admin_console.submit_form()

    @PageService(hide_args=True)
    def add_informix_instance(self, server_name, instance_name, plan,
                              informix_username, informix_home, onconfig,
                              sqlhosts, is_windows_os, informix_password=None):
        """ Adds new informix instance

            Args:
                server_name             (str)   --  Server name
                instance_name           (str)   --  informix instance name, INFORMIXSERVER
                plan                    (str)   --  Plan to be associated with the instance
                informix_username       (str)   --  informix user name
                informix_home           (str)   --  informix home directory, INFORMIXDIR
                onconfig                (str)   --  onconfig filename, ONCONFIG
                sqlhosts                (str)   --  sqlhosts file path, INFORMIXSQLHOSTS
                is_windows_os           (bool)  --  True if server OS is windows
                informix_password       (str)   --  informix user password

        """

        self.__admin_console.access_menu(self.props['pageHeader.addInstance'])
        self.__admin_console.access_sub_menu('Informix')
        self._panel_dropdown.select_drop_down_values(values=[server_name], drop_down_id='cvClients')
        self.__admin_console.fill_form_by_name('instanceName', instance_name)
        self._panel_dropdown.select_drop_down_values(values=[plan],
                                                     drop_down_id='planSummaryDropdown')
        self.__admin_console.fill_form_by_name('dbUsername', informix_username)
        if is_windows_os and informix_password:
            if "\\" in informix_username:
                self.__admin_console.fill_form_by_name('dbUserPassword', informix_password)
            else:
                raise Exception("Domain name is missing in user name or password is not provided")
        self.__admin_console.fill_form_by_name('dbHome', informix_home)
        self.__admin_console.fill_form_by_name('onconfigFile', onconfig)
        self.__admin_console.fill_form_by_name('sqlHostsFile', sqlhosts)
        self.__admin_console.submit_form()

    @PageService(hide_args=True)
    def add_oracle_instance(self, server_name, oracle_sid, plan, oracle_home,
                            username, password, service_name, use_catalog_connect,
                            catalog_username, catalog_password, catalog_service_name):
        """Adds new oracle instance

            Args:
                server_name             (str)   --  Server name
                oracle_sid              (str)   --  Oracle server SID
                plan                    (str)   --  Plan to be associated with the instance
                oracle_home             (str)   --  oracle home directory
                username                (str)   --  Oracle server username
                password                (str)   --  Oracle server password
                service_name            (str)   --  Oracle service name
                use_catalog_connect     (bool)  --  True if catalog connect is to be enabled
                catalog_username        (str)   --  Catalog username
                catalog_password        (str)   --  Catalog password
                catalog_service_name    (str)   --  Connect service name
        """

        self.__admin_console.access_menu(self.props['pageHeader.addInstance'])
        self.__admin_console.access_sub_menu('Oracle')
        self._panel_dropdown.select_drop_down_values(values=[server_name],
                                                     drop_down_id='cvClients')
        self.__admin_console.fill_form_by_name('instanceName', oracle_sid)
        self._panel_dropdown.select_drop_down_values(
            values=[plan], drop_down_id='oracleCreateInstance_isteven-multi-select_#9446')
        self.__admin_console.fill_form_by_name('oracleHome', oracle_home)
        self.__admin_console.fill_form_by_name('dbUserName', username)
        self.__admin_console.fill_form_by_name('dbPassword', password)
        self.__admin_console.fill_form_by_name('dbInstanceName', service_name)
        if use_catalog_connect:
            self.__admin_console.enable_toggle(index=0)
            self.__admin_console.fill_form_by_name('catalogUserName', catalog_username)
            self.__admin_console.fill_form_by_name('catalogPassword', catalog_password)
            self.__admin_console.fill_form_by_name('catalogInstanceName', catalog_service_name)
        self.__admin_console.submit_form()
