# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This file contains _NewCloudDBInstance class which inherits from Modal Panel.
This is specific to cloud DBs and implements private methods to set the common inputs like
vendor name, DB engine, cloud account, backup plan when creating a cloud database instance
for DynamoDB, Redshift, RDS, DocumentDB.
This file will also contain classes specific to each cloud DB type which
implements methods to add cloud DB instances.

Perform the following steps when trying to add support for a new cloud database:
    1. Create a class for the new agent and inherit from _NewCloudDBInstance
    2. Implement the vendor_name() and cloud_db_engine() in the class
        to return the values specific to the agent
    3. Define a method in the class to create instance for the specific agent

---------------------------------------------------------------------------

_NewCloudDBInstance:
-------------------

    vendor_name()       --  Abstract method to set the cloud vendor name

    cloud_db_engine()   --  Abstract method to set the cloud database engine

    select_cloud_regions()  --  Selects cloud regions from cloud browse

    select_items_under_regions()  --   Selects one or more items under cloud regions


DynamoDBInstance:
----------------
    vendor_name()       -- Returns vendor for database engine

    cloud_db_engine()   --  Returns the database engine name

    create_instance()   --  Creates Amazon DynamoDB Engine

AmazonRDSInstance:
-----------------
    vendor_name()       -- Returns vendor for database engine

    cloud_db_engine()   --  Returns the database engine name

    create_instance()   --  Creates Amazon RDS Engine

RedshiftInstance:
----------------
    vendor_name()       --  Returns vendor for database engine

    cloud_db_engine()   --  Returns the database engine name

    create_instance()   --  Creates Amazon Redshift Engine

CosmosDBSQLInstance:
-------------------
    vendor_name()       --  Returns vendor for database engine

    cloud_db_engine()   --  Returns the database engine name

    create_instance()   --  Creates new Azure CosmosDB SQL API Instance

"""

from abc import abstractmethod
from Web.Common.page_object import (
    PageService,
    WebAction
)
from Web.AdminConsole.Components.panel import ModalPanel


class _NewCloudDBInstance(ModalPanel):
    """Class to set common options while creating cloud DB instances"""
    @property
    @abstractmethod
    def vendor_name(self):
        """Override this method and implement it as a variable
        whose value needs to be set for vendor name"""
        raise NotImplementedError

    @WebAction()
    def _select_vendor(self):
        """Selects vendor"""
        self._dropdown.select_drop_down_values(
            values=[self.vendor_name],
            drop_down_id='addServerContent_isteven-multi-select_#8657',
            partial_selection=True)

    @property
    @abstractmethod
    def cloud_db_engine(self):
        """Override this method and implement it as a variable
                whose value needs to be set for Cloud DB engine"""
        raise NotImplementedError

    @WebAction()
    def _set_database_engine(self):
        """Selects the cloud database engine"""
        self._dropdown.select_drop_down_values(
            values=[self.cloud_db_engine],
            drop_down_id='databaseServiceSelection')

    @WebAction()
    def _set_cloud_account(self, cloud_account):
        """
        Sets the cloud account
        Args:
            cloud_account   (str) : The cloud account that needs to be used for
                                    configuring instance
        """
        self._dropdown.select_drop_down_values(
            values=[cloud_account],
            drop_down_id='addCloudDatabase_isteven-multi-select_#5611')

    @WebAction()
    def _set_plan(self, name):
        """
        Sets the plan
        Args:
            name    (str)  :    The name of the plan
        """

        self._dropdown.select_drop_down_values(
            values=[name],
            drop_down_id='planSummaryDropdown')

    @WebAction()
    def _click_edit(self):
        """click edit"""
        self._driver.find_element_by_xpath(
            "//a[@data-ng-click='addCloudDB.editContent = true']"
        ).click()

    @WebAction()
    def _click_clear_all(self):
        """clear all"""
        self._driver.find_element_by_xpath(
            "//a[@ng-click='cappsBC.clearAll()']").click()

    @WebAction()
    def _click_select_all(self):
        """select all"""
        self._driver.find_element_by_xpath(
            "//a[@ng-click='cappsBC.selectAll()']").click()

    @WebAction()
    def _click_on_cloud_region(self, region):
        """Clicks on the region to add cloud content
        Args:
            region (str):  Name of the region to be selected

                           Example: Asia Pacific (Mumbai) (ap-south-1)
        """
        self._driver.find_element_by_xpath(
            f"//span[text()='{region}']").click()

    @WebAction()
    def _expand_cloud_region(self, region):
        """Expands the cloud region by clicking on the arrow near region
        Args:
            region  (str):  Full nmae of the cloud region

                            Example: Asia Pacific (Mumbai) (ap-south-1)
        """
        self._driver.find_element_by_xpath(
            f"//span[@title='{region}']//parent::div/button").click()
        self._admin_console.wait_for_completion()

    @WebAction()
    def _click_on_items_inside_region(self, region, items_list):
        """Clicks on the items inside the cloud regions
        Args:
            region  (str):  Full nmae of the cloud region

                            Example: Asia Pacific (Mumbai) (ap-south-1)

            items_list  (list)  : List of items to be selected under region
        """
        for each_item in items_list:
            self._driver.find_element_by_xpath(
                f"//span[@title='{region}']//parent::div//following-sibling::div"
                f"//span[text()='{each_item}']").click()

    @PageService()
    def select_cloud_regions(self, region_list):
        """Clicks on the region to add cloud content
        Args:
            region_list (list):  list of names of the regions to be selected

                                   Example: ['Asia Pacific (Mumbai) (ap-south-1)',
                                   'Asia Pacific (Singapore) (ap-southeast-1)']
        """
        for region in region_list:
            self._click_on_cloud_region(region)

    @PageService()
    def select_items_under_regions(self, mapping_dict):
        """Selects one or more items (like tables, clusters) under cloud regions
        Args:
            mapping_dict (dict) : The dictionary containing the full region names as keys
                                and LIST of items to be selected under them as value
                                Example --
                                mapping_dict={
                                'full region-1 name':['table1','table2','table3']
                                'full region-2 name':['item1','item2','item3']
                                }
        """
        for key, value in mapping_dict.items():
            self._expand_cloud_region(key)
            self._click_on_items_inside_region(key, value)


class DynamoDBInstance(_NewCloudDBInstance):
    """Class for creating DynamoDB instance"""
    @property
    def vendor_name(self):
        """returns the vendor"""
        return self._admin_console.props['label.vendor.amazon']

    @property
    def cloud_db_engine(self):
        """returns the database engine type"""
        return self._admin_console.props['label.dynamodb']

    @PageService()
    def create_instance(self, cloud_account, plan, adjust_read_capacity, content='default'):
        """
        Creates DynamoDB instance
        Args:
            cloud_account   (str) : The cloud account that needs to be used for
                                    configuring instance

            plan            (str):  The name of the plan

            adjust_read_capacity(int)   :   The value that needs to be set for
                                        adjust read capacity parameter

            content         (dict or list):  The content to be selected

                            1. To set complete regions as content:
                            Provide list of strings of region names
                            Example: ['Asia Pacific (Mumbai) (ap-south-1)',
                                   'Asia Pacific (Singapore) (ap-southeast-1)']


                            2. To set one more tables under regions as content:
                            Provide a dictionary containing the full region names as keys
                            and LIST of strings with items to be selected under them as value
                                Example:
                                {
                                'US East (Ohio) (us-east-2)':['table1','table2','table3']
                                'US East (Virginia) (us-east-1)':['tableA','tableB'']
                                }

                            Default value is 'default', default content set in UI will be used
        """
        self._select_vendor()
        self._set_database_engine()
        self._set_cloud_account(cloud_account)
        self._set_plan(plan)
        if adjust_read_capacity != 0:
            self._admin_console.enable_toggle(index=0)
            self._admin_console.fill_form_by_id('capacity', adjust_read_capacity)
        if content != 'default':
            self._click_edit()
            self._admin_console.wait_for_completion()
            self._click_clear_all()
            if isinstance(content, dict):
                self.select_items_under_regions(content)
            elif isinstance(content, list):
                self.select_cloud_regions(content)
            else:
                raise Exception("Did not find the content in expected format, "
                                "Expected dict or list")
        self.submit()


class AmazonRDSInstance(_NewCloudDBInstance):
    """Class for creating Amazon RDS instance"""
    @property
    def vendor_name(self):
        """returns the vendor"""
        return self._admin_console.props['label.vendor.amazon']

    @property
    def cloud_db_engine(self):
        """returns the database engine type"""
        return self._admin_console.props['label.rds']

    @PageService()
    def create_rds_instance(self, cloud_account, plan, content='default'):
        """
        Creates Amazon RDS instance
        Args:
            cloud_account   (str) : The cloud account that needs to be used for
                                        configuring instance

            plan            (str):  The name of the plan

            content         (dict or list):  The content to be selected

                            1. To set complete regions as content:
                            Provide list of strings of region names
                            Example: ['Asia Pacific (Mumbai) (ap-south-1)',
                                    'Asia Pacific (Singapore) (ap-southeast-1)']


                            2. To set one more tables under regions as content:
                            provide a dictionary containing the full region names as keys
                            and LIST of strings with items to be selected under them as value
                            Example:
                            {
                            'US East (Ohio) (us-east-2)':['table1','table2','table3']
                            'US East (Virginia) (us-east-1)':['tableA','tableB'']
                            }

                            Default value is 'default', default content set in UI will be used
                """
        self._select_vendor()
        self._set_database_engine()
        self._set_cloud_account(cloud_account)
        self._set_plan(plan)
        if content != 'default':
            self._click_edit()
            self._admin_console.wait_for_completion()
            self._click_clear_all()
            if isinstance(content, dict):
                self.select_items_under_regions(content)
            elif isinstance(content, list):
                self.select_cloud_regions(content)
            else:
                raise Exception("Did not find the content in expected format, "
                                "Expected dict or list")
        self.submit()

class RedshiftInstance(_NewCloudDBInstance):
    """Class for creating RedShift instance"""
    @property
    def vendor_name(self):
        """returns the vendor"""
        return self._admin_console.props['label.vendor.amazon']

    @property
    def cloud_db_engine(self):
        """returns the database engine type"""
        return self._admin_console.props['viewname.dbInstancesTable.redshift']

    @PageService()
    def create_instance(self, cloud_account, plan, content='default'):
        """
        Creates Redshift instance
        Args:
            cloud_account   (str) : The cloud account that needs to be used for
                                    configuring instance

            plan            (str):  The name of the plan

            content         (dict or list):  The content to be selected

                            1. To set complete regions as content:
                            Provide list of strings of region names
                            Example: ['Asia Pacific (Mumbai) (ap-south-1)',
                                   'Asia Pacific (Singapore) (ap-southeast-1)']


                            2. To set one more tables under regions as content:
                            Provide a dictionary containing the full region names as keys
                            and LIST of strings with items to be selected under them as value
                                Example:
                                {
                                'US East (Ohio) (us-east-2)':['table1','table2','table3']
                                'US East (Virginia) (us-east-1)':['tableA','tableB'']
                                }

                            Default value is 'default', default content set in UI will be used
        """
        self._select_vendor()
        self._set_database_engine()
        self._set_cloud_account(cloud_account)
        self._set_plan(plan)
        if content != 'default':
            self._click_edit()
            self._admin_console.wait_for_completion()
            self._click_clear_all()
            if isinstance(content, dict):
                self.select_items_under_regions(content)
            elif isinstance(content, list):
                self.select_cloud_regions(content)
            else:
                raise Exception("Did not find the content in expected format, "
                                "Expected dict or list")
        self.submit()


class CosmosDBSQLInstance(_NewCloudDBInstance):
    """class for creating azure CosmosDB SQL API instance"""
    @property
    def vendor_name(self):
        """returns the vendor"""
        return self._admin_console.props['label.vendor.azure_v2']

    @property
    def cloud_db_engine(self):
        """returns the database engine or service type"""
        return self._admin_console.props['label.cosmosDB']

    @WebAction()
    def _expand_cosmosdb_database(self, account_name, database_name):
        """Expands the given database under the cosmosdb account
        Args:
            account_name    (str)   --   CosmosDB account name

            database_name   (str)   --  CosmosDB database name
        """
        self._driver.find_element_by_xpath(
            f"//span[text()='{account_name}']/following::div/div"
            f"/span[@title='{database_name}']//parent::div/button").click()
        self._admin_console.wait_for_completion()

    @WebAction()
    def _select_containers(self, account_name, database_name, container_list):
        """Selects on one or more containers under given account and database
        Args:
            account_name    (str)   --   CosmosDB account name

            database_name   (str)   --  CosmosDB database name

            container_list  (list)  --  List of containers to be selected
        """
        for container_name in container_list:
            self._driver.find_element_by_xpath(
                f"//span[text()='{account_name}']/parent::div//following-sibling::div"
                f"//span[text()='{database_name}']/parent::div//following-sibling::div"
                f"//span[text()='{container_name}']"
            ).click()

    @PageService()
    def create_instance(self, cloud_account, plan, content='default'):
        """Creates new Azure CosmosDB SQL API instance
        Args:
            cloud_account   (str) : The cloud account that needs to be used for
                                    configuring instance

            plan            (str):  The name of the plan

            content         (List or nested dict):  The content to be selected
                Default value is 'default', default content set in UI will be used

                            1. To set complete CosmosDB account as content:

                            Provide a list of strings of account names
                                Example: ['cosmos-account-1', cosmos-account-2]

                            2. To set one more databases and containers under database as content:

                            Provide a nested dictionary containing the account names as keys
                            and value as another dictionary whose keys are the database names
                            and values is a LIST of containers under the database
                                Example:
                                {
                                'cosmos-account-1': {
                                'database1':['container1', 'container2'],
                                'database2':['container3', 'container4']
                                        }
                                'cosmos-account-2':{
                                'database3': ['container5', 'container6'],
                                'database4': ['container7', 'container8']
                                        }
                                }

                            3. To set complete database as content:

                            Provide the nested dictionary same as above #2 but
                            in the nested dictionary, provide an empty LIST as value
                            instead of list of containers
                                Example:
                                    {
                                'cosmos-account-1': {
                                'database1':[],
                                        }
                                'cosmos-account-2':{
                                'database2': [],
                                'database3': []
                                        }
                                }
        """
        self._select_vendor()
        self._set_database_engine()
        self._set_cloud_account(cloud_account)
        self._set_plan(plan)
        if content != 'default':
            self._click_edit()
            self._admin_console.wait_for_completion()
            self._click_clear_all()
            if isinstance(content, list):
                self.select_cloud_regions(content)
            elif isinstance(content, dict):
                for account, details in content.items():
                    self._expand_cloud_region(account)
                    for database, container_list in details.items():
                        if not container_list:
                            self._click_on_items_inside_region(account, [database])
                        else:
                            self._expand_cosmosdb_database(account, database)
                            self._select_containers(account, database, container_list)
        self.submit()
