# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
Use this module to configure vitualization replication group.
"""

from abc import abstractmethod

from time import sleep

from Web.AdminConsole.Components.panel import DropDown, ModalPanel
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.VSAPages.hypervisor_details import HypervisorDetails
from Web.Common.page_object import (WebAction, PageService)
from Web.Common.exceptions import CVWebAutomationException

SOURCE_HYPERVISOR_AZURE = 'Azure'
SOURCE_HYPERVISOR_VMWARE = 'VMware'
SOURCE_HYPERVISOR_HYPERV = 'Hyper-V'

class _Content:
    """use this module for configure replication group content page operations"""

    def __init__(self, admin_console):
        """
        Args:
            admin_console: adminconsole base object
        """
        self.__admin_console = admin_console
        self.__driver = admin_console.driver
        self.__drop_down = DropDown(admin_console)
        self.__table = Table(admin_console)
        self.__modal_panel = ModalPanel(admin_console)
        self._hypervisor_details = HypervisorDetails(self.__admin_console)

    @PageService()
    def set_name(self, name):
        """
        Set name in content page
        Args:
            name      (str):   Name of the replication group

        """
        self.__admin_console.fill_form_by_id('name', name)

    @PageService()
    def set_vm_search_string(self, vm_name):
        """
        set name to search for vm
        Args:
            vm_name      (String)   : specify the name of the vm to search

        """
        self.__admin_console.search_vm(vm_name)

    @PageService()
    def select_vm_from_browse_tree(self, hypervisor, vm_content):
        """
        Select content for subclient from the browse tree

        Args:
            hypervisor   (str): the name of the hypervisor to select, if none, follows SP21 guidelines
            vm_content  (dict):     the content to be selected
                Sample value:   {'Datastores and datastore clusters':[ds1,ds2],
                                 'Hosts and clusters':[h1,h2],
                                 'Tags and categories':[tag1,category1],
                                 'VMs and templates':[vm1,vm2],
                                 'Storage':[strg1, strg2],
                                 'By region': [region1, region2],
                                 'By zone': [zone1, zone2]
                                }

        Raises:
            Exception:
                if the content could not be selected properly

        """
        self.__drop_down.select_drop_down_values(0, [hypervisor])
        self.__table.access_toolbar_menu('ADD_VIRTUALMACHINES')
        self._hypervisor_details.select_vm_from_browse_tree(vm_content)
        self.__admin_console.click_button("OK")


class _Target:
    """use this module for configure replication group target page operations"""
    FREQUENCY_HOURS = "Hour(s)"
    FREQUENCY_MINUTES = "Minute(s)"
    FREQUENCY_DAYS = "Day(s)"

    def __init__(self, admin_console):
        """
        Args:
            admin_console: adminconsole base object
        """
        self.__admin_console = admin_console
        self.__drop_down = DropDown(self.__admin_console)

    @PageService()
    def select_recovery_target(self, target):
        """
        Select recovery target
        Args:
            target              (str):   Specify the recovery target
        """
        drop_down_id = 'vsaPlanReplicationTemplate_isteven-multi-select_#0954'
        self.__drop_down.select_drop_down_values(values=[target], drop_down_id=drop_down_id)

    @PageService()
    def _set_frequency_number(self, number):
        """
        Select frequency number
        Args:
            number              (int):   frequency to be set
        """
        self.__admin_console.fill_form_by_id('rpo', number)

    @PageService()
    def _select_frequency_period(self, period):
        """
        Select frequency period from declaration FREQUENCY_DAYS/FREQUENCY_MINUTES/
        FREQUENCY_HOURS
        Args:
            period             (str):   frequency period(Hours/Minutes/Days)
        """
        self.__admin_console.select_value_from_dropdown('rpoTime', period)

    @PageService()
    def set_frequency(self, frequency_number, frequency_period=FREQUENCY_HOURS):
        """
        Set frequency
        Args:
            frequency_number       (int):   frequency to be set
            frequency_period       (str):   frequency period(Hours/Minutes/Days)
        """
        self._set_frequency_number(frequency_number)
        self._select_frequency_period(frequency_period)

    @PageService()
    def validate_destination_vm(self, enable=True):
        """
        Select/deselect validate destination VM
        Args:
            enable                (bool): True/False to select/deselect
        """
        if enable:
            self.__admin_console.checkbox_select("powerOn")
        else:
            self.__admin_console.checkbox_deselect("powerOn")

    @PageService()
    def unconditionally_overwrite_vm(self, enable=True):
        """
        Select/Deselct Unconditionally overwrite vm
        Args:
            enable                (bool): True/False to select/deselect
        """
        toggle_id = "overwrite"
        if enable:
            self.__admin_console.enable_toggle(toggle_id=toggle_id, cv_toggle=True)
        else:
            self.__admin_console.disable_toggle(toggle_id=toggle_id, cv_toggle=True)

    @PageService()
    def set_snapshots_to_retain_on_destination_vm(self, snapshots):
        """
        set snapshots to retain on destination vm
        Args:
            snapshots           (int): number of snapshots
        """
        self.__admin_console.fill_form_by_id('nMaxSnapshotsPerDRVM', snapshots)


class _TargetAzure(_Target):
    """use this module for configure azure replication group target page operations"""

    def __init__(self, admin_console):
        super().__init__(admin_console)
        self.__admin_console = admin_console

    @PageService()
    def deploy_vm_during_failover(self, enable=True):
        """
        Select/Deselct deploy vm only during failover
        Args:
            enable                (bool): True/False to select/deselect
        """
        toggle_id = "deployVmWhenFailover"
        if enable:
            self.__admin_console.enable_toggle(toggle_id=toggle_id, cv_toggle=True)
        else:
            self.__admin_console.disable_toggle(toggle_id=toggle_id, cv_toggle=True)


class _StorageOrCache:
    """use this module for configure replication group storage/cache page operations"""

    class RetentionPeriodConstants:
        """Retention Period constants"""
        CONSTANT_DAYS = "Day(s)"
        CONSTANT_WEEKS = "Week(s)"
        CONSTANT_MONTHS = "Month(s)"
        CONSTANT_YEARS = "Year(s)"
        CONSTANT_INFINITE = "Infinite"

    def __init__(self, admin_console):
        """
        Args:
            admin_console: adminconsole base object
        """
        self.__admin_console = admin_console
        self.__driver = admin_console.driver
        self.__drop_down = DropDown(self.__admin_console)

    @PageService()
    def select_storage(self, storage_name):
        """
        Select storage
        Args:
            storage_name              (str): Provide storage name
        """
        self.__drop_down.select_drop_down_values(values=[storage_name], drop_down_id='primaryStorage')

    @PageService()
    def use_snapshot_on_source(self, enable=False, snapshot_source=""):
        """
        Select/deselect use snapshot on source option
        Args:
            enable (bool): True/False to select/deselect
        """
        if enable:
            self.__admin_console.checkbox_select("isSnapbackupEnabled")
            if snapshot_source:
                self.__drop_down.select_drop_down_values(values=[snapshot_source],
                                                         drop_down_id="snapEngines")
        else:
            self.__admin_console.checkbox_deselect("isSnapbackupEnabled")

    @PageService()
    def select_secondary_storage(self, secondary_storage_name):
        """
        Select secondary storage
        Args:
            secondary_storage_name(str):specify the name of the secondary storage
        """
        self.__admin_console.enable_toggle(index=0, cv_toggle=True)
        self.__drop_down.select_drop_down_values(values=[secondary_storage_name],
                                                 drop_down_id='secondaryStorage')


class _OverrideOptions:
    """use this module for configure replication group opverride options page operations"""

    def __init__(self, admin_console, hypervisor_type):
        """
        Args:
            admin_console   : adminconsole base object
            hypervisor_type : SOURCE_HYPERVISOR constants
        """
        self.__admin_console = admin_console
        self.__driver = admin_console.driver
        self.__table = Table(self.__admin_console)
        self.__drop_down = DropDown(self.__admin_console)
        self.__type = hypervisor_type

    @PageService()
    def override_vms(self, source_vm):
        """Enable override vms toggle and edit virtual machine parameters"""
        vm_options = None
        if self.__type == SOURCE_HYPERVISOR_AZURE:
            vm_options = _AzureVMOptions(self.__admin_console)
        if self.__type == SOURCE_HYPERVISOR_VMWARE:
            vm_options = _VMwareVMOptions(self.__admin_console)
        if self.__type == SOURCE_HYPERVISOR_HYPERV:
            vm_options = _HyperVOptions(self.__admin_console)
        self.__admin_console.enable_toggle(index=0, cv_toggle=True)
        sleep(10)
        self.__table.select_rows([source_vm])
        self.__table.access_toolbar_menu('OVERRIDE_REPLICATION')
        self.__admin_console.wait_for_completion()
        return vm_options

    @PageService()
    def __get_field_value(self, field_id, use_value=False):
        """Gets the value of the field id"""
        if self.__admin_console.check_if_entity_exists("xpath", f"//select[@id='{field_id}']//option"):
            options = ([element for element in
                        self.__driver.find_elements_by_xpath(f"//select[@id='{field_id}']//option")
                        if element.get_attribute("selected")])
            return options[0].get_attribute("value") if use_value else options[0].text
        if self.__admin_console.check_if_entity_exists("xpath", f"//*[@id='{field_id}']//button"):
            return (self.__driver.find_element_by_xpath(f"//*[@id='{field_id}']//button//div[@class='buttonLabel']")
                    .text)
        if self.__admin_console.check_if_entity_exists("xpath", f"//input[@name='{field_id}']"):
            return self.__driver.find_element_by_xpath(f"//input[@name='{field_id}']").get_property("value")
        return None

    @PageService()
    def get_vm_details(self, source_vm):
        """Get the VM details from the override options page"""
        self.override_vms(source_vm)
        field_ids = ['displayNamePrefixSuffix', 'azureContainer', 'azureRegion', 'azureStorageAccount',
                     'azureVmSize', 'selectVirtualNetworkAzure_isteven-multi-select_#2554',
                     'azureSecurityGroup_isteven-multi-select']
        report = [self.__get_field_value(field_id) for field_id in field_ids]
        self.__admin_console.click_button('Cancel')
        return report

    @PageService()
    def get_hyperv_details(self, source_vm):
        """Get the VM details from the override options page"""
        self.override_vms(source_vm)
        field_id = 'networkName'
        report = self.__get_field_value(field_id)
        self.__admin_console.click_button('Cancel')
        return report


class _EditVMCommon:
    """Edit virtual machine parameters using this class"""

    def __init__(self, admin_console):
        self._admin_console = admin_console
        self._drop_down = DropDown(self._admin_console)
        self._modal_panel = ModalPanel(self._admin_console)
        self.__driver = self._admin_console.driver

    @WebAction()
    def __is_tab_open(self, tab_name) -> bool:
        """Checks to see if the tab is open or not"""
        return 'expanded' in (self._admin_console.
                              driver.find_element_by_xpath(f"//span[contains(@class, 'cv-accordion-text')]"
                                                           f"//span[contains(text(), '{tab_name}')]"
                                                           f"/ancestor::div[contains(@class, 'cv-accordion-header')]")
                              .get_attribute('class'))

    @WebAction()
    def expand_tab(self, tab_name):
        """Expands the additional optional tags"""
        if not self.__is_tab_open(tab_name):
            self._admin_console.driver.find_element_by_xpath(f"//span[contains(@class, 'cv-accordion-text')]"
                                                             f"//span[contains(text(), '{tab_name}')]").click()

    @PageService()
    def set_vm_display_name(self, name):
        """
        Update display name
        Args:
            name(String): specify the vm display name
        """
        self._admin_console.fill_form_by_name('displayNamePrefixSuffix', name)

    @PageService()
    def cancel(self):
        """Click on cancel"""
        self._modal_panel.cancel()

    @PageService()
    def save(self):
        """Click on save"""
        self._modal_panel.submit()

    @WebAction()
    def get_selected_multi_select(self, multi_select_id: str) -> str:
        """Gets the current selected value from the multi-select"""
        return self.__driver.find_element_by_xpath(f"//*[@id='{multi_select_id}']//button").text

    @WebAction()
    def is_field_disabled(self, field_id: str) -> bool:
        """Checks if field is disabled and then waits for delay"""
        if self._admin_console.check_if_entity_exists("xpath", f"//select[@name='{field_id}']//option"):
            return bool([element for element in
                         self.__driver.find_elements_by_xpath(f"//select[@name='{field_id}']//option")
                         if element.get_attribute("disabled") is not None])
        if self._admin_console.check_if_entity_exists("xpath", f"//*[@id='{field_id}']//button"):
            return bool(self.__driver.find_element_by_xpath(f"//*[@id='{field_id}']//button")
                        .get_attribute('ng-disabled'))
        if self._admin_console.check_if_entity_exists("xpath", f"//input[@name='{field_id}']"):
            disabled_attr = (self.__driver.find_element_by_xpath(f"//input[@name='{field_id}']")
                             .get_attribute('data-ng-disabled'))
            return disabled_attr is not None and "editMode" in disabled_attr
        raise CVWebAutomationException(f'In the edit VM, the field with id {field_id} does not exist')


class _AzureVMOptions(_EditVMCommon):
    """Class for editing azure vm"""

    @PageService()
    def select_resource_group(self, group):
        """
        Select resource group
        Args:
            group(String): specify the resource group name
        """
        self._admin_console.select_value_from_dropdown('azureContainer', group)

    @PageService()
    def select_region(self, region):
        """
        Select region
        Args:
            region(string): specify the region
        """
        self._admin_console.select_value_from_dropdown('azureRegion', region)

    @PageService()
    def select_storage_account(self, account):
        """
        Select storage account
        Args:
            account(string): specify the storage account
        """
        drop_down_id = 'azureStorageAccount'
        self._drop_down.select_drop_down_values(values=[account], drop_down_id=drop_down_id)

    @PageService()
    def select_vm_size(self, size, use_value=True):
        """
        Select vm size
        Args:
            size(string): specify the vm size
            use_value(bool): uses the value tag to select the element from the dropdown
        """
        self._admin_console.select_value_from_dropdown('azureVmSize', size, attribute=use_value)

    @PageService()
    def virtual_network(self, network):
        """
        Select virtual network
        Args:
            network(string): specify the virtual network
        """
        drop_down_id = 'selectVirtualNetworkAzure_isteven-multi-select_#2554'
        self._drop_down.select_drop_down_values(values=[network], drop_down_id=drop_down_id,
                                                partial_selection=True)

    @PageService()
    def select_security_group(self, group):
        """
        Select vm size
        Args:
            group(string): specify the security group name
        """
        self._drop_down.select_drop_down_values(drop_down_id="azureSecurityGroup_isteven-multi-select",
                                                values=[group], partial_selection=True)

    @PageService()
    def create_public_ip(self, enable=True):
        """
        select/Deselect create public ip
        Args:
            enable(bool): True/False to select/deselect
        """
        if enable:
            self._admin_console.checkbox_select("createPublicIp")
        else:
            self._admin_console.checkbox_deselect("createPublicIp")

    @PageService()
    def restore_as_managed_vm(self, enable=True):
        """
        Select/Deselect restore as a managed vm
        Args:
            enable(bool): True/False to select/deselect
        """
        if enable:
            self._admin_console.checkbox_select("restoreAsManagedVM")
        else:
            self._admin_console.checkbox_deselect("restoreAsManagedVM")


class _VMwareVMOptions(_EditVMCommon):
    """Class for editing VMware vm"""

    def __init__(self, admin_console):
        """
        Args:
            admin_console: adminconsole base object
        """
        _EditVMCommon.__init__(self, admin_console)

        self._admin_console._load_properties(self, unique=True)
        self._label = self._admin_console.props[self.__class__.__name__]

    @WebAction()
    def __click_link_by_title(self, title, index=0, network=True):
        """
        Click the link by title
        Args:
            title: (str) click the link by title
            index: (int) the index of the button to click
            network: (bool) whether to click in network tab or ip tab
        """
        parent_xp = "//cv-vmware-network-settings" if network else "//cv-vmware-ipaddress-settings"
        buttons = self._admin_console.driver.find_elements_by_xpath(f"{parent_xp}//a[@title='{title}']")
        buttons[index].click()

    @property
    @WebAction()
    def vm_display_name(self):
        """Returns the VM display name"""
        return (self._admin_console.driver
                .find_element_by_xpath("//input[@name='displayNamePrefixSuffix']")
                .get_attribute("value"))

    @PageService()
    def set_vm_display_name(self, name):
        """Sets the VM Display Name"""
        self._admin_console.fill_form_by_name("displayNamePrefixSuffix", name)

    @PageService()
    def set_destination_host(self, host):
        """
        Set destination host
        Args:
            host(str): set destination host name
        """
        self._admin_console.click_by_id('selectVMwareEsxHost_button_#9930')  # click Browse
        sleep(2)
        self._admin_console.wait_for_completion()
        self._admin_console.select_destination_host(host)
        self._admin_console.click_button('OK')
        self._admin_console.wait_for_completion()

    @PageService()
    def select_datastore(self, datastore):
        """
        Select datastore
        Args:
            datastore(str): datastore name
        """
        drop_down_id = 'dataStore'
        self._drop_down.select_drop_down_values(values=[datastore], drop_down_id=drop_down_id,
                                                partial_selection=True)

    @PageService()
    def select_resource_pool(self, resource_pool_name):
        """
        Select resource pool
        Args:
            resource_pool_name(str): select resource pool name
        """
        self._admin_console.select_value_from_dropdown('resourcePool', resource_pool_name)

    @PageService()
    def set_vm_folder(self, name):
        """Set vm folder
        Args:
            name(str): vm folder name
        """
        self._admin_console.click_by_id('selectVMwareFolder_button_#3728')  # click Browse
        sleep(2)
        self._admin_console.wait_for_completion()
        self._admin_console.select_destination_host(name)
        self._admin_console.click_button('OK')
        self._admin_console.wait_for_completion()

    @PageService()
    def add_network(self):
        """Adds a network to the override options"""
        network_settings = _NetworkSettings(self._admin_console)
        self.expand_tab(self._label['label.networkSettingsVMWare'])
        self._admin_console.select_hyperlink(self._label['label.add'])
        return network_settings

    @PageService()
    def edit_network(self, index=0):
        """
        Edits the network at the particular index
        Args:
            index: (int) index of the network row
        Returns:
        """
        network_settings = _NetworkSettings(self._admin_console)
        self.expand_tab(self._label['label.networkSettingsVMWare'])
        self.__click_link_by_title(self._label['label.editSetting'], index, network=True)
        return network_settings

    @PageService()
    def remove_network(self, index=0):
        """
        Removes the network at the given index
        Args:
            index: (int) index of the network row
        """
        network_settings = _NetworkSettings(self._admin_console)
        self.expand_tab(self._label['label.networkSettingsVMWare'])
        self.__click_link_by_title(self._label['label.removeSetting'], index, network=True)
        return network_settings

    @PageService()
    def add_ip(self):
        """Adds an IP address to the override options"""
        ip_settings = _IpSettings(self._admin_console)
        self.expand_tab(self._label['label.addressSettings'])
        self._admin_console.select_hyperlink(self._label['label.add'])
        return ip_settings

    @PageService()
    def edit_ip(self, index=0):
        """
        Edits the ip at the particular index
        Args:
            index: (int) index of the ip row
        Returns:
        """
        ip_settings = _IpSettings(self._admin_console)
        self.expand_tab(self._label['label.addressSettings'])
        self.__click_link_by_title(self._label['label.editSetting'], index, network=False)
        return ip_settings

    @PageService()
    def remove_ip(self, index=0):
        """
        Removes the ip at the given index
        Args:
            index: (int) ip of the network row
        """
        ip_settings = _IpSettings(self._admin_console)
        self.expand_tab(self._label['label.addressSettings'])
        self.__click_link_by_title(self._label['label.removeSetting'], index, network=False)
        return ip_settings


class _HyperVOptions(_EditVMCommon):
    """Class for editing HyperV vm"""
    @PageService()
    def set_vm_display_name(self, name):
        """
        Set VM display name
            Args:
                name =  Str
        """
        self._admin_console.fill_form_by_name('displayNamePrefixSuffix', name)

    @PageService()
    def select_network(self, network):
        """
        Select network
        Args:
            network(str): network name
        """
        drop_down_id = 'networkName'
        self._drop_down.select_drop_down_values(values=[network], drop_down_id=drop_down_id, partial_selection=True)


class _NetworkSettings:
    """Class for configuring the network settings in the VMWare override options"""
    def __init__(self, admin_console):
        """Initialises the class"""
        self.__admin_console = admin_console
        self.__drop_down = DropDown(self.__admin_console)
        self.__modal_panel = ModalPanel(self.__admin_console)

    @PageService()
    def select_source_network(self, source_network: str):
        """Selects the network for the source VM"""
        self.__drop_down.select_drop_down_values(
            drop_down_id="addVMwareNetworkSettingsModal_isteven-multi-select_#0160",
            values=[source_network]
        )

    @PageService()
    def select_destination_network(self, destination_network: str):
        """Selects the network for the destination VM"""
        self.__drop_down.select_drop_down_values(
            drop_down_id="addVMwareNetworkSettingsModal_isteven-multi-select_#9141",
            values=[destination_network]
        )

    @PageService()
    def save(self):
        """Saves the network settings"""
        self.__admin_console.click_button("OK")  # TODO: Fix cvBusyOnAjax self.__modal_panel.submit()

    @PageService()
    def cancel(self):
        """Cancels the network setting form"""
        self.__modal_panel.cancel()


class _IpSettings:
    """Class for setting the IP settings for a VMware replication group"""
    def __init__(self, admin_console):
        """Initialises the page object"""
        self.__admin_console = admin_console
        self.__modal_panel = ModalPanel(self.__admin_console)
        self.__drop_down = DropDown(self.__admin_console)

    @property
    @WebAction()
    def source_ip(self):
        """Gets the IP address settings for the source VM"""
        ids = ["sourceIpAddress", "sourceSubnetMask", "sourceDefaultGateway"]
        return {element_id: self.__admin_console.driver.find_element_by_id(element_id).get_attribute("value")
                for element_id in ids}

    @property
    @WebAction()
    def destination_ip(self):
        """Gets the destination IP settings"""
        ids = ["destinationIpAddress", "destinationSubnetMask", "destinationDefaultGateway",
               "destinationPrefDnsServer", "destinationAltDnsServer", "destinationPrefWinsServer",
               "destinationAltWinsServer"]
        return {element_id: self.__admin_console.driver.find_element_by_id(element_id).get_attribute("value")
                for element_id in ids}

    @property
    @WebAction()
    def dhcp_enabled(self):
        """Returns true if DHCP is enabled"""
        return ('enabled' in self.__admin_console.driver
                .find_element_by_xpath("//div[contains(@class, 'cv-material-toggle')]")
                .get_attribute('class'))

    @property
    @WebAction()
    def hostname(self):
        """Gets the hostname value"""
        return self.__admin_console.driver.find_element_by_id("destComputerName").get_attribute("value")

    @PageService()
    def set_source_ip(self, ip_address: str, subnet_mask: str = '', default_gateway: str = ''):
        """Sets the IP address settings for the Source VM"""
        self.__admin_console.fill_form_by_id("sourceIpAddress", ip_address)
        if subnet_mask:
            self.__admin_console.fill_form_by_id("sourceSubnetMask", subnet_mask)
        if default_gateway:
            self.__admin_console.fill_form_by_id("sourceDefaultGateway", default_gateway)

    @PageService()
    def toggle_dhcp(self, enable=False):
        """DCHP toggler"""
        if enable:
            self.__admin_console.enable_toggle(index=0, cv_toggle=False)
        else:
            self.__admin_console.disable_toggle(index=0, cv_toggle=False)

    @PageService()
    def set_destination_ip(self, ip_address: str, subnet_mask: str = '', default_gateway: str = '',
                           primary_dns: str = '', secondary_dns: str = '',
                           primary_wins: str = '', secondary_wins: str = ''):
        """Sets the Destination IP settings for the Destination VM"""
        self.__admin_console.fill_form_by_id("destinationIpAddress", ip_address)
        if subnet_mask:
            self.__admin_console.fill_form_by_id("destinationSubnetMask", subnet_mask)
        if default_gateway:
            self.__admin_console.fill_form_by_id("destinationDefaultGateway", default_gateway)
        if primary_dns:
            self.__admin_console.fill_form_by_id("destinationPrefDnsServer", primary_dns)
        if secondary_dns:
            self.__admin_console.fill_form_by_id("destinationAltDnsServer", secondary_dns)
        if primary_wins:
            self.__admin_console.fill_form_by_id("destinationPrefWinsServer", primary_wins)
        if secondary_wins:
            self.__admin_console.fill_form_by_id("destinationAltWinsServer", secondary_wins)

    @PageService()
    def set_hostname(self, hostname: str):
        """Sets the hostname for the destination VM"""
        self.__admin_console.fill_form_by_id("destComputerName", hostname)

    @PageService()
    def save(self):
        """Saves the IP settings"""
        self.__admin_console.click_button("OK")

    @PageService()
    def cancel(self):
        """Cancels the IP setting form"""
        self.__admin_console.click_button("Cancel")


class _ConfigureCommon:
    """Class for configuring virtualization replication group"""

    def __init__(self, admin_console):
        """
        Args:
            admin_console: adminconsole base object
        """
        self._admin_console = admin_console
        self.content = _Content(self._admin_console)
        self.storage_cache = _StorageOrCache(self._admin_console)
        self.target = _Target(self._admin_console)
        self.override_options = _OverrideOptions(admin_console, self.hypervisor_type)
        self.__vm_type = self.hypervisor_type

        self._admin_console._load_properties(self, unique=True)
        self._label = self._admin_console.props[self.__class__.__name__]

    @property
    @abstractmethod
    def hypervisor_type(self):
        """Provide vm type"""
        raise NotImplementedError

    @PageService()
    @abstractmethod
    def add_default_group(self):
        """Create replication group with default options"""
        raise NotImplementedError

    @PageService()
    def next(self):
        """Click on next"""
        self._admin_console.button_next()
        self._admin_console.wait_for_completion()

    @PageService()
    def previous(self):
        """Click on previous"""
        self._admin_console.click_button("Previous")
        self._admin_console.wait_for_completion()

    @PageService()
    def finish(self):
        """click on finish"""
        self._admin_console.click_button("Finish")
        sleep(5)
        self._admin_console.wait_for_completion()


class ConfigureVMWareVM(_ConfigureCommon):
    """Class for configuring azure replication group"""

    @property
    def hypervisor_type(self):
        """VM type"""
        return SOURCE_HYPERVISOR_VMWARE

    @PageService()
    def add_default_group(self, replication_group_name, source_hypervisor,
                          virtual_machines, recovery_target, storage,
                          secondary_storage=None):
        """Create replication group with default options"""

        # set content fields
        self.content.set_name(replication_group_name)
        self.content.select_vm_from_browse_tree(
            source_hypervisor, {self._label['groupingOption.vmware.vmsAndTemplates']: virtual_machines}
        )
        self.next()

        # set target fields
        self.target.select_recovery_target(recovery_target)

        self.target.validate_destination_vm(False)
        self.target.unconditionally_overwrite_vm(True)
        self.next()

        # set storage/cache fields
        self.storage_cache.select_storage(storage)
        if secondary_storage:
            self.storage_cache.select_secondary_storage(secondary_storage)
        self.next()

        #  Override options
        self.next()

        #  Summary
        self.finish()


class ConfigureAzureVM(_ConfigureCommon):
    """Class for configuring azure virtualization replication group"""

    def __init__(self, admin_console):
        """
        Args:
            admin_console: adminconsole base object
        """
        super().__init__(admin_console)
        self.target = _TargetAzure(admin_console)

    @property
    def hypervisor_type(self):
        """VM type"""
        return SOURCE_HYPERVISOR_AZURE

    @PageService()
    def add_default_group(self, name, source_hypervisor, virtual_machines,
                          recovery_target, storage, secondary_storage=None):
        """Create replication group with default options"""

        # Target fields
        self.content.set_name(name)
        self.content.select_vm_from_browse_tree(source_hypervisor, {self._label['groupingOption.vms']:
                                                                    virtual_machines})
        self.next()

        # set target fields
        self.target.select_recovery_target(recovery_target)

        self.target.unconditionally_overwrite_vm(True)
        self.target.deploy_vm_during_failover()
        self.next()

        # set storage/cache fields
        self.storage_cache.select_storage(storage)
        if secondary_storage:
            self.storage_cache.select_secondary_storage(secondary_storage)
        self.next()

        #  Override options
        self.next()

        #  Summary
        self.finish()


class ConfigureHypervVM(_ConfigureCommon):

    @property
    def hypervisor_type(self):
        """VM type"""
        return SOURCE_HYPERVISOR_HYPERV

    @PageService()
    def add_default_group(self, replication_group_name, source_hypervisor,
                          virtual_machines, recovery_target, storage,
                          secondary_storage=None):
        """Create replication group with default options"""
        self.content.set_name(replication_group_name)
        self.content.select_vm_from_browse_tree(
            source_hypervisor, {self._label['groupingOption.vms']: virtual_machines}
        )
        self.next()

        # set target fields
        self.target.select_recovery_target(recovery_target)
        self.target.unconditionally_overwrite_vm(True)
        self.next()

        # set storage/cache fields
        self.storage_cache.select_storage(storage)
        if secondary_storage:
            self.storage_cache.select_secondary_storage(secondary_storage)
        self.next()

        #  Override options
        self.next()

        #  Summary
        self.finish()
