# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on the
replication page of the AdminConsole

"""

from Web.AdminConsole.DR.virtualization_replication import ConfigureVMWareVM, \
    ConfigureAzureVM, ConfigureHypervVM
from Web.AdminConsole.DR.fs_replication import ConfigureBLR
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.Common.exceptions import CVWebAutomationException
from Web.Common.page_object import WebAction, PageService
from Web.AdminConsole.Components.table import Table


class ReplicationGroup:
    """Class for Replication Groups Page"""

    def __init__(self, admin_console):
        self.__admin_console = admin_console
        self.__driver = admin_console.driver
        self.__table = Table(self.__admin_console)
        self.__modal_dialog = ModalDialog(self.__admin_console)

        self.__admin_console._load_properties(self, unique=True)
        self.__label = self.__admin_console.props[self.__class__.__name__]

    @WebAction()
    def __click_menu_link(self, menu_id):
        """
        Clicks on menu link
        Args:
            menu_id: Id of menu item
        """
        self.__driver.find_element_by_id(menu_id).click()

    @WebAction()
    def __click_row_selection(self, name):
        """
        Clicks on the row selection menu for the given title
        Args:
            name: the title of the row selection menu in FS Replication Group creation
        """
        self.__driver.find_element_by_xpath("//*[contains(@class, 'row-selection-menu')]"
                                            "/div/div/h3[contains(text(),'{}')]".format(name)).click()

    @PageService()
    def configure_sql_server_replication_group(self):
        """Configures a SQL Server replication group"""
        self.__admin_console.access_menu(self.__label['label.configureReplicationGroup'])
        self.__click_menu_link('ConfigureSQLReplicationGroup')
        self.__admin_console.wait_for_completion()

    @PageService()
    def delete_group(self, group_name):
        """Deletes the given group
        Args:
            group_name  (str):  Name of the group to be deleted
        """
        self.__table.access_action_item(group_name, self.__label['label.delete'])
        self.__modal_dialog.click_submit()
        self.__admin_console.refresh_page()
        if self.has_group(group_name) is False:
            return
        raise CVWebAutomationException(f"The group {group_name} still exists after deletion")

    @PageService()
    def access_group(self, group_name):
        """Access the given replication group
        Args:
            group_name(str):    Name of the group to be accessed
        """
        self.__table.access_link(group_name)

    @PageService()
    def has_group(self, group_name):
        """Returns True if group exists
        Args:
            group_name(str):    Name of the group
        """
        return self.__table.is_entity_present_in_column(self.__label['header.replicationGroup.name'], group_name)

    @PageService()
    def __configure_virtulization(self):
        """Configuring virtualization common steps"""
        self.__admin_console.access_menu(self.__label['label.configureReplicationGroup'])
        self.__click_menu_link('ConfigureVSAReplicationGroup')
        self.__admin_console.wait_for_completion()

    @PageService()
    def configure_vmware(self):
        """Selects configure replication group vmware vm"""
        vmware_vm = ConfigureVMWareVM(self.__admin_console)
        self.__configure_virtulization()
        return vmware_vm

    @PageService()
    def configure_azure(self):
        """Selects configure replication group azure vm"""
        azure_vm = ConfigureAzureVM(self.__admin_console)
        self.__configure_virtulization()
        return azure_vm

    @PageService()
    def configure_hyperv(self):
        """Selects configure replication group hyperv vm"""
        hyperv_vm = ConfigureHypervVM(self.__admin_console)
        self.__configure_virtulization()
        return hyperv_vm

    @PageService()
    def get_replication_group_details_by_name(self, name):
        """
        Read table content of replication group
        Args:
            name                     (str):   replication group name

        Returns                      (dict):  table content

        """
        group_name_label = self.__label['header.replicationGroup.name']
        if not self.__table.is_entity_present_in_column(group_name_label, name):
            raise CVWebAutomationException("Replication group [%s] not found in replications "
                                           "group page")
        self.__table.apply_filter_over_column(group_name_label, name)
        return self.__table.get_table_data()

    @staticmethod
    def get_schedule_name_by_replication_group(group_name):
        """
        Get schedule name by replication name
        Args:
            group_name               (str):  replication group name

        Returns                      (str):  Schedule name

        """
        return group_name + '_ReplicationPlan__ReplicationGroup'

    @PageService()
    def __configure_fs(self):
        """
        Selects configure File System under the Configure replication group
        """
        self.__admin_console.access_menu(self.__label['label.configureReplicationGroup'])
        self.__click_menu_link('ConfigureFSReplicationGroup')
        self.__admin_console.wait_for_completion()

    @PageService()
    def configure_blr(self):
        """
        Configure a Block Level replication in the admin console
        """
        blr = ConfigureBLR(self.__admin_console)
        self.__configure_fs()
        self.__click_row_selection(self.__label['label.replication.fs.continuous'])
        self.__admin_console.wait_for_completion()
        return blr

    @PageService()
    def replicate(self, source, group_name):
        """
        Args:
            source(str): Source name
            group_name(str): replication group name
        """
        self.__table.apply_filter_over_column(self.__label['label.selectSource'], source)
        self.__table.apply_filter_over_column("Group name", group_name)

        self.__table.access_action_item(group_name, "Replicate now")
        self.__admin_console.wait_for_completion()

    @PageService()
    def disable(self, group_name):
        """Disable a group"""
        self.__table.access_action_item(group_name, "Disable")
        self.__admin_console.wait_for_completion()

    @PageService()
    def enable(self, group_name):
        """enable a group"""
        self.__table.access_action_item(group_name, "Enable")
        self.__admin_console.wait_for_completion()
