# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on the
recovery targets page of the AdminConsole

"""
from time import sleep
from abc import ABC
from abc import abstractmethod
from Web.AdminConsole.Components.panel import (PanelInfo, DropDown, ModalPanel)
from Web.Common.exceptions import CVWebAutomationException
from Web.AdminConsole.Components.table import Table
from Web.Common.page_object import PageService, WebAction
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.Components.browse import ContentBrowse, CVAdvancedTree


class TargetConstants:
    """Recovery target string constants"""
    # vendors
    AMAZON = "Amazon"
    MICROSOFT_AZURE = "Microsoft Azure"
    MICROSOFT_HYPERV = "Microsoft Hyper-V"
    OPENSTACK = "OpenStack"
    ORACLE_CLOUD_INFRASTRUCTURE = "Oracle Cloud Infrastructure"
    VMWARE_VCENTER = "VMware vCenter"

    # Application Type
    REPLICATION = "Replication"
    REGULAR = "Regular"


class _RecoveryTargetCommon(ABC):
    """All component on recovery targets class should inherit this class"""

    def __init__(self, admin_console, name, vendor, application_type):
        self._admin_console = admin_console
        self.__driver: admin_console.driver = admin_console.driver
        self.__recovery_target_name = name
        self.__vendor = vendor
        self.__application_type = application_type
        self._drop_down = DropDown(self._admin_console)
        self._modal_panel = ModalPanel(self._admin_console)
        self._table = Table(self._admin_console)
        self._browse = ContentBrowse(self._admin_console)
        if vendor:
            # For edit panel vendor will be None as this is not editable field
            self.select_vendor()
        if application_type:
            # For edit panel application_type will be None as this is not editable field
            self.select_application_type()
        if name:
            # For edit panel target name might not be edited, in that case name will be None
            self.set_recovery_target_name()

    @property
    @abstractmethod
    def vendor(self):
        """Vendor name"""
        raise NotImplementedError

    @property
    def application_type(self):
        """Application type"""
        return self.__application_type

    @property
    def recovery_target_name(self):
        """Get recovery target name"""
        return self.__recovery_target_name

    @recovery_target_name.setter
    def recovery_target_name(self, recovery_target_name):
        self.__recovery_target_name = recovery_target_name

    @PageService()
    def set_recovery_target_name(self, name=None):
        """
        Set recovery target name
        Args:
            name =  Str
        """
        if not name:
            self._admin_console.fill_form_by_name('replicationTargetName', self.recovery_target_name)
        else:
            self._admin_console.fill_form_by_name('replicationTargetName', name)

    @PageService()
    def select_vendor(self):
        """
        Select vendor from TargetConstants vendors
        """
        drop_down_id = 'vendorTypes'
        self._drop_down.select_drop_down_values(values=[self.vendor], drop_down_id=drop_down_id)

    @PageService()
    def select_application_type(self):
        """
        Select vendor type TargetConstants vendor type
        """
        drop_down_id = 'applicationTypeSelection'
        self._drop_down.select_drop_down_values(values=[self.application_type],
                                                drop_down_id=drop_down_id)

    @PageService()
    def select_destination_hypervisor(self, name):
        """
        Set destination hypervisor
        Args:
            name(str): set destination hypervisor name
        """
        drop_down_id = 'client'
        self._drop_down.select_drop_down_values(values=[name], drop_down_id=drop_down_id)

    @PageService()
    def select_users_and_user_groups(self, values):
        """
        Select users and user groups
        Args:
            values(list): list of users and user groups
        """
        drop_down_id = 'usersAndUserGroups'
        self._drop_down.select_drop_down_values(values=values, drop_down_id=drop_down_id)

    @WebAction()
    def __set_display_name(self, name):
        """Set display name"""
        xpath = "(//*[@name='displayNamePrefixSuffix'])[3]"
        element = self._admin_console.driver.find_element_by_xpath(xpath)
        element.clear()
        element.send_keys(name)

    @PageService()
    def set_vm_display_name(self, name, suffix=False):
        """
        Set display name
        Args:
            name(str): Display name
            suffix(bool): if False 'prefix' will be selected otherwise suffix will be selected
        """
        if suffix:
            self._admin_console.select_radio('azureDisplayNameSuffix')
        else:
            self._admin_console.select_radio('azureDisplayNamePrefix')
        self.__set_display_name(name)

    @PageService()
    def add(self):
        """Click on add"""
        self._table.access_toolbar_menu("AddReplicationTargetButton")

    @PageService()
    def select_access_node(self, node):
        """
        Select access node
        Args:
            node(str): specify access node
        """
        drop_down_id = 'selectProxy_isteven-multi-select_#6148'
        self._drop_down.select_drop_down_values(values=[node], drop_down_id=drop_down_id)

    @WebAction()
    def is_field_disabled(self, field_id: str) -> bool:
        """Checks if field is disabled"""
        if self._admin_console.check_if_entity_exists("xpath", f"//select[@name='{field_id}']//option"):
            return (bool([element for element in
                          self.__driver.find_elements_by_xpath(f"//select[@name='{field_id}']//option")
                          if element.get_attribute("disabled") is not None]) or self.__driver.
                    find_element_by_xpath(f"//select[@name='{field_id}']").get_attribute("disabled") is not None)
        if self._admin_console.check_if_entity_exists("xpath", f"//*[@id='{field_id}']//button"):
            return bool(self.__driver.find_element_by_xpath(f"//*[@id='{field_id}']//button")
                        .get_attribute('ng-disabled'))
        if self._admin_console.check_if_entity_exists("xpath", f"//input[@name='{field_id}']"):
            disabled_attr = (self.__driver.find_element_by_xpath(f"//input[@name='{field_id}']")
                             .get_attribute('data-ng-disabled'))
            return disabled_attr is not None and "editMode" in disabled_attr
        raise CVWebAutomationException(f'In the edit VM, the field with id {field_id} does not exist')

    @PageService()
    def save(self):
        """Save target"""
        self._modal_panel.submit()

    @PageService()
    def cancel(self):
        """Save target"""
        self._modal_panel.cancel()


class _VMWareRecoveryTarget(_RecoveryTargetCommon):
    """Class for Recovery targets Page"""

    def __init__(self, admin_console, recovery_target_name, application_type):
        super().__init__(admin_console, recovery_target_name, self.vendor, application_type)

    @property
    def vendor(self):
        """Vendor name"""
        return TargetConstants.VMWARE_VCENTER

    # -------------------  General Options -------------------------------------------------------

    @PageService()
    def set_destination_host(self, host):
        """
        Set destination host
        Args:
            host(str): set destination host name
        """
        self._admin_console.click_by_id('selectVMwareEsxHost_button_#9930')  # click Browse
        sleep(2)
        self._admin_console.wait_for_completion()
        self._admin_console.select_destination_host(host)
        self._admin_console.click_button('OK')
        self._admin_console.wait_for_completion()

    @PageService()
    def select_datastore(self, datastore):
        """
        Select datastore
        Args:
            datastore(str): datastore name
        """
        drop_down_id = 'dataStore'
        self._drop_down.select_drop_down_values(values=[datastore], drop_down_id=drop_down_id,
                                                partial_selection=True)

    @PageService()
    def select_resource_pool(self, resource_pool_name):
        """
        Select resource pool
        Args:
            resource_pool_name(str): select resource pool name
        """
        self._admin_console.select_value_from_dropdown('resourcePool', resource_pool_name)

    @PageService()
    def set_vm_folder(self, name):
        """Set vm folder
        Args:
            name(str): vm folder name
        """
        self._admin_console.click_by_id('selectVMwareFolder_button_#3728')  # click Browse
        sleep(2)
        self._admin_console.wait_for_completion()
        self._admin_console.select_destination_host(name)
        self._admin_console.click_button('OK')
        self._admin_console.wait_for_completion()

    @PageService()
    def select_destination_network(self, destination_network):
        """
        select destination network
        Args:
            destination_network(str): select destination network
        """
        drop_down_id = 'networkSettingsDestination'
        self._drop_down.select_drop_down_values(values=[destination_network],
                                                drop_down_id=drop_down_id)

    # -------------------  Test failover options --------------------------------------------------
    @PageService()
    def set_expiration_time(self, expiration_interval, select_days=True):
        """
        set expiration time
        Args:
            expiration_interval(int): specify expiration time
            select_days(Bool): select days otherwise 'Hour(s)' will be selected
        """
        if select_days:
            self.__select_days()
            self._admin_console.fill_form_by_name('expirationTimeInDays', expiration_interval)
            return
        self._admin_console.fill_form_by_name('expirationTimeInHours', expiration_interval)

    def __select_days(self):
        """Select Day(s) in expiration time"""
        self._admin_console.select_radio('expirationTimeDays')

    @PageService()
    def select_mediaagent(self, mediaagent):
        """
        Select media agent
        Args:
            mediaagent(str): specify the media agent to be selected
        """
        drop_down_id = 'mediaAgent'
        self._drop_down.select_drop_down_values(values=[mediaagent],
                                                drop_down_id=drop_down_id)

    # -----------------------  Virtual lab options  ----------------------------------------------

    @PageService()
    def select_configure_isolated_network(self):
        """Select configure isolated network"""
        self._admin_console.select_radio('isolatedNetwork')

    @PageService()
    def select_configure_existing_network(self):
        """Select configure existing network"""
        self._admin_console.select_radio('existingNetwork')

    @PageService()
    def set_gateway_template(self, name):
        """Set gateway template
        Args:
            name(str): Gateway template name
        """
        self._admin_console.click_by_id('vmLCPolicyVMWare_button_#1871')  # click Browse
        sleep(2)
        self._admin_console.wait_for_completion()
        self._admin_console.select_destination_host(name)
        self._admin_console.click_button('OK')
        self._admin_console.wait_for_completion()

    @PageService()
    def select_gateway_network(self, network):
        """
        Select gateway network
        Args:
            network(str):specify the gateway netwrok
        """
        drop_down_id = 'gatewayNetwork'
        self._drop_down.select_drop_down_values(values=[network],
                                                drop_down_id=drop_down_id)

    def select_network(self, network):
        """
        select a network when 'Configure existing network is selected'
        Args:
            network(str): select network
        """
        drop_down_id = 'existingNetwork'
        self._drop_down.select_drop_down_values(values=[network],
                                                drop_down_id=drop_down_id)


class _EditVMWareRecoveryTarget(_VMWareRecoveryTarget):
    """Edit vmware recovery target"""

    def __init__(self, admin_console, recovery_target_name=None):
        super().__init__(admin_console, recovery_target_name, self.application_type)

    @property
    def vendor(self):
        """Vendor name"""
        # In recovery target edit panel, vendor is not editable field, so setting it as None so
        # that in base class(_VMWareRecoveryTarget) vendor selection should not be called
        return None

    @property
    def application_type(self):
        """Application type"""
        # In recovery target edit panel, vendor is not editable field, so setting it as None so
        # that in baseclass(_VMWareRecoveryTarget) application_type selection should not be called
        return None


class _AzureRecoveryTarget(_RecoveryTargetCommon):
    """Class for creating a Azure Recovery Target in DR"""

    def __init__(self, admin_console: AdminConsole, name: str or None, application_type: str or None):
        """Initialises the recovery target class for Azure VMs
        application_type: Either 'Regular' or 'Replication'
        """
        super(_AzureRecoveryTarget, self).__init__(admin_console, name, self.vendor, application_type)
        self._advanced_tree = CVAdvancedTree(admin_console)

        self._admin_console._load_properties(self, unique=True)
        self.__label = self._admin_console.props[self.__class__.__name__]

    @property
    def vendor(self) -> str:
        """The vendor name of the recovery target"""
        return TargetConstants.MICROSOFT_AZURE

    @WebAction()
    def expand_options(self) -> bool:
        """Expands the additional options tab"""
        xpath = "//a[@role='button']//span//i[contains(@class, 'chevron-right')]"
        if self._admin_console.check_if_entity_exists("xpath", xpath):
            self._admin_console.driver.find_element_by_xpath(xpath).click()
            return True
        return False

    @PageService()
    def select_resource_group(self, resource_group: str):
        """Selects the resource group if it is available"""
        self._admin_console.select_value_from_dropdown("azureContainer", value=resource_group)

    @PageService()
    def select_region(self, region_name: str):
        """Selects the region for the resource group"""
        self._admin_console.select_value_from_dropdown("azureRegion", value=region_name)

    @PageService()
    def select_storage_account(self, storage_name: str):
        """Selects the storage account for the resource group"""
        self._drop_down.select_drop_down_values(drop_down_id="azureStorageAccount", values=[storage_name])

    @PageService()
    def set_vm_size(self, vm_size: str = ''):
        """Sets the VM size in additional options
            vm_size: Usually Standard_* or Basic_*
        """
        if not vm_size:
            vm_size = self.__label['label.autoSelect']
        self._admin_console.select_value_from_dropdown("azureVmSize", vm_size, attribute=True)

    @PageService()
    def select_disk_type(self, disk_type: str):
        """Selects the disk type"""
        self._admin_console.select_value_from_dropdown("azureDiskTypeModel", disk_type)

    @PageService()
    def select_virtual_network(self, network: str = ''):
        """Selects the virtual gateway network"""
        if not network:
            network = self.__label['label.autoSelect']
        self._drop_down.select_drop_down_values(drop_down_id="selectVirtualNetworkAzure_isteven-multi-select_#2554",
                                                values=[network], partial_selection=True)

    @PageService()
    def set_security_group(self, security_group_name: str = ''):
        """Sets the security group for the resource group"""
        if not security_group_name:
            security_group_name = self.__label['label.autoSelect']
        self._drop_down.select_drop_down_values(drop_down_id='azureSecurityGroup_isteven-multi-select',
                                                values=[security_group_name])

    @PageService()
    def set_public_ip(self, create: bool):
        """Sets whether to create a public IP"""
        checkbox_id = "createPublicIp"
        if create:
            self._admin_console.checkbox_select(checkbox_id)
        else:
            self._admin_console.checkbox_deselect(checkbox_id)

    @PageService()
    def restore_as_managed_vm(self, enable: bool):
        """Checks the box for restore as a managed VM, only applicable for 'Replication' application type"""
        if enable:
            self._admin_console.checkbox_select("restoreAsManagedVM")
        else:
            self._admin_console.checkbox_deselect("restoreAsManagedVM")

    @PageService()
    def set_expiration_time(self, expiration_time: str):
        """Sets the expiration time in hours or days"""
        num, unit = expiration_time.split()
        if 'hour' in unit.lower():
            self._admin_console.select_radio("expirationTimeHoursAz")
            self._admin_console.fill_form_by_id("expirationTimeInHours", num)
        else:
            self._admin_console.select_radio("expirationTimeDaysAz")
            self._admin_console.fill_form_by_id("expirationTimeInDaysAz", num)

    @PageService()
    def select_test_virtual_network(self, virtual_network: str = ''):
        """Selects the test virtual network"""
        if not virtual_network:
            virtual_network = self.__label['label.autoSelect']
        self._drop_down.select_drop_down_values(drop_down_id='selectVirtualTestNetworkAzure',
                                                values=[virtual_network])

    @PageService()
    def select_test_vm_size(self, vm_size: str = ''):
        """Selects the test virtual network"""
        if not vm_size:
            vm_size = self.__label['label.autoSelect']
        self._drop_down.select_drop_down_values(drop_down_id='azureTestVMSize',
                                                values=[vm_size])

    @PageService()
    def select_templates(self, template_path):
        """Selects the template path from the template browser, only applicable for 'Regular' application type
        template_path   (list): List of strings delimited by '\' or '/' to be selected
        """
        self._advanced_tree.select_elements_by_full_path(template_path)


class _EditAzureRecoveryTarget(_AzureRecoveryTarget):
    """Class for editing a Azure Recovery Target"""
    def __init__(self, admin_console: AdminConsole, name: str = None):
        """Initialises the class for editing recovery target"""
        super(_EditAzureRecoveryTarget, self).__init__(admin_console, name, application_type=None)

    @property
    def vendor(self) -> None:
        """The vendor name is returned as None, as it is uneditable"""
        return

    def select_resource_group(self, resource_group: str):
        """Not editable"""
        raise NotImplementedError("Azure resource group is not editable")

    def select_region(self, region_name: str):
        """Not editable"""
        raise NotImplementedError("Region name is not editable")

    def select_availability_zone(self, zone: str):
        """Not editable"""
        raise NotImplementedError("Availability zone is not editable")

    def select_storage_account(self, storage_name: str):
        """Not editable"""
        raise NotImplementedError("Storage account is not editable")


class _HyperVRecoveryTarget(_RecoveryTargetCommon):
    """Class for Recovery targets Page"""
    def __init__(self, admin_console, recovery_target_name, application_type):
        super().__init__(admin_console, recovery_target_name, self.vendor, application_type)

    @property
    def vendor(self):
        """Vendor name"""
        return TargetConstants.MICROSOFT_HYPERV

    @PageService()
    def set_destination_folder(self, path):
        """
        Set destination path
        Args:
            path(str): set destination path name
        """
        self._admin_console.checkbox_deselect('restoreLocation')
        self._admin_console.wait_for_completion()
        self._admin_console.click_button('Browse')  # click Browse
        self._browse.select_path(path)
        self._admin_console.wait_for_completion()
        self._browse.save_path()

    @PageService()
    def select_network(self, network):
        """
        Select network
        Args:
            network(str): network name
        """
        drop_down_id = 'networkName'
        self._drop_down.select_drop_down_values(values=[network], drop_down_id=drop_down_id, partial_selection=True)


class _EditHyperVRecoveryTarget(_HyperVRecoveryTarget):
    """Edit Microsoft Hyper V recovery target"""
    def __init__(self, admin_console, recovery_target_name=None):
        super().__init__(admin_console, recovery_target_name, self.application_type)

    @property
    def vendor(self):
        """Vendor name"""
        # In recovery target edit panel, vendor is not editable field
        return None

    @property
    def application_type(self):
        """Application type"""
        # In recovery target edit panel, application Type is not editable field
        return None

    @PageService()
    def select_destination_hypervisor(self, name):
        """In recovery target edit panel, Not editable destination Hyper"""
        return None


class RecoveryTargets:
    """Operations on recovery target page"""

    def __init__(self, admin_console):
        self.__admin_console = admin_console
        self.__driver: admin_console.driver = admin_console.driver
        self.__table = Table(self.__admin_console)

    @PageService()
    def configure_vmware_recovery_target(self, recovery_target_name, application_type):
        """Configure vmware recovery target"""
        self.__admin_console.select_hyperlink('Add')
        sleep(2)
        return _VMWareRecoveryTarget(self.__admin_console, recovery_target_name, application_type)

    @PageService()
    def configure_azure_recovery_target(self, recovery_target_name: str,
                                        application_type: str) -> _AzureRecoveryTarget:
        """Configures an Azure Recovery Target"""
        self.__admin_console.select_hyperlink('Add')
        self.__admin_console.wait_for_completion()
        return _AzureRecoveryTarget(self.__admin_console, recovery_target_name, application_type)

    @PageService()
    def configure_hyperv_recovery_target(self, recovery_target_name, application_type):
        """Configure Microsoft Hyperv recovery target"""
        self.__admin_console.select_hyperlink('Add')
        self.__admin_console.wait_for_completion()
        return _HyperVRecoveryTarget(self.__admin_console, recovery_target_name, application_type)

    @PageService()
    def access_target(self, target_name):
        """
        Access specified recovery target
        Args:
            target_name(Str): recovery target name
        Returns: TargetDetails object
        """
        if self.has_target(target_name):
            target = TargetDetails(self.__admin_console)
            self.__admin_console.select_hyperlink(target_name)
            self.__admin_console.wait_for_completion()
            return target
        raise CVWebAutomationException("Target [%s] does not exists" % target_name)

    @PageService()
    def has_target(self, target_name):
        """
        Check recovery target exists
        Args:
            target_name(str): Specify recovery target name
        Returns(bool):True if target exists else returns false
        """
        return self.__table.is_entity_present_in_column('Name', target_name)

    @PageService()
    def delete_recovery_target(self, target_name):
        """Delete recovery target"""
        if self.has_target(target_name):
            self.__table.access_action_item(target_name, 'Delete')
            self.__admin_console.click_button('Yes')
        else:
            raise CVWebAutomationException("Target [%s] does not exists to delete" % target_name)

    @PageService()
    def get_target_details(self, target_name):
        """
        Read specified recovery target all the column fields
        Args:
            target_name(str): Recovery target name
        Returns(dict): table content of specified recovery target
        """
        self.__table.apply_filter_over_column('Name', target_name)
        data = self.__table.get_table_data()
        if data['Name']:
            return data
        raise CVWebAutomationException("Recovery Target [%s] does not exists in Recovery "
                                       "Targets Page" % target_name)


class TargetDetails:
    """Target details page operations"""

    def __init__(self, admin_console):
        self.__admin_console = admin_console
        self.__table = Table(self.__admin_console)

    @PageService()
    def get_target_summary(self):
        """Read recovery target summary panel"""
        summary_panel = PanelInfo(self.__admin_console, 'Summary')
        return summary_panel.get_details()

    @PageService()
    def edit_target(self, name=None, hypervisor_type=TargetConstants.VMWARE_VCENTER):
        """
        Edit target
        Args:
            name(str): specify name to be updated
            hypervisor_type(str): The type of hypervisor to edit
        Returns:_EditVMWareRecoveryTarget object
        """
        summary_panel = PanelInfo(self.__admin_console, 'Summary')
        summary_panel.edit_tile()
        target = None
        if hypervisor_type == TargetConstants.VMWARE_VCENTER:
            target = _EditVMWareRecoveryTarget(self.__admin_console, recovery_target_name=name)
        elif hypervisor_type == TargetConstants.MICROSOFT_AZURE:
            target = _EditAzureRecoveryTarget(self.__admin_console, name=name)
        elif hypervisor_type == TargetConstants.MICROSOFT_HYPERV:
            target = _EditHyperVRecoveryTarget(self.__admin_console, recovery_target_name=name)
        return target


class RecoveryPointStore:
    """Class for recovery point store configuration for continuous mode replication pairs"""

    def __init__(self, admin_console):
        """
        Args:
            admin_console: adminconsole base object
        """
        self.__admin_console = admin_console
        self.__driver = admin_console.driver
        self.__modal_panel = ModalPanel(admin_console)
        self.__drop_down = DropDown(admin_console)
        self.__browse = ContentBrowse(admin_console)

        self.__admin_console._load_properties(self, unique=True)
        self.__label = self.__admin_console.props[self.__class__.__name__]

    @WebAction()
    def __click_new_rpstore_button(self):
        """
        Click the hyperlink by finding using the title
        """
        title = self.__label['label.createRPStore']
        self.__driver.find_element_by_xpath(f"//*[@title='{title}']").click()

    @WebAction()
    def __fill_form_password(self, value):
        """
        Enters the password for the network path
        Args:
            value (str): The password value for the network path access
        """
        element = self.__driver.find_element_by_xpath('//input[@type="password"]')
        element.clear()
        element.send_keys(value)

    @WebAction()
    def __click_interval_link(self):
        """Clicks the peak interval hyperlink to open the interval selection window"""
        self.__driver.find_element_by_xpath("//*[@class='backup-window-list']//li").click()

    @WebAction()
    def __get_all_intervals(self, day):
        """Returns all the interval elements for a day"""
        return self.__driver.find_elements_by_xpath("//a[contains(text(),'{}')]/../../"
                                                    "td[contains(@class, 'week-time')]".format(day))

    @WebAction()
    def __create_store_button_click(self):
        """Recovery Store Creation Page Create Button click"""
        self.__driver.find_element_by_xpath("//*[@id='addRPStore_button_#9443']").click()

    @WebAction()
    def __time_picker(self, picker, time):
        """
        Picks a time for an interval for the selected time picker
        picker        : Picker element produced by driver
        time        (str): Time interval for the picker
        """
        time, unit = time.split()
        picker.click()
        time_element = picker.find_element_by_xpath(".//div/div[@class='popover-inner']/div//div[2]/input")
        time_element.clear()
        time_element.send_keys(time)
        self.__drop_down.select_drop_down_values(
            values=[unit],
            drop_down_id="cvTimeRelativePicker_isteven-multi-select_#6209")
        picker.click()

    @PageService()
    def __peak_interval_selection(self, peak_interval):
        """
        Selects the intervals which are marked as peak for the recovery point store
        Args:
            peak_interval (dict): the intervals at which recovery point store is marked at peak
                Must be a dict of keys as days, and values as list of date time ids(0-23)
                eg: {'Monday': [0,1,2,3], 'Tuesday': [0,1,2,3], 'Wednesday': [0,1,2,3]}
        """
        self.__click_interval_link()
        self.__admin_console.wait_for_completion()

        self.__admin_console.select_hyperlink("Clear")
        keys = peak_interval.keys()
        days = ["Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday", "Sunday"]
        for day in days:
            if day in keys:
                intervals = self.__get_all_intervals(day)
                for slot in peak_interval[day]:
                    if "selected" not in intervals[slot].get_attribute('class'):
                        intervals[slot].click()
        self.__modal_panel.submit()

    @PageService()
    def _pick_time(self, picker_id, time=None):
        """
        This page service is used to select a time picker on the basis of label or id and then pick the time
        Args:
            picker_id (str): ID of the picker
            time      (str): If time is given, the checkbox is selected(only for optional time pickers)
        """
        picker_xpath = "//*[@{}='{}']/../../div[2]/div"
        if self.__admin_console.check_if_entity_exists("xpath", picker_xpath.format("id", picker_id)):
            if not time:
                self.__admin_console.checkbox_deselect(picker_id)
                return
            self.__admin_console.checkbox_select(picker_id)
            picker = self.__driver.find_element_by_xpath(picker_xpath.format("id", picker_id))
        elif self.__admin_console.check_if_entity_exists("xpath", picker_xpath.format("for", picker_id)):
            picker = self.__driver.find_element_by_xpath(picker_xpath.format("for", picker_id))
        else:
            raise CVWebAutomationException("No time picker available with this ID")
        self.__time_picker(picker, time)

    @PageService()
    def select_recovery_type(self, recovery_type):
        """Selects the recovery type from the dropdown
        Args:
            recovery_type (int): 0 for latest recovery, 1 for point in time recovery
        """
        recovery_types = [self.__label['label.recoveryLive'], self.__label['label.recoveryGranular']]
        self.__drop_down.select_drop_down_values(
            values=[recovery_types[recovery_type]],
            drop_down_id="selectRecoveryOptions_isteven-multi-select_#1569")

    @PageService()
    def select_store(self, name):
        """
        Checks the recovery point store list to select the recovery point store
        Args:
            name (str): Name of the store
        """
        self.__drop_down.select_drop_down_values(
            values=[name],
            drop_down_id='selectRecoveryOptions_isteven-multi-select_#1717')

    @PageService()
    def create_recovery_store(self, name, media_agent, max_size, path, path_type='',
                              path_username=None, path_password=None, peak_interval=None):
        """
        Args:
            name            (str): name of the recovery point store
            media_agent     (str): name of the media agent on which the store will reside on
            max_size        (int): the maximum size of the recovery point store in GB
            path            (str): the path at which the store will be present
            path_type       (str): the path type as 'Local path' or 'Network path'
            path_username   (str): the path access username, only for network path
            path_password   (str): the path access password, only for network path
            peak_interval  (dict): the intervals at which recovery point store is marked at peak
                Must be a dict of keys as days, and values as list of date time ids(0-23)
        """
        self.__click_new_rpstore_button()
        self.__admin_console.wait_for_completion()

        self.__admin_console.fill_form_by_name("libraryName", name)
        self.__admin_console.select_value_from_dropdown('mediaAgent', media_agent)
        self.__admin_console.fill_form_by_name("maxThreshold", max_size)
        self.__admin_console.check_radio_button(path_type)

        if not path_type:
            path_type = self.__label['Local_Path']

        if path_type == self.__label['Local_Path']:
            self.__admin_console.click_button(self.__label['Browse'])
            self.__browse.select_path(path)
            self.__browse.save_path()
        elif path_type == self.__label['Network_Path']:
            self.__admin_console.fill_form_by_name("loginName", path_username)
            self.__fill_form_password(path_password)
            self.__admin_console.wait_for_completion()

        if peak_interval:
            self.__peak_interval_selection(peak_interval)
            self.__admin_console.wait_for_completion()
            self.__admin_console.check_error_message()

        self.__create_store_button_click()
        self.__admin_console.wait_for_completion()
        self.__admin_console.check_error_message()

    @PageService()
    def configure_intervals(self, ccrp=None, acrp=None):
        """
        Configures the recovery point store interval options
        Args:
            ccrp      (str): Crash consistent recovery points interval '<time> <unit>'
                             unit -> ('seconds', 'minutes', 'days', 'hours')
            acrp      (str): Application Consistent recovery points interval '<time> <unit>'
                             unit -> ('seconds', 'minutes', 'days', 'hours')
        """
        if not (ccrp or acrp):
            raise CVWebAutomationException("Either CCRP or ACRP must be set")

        self._pick_time("useCcrp", ccrp)
        self._pick_time("useAcrp", acrp)

    @PageService()
    def configure_retention(self, retention, merge=False, merge_delay=None,
                            max_rp_interval=None, max_rp_offline=None, off_peak_only=False):
        """
        Args:
            retention            (str): Duration for which a recovery point is retained for
            merge               (bool): Whether to merge recovery points
            merge_delay          (str): Merge recovery points older than this interval
            max_rp_interval      (str): Recovery point max retention interval
            max_rp_offline       (str): After what time to switch to latest recovery if RPstore is offline
            off_peak_only       (bool): Whether to prune and merge only on non-peak time
        """
        self._pick_time("useRpRetention", retention)
        if merge:
            self.__admin_console.checkbox_select("useRpMerge")

            self._pick_time("useRpMergeDelay", merge_delay)
            self._pick_time("useMaxRpInterval", max_rp_interval)
        self._pick_time("useMaxRpStoreOfflineTime", max_rp_offline)
        if off_peak_only:
            self.__admin_console.checkbox_select("useOffPeakSchedule")
        else:
            self.__admin_console.checkbox_deselect("useOffPeakSchedule")
