# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on replication monitor page

"""
from Web.Common.page_object import PageService, WebAction
from Web.AdminConsole.Components.table import Table, CVTable
from Web.AdminConsole.Components.panel import ModalPanel
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.AdminConsole.DR.recovery_targets import RecoveryPointStore
from Web.AdminConsole.DR.fs_replication import ConfigureBLR, ReplicaCopy
from Web.Common.exceptions import CVWebAutomationException


class ReplicationMonitor:
    """All operation on s specific to Replication monitors goes here"""

    def __init__(self, admin_console):
        self.__admin_console = admin_console
        self.__table = Table(self.__admin_console)
        self.__modal_panel = ModalPanel(self.__admin_console)

        self.__admin_console._load_properties(self, unique=True)
        self.__label = self.__admin_console.props[self.__class__.__name__]

    @PageService()
    def __filter_row_select(self, source, replication_group=None, select=True):
        """ Selects the row after applying the filters for source and replication group
        Args:
            source(str): Source name
            replication_group(str or None): replication group name
            select(bool): whether to select the row
        """
        self.__table.apply_filter_over_column(self.__label['label.replicationSource'], source)
        if replication_group:
            self.__table.apply_filter_over_column(self.__label['label.replicationGroup'], replication_group)
        if select:
            self.__table.select_rows([source])

    @PageService()
    def has_replication_group(self, group_name):
        """
        Check if specified replication group exists
        Args:
            group_name           (str):      Replication group name
        Returns                  (bool):     True if replication group exists or False otherwise
        """
        return self.__table.is_entity_present_in_column(self.__label['label.replicationGroup'], group_name)

    @PageService()
    def get_replication_group_details(self, source, replication_group=None):
        """
        Read replication monitor page content for specific group name
        Args:
            replication_group   (str or None): Specify replication group name
        Returns                 (dict): table content
        """
        self.__filter_row_select(source, replication_group, select=False)
        data = self.__table.get_table_data()
        if data.get(self.__label['label.replicationSource']):
            return data
        if replication_group:
            raise CVWebAutomationException("Replication group [%s] not in replication monitor page"
                                           % replication_group)
        return

    @PageService()
    def access_source(self, source, replication_group):
        """
        Opens the vm with the given name
        Args:
            source              (str):  name of vm
            replication_group    (str or None): replication group name
        Raises:
            Exception:
                If there is no vm with given name
        """
        self.__filter_row_select(source, replication_group, select=False)
        self.__table.access_link(source)

    @PageService()
    def planned_failover(self, source, replication_group):
        """
        Planned failover for the replication group.
        Args:
            source            (str): source name
            replication_group (str or None): replication group name
        Returns: job id for planned failover job
        """
        self.__filter_row_select(source, replication_group)
        self.__table.access_toolbar_menu('plannedFailover')
        self.__modal_panel.submit(wait_for_load=False)
        job_id = self.__admin_console.get_jobid_from_popup()
        return job_id

    @PageService()
    def unplanned_failover(self, source, replication_group):
        """
        Unplanned failover for the replication group.
        Args:
            source            (str): source name
            replication_group (str or None): replication group name
        Returns: job id for unplanned failover job
        """
        self.__filter_row_select(source, replication_group)
        self.__table.access_toolbar_menu('unplannedFailover')
        self.__modal_panel.submit(wait_for_load=False)
        job_id = self.__admin_console.get_jobid_from_popup()
        return job_id

    @PageService()
    def failback(self, source, replication_group):
        """
        Failback for the replication group.
        Args:
            source            (str): source name
            replication_group (str or None): replication group name
        Returns: job id for unplanned failover job
        """
        self.__filter_row_select(source, replication_group)
        self.__table.access_toolbar_menu('VMFailback')
        self.__modal_panel.submit(wait_for_load=False)
        job_id = self.__admin_console.get_jobid_from_popup()
        return job_id

    @PageService()
    def undo_failover(self, source, replication_group):
        """
        Undo failover for the replication group.
        Args:
            source            (str): source name
            replication_group (str or None): replication group name
        Returns: job id for unplanned failover job
        """
        self.__filter_row_select(source, replication_group)
        self.__table.access_toolbar_menu('VMUndoFailover')
        self.__modal_panel.submit(wait_for_load=False)
        job_id = self.__admin_console.get_jobid_from_popup()
        return job_id

    @PageService()
    def mark_for_full_replication(self, source, replication_group):
        """
         This method marks the VM for full replication

        Args:
            source            (string)    : vm whose replication is to be started immediately
            replication_group (string)    : replication group name
        Raises:
            Exception:
                Replication action item is not present

                Wrong job text is displayed
        """
        sync_status = self.sync_status(source, replication_group)
        expected_sync_status = ['Sync Enabled', 'In Sync', 'Sync Pending']
        if sync_status in expected_sync_status:
            self.__table.select_rows([source])
            self.__table.access_toolbar_menu('VSAREP_PENDING')
        else:
            raise CVWebAutomationException("status [%s] of source is not in expected state[%s]"
                                           % (sync_status, str(expected_sync_status)))

    @PageService()
    def test_boot_vm(self, source, replication_group):
        """
        This method test the boot for VM
        Args:
            source            (string)    : specify the source name
            replication_group (string)    : replication group name
        Raises:
            Exception:
                Test boot VM action item is not present
                Wrong job text is displayed
        """
        sync_status = self.sync_status(source, replication_group)
        expected_sync_status = ['In Sync', 'Needs Sync']
        if sync_status not in expected_sync_status:
            raise CVWebAutomationException("status [%s] of source is not in expected state[%s]"
                                           % (sync_status, str(expected_sync_status)))
        self.__table.select_rows([source])
        self.__table.access_menu_from_dropdown('failoverTestBootVM')
        self.__modal_panel.submit(wait_for_load=False)
        return self.__admin_console.get_jobid_from_popup()

    @PageService()
    def sync_status(self, source, replication_group):
        """
        Returns the sync status of vm
        Args:
            source              (String) : source name whose sync status has to be checked
            replication_group   (string) : replication group name
        Returns:
            Sync Status of the VM
        Raises:
            Exception:
                Not able to find the staus for VM
        """
        self.__filter_row_select(source, replication_group, select=False)
        sync_status = self.__table.get_column_data(self.__label['label.column.status'])
        if not sync_status:
            raise CVWebAutomationException("Source[%s] and replication_group[%s] row is not "
                                           "found in monitor page" % (source, replication_group))
        return sync_status[0]

    def replicate_now(self, source, replication_group):
        """
        Triggers replicate now by filtering source and replication group name in monitor page
        Args:
            source(str): Source name
            replication_group(str): replication group name
        """
        self.__filter_row_select(source, replication_group)
        self.__table.access_toolbar_menu('VMReplicateNow')

    @PageService()
    def view_details(self, source, replication_group):
        """
        Args:
            source(str): Source name
            replication_group(str): replication group name
        """
        self.__filter_row_select(source, replication_group)
        self.__table.access_toolbar_menu("Details")

    @PageService()
    def delete(self, source, replication_group):
        """
        Delete VM from Replication Monitor
        Args:
            source(str): Source name
            replication_group(str): replication group name
        """
        self.__filter_row_select(source, replication_group)
        self.__table.access_toolbar_menu("Delete")
        self.__modal_panel.submit()

    @PageService()
    def enable_validation(self, source, replication_group):
        """
        Args:
            source(str): Source name
            replication_group(str): replication group name
        """
        self.__filter_row_select(source, replication_group)
        self.__table.access_toolbar_menu("EnabledValidation")

    @PageService()
    def disable_validation(self, source, replication_group):
        """
        Args:
            source(str): Source name
            replication_group(str): replication group name
        """
        self.__filter_row_select(source, replication_group)
        self.__table.access_toolbar_menu("DisabledValidation")

    @PageService()
    def disable_replication(self, source, replication_group):
        """
        Args:
            source(str): Source name
            replication_group(str): replication group name
        """
        self.__filter_row_select(source, replication_group)
        self.__table.access_toolbar_menu("VSAREP_DISABLED")

    @PageService()
    def enable_replication(self, source, replication_group):
        """
        Args:
            source(str): Source name
            replication_group(str): replication group name
        """
        self.__filter_row_select(source, replication_group)
        self.__table.access_toolbar_menu('VSAREP_ENABLED')

    @PageService()
    def validate_as_per_replication_group(self, source, replication_group):
        """
        Validate as Per Replication Group setting
        Args:
            source(str): Source name
            replication_group(str): replication group name
        """
        self.__filter_row_select(source, replication_group)
        self.__table.access_toolbar_menu("AsPerScheduleValidation")

    @PageService()
    def pit_fail_over(self, source, replication_group):
        """
        Point in Time Failover
        Args:
            source(str): Source name
            replication_group(str): replication group name
        """
        self.__filter_row_select(source, replication_group)
        self.__table.access_toolbar_menu("PITFailover")
        self.__modal_panel.submit()


class ContinuousReplicationMonitor:
    """
    This class is used to perform monitoring operations on the Continuous tab of the Replication monitor
    """

    # TODO: Move all functions to not use views

    def __init__(self, admin_console):
        """
        Args:
            admin_console: adminconsole base object
        """
        self.__admin_console = admin_console
        self.__driver = admin_console.driver
        self.__table = CVTable(admin_console)
        self.__dialog = ModalDialog(admin_console)
        self.__modal_panel = ModalPanel(admin_console)

        self.__admin_console._load_properties(self, unique=True)
        self.__label = self.__admin_console.props[self.__class__.__name__]

    @WebAction()
    def __get_report(self):
        """Gets the values for each entity in the dash pane component"""
        values = [int(element.text) for element in
                  self.__driver.find_elements_by_xpath("//div[contains(@class, 'entities')]"
                                                       "//span[contains(@class, 'entity-value')]")]
        labels = [element.get_attribute('title') for element in
                  self.__driver.find_elements_by_xpath("//div[contains(@class, 'entities')]"
                                                       "//a[contains(@class, 'entity-container')]")]
        return dict(zip(labels, values))

    @WebAction()
    def __open_grid_menu(self):
        """Opens the filter menu present in the table for views and column selection"""
        self.__driver.find_element_by_xpath("//*[@class='ui-grid-icon-container']").click()

    @WebAction()
    def __select_menu_option(self, action_name):
        """
        Selects the action in the grid menu
        Args:
            action_name (str): Name of the action to be selected
        """
        self.__driver.find_element_by_xpath("//*[@class='ui-grid-menu-inner']//button[contains(text(), '{}')]"
                                            .format(action_name)).click()

    @WebAction()
    def __click_view_dropdown(self):
        """Clicks on the filters dropdown"""
        self.__driver.find_element_by_xpath("//*[contains(@class, 'view-selection')]").click()

    @WebAction()
    def __click_view(self, name='All'):
        """Clicks on the view name available on the view dropdown"""
        self.__driver.find_element_by_xpath("//*[contains(@class, 'view-selection')]//*[contains(text(), '{}')]"
                                            .format(name)).click()

    @WebAction()
    def __fill_rule_form(self, rule_id, value):
        """
        Fills the rules inputs using the class name
            index: index# of the rule
            value: Value to be entered
        """
        element = self.__driver.find_element_by_xpath("//*[@id='{}']//input".format(rule_id))
        element.clear()
        element.send_keys(value)

    @WebAction()
    def __click_action_menu(self):
        """Clicks the action menu button for the first row in the table"""
        self.__driver.find_element_by_xpath("//*[contains(@class, 'cv-permitted-actions')]").click()

    @WebAction()
    def __click_action(self, action_name):
        """Clicks on the action name provided"""
        self.__driver.find_element_by_xpath("//*[contains(@class, 'cv-permitted-actions')]"
                                            "//li/a[contains(text(), '{}')]".format(action_name)).click()

    @property
    @PageService()
    def report(self):
        """Gets the report from the web action
        eg: {
            'Total pairs': 0,
            'Point in time recovery pairs': 0,
            'Latest recovery pairs': 0,
            'Application consistent pairs': 0,
            'Pairs with lag': 0
            }
        """
        return self.__get_report()

    @PageService()
    def create_view(self, view_name, rules, set_default=False):
        """
        Creates a new view in the replication pairs for sorting and filtering
        Args:
            view_name: Name of the view to be created
            rules: A dictionary of rules in the form of {<column-name>: <value>}
                    eg: {'Source': 'client1'}
            set_default: Sets the view as default when opening the replication monitor
        """
        self.__open_grid_menu()
        self.__select_menu_option(self.__label['label.createView'])
        self.__admin_console.wait_for_completion()

        self.__admin_console.fill_form_by_id("viewName", view_name)
        if set_default:
            self.__admin_console.checkbox_select("chkSetAsDefault")
        num_rules = len(rules)
        for rule_idx, rule_key in enumerate(rules):
            self.__admin_console.select_value_from_dropdown(f'rule-{rule_idx}', rule_key)
            self.__fill_rule_form(f'ruleFilter-{rule_idx}', rules[rule_key])
            if rule_idx < num_rules - 1:
                # Click add rule only if rule is not the last
                self.__admin_console.click_button(self.__label['label.addRule'])
        self.__modal_panel.submit()
        self.__admin_console.check_error_message()
        self.__admin_console.wait_for_completion()

    @PageService()
    def check_view(self, name):
        """Checks whether a given view exists or not"""
        return self.__admin_console.check_if_entity_exists("xpath", "//*[contains(@class, 'view-selection')]"
                                                                    "//*[contains(text(), '{}')]".format(name))

    @PageService()
    def select_view(self, view_name):
        """Selects a view based on the name provided"""
        if not self.check_view(view_name):
            raise CVWebAutomationException('The view {} does not exist'.format(view_name))
        self.__click_view_dropdown()
        self.__click_view(view_name)
        self.__admin_console.wait_for_completion()

    @PageService()
    def delete_view(self, view_name):
        """Removes the view from the command center"""
        if not self.check_view(view_name):
            raise CVWebAutomationException('Cannot delete the view {} because it does not exist'.format(view_name))
        self.__open_grid_menu()
        self.__select_menu_option(self.__label['label.deleteView'])
        self.__admin_console.wait_for_completion()

        self.__admin_console.select_value_from_dropdown("deleteViewName", view_name)
        self.__modal_panel.submit()
        self.__admin_console.check_error_message()

    @PageService()
    def access_action(self, action_name):
        """Access the actions options present in the action button for the first row"""
        self.__click_action_menu()
        self.__admin_console.wait_for_completion()
        self.__click_action(action_name)
        self.__admin_console.wait_for_completion()

    @PageService()
    def has_replication_group(self, source_client, destination_client):
        """
        Returns the boolean on whether the pair exists or not
        Args:
            source_client       (str): Name of the source client
            destination_client  (str): Name of the destination client
        Returns: true, if row exists, false otherwise
        """
        source_exists = source_client in self.__table.get_column_data(self.__label['label.source'])
        destination_exists = destination_client in self.__table.get_column_data(self.__label['label.destination'])
        return source_exists and destination_exists

    @PageService()
    def get_replication_group_details(self, source_client, destination_client):
        """
        Gets the row data for the selected replication pair
        Args:
            source_client       (str): Name of the source client
            destination_client  (str): Name of the destination client
        Returns: dict of table content
        """
        if not self.has_replication_group(source_client, destination_client):
            raise CVWebAutomationException("No pair {}, {} found".format(source_client, destination_client))
        details = self.__table.get_values_from_table([source_client])
        return details[0]

    @PageService()
    def sync_status(self, source_client, destination_client):
        """
        Returns the synchronisation status of the FIRST replication pair of the source client
        """
        return self.get_replication_group_details(source_client, destination_client)[self.__label['header.status']]

    @PageService()
    def access_source(self, source_client):
        """
        Clicks on the FIRST replication pair for the source client name
        To make sure you select the right pair, ensure to use `create_view` and `select_view` functions beforehand for
        filtering the rows
        """
        self.__admin_console.select_hyperlink(source_client)
        self.__admin_console.wait_for_completion()

    @PageService()
    def resync(self):
        """
        Performs the resync operation on the FIRST replication pair for the source client
        To make sure you select the right pair, ensure to use `create_view` and `select_view` functions beforehand for
        filtering the rows
        """
        self.access_action(self.__label['action.resync'])
        self.__admin_console.wait_for_completion()

    @PageService()
    def stop(self):
        """
        Performs the stop operation on the FIRST replication pair for the source client
        To make sure you select the right pair, ensure to use `create_view` and `select_view` functions beforehand for
        filtering the rows
        """
        self.access_action(self.__label['action.stop'])
        self.__admin_console.wait_for_completion()

    @PageService()
    def suspend(self):
        """
        Performs the suspend operation on the FIRST replication pair for the source client
        To make sure you select the right pair, ensure to use `create_view` and `select_view` functions beforehand for
        filtering the rows
        """
        self.access_action(self.__label['action.suspend'])
        self.__admin_console.wait_for_completion()

    @PageService()
    def resume(self):
        """
        Performs the resume operation on the FIRST replication pair for the source client
        To make sure you select the right pair, ensure to use `create_view` and `select_view` functions beforehand for
        filtering the rows
        """
        self.access_action(self.__label['action.resume'])
        self.__admin_console.wait_for_completion()

    @PageService()
    def start(self):
        """
        Performs the start operation on the FIRST replication pair for the source client
        To make sure you select the right pair, ensure to use `create_view` and `select_view` functions beforehand for
        filtering the rows
        """
        self.access_action(self.__label['action.start'])
        self.__admin_console.wait_for_completion()

    @PageService()
    def delete_pair(self):
        """
        Performs the suspend operation on the FIRST replication pair for the source client
        To make sure you select the right pair, ensure to use `create_view` and `select_view` functions beforehand for
        filtering the rows
        """
        self.access_action(self.__label['action.deletePair'])
        self.__dialog.click_submit()
        self.__admin_console.wait_for_completion()

    @PageService()
    def edit_recovery_options(self):
        """
        Accesses the Edit recovery options and then returns the RecoveryPointStore used to configure the recovery
        options on the page
        """
        rpstore = RecoveryPointStore(self.__admin_console)
        self.access_action(self.__label['action.editRecoveryOptions'])
        self.__admin_console.wait_for_completion()
        return rpstore

    @PageService()
    def edit_replication_volumes(self):
        """
        Accesses the Edit replication volumes and then returns the ConfigureBLR object used to configure the volumes
        on the page
        """
        blr = ConfigureBLR(self.__admin_console)
        self.access_action(self.__label['label.editReplicationVolume'])
        self.__admin_console.wait_for_completion()
        return blr

    @PageService()
    def create_replica_copy(self):
        """
        Accesses the replica copy pair action for a replication pair, and return the ReplicaCopy object which
        can be used to configure a replica copy
        """
        replica_copy = ReplicaCopy(self.__admin_console)
        self.access_action(self.__label['action.createNewPermMount'])
        self.__admin_console.wait_for_completion()
        return replica_copy
