# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on the
specific replication page of the AdminConsole
"""
from time import sleep
from selenium.webdriver.common.keys import Keys
from Web.Common.exceptions import CVWebAutomationException
from Web.Common.page_object import (
    PageService, WebAction
)
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.Components.panel import (PanelInfo, DropDown, ModalPanel)
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.AdminConsole.Components.table import Table, CVTable
from Web.AdminConsole.DR.monitor import ReplicationMonitor, ContinuousReplicationMonitor
from Web.AdminConsole.DR.virtualization_replication import (SOURCE_HYPERVISOR_AZURE,
                                                            SOURCE_HYPERVISOR_VMWARE,
                                                            SOURCE_HYPERVISOR_HYPERV,
                                                            _Content, _AzureVMOptions,
                                                            _VMwareVMOptions, _HyperVOptions,
                                                            _OverrideOptions)


class OverviewTab:
    """Class for overview tab of the group details page"""

    def __init__(self, admin_console: AdminConsole):
        self.__admin_console = admin_console
        self.__table = Table(self.__admin_console)
        self.__panel = ModalPanel(self.__admin_console)

        self.monitor = ReplicationMonitor(self.__admin_console)
        self.continuous_monitor = ContinuousReplicationMonitor(self.__admin_console)

        self.__admin_console._load_properties(self, unique=True)
        self.__label = self.__admin_console.props[self.__class__.__name__]

    @PageService()
    def get_summary_details(self):
        """Read summary information from overview tab page"""

        panel = PanelInfo(self.__admin_console,
                          self.__label['label.replicationGroupsDetail'])
        return panel.get_details()

    @PageService()
    def get_sync_status(self, source):
        """
        Returns Sync Status for source
        Args:
            source(string): source name
        """
        return self.monitor.get_replication_group_details(source)[self.__label['header.syncStatus']]

    @PageService()
    def get_failover_status(self, source):
        """
        Returns: failover status
        Args:
            source(string): source name
        """
        return (self.monitor.
                get_replication_group_details(source)[self.__label['header.failoverStatus']])

    @PageService()
    def get_recovery_target(self):
        """
        Return Recovery Target
        """
        return self.get_summary_details()[self.__label['label.replicationTargetDestination']]

    @PageService()
    def get_vm_details(self, vm_name):
        """Returns the replication VMs and their configuration from the table"""
        table_data = self.monitor.get_replication_group_details(vm_name)
        if self.__label['header.clientName'] not in table_data.keys():
            raise CVWebAutomationException("Get VM details for overview tab is called "
                                           "from configuration tab or the column does not exist")
        return table_data

    @PageService()
    def remove_virtual_machines(self, vm_name: str):
        """Removes the virtual machines from the replication group"""
        self.__table.select_rows([vm_name])
        self.__table.access_toolbar_menu("Delete")
        self.__panel.submit()

    # TODO: Continuous monitor refactors to not use views,
    #  since views are not supported in group page for VSA BLR


class ConfigurationTab:
    """Class for configuration tab of the group details page"""

    def __init__(self, admin_console: AdminConsole):
        self.__admin_console = admin_console
        self.__table = Table(self.__admin_console)
        self.__grid_table = CVTable(self.__admin_console)
        self.__dropdown = DropDown(self.__admin_console)
        self.__dialog = ModalDialog(self.__admin_console)
        self.__modal_panel = ModalPanel(self.__admin_console)

        self.__admin_console._load_properties(self, unique=True)
        self.__label = self.__admin_console.props[self.__class__.__name__]

    @WebAction()
    def __click_window_edit_hyperlink(self):
        """Clicks on the replication window edit hyperlink"""
        label_name = self.__label['label.replicationWindow']
        elem_xpath = (f"//div[contains(@text, '{label_name}')]"
                      f"/following-sibling::div[contains(@class, 'pageDetailColumn')]//a")
        self.__admin_console.driver.find_element_by_xpath(elem_xpath).click()

    @WebAction()
    def __get_all_intervals(self, day):
        """Returns all the interval elements for a day"""
        return (self.__admin_console.driver.
                find_elements_by_xpath("//a[contains(text(),'{}')]/../../"
                                       "td[contains(@class, 'week-time')]".format(day)))

    @PageService()
    def __interval_selection(self, interval):
        """
        Selects the intervals which are marked
        Args:
         interval (dict): the intervals at which recovery point store is marked at peak
                Must be a dict of keys as days, and values as list of date time ids(0-23)
                eg: {'Monday': [0,1,2,3], 'Tuesday': [0,1,2,3], 'Wednesday': [0,1,2,3]}
        """
        self.__admin_console.select_hyperlink("Clear")
        keys = interval.keys()
        days = ["Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday", "Sunday"]
        for day in days:
            if day in keys:
                intervals = self.__get_all_intervals(day)
                for slot in interval[day]:
                    if "selected" not in intervals[slot].get_attribute('class'):
                        intervals[slot].click()
        self.__modal_panel.submit()

    @PageService()
    def get_rpo_details(self):
        """Read RPO information from configuration tab page"""
        panel = PanelInfo(self.__admin_console, self.__label['label.rpo'])
        return panel.get_details()

    @PageService()
    def get_advanced_options_details(self):
        """Get advanced options details"""
        # label = 'Advanced options' in below line
        panel = PanelInfo(self.__admin_console, self.__label['label.advanced.options'])
        return panel.get_details()

    @PageService()
    def get_storage_details(self, storage_only=True):
        """Returns the details of the storage from the configuration tab"""
        if storage_only:
            return self.__grid_table.get_column_data(self.__label['label.storagePool'])
        row_names = self.__grid_table.get_column_data(self.__label['label.name'])
        return self.__grid_table.get_values_from_table(row_names)

    @PageService()
    def get_vm_details(self, vm_name):
        """Returns the replication VMs and their configuration from the table
            List with a dict for each row as {""}
            Use new classes to fetch the information
        """
        table_data = self.__table.get_table_data()

        row_num = [idx for idx, name in enumerate(table_data.get(self.__label['header.source']))
                   if name == vm_name]
        if not row_num:
            raise CVWebAutomationException(f"Cannot find the VM with the name {vm_name}")
        return {col: row[row_num[0]] for col, row in table_data.items()}

    @PageService()
    def edit_transport_mode(self, mode):
        """
        Change transport mode
        Args:
            mode            (str):   transport mode to be selected
        """
        panel = PanelInfo(self.__admin_console, self.__label['label.advanced.options'])
        panel.edit_tile_entity(self.__label['header.transportMode'])
        self.__dropdown.select_drop_down_values(
            values=[mode], drop_down_id='transportMode_isteven-multi-select_#5064')
        panel.confirm()

        notification = self.__admin_console.get_notification()
        if notification != self.__label['msg.updated']:
            raise CVWebAutomationException("Expected notification [%s], "
                                           "found:[%s]" %
                                           (self.__label['msg.updated'], notification))

    @PageService()
    def edit_snapshots(self, snapshots):
        """
        Edit snapshots to retain on destination vm
        Args:
            snapshots            (int):   number of snapshots
        """
        panel = PanelInfo(self.__admin_console, self.__label['label.advanced.options'])
        panel.edit_tile_entity(self.__label['label.noOfRecoveryPoints'])
        self.__admin_console.fill_form_by_id('nMaxSnapshotsPerDRVM', snapshots)
        panel.confirm()

        notification = self.__admin_console.get_notification()
        if notification != self.__label['msg.updated']:
            raise CVWebAutomationException("Expected notification [%s], "
                                           "found:[%s]" %
                                           (self.__label['msg.updated'], notification))

    @PageService()
    def edit_replication_frequency(self, frequency):
        """
        Edit replication frequency
        Args:
            frequency(int): frequency to set
        """
        panel = PanelInfo(self.__admin_console, self.__label['label.rpo'])
        panel.edit_tile_entity(self.__label['label.replicationFrequency'])
        self.__admin_console.fill_form_by_id('rpo', str(frequency))
        panel.confirm()

        notification = self.__admin_console.get_notification()
        if notification != self.__label['msg.updated']:
            raise CVWebAutomationException("Expected notification [%s], "
                                           "found:[%s]" %
                                           (self.__label['msg.updated'], notification))

    @PageService()
    def edit_replication_window(self, interval):
        """Edits the replication window and assign it values
        Args:
         interval (dict): the intervals at which recovery point store is marked at peak
                Must be a dict of keys as days, and values as list of date time ids(0-23)
                eg: {'Monday': [0,1,2,3], 'Tuesday': [0,1,2,3], 'Wednesday': [0,1,2,3]}
        """
        self.__click_window_edit_hyperlink()
        self.__admin_console.wait_for_completion()
        self.__interval_selection(interval)

    @PageService()
    def toggle_validate_destination_vm(self, enable=True):
        """Toggles the validate destination VM to flag"""
        panel = PanelInfo(self.__admin_console, self.__label['label.advanced.options'])
        toggle_name = self.__label['label.powerOn.replication']
        if enable:
            panel.enable_toggle(toggle_name)
        else:
            panel.disable_toggle(toggle_name)

    @PageService()
    def toggle_unconditional_overwrite(self, enable=True):
        """Toggles the validate destination VM to flag"""
        panel = PanelInfo(self.__admin_console, self.__label['label.advanced.options'])
        toggle_name = self.__label['warning.overwriteVM']
        if enable:
            panel.enable_toggle(toggle_name)
        else:
            panel.disable_toggle(toggle_name)

    @PageService()
    def add_virtual_machines(self, vm_type):
        """Clicks on the add virtual machines and returns the particular AddVMMachineClass"""
        add_vm = _AddVirtualMachine(self.__admin_console, vm_type)
        self.__table.access_toolbar_menu("ADD_VMS")
        return add_vm

    @PageService()
    def edit_virtual_machines(self, source_vm: str, vm_type: str = SOURCE_HYPERVISOR_VMWARE):
        """
        Edit virtual machine
        Args:
            source_vm         (str):   source virtual machine
            vm_type           (str):   Source hypervisor name as a constant from
                                       Virtualization_replication.py

        Returns       (object): object of VirtualMachine
        """
        edit_vm = None
        if vm_type == SOURCE_HYPERVISOR_VMWARE:
            edit_vm = EditVMwareVirtualMachine(self.__admin_console)
        if vm_type == SOURCE_HYPERVISOR_AZURE:
            edit_vm = EditAzureVirtualMachine(self.__admin_console)
        if vm_type == SOURCE_HYPERVISOR_HYPERV:
            edit_vm = EditHyperVVirtualMachine(self.__admin_console)
        self.__table.select_rows([source_vm])
        self.__table.access_toolbar_menu('OVERRIDE_REPLICATION')
        self.__admin_console.wait_for_completion()
        return edit_vm

    @PageService()
    def remove_virtual_machines(self, vm_name: str):
        """Removes the virtual machines from the replication group"""
        self.__table.select_rows([vm_name])
        self.__table.access_toolbar_menu("DELETE")
        self.__dialog.click_submit()


class ReplicationDetails:
    """Class for virtualiztion Replication Group details Page"""

    def __init__(self, admin_console):
        self.__admin_console = admin_console
        self.overview = OverviewTab(self.__admin_console)
        self.configuration = ConfigurationTab(self.__admin_console)

        self.__admin_console._load_properties(self.__admin_console.navigator)
        self.__admin_console._load_properties(self, unique=True)
        self.__label = self.__admin_console.props[self.__class__.__name__]

    @WebAction()
    def __edit_replication_group_name(self, group_name):
        """
        Edit the replication group name
        Args:
            group_name(string): New name of the replication group
        """
        element = self.__admin_console.driver.find_element_by_xpath(
            "//h1[contains(@id,'changeNameTitle')]")
        element.send_keys(Keys.CONTROL, 'a', Keys.DELETE)
        element.send_keys(group_name, '\n')

    @PageService()
    def edit_replication_group_name(self, group_name):
        """
        Edit the replication group name
        Args:
            group_name(string): New name of the replication group
        """
        self.__edit_replication_group_name(group_name)
        self.__admin_console.wait_for_completion()

    @WebAction()
    def get_replication_group_name(self):
        """Return replication name"""
        return self.__admin_console.driver.find_element_by_xpath(
            "//h1[contains(@id,'changeNameTitle')]").text

    @WebAction()
    def replicate_now(self):
        """
        click on replicate now
        """
        self.__admin_console.driver.find_element_by_id('actionDetail_REPLICATE_').click()

    @WebAction()
    def __visible_option(self, option):
        """Checks to see whether the option is visible or not"""
        return (self.__admin_console.driver.
                find_element_by_xpath(f"//a[contains(text(), '{option}')]").is_displayed())

    @WebAction()
    def __click_more_options(self):
        """Click on more options in page"""
        self.__admin_console.driver.find_element_by_xpath("//*[@class='uib-dropdown "
                                                          "dropdown']").click()

    @PageService()
    def __access_more_option_menu_item(self, option):
        """
        click on more options and select specified item
        Args:
            option (string) : option label to select replication details page(enable/disable)
        """
        if not self.__visible_option(option):
            self.__click_more_options()
            sleep(1)
        self.__admin_console.select_hyperlink(option)

    @WebAction()
    def __click_by_id(self, ide):
        """click by id"""
        self.__admin_console.driver.find_element_by_id(ide).click()

    @PageService()
    def disable_replication_group(self):
        """
        Disables replication group
        """
        id_disable = "actionDetail_DISABLE_"
        self.__click_by_id(id_disable)
        self.__admin_console.wait_for_completion()

    @PageService()
    def enable_replication_group(self):
        """
        Enables replication group
        """
        id_enable = "actionDetail_ENABLE_"
        self.__click_by_id(id_enable)
        self.__admin_console.wait_for_completion()

    @PageService()
    def access_overview_tab(self):
        """Access overview tab"""
        self.__admin_console.select_overview_tab()

    @PageService()
    def access_configuration_tab(self):
        """Access configuration tab"""
        self.__admin_console.select_configuration_tab()


class EditVMwareVirtualMachine(_VMwareVMOptions):
    """Edit virtual machine parameters using this class"""

    def __init__(self, admin_console):
        _VMwareVMOptions.__init__(self, admin_console)

    @WebAction()
    def __is_general_settings_editable(self):
        """
        If parameters are editable then return true otherwise false
        Returns  (Boolean): True if the General settings is editable otherwise false
        """
        xp = "//div[contains(text(), 'General settings')]/..//div[contains(@data-ng-disabled, " \
             "'repTarget')]"
        parameters = self._admin_console.driver.find_elements_by_xpath(xp)
        # status below will be listof 'true' if its disabled
        status = [each_param.get_attribute('disabled') for each_param in parameters]
        if [each_status for each_status in status if each_status != 'true']:
            # if disabled is not 'true' then its editable, so returning True
            return True
        return False

    @PageService()
    def is_general_settings_editable(self):
        """
        Returns  (Boolean): True if the General settings is editable otherwise false
        """
        return self.__is_general_settings_editable()

    @PageService()
    def edit_network_settings(self, source_network, destination_network):
        """
        Edit network setting
        Args:
            source_network                (str): specify the source network
            destination_network           (str): specify the destination network
        """
        network_settings = self.edit_network()
        network_settings.select_source_network(source_network)
        network_settings.select_destination_network(destination_network)
        network_settings.save()

    @WebAction()
    def __get_network_settings(self):
        """
        Read network settings information
        Returns          (list): table content
        """
        xp = "//*[@accordion-label='label.networkSettingsVMWare']//span[@title]"
        return [each_element.text for each_element in
                self._admin_console.driver.find_elements_by_xpath(xp)]

    @PageService()
    def get_network_settings(self):
        """
        Read network settings
        Returns(list): list of column data
        """
        self.expand_tab(self._label['label.networkSettingsVMWare'])
        return self.__get_network_settings()

    @PageService()
    def set_vm_display_name(self, name):
        """Not implemented for editing since field is disabled"""
        raise NotImplementedError("Field VM display name is disabled")

    @PageService()
    def set_destination_host(self, host):
        """Not implemented for editing since field is disabled"""
        raise NotImplementedError("Field Destination host is disabled")

    @PageService()
    def select_datastore(self, datastore):
        """Not implemented for editing since field is disabled"""
        raise NotImplementedError("Field datastore is disabled")

    @PageService()
    def select_resource_pool(self, resource_pool_name):
        """Not implemented for editing since field is disabled"""
        raise NotImplementedError("Field resource pool is disabled")

    @PageService()
    def set_vm_folder(self, name):
        """Not implemented for editing since field is disabled"""
        raise NotImplementedError("Field Vm folder is disabled")


class EditAzureVirtualMachine(_AzureVMOptions):
    """Edit the Azure virtual machine options using this class"""

    def __init__(self, admin_console: AdminConsole):
        """Initialises the components required for functions"""
        _AzureVMOptions.__init__(self, admin_console)
        self.__driver = self._admin_console.driver
        self.__modal_panel = ModalPanel(admin_console)

    @WebAction()
    def __get_selected_dropdown_option(self, select_id: str) -> str or None:
        """Gets the current value from the select"""
        options = self.__driver.find_elements_by_xpath(f"//*[@id='{select_id}']//option")
        for option in options:
            if option.get_attribute("selected") is not None:
                return option.text
        return None

    @WebAction()
    def __get_checkbox_value(self, checkbox_id: str) -> bool:
        """Gets the value of the given checkbox"""
        return (self.__driver.
                find_element_by_xpath(f"//input[@type='checkbox' and @id='{checkbox_id}']").
                is_selected())

    @property
    @PageService()
    def vm_size(self) -> str:
        """Returns the selected VM size from the panel"""
        return self.__get_selected_dropdown_option('azureVmSize')

    @property
    @PageService()
    def virtual_network(self) -> str:
        """Returns the current value of virtual network"""
        return self.get_selected_multi_select(
            'selectVirtualNetworkAzure_isteven-multi-select_#2554')

    @property
    @PageService()
    def security_group(self) -> str:
        """Returns the current security group"""
        return self.get_selected_multi_select('azureSecurityGroup_isteven-multi-select')

    @property
    @PageService()
    def public_ip(self) -> bool:
        """Returns the value of the checkbox create public IP"""
        return self.__get_checkbox_value('createPublicIp')

    @property
    @PageService()
    def restore_managed_vm(self) -> bool:
        """Returns the value of the checkbox restore as a managed VM"""
        return self.__get_checkbox_value('restoreAsManagedVM')

    @PageService()
    def select_virtual_network(self, virtual_network: str):
        """Selects the virtual network from the options"""
        _AzureVMOptions.virtual_network(self, virtual_network)

    @PageService()
    def select_public_ip(self, enabled: bool):
        """Checks/unchecks the box for create public IP option"""
        _AzureVMOptions.create_public_ip(self, enabled)


class EditHyperVVirtualMachine(_HyperVOptions):
    """Edit virtual machine parameters using this class"""

    def __init__(self, admin_console):
        _HyperVOptions.__init__(self, admin_console)
        self.__driver = self._admin_console.driver
        self.__modal_panel = ModalPanel(admin_console)

    @PageService()
    def set_vm_display_name(self, name):
        """Not implemented for editing since field is disabled"""
        raise NotImplementedError("Vm display name is disabled")

    @PageService()
    def select_network(self, network):
        """
        Select network
        Args:
            network(str): network name
        """
        drop_down_id = 'networkName'
        (self._drop_down.select_drop_down_values(values=[network],
                                                 drop_down_id=drop_down_id, partial_selection=True))

    @property
    @PageService()
    def network(self) -> str:
        """Returns the current value of network"""
        return self.get_selected_multi_select('networkName')


class _AddVirtualMachine:
    """Class to introduce common functionality for adding cross-platform VMs"""

    def __init__(self, admin_console: AdminConsole, vm_type: str = SOURCE_HYPERVISOR_VMWARE):
        """Add content options from virtualization_replication"""
        self._admin_console = admin_console
        self._modal_panel = ModalPanel(admin_console)
        self._dialog = ModalDialog(admin_console)
        self.vm_type = vm_type
        self.override_options = _OverrideOptions(self._admin_console, vm_type)

        # Added for future purpose of removing select_vm_from_browse_tree
        self.content = _Content(admin_console)

    def select_vm_from_browse_tree(self, virtual_machines: list):
        """Selects the given VMs from the browse tree"""
        for vm_name in virtual_machines:
            self._admin_console.search_vm(vm_name)

    def add_vm(self, virtual_machines: list):
        """Add the VM with default options
        Args:
            virtual_machines (list): The virtual machines to add in the form
            {'Datastores and datastore clusters':[ds1,ds2],
                                 'Hosts and clusters':[h1,h2],
                                 'Tags and categories':[tag1,category1],
                                 'VMs and templates':[vm1,vm2],
                                 'Storage':[strg1, strg2],
                                 'By region': [region1, region2],
                                 'By zone': [zone1, zone2]
            }
        """
        self.select_vm_from_browse_tree(virtual_machines)
        self._modal_panel.submit()
        # TODO :  Will Add override Support
        # override = self.override_options.override_vms(virtual_machines[0])
        # override.save()
        self._modal_panel.submit()
